﻿/*! DataTables 2.0.8
 * © SpryMedia Ltd - datatables.net/license
 */

/**
 * @summary     DataTables
 * @description Paginate, search and order HTML tables
 * @version     2.0.8
 * @author      SpryMedia Ltd
 * @contact     www.datatables.net
 * @copyright   SpryMedia Ltd.
 *
 * This source file is free software, available under the following license:
 *   MIT license - https://datatables.net/license
 *
 * This source file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
 *
 * For details please refer to: https://www.datatables.net
 */

(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		// jQuery's factory checks for a global window - if it isn't present then it
		// returns a factory function that expects the window object
		var jq = require('jquery');

		if (typeof window === 'undefined') {
			module.exports = function (root, $) {
				if ( ! root ) {
					// CommonJS environments without a window global must pass a
					// root. This will give an error otherwise
					root = window;
				}

				if ( ! $ ) {
					$ = jq( root );
				}

				return factory( $, root, root.document );
			};
		}
		else {
			module.exports = factory( jq, window, window.document );
		}
	}
	else {
		// Browser
		window.DataTable = factory( jQuery, window, document );
	}
}(function( $, window, document ) {
	"use strict";

	
	var DataTable = function ( selector, options )
	{
		// Check if called with a window or jQuery object for DOM less applications
		// This is for backwards compatibility
		if (DataTable.factory(selector, options)) {
			return DataTable;
		}
	
		// When creating with `new`, create a new DataTable, returning the API instance
		if (this instanceof DataTable) {
			return $(selector).DataTable(options);
		}
		else {
			// Argument switching
			options = selector;
		}
	
		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;
	
		if ( emptyInit ) {
			options = {};
		}
	
		// Method to get DT API instance from jQuery object
		this.api = function ()
		{
			return new _Api( this );
		};
	
		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;
	
			
			var i=0, iLen;
			var sId = this.getAttribute( 'id' );
			var bInitHandedOff = false;
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			
			/* Sanity check */
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			$(this).trigger( 'options.dt', oInit );
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, $this.data() ), true );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						new DataTable.Api(s).destroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				/* If the element we are initialising has the same ID as a table which was previously
				 * initialised, but the table nodes don't match (from before) then we destroy the old
				 * instance by simply deleting it. This is under the assumption that the table has been
				 * destroyed by other methods. Anyone using non-id selectors will need to do this manually
				 */
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId,
				colgroup: $('<colgroup>').prependTo(this),
				fastData: function (row, column, type) {
					return _fnGetCellData(oSettings, row, column, type);
				}
			} );
			oSettings.nTable = this;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Make a single API instance available for internal handling
			oSettings.api = new _Api( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = Array.isArray(oInit.aLengthMenu[0])
					? oInit.aLengthMenu[0][0]
					: $.isPlainObject( oInit.aLengthMenu[0] )
						? oInit.aLengthMenu[0].value
						: oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"ajax",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"iStateDuration",
				"bSortCellsTop",
				"iTabIndex",
				"sDom",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				"caption",
				"layout",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.table );
			
			if (! oSettings.oFeatures.bPaginate) {
				oInit.iDisplayStart = 0;
			}
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			/* Language definitions */
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl )
			{
				/* Get the language definitions from a file - because this Ajax call makes the language
				 * get async to the remainder of this function we use bInitHandedOff to indicate that
				 * _fnInitialise will be fired by the returned Ajax handler, rather than the constructor
				 */
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnCamelToHungarian( defaults.oLanguage, json );
						$.extend( true, oLanguage, json, oSettings.oInit.oLanguage );
			
						_fnCallbackFire( oSettings, null, 'i18n', [oSettings], true);
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file
						_fnLog( oSettings, 0, 'i18n file loading error', 21 );
			
						// continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
				bInitHandedOff = true;
			}
			else {
				_fnCallbackFire( oSettings, null, 'i18n', [oSettings]);
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var columnsInit = [];
			var thead = this.getElementsByTagName('thead');
			var initHeaderLayout = _fnDetectHeader( oSettings, thead[0] );
			
			// If we don't have a columns array, then generate one with nulls
			if ( oInit.aoColumns ) {
				columnsInit = oInit.aoColumns;
			}
			else if ( initHeaderLayout.length ) {
				for ( i=0, iLen=initHeaderLayout[0].length ; i<iLen ; i++ ) {
					columnsInit.push( null );
				}
			}
			
			// Add the columns
			for ( i=0, iLen=columnsInit.length ; i<iLen ; i++ ) {
				_fnAddColumn( oSettings );
			}
			
			// Apply the column definitions
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, columnsInit, initHeaderLayout, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			var rowOne = $this.children('tbody').find('tr').eq(0);
			
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if (! col) {
						_fnLog( oSettings, 0, 'Incorrect column count', 18 );
					}
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
							col._isArrayHost = true;
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			var features = oSettings.oFeatures;
			var loadedInit = function () {
				/*
				 * Sorting
				 * @todo For modularisation (1.11) this needs to do into a sort start up handler
				 */
			
				// If aaSorting is not defined, then we use the first indicator in asSorting
				// in case that has been altered, so the default sort reflects that option
				if ( oInit.aaSorting === undefined ) {
					var sorting = oSettings.aaSorting;
					for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
						sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
					}
				}
			
				/* Do a first pass on the sorting classes (allows any size changes to be taken into
				 * account, and also will apply sorting disabled classes if disabled
				 */
				_fnSortingClasses( oSettings );
			
				_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
					if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
						_fnSortingClasses( oSettings );
					}
				} );
			
			
				/*
				 * Final init
				 * Cache the header, body and footer as required, creating them if needed
				 */
				var caption = $this.children('caption');
			
				if ( oSettings.caption ) {
					if ( caption.length === 0 ) {
						caption = $('<caption/>').appendTo( $this );
					}
			
					caption.html( oSettings.caption );
				}
			
				// Store the caption side, so we can remove the element from the document
				// when creating the element
				if (caption.length) {
					caption[0]._captionSide = caption.css('caption-side');
					oSettings.captionNode = caption[0];
				}
			
				if ( thead.length === 0 ) {
					thead = $('<thead/>').appendTo($this);
				}
				oSettings.nTHead = thead[0];
				$('tr', thead).addClass(oClasses.thead.row);
			
				var tbody = $this.children('tbody');
				if ( tbody.length === 0 ) {
					tbody = $('<tbody/>').insertAfter(thead);
				}
				oSettings.nTBody = tbody[0];
			
				var tfoot = $this.children('tfoot');
				if ( tfoot.length === 0 ) {
					// If we are a scrolling table, and no footer has been given, then we need to create
					// a tfoot element for the caption element to be appended to
					tfoot = $('<tfoot/>').appendTo($this);
				}
				oSettings.nTFoot = tfoot[0];
				$('tr', tfoot).addClass(oClasses.tfoot.row);
			
				// Check if there is data passing into the constructor
				if ( oInit.aaData ) {
					for ( i=0 ; i<oInit.aaData.length ; i++ ) {
						_fnAddData( oSettings, oInit.aaData[ i ] );
					}
				}
				else if ( _fnDataSource( oSettings ) == 'dom' ) {
					// Grab the data from the page
					_fnAddTr( oSettings, $(oSettings.nTBody).children('tr') );
				}
			
				/* Copy the data index array */
				oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
				/* Initialisation complete - table can be drawn */
				oSettings.bInitialised = true;
			
				/* Check if we need to initialise the table (it might not have been handed off to the
				 * language processor)
				 */
				if ( bInitHandedOff === false ) {
					_fnInitialise( oSettings );
				}
			};
			
			/* Must be done after everything which can be overridden by the state saving! */
			_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState );
			
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
				_fnLoadState( oSettings, oInit, loadedInit );
			}
			else {
				loadedInit();
			}
			
		} );
		_that = null;
		return this;
	};
	
	
	
	/**
	 * DataTables extensions
	 * 
	 * This namespace acts as a collection area for plug-ins that can be used to
	 * extend DataTables capabilities. Indeed many of the build in methods
	 * use this method to provide their own capabilities (sorting methods for
	 * example).
	 *
	 * Note that this namespace is aliased to `jQuery.fn.dataTableExt` for legacy
	 * reasons
	 *
	 *  @namespace
	 */
	DataTable.ext = _ext = {
		/**
		 * Buttons. For use with the Buttons extension for DataTables. This is
		 * defined here so other extensions can define buttons regardless of load
		 * order. It is _not_ used by DataTables core.
		 *
		 *  @type object
		 *  @default {}
		 */
		buttons: {},
	
	
		/**
		 * Element class names
		 *
		 *  @type object
		 *  @default {}
		 */
		classes: {},
	
	
		/**
		 * DataTables build type (expanded by the download builder)
		 *
		 *  @type string
		 */
		builder: "-source-",
	
	
		/**
		 * Error reporting.
		 * 
		 * How should DataTables report an error. Can take the value 'alert',
		 * 'throw', 'none' or a function.
		 *
		 *  @type string|function
		 *  @default alert
		 */
		errMode: "alert",
	
	
		/**
		 * Legacy so v1 plug-ins don't throw js errors on load
		 */
		feature: [],
	
		/**
		 * Feature plug-ins.
		 * 
		 * This is an object of callbacks which provide the features for DataTables
		 * to be initialised via the `layout` option.
		 */
		features: {},
	
	
		/**
		 * Row searching.
		 * 
		 * This method of searching is complimentary to the default type based
		 * searching, and a lot more comprehensive as it allows you complete control
		 * over the searching logic. Each element in this array is a function
		 * (parameters described below) that is called for every row in the table,
		 * and your logic decides if it should be included in the searching data set
		 * or not.
		 *
		 * Searching functions have the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{array|object}` Data for the row to be processed (same as the
		 *    original format that was passed in as the data source, or an array
		 *    from a DOM data source
		 * 3. `{int}` Row index ({@link DataTable.models.oSettings.aoData}), which
		 *    can be useful to retrieve the `TR` element if you need DOM interaction.
		 *
		 * And the following return is expected:
		 *
		 * * {boolean} Include the row in the searched result set (true) or not
		 *   (false)
		 *
		 * Note that as with the main search ability in DataTables, technically this
		 * is "filtering", since it is subtractive. However, for consistency in
		 * naming we call it searching here.
		 *
		 *  @type array
		 *  @default []
		 *
		 *  @example
		 *    // The following example shows custom search being applied to the
		 *    // fourth column (i.e. the data[3] index) based on two input values
		 *    // from the end-user, matching the data in a certain range.
		 *    $.fn.dataTable.ext.search.push(
		 *      function( settings, data, dataIndex ) {
		 *        var min = document.getElementById('min').value * 1;
		 *        var max = document.getElementById('max').value * 1;
		 *        var version = data[3] == "-" ? 0 : data[3]*1;
		 *
		 *        if ( min == "" && max == "" ) {
		 *          return true;
		 *        }
		 *        else if ( min == "" && version < max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && "" == max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && version < max ) {
		 *          return true;
		 *        }
		 *        return false;
		 *      }
		 *    );
		 */
		search: [],
	
	
		/**
		 * Selector extensions
		 *
		 * The `selector` option can be used to extend the options available for the
		 * selector modifier options (`selector-modifier` object data type) that
		 * each of the three built in selector types offer (row, column and cell +
		 * their plural counterparts). For example the Select extension uses this
		 * mechanism to provide an option to select only rows, columns and cells
		 * that have been marked as selected by the end user (`{selected: true}`),
		 * which can be used in conjunction with the existing built in selector
		 * options.
		 *
		 * Each property is an array to which functions can be pushed. The functions
		 * take three attributes:
		 *
		 * * Settings object for the host table
		 * * Options object (`selector-modifier` object type)
		 * * Array of selected item indexes
		 *
		 * The return is an array of the resulting item indexes after the custom
		 * selector has been applied.
		 *
		 *  @type object
		 */
		selector: {
			cell: [],
			column: [],
			row: []
		},
	
	
		/**
		 * Legacy configuration options. Enable and disable legacy options that
		 * are available in DataTables.
		 *
		 *  @type object
		 */
		legacy: {
			/**
			 * Enable / disable DataTables 1.9 compatible server-side processing
			 * requests
			 *
			 *  @type boolean
			 *  @default null
			 */
			ajax: null
		},
	
	
		/**
		 * Pagination plug-in methods.
		 * 
		 * Each entry in this object is a function and defines which buttons should
		 * be shown by the pagination rendering method that is used for the table:
		 * {@link DataTable.ext.renderer.pageButton}. The renderer addresses how the
		 * buttons are displayed in the document, while the functions here tell it
		 * what buttons to display. This is done by returning an array of button
		 * descriptions (what each button will do).
		 *
		 * Pagination types (the four built in options and any additional plug-in
		 * options defined here) can be used through the `paginationType`
		 * initialisation parameter.
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{int} page` The current page index
		 * 2. `{int} pages` The number of pages in the table
		 *
		 * Each function is expected to return an array where each element of the
		 * array can be one of:
		 *
		 * * `first` - Jump to first page when activated
		 * * `last` - Jump to last page when activated
		 * * `previous` - Show previous page when activated
		 * * `next` - Show next page when activated
		 * * `{int}` - Show page of the index given
		 * * `{array}` - A nested array containing the above elements to add a
		 *   containing 'DIV' element (might be useful for styling).
		 *
		 * Note that DataTables v1.9- used this object slightly differently whereby
		 * an object with two functions would be defined for each plug-in. That
		 * ability is still supported by DataTables 1.10+ to provide backwards
		 * compatibility, but this option of use is now decremented and no longer
		 * documented in DataTables 1.10+.
		 *
		 *  @type object
		 *  @default {}
		 *
		 *  @example
		 *    // Show previous, next and current page buttons only
		 *    $.fn.dataTableExt.oPagination.current = function ( page, pages ) {
		 *      return [ 'previous', page, 'next' ];
		 *    };
		 */
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
	
		/**
		 * Ordering plug-ins - custom data source
		 * 
		 * The extension options for ordering of data available here is complimentary
		 * to the default type based ordering that DataTables typically uses. It
		 * allows much greater control over the the data that is being used to
		 * order a column, but is necessarily therefore more complex.
		 * 
		 * This type of ordering is useful if you want to do ordering based on data
		 * live from the DOM (for example the contents of an 'input' element) rather
		 * than just the static string that DataTables knows of.
		 * 
		 * The way these plug-ins work is that you create an array of the values you
		 * wish to be ordering for the column in question and then return that
		 * array. The data in the array much be in the index order of the rows in
		 * the table (not the currently ordering order!). Which order data gathering
		 * function is run here depends on the `dt-init columns.orderDataType`
		 * parameter that is used for the column (if any).
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{int}` Target column index
		 *
		 * Each function is expected to return an array:
		 *
		 * * `{array}` Data for the column to be ordering upon
		 *
		 *  @type array
		 *
		 *  @example
		 *    // Ordering using `input` node values
		 *    $.fn.dataTable.ext.order['dom-text'] = function  ( settings, col )
		 *    {
		 *      return this.api().column( col, {order:'index'} ).nodes().map( function ( td, i ) {
		 *        return $('input', td).val();
		 *      } );
		 *    }
		 */
		order: {},
	
	
		/**
		 * Type based plug-ins.
		 *
		 * Each column in DataTables has a type assigned to it, either by automatic
		 * detection or by direct assignment using the `type` option for the column.
		 * The type of a column will effect how it is ordering and search (plug-ins
		 * can also make use of the column type if required).
		 *
		 * @namespace
		 */
		type: {
			/**
			 * Automatic column class assignment
			 */
			className: {},
	
			/**
			 * Type detection functions.
			 *
			 * The functions defined in this object are used to automatically detect
			 * a column's type, making initialisation of DataTables super easy, even
			 * when complex data is in the table.
			 *
			 * The functions defined take two parameters:
			 *
		     *  1. `{*}` Data from the column cell to be analysed
		     *  2. `{settings}` DataTables settings object. This can be used to
		     *     perform context specific type detection - for example detection
		     *     based on language settings such as using a comma for a decimal
		     *     place. Generally speaking the options from the settings will not
		     *     be required
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Data type detected, or null if unknown (and thus
			 *   pass it on to the other type detection functions.
			 *
			 *  @type array
			 *
			 *  @example
			 *    // Currency type detection plug-in:
			 *    $.fn.dataTable.ext.type.detect.push(
			 *      function ( data, settings ) {
			 *        // Check the numeric part
			 *        if ( ! data.substring(1).match(/[0-9]/) ) {
			 *          return null;
			 *        }
			 *
			 *        // Check prefixed by currency
			 *        if ( data.charAt(0) == '$' || data.charAt(0) == '&pound;' ) {
			 *          return 'currency';
			 *        }
			 *        return null;
			 *      }
			 *    );
			 */
			detect: [],
	
			/**
			 * Automatic renderer assignment
			 */
			render: {},
	
	
			/**
			 * Type based search formatting.
			 *
			 * The type based searching functions can be used to pre-format the
			 * data to be search on. For example, it can be used to strip HTML
			 * tags or to de-format telephone numbers for numeric only searching.
			 *
			 * Note that is a search is not defined for a column of a given type,
			 * no search formatting will be performed.
			 * 
			 * Pre-processing of searching data plug-ins - When you assign the sType
			 * for a column (or have it automatically detected for you by DataTables
			 * or a type detection plug-in), you will typically be using this for
			 * custom sorting, but it can also be used to provide custom searching
			 * by allowing you to pre-processing the data and returning the data in
			 * the format that should be searched upon. This is done by adding
			 * functions this object with a parameter name which matches the sType
			 * for that target column. This is the corollary of <i>afnSortData</i>
			 * for searching data.
			 *
			 * The functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for searching
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Formatted string that will be used for the searching.
			 *
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    $.fn.dataTable.ext.type.search['title-numeric'] = function ( d ) {
			 *      return d.replace(/\n/g," ").replace( /<.*?>/g, "" );
			 *    }
			 */
			search: {},
	
	
			/**
			 * Type based ordering.
			 *
			 * The column type tells DataTables what ordering to apply to the table
			 * when a column is sorted upon. The order for each type that is defined,
			 * is defined by the functions available in this object.
			 *
			 * Each ordering option can be described by three properties added to
			 * this object:
			 *
			 * * `{type}-pre` - Pre-formatting function
			 * * `{type}-asc` - Ascending order function
			 * * `{type}-desc` - Descending order function
			 *
			 * All three can be used together, only `{type}-pre` or only
			 * `{type}-asc` and `{type}-desc` together. It is generally recommended
			 * that only `{type}-pre` is used, as this provides the optimal
			 * implementation in terms of speed, although the others are provided
			 * for compatibility with existing Javascript sort functions.
			 *
			 * `{type}-pre`: Functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for ordering
			 *
			 * And return:
			 *
			 * * `{*}` Data to be sorted upon
			 *
			 * `{type}-asc` and `{type}-desc`: Functions are typical Javascript sort
			 * functions, taking two parameters:
			 *
		     *  1. `{*}` Data to compare to the second parameter
		     *  2. `{*}` Data to compare to the first parameter
			 *
			 * And returning:
			 *
			 * * `{*}` Ordering match: <0 if first parameter should be sorted lower
			 *   than the second parameter, ===0 if the two parameters are equal and
			 *   >0 if the first parameter should be sorted height than the second
			 *   parameter.
			 * 
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    // Numeric ordering of formatted numbers with a pre-formatter
			 *    $.extend( $.fn.dataTable.ext.type.order, {
			 *      "string-pre": function(x) {
			 *        a = (a === "-" || a === "") ? 0 : a.replace( /[^\d\-\.]/g, "" );
			 *        return parseFloat( a );
			 *      }
			 *    } );
			 *
			 *  @example
			 *    // Case-sensitive string ordering, with no pre-formatting method
			 *    $.extend( $.fn.dataTable.ext.order, {
			 *      "string-case-asc": function(x,y) {
			 *        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
			 *      },
			 *      "string-case-desc": function(x,y) {
			 *        return ((x < y) ? 1 : ((x > y) ? -1 : 0));
			 *      }
			 *    } );
			 */
			order: {}
		},
	
		/**
		 * Unique DataTables instance counter
		 *
		 * @type int
		 * @private
		 */
		_unique: 0,
	
	
		//
		// Depreciated
		// The following properties are retained for backwards compatibility only.
		// The should not be used in new projects and will be removed in a future
		// version
		//
	
		/**
		 * Version check function.
		 *  @type function
		 *  @depreciated Since 1.10
		 */
		fnVersionCheck: DataTable.fnVersionCheck,
	
	
		/**
		 * Index for what 'this' index API functions should use
		 *  @type int
		 *  @deprecated Since v1.10
		 */
		iApiIndex: 0,
	
	
		/**
		 * Software version
		 *  @type string
		 *  @deprecated Since v1.10
		 */
		sVersion: DataTable.version
	};
	
	
	//
	// Backwards compatibility. Alias to pre 1.10 Hungarian notation counter parts
	//
	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		container: 'dt-container',
		empty: {
			row: 'dt-empty'
		},
		info: {
			container: 'dt-info'
		},
		length: {
			container: 'dt-length',
			select: 'dt-input'
		},
		order: {
			canAsc: 'dt-orderable-asc',
			canDesc: 'dt-orderable-desc',
			isAsc: 'dt-ordering-asc',
			isDesc: 'dt-ordering-desc',
			none: 'dt-orderable-none',
			position: 'sorting_'
		},
		processing: {
			container: 'dt-processing'
		},
		scrolling: {
			body: 'dt-scroll-body',
			container: 'dt-scroll',
			footer: {
				self: 'dt-scroll-foot',
				inner: 'dt-scroll-footInner'
			},
			header: {
				self: 'dt-scroll-head',
				inner: 'dt-scroll-headInner'
			}
		},
		search: {
			container: 'dt-search',
			input: 'dt-input'
		},
		table: 'dataTable',	
		tbody: {
			cell: '',
			row: ''
		},
		thead: {
			cell: '',
			row: ''
		},
		tfoot: {
			cell: '',
			row: ''
		},
		paging: {
			active: 'current',
			button: 'dt-paging-button',
			container: 'dt-paging',
			disabled: 'disabled'
		}
	} );
	
	
	/*
	 * It is useful to have variables which are scoped locally so only the
	 * DataTables functions can access them and they don't leak into global space.
	 * At the same time these functions are often useful over multiple files in the
	 * core and API, so we list, or at least document, all variables which are used
	 * by DataTables as private variables here. This also ensures that there is no
	 * clashing of variable names and that they can easily referenced for reuse.
	 */
	
	
	// Defined else where
	//  _selector_run
	//  _selector_opts
	//  _selector_row_indexes
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n\u2028]/g;
	var _re_html = /<([^>]*>)/g;
	var _max_str_len = Math.pow(2, 28);
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[./-]\d{1,2}[./-]\d{1,2}([T ]{1}\d{1,2}[:.]\d{2}([.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	// https://en.wikipedia.org/wiki/Foreign_exchange_market
	// - \u20BD - Russian ruble.
	// - \u20a9 - South Korean Won
	// - \u20BA - Turkish Lira
	// - \u20B9 - Indian Rupee
	// - R - Brazil (R$) and South Africa
	// - fr - Swiss Franc
	// - kr - Swedish krona, Norwegian krone and Danish krone
	// - \u2009 is thin space and \u202F is narrow no-break space, both used in many
	// - Ƀ - Bitcoin
	// - Ξ - Ethereum
	//   standards as thousands separators.
	var _re_formatted_numeric = /['\u00A0,$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted ) {
		var type = typeof d;
		var strType = type === 'string';
	
		if ( type === 'number' || type === 'bigint') {
			return true;
		}
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	// Is a string a number surrounded by HTML?
	var _htmlNumeric = function ( d, decimalPoint, formatted ) {
		if ( _empty( d ) ) {
			return true;
		}
	
		// input and select strings mean that this isn't just a number
		if (typeof d === 'string' && d.match(/<(input|select)/i)) {
			return null;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ] ) {
					out.push( a[ order[i] ][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	// Replaceable function in api.util
	var _stripHtml = function (input) {
		// Irrelevant check to workaround CodeQL's false positive on the regex
		if (input.length > _max_str_len) {
			throw new Error('Exceeded max str len');
		}
	
		var previous;
	
		input = input.replace(_re_html, ''); // Complete tags
	
		// Safety for incomplete script tag - use do / while to ensure that
		// we get all instances
		do {
			previous = input;
			input = input.replace(/<script/i, '');
		} while (input !== previous);
	
		return previous;
	};
	
	// Replaceable function in api.util
	var _escapeHtml = function ( d ) {
		if (Array.isArray(d)) {
			d = d.join(',');
		}
	
		return typeof d === 'string' ?
			d
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;')
				.replace(/"/g, '&quot;') :
			d;
	};
	
	// Remove diacritics from a string by decomposing it and then removing
	// non-ascii characters
	var _normalize = function (str, both) {
		if (typeof str !== 'string') {
			return str;
		}
	
		// It is faster to just run `normalize` than it is to check if
		// we need to with a regex!
		var res = str.normalize("NFD");
	
		// Equally, here we check if a regex is needed or not
		return res.length !== str.length
			? (both === true ? str + ' ' : '' ) + res.replace(/[\u0300-\u036f]/g, "")
			: res;
	}
	
	/**
	 * Determine if all values in the array are unique. This means we can short
	 * cut the _unique method at the cost of a single loop. A sorted array is used
	 * to easily check the values.
	 *
	 * @param  {array} src Source array
	 * @return {boolean} true if all unique, false otherwise
	 * @ignore
	 */
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if (Array.from && Set) {
			return Array.from(new Set(src));
		}
	
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.app/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	// Surprisingly this is faster than [].concat.apply
	// https://jsperf.com/flatten-an-array-loop-vs-reduce/2
	var _flatten = function (out, val) {
		if (Array.isArray(val)) {
			for (var i=0 ; i<val.length ; i++) {
				_flatten(out, val[i]);
			}
		}
		else {
			out.push(val);
		}
	
		return out;
	}
	
	// Similar to jQuery's addClass, but use classList.add
	function _addClass(el, name) {
		if (name) {
			name.split(' ').forEach(function (n) {
				if (n) {
					// `add` does deduplication, so no need to check `contains`
					el.classList.add(n);
				}
			});
		}
	}
	
	/**
	 * DataTables utility methods
	 * 
	 * This namespace provides helper methods that DataTables uses internally to
	 * create a DataTable, but which are not exclusively used only for DataTables.
	 * These methods can be used by extension authors to save the duplication of
	 * code.
	 *
	 *  @namespace
	 */
	DataTable.util = {
		/**
		 * Return a string with diacritic characters decomposed
		 * @param {*} mixed Function or string to normalize
		 * @param {*} both Return original string and the normalized string
		 * @returns String or undefined
		 */
		diacritics: function (mixed, both) {
			var type = typeof mixed;
	
			if (type !== 'function') {
				return _normalize(mixed, both);
			}
			_normalize = mixed;
		},
	
		/**
		 * Debounce a function
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		debounce: function ( fn, timeout ) {
			var timer;
	
			return function () {
				var that = this;
				var args = arguments;
	
				clearTimeout(timer);
	
				timer = setTimeout( function () {
					fn.apply(that, args);
				}, timeout || 250 );
			};
		},
	
		/**
		 * Throttle the calls to a function. Arguments and context are maintained
		 * for the throttled function.
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
		/**
		 * Escape a string such that it can be used in a regular expression
		 *
		 *  @param {string} val string to escape
		 *  @returns {string} escaped string
		 */
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		},
	
		/**
		 * Create a function that will write to a nested object or array
		 * @param {*} source JSON notation string
		 * @returns Write function
		 */
		set: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				/* Unlike get, only the underscore (global) option is used for for
				 * setting data since we don't know the type here. This is why an object
				 * option is not documented for `mData` (which is read/write), but it is
				 * for `mRender` which is read only.
				 */
				return DataTable.util.set( source._ );
			}
			else if ( source === null ) {
				// Nothing to do when the data source is null
				return function () {};
			}
			else if ( typeof source === 'function' ) {
				return function (data, val, meta) {
					source( data, 'set', val, meta );
				};
			}
			else if (
				typeof source === 'string' && (source.indexOf('.') !== -1 ||
				source.indexOf('[') !== -1 || source.indexOf('(') !== -1)
			) {
				// Like the get, we need to get data from a nested object
				var setData = function (data, val, src) {
					var a = _fnSplitObjNotation( src ), b;
					var aLast = a[a.length-1];
					var arrayNotation, funcNotation, o, innerSrc;
		
					for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ ) {
						// Protect against prototype pollution
						if (a[i] === '__proto__' || a[i] === 'constructor') {
							throw new Error('Cannot set prototype values');
						}
		
						// Check if we are dealing with an array notation request
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
		
						if ( arrayNotation ) {
							a[i] = a[i].replace(__reArray, '');
							data[ a[i] ] = [];
		
							// Get the remainder of the nested object to set so we can recurse
							b = a.slice();
							b.splice( 0, i+1 );
							innerSrc = b.join('.');
		
							// Traverse each entry in the array setting the properties requested
							if ( Array.isArray( val ) ) {
								for ( var j=0, jLen=val.length ; j<jLen ; j++ ) {
									o = {};
									setData( o, val[j], innerSrc );
									data[ a[i] ].push( o );
								}
							}
							else {
								// We've been asked to save data to an array, but it
								// isn't array data to be saved. Best that can be done
								// is to just save the value.
								data[ a[i] ] = val;
							}
		
							// The inner call to setData has already traversed through the remainder
							// of the source and has set the data, thus we can exit here
							return;
						}
						else if ( funcNotation ) {
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]( val );
						}
		
						// If the nested object doesn't currently exist - since we are
						// trying to set the value - create it
						if ( data[ a[i] ] === null || data[ a[i] ] === undefined ) {
							data[ a[i] ] = {};
						}
						data = data[ a[i] ];
					}
		
					// Last item in the input - i.e, the actual set
					if ( aLast.match(__reFn ) ) {
						// Function call
						data = data[ aLast.replace(__reFn, '') ]( val );
					}
					else {
						// If array notation is used, we just want to strip it and use the property name
						// and assign the value. If it isn't used, then we get the result we want anyway
						data[ aLast.replace(__reArray, '') ] = val;
					}
				};
		
				return function (data, val) { // meta is also passed in, but not used
					return setData( data, val, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data, val) { // meta is also passed in, but not used
					data[source] = val;
				};
			}
		},
	
		/**
		 * Create a function that will read nested objects from arrays, based on JSON notation
		 * @param {*} source JSON notation string
		 * @returns Value read
		 */
		get: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				// Build an object of get functions, and wrap them in a single call
				var o = {};
				$.each( source, function (key, val) {
					if ( val ) {
						o[key] = DataTable.util.get( val );
					}
				} );
		
				return function (data, type, row, meta) {
					var t = o[type] || o._;
					return t !== undefined ?
						t(data, type, row, meta) :
						data;
				};
			}
			else if ( source === null ) {
				// Give an empty string for rendering / sorting etc
				return function (data) { // type, row and meta also passed, but not used
					return data;
				};
			}
			else if ( typeof source === 'function' ) {
				return function (data, type, row, meta) {
					return source( data, type, row, meta );
				};
			}
			else if (
				typeof source === 'string' && (source.indexOf('.') !== -1 ||
				source.indexOf('[') !== -1 || source.indexOf('(') !== -1)
			) {
				/* If there is a . in the source string then the data source is in a
				 * nested object so we loop over the data for each level to get the next
				 * level down. On each loop we test for undefined, and if found immediately
				 * return. This allows entire objects to be missing and sDefaultContent to
				 * be used if defined, rather than throwing an error
				 */
				var fetchData = function (data, type, src) {
					var arrayNotation, funcNotation, out, innerSrc;
		
					if ( src !== "" ) {
						var a = _fnSplitObjNotation( src );
		
						for ( var i=0, iLen=a.length ; i<iLen ; i++ ) {
							// Check if we are dealing with special notation
							arrayNotation = a[i].match(__reArray);
							funcNotation = a[i].match(__reFn);
		
							if ( arrayNotation ) {
								// Array notation
								a[i] = a[i].replace(__reArray, '');
		
								// Condition allows simply [] to be passed in
								if ( a[i] !== "" ) {
									data = data[ a[i] ];
								}
								out = [];
		
								// Get the remainder of the nested object to get
								a.splice( 0, i+1 );
								innerSrc = a.join('.');
		
								// Traverse each entry in the array getting the properties requested
								if ( Array.isArray( data ) ) {
									for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
										out.push( fetchData( data[j], type, innerSrc ) );
									}
								}
		
								// If a string is given in between the array notation indicators, that
								// is used to join the strings together, otherwise an array is returned
								var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
								data = (join==="") ? out : out.join(join);
		
								// The inner call to fetchData has already traversed through the remainder
								// of the source requested, so we exit from the loop
								break;
							}
							else if ( funcNotation ) {
								// Function call
								a[i] = a[i].replace(__reFn, '');
								data = data[ a[i] ]();
								continue;
							}
		
							if (data === null || data[ a[i] ] === null) {
								return null;
							}
							else if ( data === undefined || data[ a[i] ] === undefined ) {
								return undefined;
							}
	
							data = data[ a[i] ];
						}
					}
		
					return data;
				};
		
				return function (data, type) { // row and meta also passed, but not used
					return fetchData( data, type, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data) { // row and meta also passed, but not used
					return data[source];
				};
			}
		},
	
		stripHtml: function (mixed) {
			var type = typeof mixed;
	
			if (type === 'function') {
				_stripHtml = mixed;
				return;
			}
			else if (type === 'string') {
				return _stripHtml(mixed);
			}
			return mixed;
		},
	
		escapeHtml: function (mixed) {
			var type = typeof mixed;
	
			if (type === 'function') {
				_escapeHtml = mixed;
				return;
			}
			else if (type === 'string' || Array.isArray(mixed)) {
				return _escapeHtml(mixed);
			}
			return mixed;
		},
	
		unique: _unique
	};
	
	
	
	/**
	 * Create a mapping object that allows camel case parameters to be looked up
	 * for their Hungarian counterparts. The mapping is stored in a private
	 * parameter called `_hungarianMap` which can be accessed on the source object.
	 *  @param {object} o
	 *  @memberof DataTable#oApi
	 */
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	
	/**
	 * Convert from camel case parameters to Hungarian, based on a Hungarian map
	 * created by _fnHungarianMap.
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 *  @memberof DataTable#oApi
	 */
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	/**
	 * Map one parameter onto another
	 *  @param {object} o Object to map
	 *  @param {*} knew The new parameter name
	 *  @param {*} old The old parameter name
	 */
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	
	/**
	 * Provide backwards compatibility for the main DT options. Note that the new
	 * options are mapped onto the old parameters, so this is an external interface
	 * change only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	
		// Enable search delay if server-side processing is enabled
		if (init.serverSide && ! init.searchDelay) {
			init.searchDelay = 400;
		}
	}
	
	
	/**
	 * Provide backwards compatibility for column options. Note that the new options
	 * are mapped onto the old parameters, so this is an external interface change
	 * only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! Array.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	
	/**
	 * Browser feature detection for capabilities, quirks
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBrowserDetect( settings )
	{
		// We don't need to do this every time DataTables is constructed, the values
		// calculated are specific to the browser and OS configuration which we
		// don't expect to change between initialisations
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: -1 * window.pageXOffset, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();
	
			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			// In rtl text layout, some browsers (most, but not all) will place the
			// scrollbar on the left, rather than the right.
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	/**
	 * Add a column to the list used for the table with default values
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddColumn( oSettings )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol,
			searchFixed: {},
			colEl: $('<col>').attr('data-dt-column', iCol)
		} );
		oSettings.aoColumns.push( oCol );
	
		// Add search object for column specific search. Note that the `searchCols[ iCol ]`
		// passed into extend can be undefined. This allows the user to give a default
		// with only some of the parameters defined, and also not give a default
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	}
	
	
	/**
	 * Apply options for a column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iCol column index to consider
	 *  @param {object} oOptions object with sType, bVisible and bSearchable etc
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions, true );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
		
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
	
			var origClass = oCol.sClass;
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			// Merge class from previously defined classes with this one, rather than just
			// overwriting it in the extend above
			if (origClass !== oCol.sClass) {
				oCol.sClass = origClass + ' ' + oCol.sClass;
			}
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
	
		// The `render` option can be given as an array to access the helper rendering methods.
		// The first element is the rendering method to use, the rest are the parameters to pass
		if ( oCol.mRender && Array.isArray( oCol.mRender ) ) {
			var copy = oCol.mRender.slice();
			var name = copy.shift();
	
			oCol.mRender = DataTable.render[name].apply(window, copy);
		}
	
		oCol._render = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return oCol._render && type ?
				oCol._render( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' && ! oCol._isArrayHost ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
		}
	}
	
	
	/**
	 * Adjust the table column widths for new data. Note: you would probably want to
	 * do a redraw after calling this function!
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAdjustColumnSizing ( settings )
	{
		_fnCalculateColumnWidths( settings );
		_fnColumnSizes( settings );
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '') {
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	/**
	 * Apply column sizes
	 *
	 * @param {*} settings DataTables settings object
	 */
	function _fnColumnSizes ( settings )
	{
		var cols = settings.aoColumns;
	
		for (var i=0 ; i<cols.length ; i++) {
			var width = _fnColumnsSumWidth(settings, [i], false, false);
	
			cols[i].colEl.css('width', width);
		}
	}
	
	
	/**
	 * Convert the index of a visible column to the index in the data array (take account
	 * of hidden columns)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iMatch Visible column index to lookup
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Convert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = aiVis.indexOf(iMatch);
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	/**
	 * Get the number of visible columns
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the number of visible columns
	 *  @memberof DataTable#oApi
	 */
	function _fnVisbleColumns( settings )
	{
		var layout = settings.aoHeader;
		var columns = settings.aoColumns;
		var vis = 0;
	
		if ( layout.length ) {
			for ( var i=0, ien=layout[0].length ; i<ien ; i++ ) {
				if ( columns[i].bVisible && $(layout[0][i].cell).css('display') !== 'none' ) {
					vis++;
				}
			}
		}
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		oSettings.aoColumns.map( function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	
	/**
	 * Calculate the 'type' of a column
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, detectedType, cache;
	
		// For each column, spin over the 
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
	
						if (! data[k]) {
							continue;
						}
	
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						detectedType = types[j]( cache[k], settings );
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-2 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string - but it
						// must not be empty
						if ( detectedType === 'html' && ! _empty(cache[k]) ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( detectedType ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
	
			// Set class names for header / footer for auto type classes
			var autoClass = _ext.type.className[col.sType];
	
			if (autoClass) {
				_columnAutoClass(settings.aoHeader, i, autoClass);
				_columnAutoClass(settings.aoFooter, i, autoClass);
			}
	
			var renderer = _ext.type.render[col.sType];
	
			// This can only happen once! There is no way to remover
			// a renderer. After the first time the renderer has
			// already been set so createTr will run the renderer itself.
			if (renderer && ! col._render) {
				col._render = DataTable.util.get(renderer);
	
				_columnAutoRender(settings, i);
			}
		}
	}
	
	/**
	 * Apply an auto detected renderer to data which doesn't yet have
	 * a renderer
	 */
	function _columnAutoRender(settings, colIdx) {
		var data = settings.aoData;
	
		for (var i=0 ; i<data.length ; i++) {
			if (data[i].nTr) {
				// We have to update the display here since there is no
				// invalidation check for the data
				var display = _fnGetCellData( settings, i, colIdx, 'display' );
	
				data[i].displayData[colIdx] = display;
				_fnWriteCell(data[i].anCells[colIdx], display);
	
				// No need to update sort / filter data since it has
				// been invalidated and will be re-read with the
				// renderer now applied
			}
		}
	}
	
	/**
	 * Apply a class name to a column's header cells
	 */
	function _columnAutoClass(container, colIdx, className) {
		container.forEach(function (row) {
			if (row[colIdx] && row[colIdx].unique) {
				_addClass(row[colIdx].cell, className);
			}
		});
	}
	
	/**
	 * Take the column definitions and static columns arrays and calculate how
	 * they relate to column indexes. The callback function will then apply the
	 * definition found for a column to a suitable configuration object.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aoColDefs The aoColumnDefs array that is to be applied
	 *  @param {array} aoCols The aoColumns array that defines columns individually
	 *  @param {array} headerLayout Layout for header as it was loaded
	 *  @param {function} fn Callback function - takes two parameters, the calculated
	 *    column index and the definition for that column.
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, headerLayout, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		if ( aoCols ) {
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ ) {
				if (aoCols[i] && aoCols[i].name) {
					columns[i].sName = aoCols[i].name;
				}
			}
		}
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.target !== undefined
					? def.target
					: def.targets !== undefined
						? def.targets
						: def.aTargets;
	
				if ( ! Array.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					var target = aTargets[j];
	
					if ( typeof target === 'number' && target >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= target )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( target, def );
					}
					else if ( typeof target === 'number' && target < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+target, def );
					}
					else if ( typeof target === 'string' )
					{
						for ( k=0, kLen=columns.length ; k<kLen ; k++ ) {
							if (target === '_all') {
								// Apply to all columns
								fn( k, def );
							}
							else if (target.indexOf(':name') !== -1) {
								// Column selector
								if (columns[k].sName === target.replace(':name', '')) {
									fn( k, def );
								}
							}
							else {
								// Cell selector
								headerLayout.forEach(function (row) {
									if (row[k]) {
										var cell = $(row[k].cell);
	
										// Legacy support. Note that it means that we don't support
										// an element name selector only, since they are treated as
										// class names for 1.x compat.
										if (target.match(/^[a-z][\w-]*$/i)) {
											target = '.' + target;
										}
	
										if (cell.is( target )) {
											fn( k, def );
										}
									}
								});
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols ) {
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ ) {
				fn( i, aoCols[i] );
			}
		}
	}
	
	
	/**
	 * Get the width for a given set of columns
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} targets Columns - comma separated string or array of numbers
	 * @param {*} original Use the original width (true) or calculated (false)
	 * @param {*} incVisible Include visible columns (true) or not (false)
	 * @returns Combined CSS value
	 */
	function _fnColumnsSumWidth( settings, targets, original, incVisible ) {
		if ( ! Array.isArray( targets ) ) {
			targets = _fnColumnsFromHeader( targets );
		}
	
		var sum = 0;
		var unit;
		var columns = settings.aoColumns;
		
		for ( var i=0, ien=targets.length ; i<ien ; i++ ) {
			var column = columns[ targets[i] ];
			var definedWidth = original ?
				column.sWidthOrig :
				column.sWidth;
	
			if ( ! incVisible && column.bVisible === false ) {
				continue;
			}
	
			if ( definedWidth === null || definedWidth === undefined ) {
				return null; // can't determine a defined width - browser defined
			}
			else if ( typeof definedWidth === 'number' ) {
				unit = 'px';
				sum += definedWidth;
			}
			else {
				var matched = definedWidth.match(/([\d\.]+)([^\d]*)/);
	
				if ( matched ) {
					sum += matched[1] * 1;
					unit = matched.length === 3 ?
						matched[2] :
						'px';
				}
			}
		}
	
		return sum + unit;
	}
	
	function _fnColumnsFromHeader( cell )
	{
		var attr = $(cell).closest('[data-dt-column]').attr('data-dt-column');
	
		if ( ! attr ) {
			return [];
		}
	
		return attr.split(',').map( function (val) {
			return val * 1;
		} );
	}
	/**
	 * Add a data array to the table, creating DOM node etc. This is the parallel to
	 * _fnGatherData, but for adding rows from a Javascript source, rather than a
	 * DOM source.
	 *  @param {object} settings dataTables settings object
	 *  @param {array} data data array to be added
	 *  @param {node} [tr] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [tds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @returns {int} >=0 if successful (index of new aoData entry), -1 if failed
	 *  @memberof DataTable#oApi
	 */
	function _fnAddData ( settings, dataIn, tr, tds )
	{
		/* Create the object for storing information about this new row */
		var rowIdx = settings.aoData.length;
		var rowModel = $.extend( true, {}, DataTable.models.oRow, {
			src: tr ? 'dom' : 'data',
			idx: rowIdx
		} );
	
		rowModel._aData = dataIn;
		settings.aoData.push( rowModel );
	
		var columns = settings.aoColumns;
	
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			// Invalidate the column types as the new data needs to be revalidated
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		settings.aiDisplayMaster.push( rowIdx );
	
		var id = settings.rowIdFn( dataIn );
		if ( id !== undefined ) {
			settings.aIds[ id ] = rowModel;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( tr || ! settings.oFeatures.bDeferRender )
		{
			_fnCreateTr( settings, rowIdx, tr, tds );
		}
	
		return rowIdx;
	}
	
	
	/**
	 * Add one or more TR elements to the table. Generally we'd expect to
	 * use this for reading data from a DOM sourced table, but it could be
	 * used for an TR element. Note that if a TR is given, it is used (i.e.
	 * it is not cloned).
	 *  @param {object} settings dataTables settings object
	 *  @param {array|node|jQuery} trs The TR element(s) to add to the table
	 *  @returns {array} Array of indexes for the added rows
	 *  @memberof DataTable#oApi
	 */
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}
	
	
	/**
	 * Get the data for a given cell from the internal cache, taking into account data mapping
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {string} type data get type ('display', 'type' 'filter|search' 'sort|order')
	 *  @returns {*} Cell data
	 *  @memberof DataTable#oApi
	 */
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		if (type === 'search') {
			type = 'filter';
		}
		else if (type === 'order') {
			type = 'sort';
		}
	
		var row = settings.aoData[rowIdx];
	
		if (! row) {
			return undefined;
		}
	
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = row._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		// Allow for a node being returned for non-display types
		if (type !== 'display' && cellData && typeof cellData === 'object' && cellData.nodeName) {
			cellData = cellData.innerHTML;
		}
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type === 'display' ) {
			return '';
		}
	
		if ( type === 'filter' ) {
			var fomatters = DataTable.ext.type.search;
	
			if ( fomatters[ col.sType ] ) {
				cellData = fomatters[ col.sType ]( cellData );
			}
		}
	
		return cellData;
	}
	
	
	/**
	 * Set the value for a specific cell, into the internal data cache
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {*} val Value to set
	 *  @memberof DataTable#oApi
	 */
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	/**
	 * Write a value to a cell
	 * @param {*} td Cell
	 * @param {*} val Value
	 */
	function _fnWriteCell(td, val)
	{
		if (val && typeof val === 'object' && val.nodeName) {
			$(td)
				.empty()
				.append(val);
		}
		else {
			td.innerHTML = val;
		}
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	/**
	 * Split string on periods, taking into account escaped periods
	 * @param  {string} str String to split
	 * @return {array} Split string
	 */
	function _fnSplitObjNotation( str )
	{
		var parts = str.match(/(\\.|[^.])+/g) || [''];
	
		return parts.map( function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	
	/**
	 * Return a function that can be used to get data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data get function
	 *  @memberof DataTable#oApi
	 */
	var _fnGetObjectDataFn = DataTable.util.get;
	
	
	/**
	 * Return a function that can be used to set data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data set function
	 *  @memberof DataTable#oApi
	 */
	var _fnSetObjectDataFn = DataTable.util.set;
	
	
	/**
	 * Return an array with the full table data
	 *  @param {object} oSettings dataTables settings object
	 *  @returns array {array} aData Master data array
	 *  @memberof DataTable#oApi
	 */
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	/**
	 * Nuke the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	/**
	 * Mark cached data as invalid such that a re-read of the data will occur when
	 * the cached data is next requested. Also update from the data source object.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {int}    rowIdx   Row index to invalidate
	 * @param {string} [src]    Source to invalidate from: undefined, 'auto', 'dom'
	 *     or 'data'
	 * @param {int}    [colIdx] Column index to invalidate. If undefined the whole
	 *     row will be invalidated
	 * @memberof DataTable#oApi
	 *
	 * @todo For the modularisation of v1.11 this will need to become a callback, so
	 *   the sort and filter methods can subscribe to it. That will required
	 *   initialisation options for sorting, which is why it is not already baked in
	 */
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
	
		// Remove the cached data for the row
		row._aSortData = null;
		row._aFilterData = null;
		row.displayData = null;
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
			var display = _fnGetRowDisplay(settings, rowIdx);
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					_fnWriteCell(cells[colIdx], display[colIdx]);
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						_fnWriteCell(cells[i], display[i]);
					}
				}
			}
		}
	
		// Column specific invalidation
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			// Type - the data might have changed
			cols[ colIdx ].sType = null;
	
			// Max length string. Its a fairly cheep recalculation, so not worth
			// something more complicated
			cols[ colIdx ].maxLenString = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
				cols[i].maxLenString = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	/**
	 * Build a data source object from an HTML row, reading the contents of the
	 * cells that are in the row.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {node|object} TR element from which to read data or existing row
	 *   object from which to re-read the data from the cells
	 * @param {int} [colIdx] Optional column index
	 * @param {array|object} [d] Data source object. If `colIdx` is given then this
	 *   parameter should also be given and will be used to write the data into.
	 *   Only the column in question will be written
	 * @returns {object} Object with two parameters: `data` the data read, in
	 *   document order, and `cells` and array of nodes (they can be useful to the
	 *   caller, so rather than needing a second traversal to get them, just return
	 *   them from here).
	 * @memberof DataTable#oApi
	 */
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = (cell.innerHTML).trim();
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	
	/**
	 * Render and cache a row's display data for the columns, if required
	 * @returns 
	 */
	function _fnGetRowDisplay (settings, rowIdx) {
		let rowModal = settings.aoData[rowIdx];
		let columns = settings.aoColumns;
	
		if (! rowModal.displayData) {
			// Need to render and cache
			rowModal.displayData = [];
		
			for ( var colIdx=0, len=columns.length ; colIdx<len ; colIdx++ ) {
				rowModal.displayData.push(
					_fnGetCellData( settings, rowIdx, colIdx, 'display' )
				);
			}
		}
	
		return rowModal.displayData;
	}
	
	/**
	 * Create a new TR element (and it's TD children) for a row
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow Row to consider
	 *  @param {node} [nTrIn] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @memberof DataTable#oApi
	 */
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen, create,
			trClass = oSettings.oClasses.tbody.row;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			_addClass(nTr, trClass);
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
				create = nTrIn && anTds[i] ? false : true;
	
				nTd = create ? document.createElement( oCol.sCellType ) : anTds[i];
	
				if (! nTd) {
					_fnLog( oSettings, 0, 'Incorrect column count', 18 );
				}
	
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
				
				var display = _fnGetRowDisplay(oSettings, iRow);
	
				// Need to create the HTML if new, or if a rendering function is defined
				if (
					create ||
					(
						(oCol.mRender || oCol.mData !== i) &&
						(!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
					)
				) {
					_fnWriteCell(nTd, display[i]);
				}
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && create )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && ! create )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', 'row-created', [nTr, rowData, iRow, cells] );
		}
		else {
			_addClass(row.nTr, trClass);
		}
	}
	
	
	/**
	 * Add attributes to a row based on the special `DT_*` parameters in a data
	 * source object.
	 *  @param {object} settings DataTables settings object
	 *  @param {object} DataTables row object for the row to be modified
	 *  @memberof DataTable#oApi
	 */
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	/**
	 * Create the HTML header for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBuildHead( settings, side )
	{
		var classes = settings.oClasses;
		var columns = settings.aoColumns;
		var i, ien, row;
		var target = side === 'header'
			? settings.nTHead
			: settings.nTFoot;
		var titleProp = side === 'header' ? 'sTitle' : side;
	
		// Footer might be defined
		if (! target) {
			return;
		}
	
		// If no cells yet and we have content for them, then create
		if (side === 'header' || _pluck(settings.aoColumns, titleProp).join('')) {
			row = $('tr', target);
	
			// Add a row if needed
			if (! row.length) {
				row = $('<tr/>').appendTo(target)
			}
	
			// Add the number of cells needed to make up to the number of columns
			if (row.length === 1) {
				var cells = $('td, th', row);
	
				for ( i=cells.length, ien=columns.length ; i<ien ; i++ ) {
					$('<th/>')
						.html( columns[i][titleProp] || '' )
						.appendTo( row );
				}
			}
		}
	
		var detected = _fnDetectHeader( settings, target, true );
	
		if (side === 'header') {
			settings.aoHeader = detected;
		}
		else {
			settings.aoFooter = detected;
		}
	
		// ARIA role for the rows
		$(target).children('tr').attr('role', 'row');
	
		// Every cell needs to be passed through the renderer
		$(target).children('tr').children('th, td')
			.each( function () {
				_fnRenderer( settings, side )(
					settings, $(this), classes
				);
			} );
	}
	
	/**
	 * Build a layout structure for a header or footer
	 *
	 * @param {*} settings DataTables settings
	 * @param {*} source Source layout array
	 * @param {*} incColumns What columns should be included
	 * @returns Layout array
	 */
	function _fnHeaderLayout( settings, source, incColumns )
	{
		var row, column, cell;
		var local = [];
		var structure = [];
		var columns = settings.aoColumns;
		var columnCount = columns.length;
		var rowspan, colspan;
	
		if ( ! source ) {
			return;
		}
	
		// Default is to work on only visible columns
		if ( ! incColumns ) {
			incColumns = _range(columnCount)
				.filter(function (idx) {
					return columns[idx].bVisible;
				});
		}
	
		// Make a copy of the master layout array, but with only the columns we want
		for ( row=0 ; row<source.length ; row++ ) {
			// Remove any columns we haven't selected
			local[row] = source[row].slice().filter(function (cell, i) {
				return incColumns.includes(i);
			});
	
			// Prep the structure array - it needs an element for each row
			structure.push( [] );
		}
	
		for ( row=0 ; row<local.length ; row++ ) {
			for ( column=0 ; column<local[row].length ; column++ ) {
				rowspan = 1;
				colspan = 1;
	
				// Check to see if there is already a cell (row/colspan) covering our target
				// insert point. If there is, then there is nothing to do.
				if ( structure[row][column] === undefined ) {
					cell = local[row][column].cell;
	
					// Expand for rowspan
					while (
						local[row+rowspan] !== undefined &&
						local[row][column].cell == local[row+rowspan][column].cell
					) {
						structure[row+rowspan][column] = null;
						rowspan++;
					}
	
					// And for colspan
					while (
						local[row][column+colspan] !== undefined &&
						local[row][column].cell == local[row][column+colspan].cell
					) {
						// Which also needs to go over rows
						for ( var k=0 ; k<rowspan ; k++ ) {
							structure[row+k][column+colspan] = null;
						}
	
						colspan++;
					}
	
					var titleSpan = $('span.dt-column-title', cell);
	
					structure[row][column] = {
						cell: cell,
						colspan: colspan,
						rowspan: rowspan,
						title: titleSpan.length
							? titleSpan.html()
							: $(cell).html()
					};
				}
			}
		}
	
		return structure;
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states.
	 *
	 *  @param object oSettings dataTables settings object
	 *  @param array aoSource Layout array from _fnDetectHeader
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( settings, source )
	{
		var layout = _fnHeaderLayout(settings, source);
		var tr, n;
	
		for ( var row=0 ; row<source.length ; row++ ) {
			tr = source[row].row;
	
			// All cells are going to be replaced, so empty out the row
			// Can't use $().empty() as that kills event handlers
			if (tr) {
				while( (n = tr.firstChild) ) {
					tr.removeChild( n );
				}
			}
	
			for ( var column=0 ; column<layout[row].length ; column++ ) {
				var point = layout[row][column];
	
				if (point) {
					$(point.cell)
						.appendTo(tr)
						.attr('rowspan', point.rowspan)
						.attr('colspan', point.colspan);
				}
			}
		}
	}
	
	
	/**
	 * Insert the required TR nodes into the table for display
	 *  @param {object} oSettings dataTables settings object
	 *  @param ajaxComplete true after ajax call to complete rendering
	 *  @memberof DataTable#oApi
	 */
	function _fnDraw( oSettings, ajaxComplete )
	{
		// Allow for state saving and a custom start position
		_fnStart( oSettings );
	
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( aPreDraw.indexOf(false) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var anRows = [];
		var iRowCount = 0;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
		var columns = oSettings.aoColumns;
		var body = $(oSettings.nTBody);
	
		oSettings.bDrawing = true;
	
		/* Server-side processing draw intercept */
		if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !ajaxComplete)
		{
			// Show loading message for server-side processing
			if (oSettings.iDraw === 0) {
				body.empty().append(_emptyRow(oSettings));
			}
	
			_fnAjaxUpdate( oSettings );
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				// Add various classes as needed
				for (var i=0 ; i<columns.length ; i++) {
					var col = columns[i];
					var td = aoData.anCells[i];
	
					_addClass(td, _ext.type.className[col.sType]); // auto class
					_addClass(td, col.sClass); // column class
					_addClass(td, oSettings.oClasses.tbody.cell); // all cells
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			anRows[ 0 ] = _emptyRow(oSettings);
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		// replaceChildren is faster, but only became widespread in 2020,
		// so a fall back in jQuery is provided for older browsers.
		if (body[0].replaceChildren) {
			body[0].replaceChildren.apply(body[0], anRows);
		}
		else {
			body.children().detach();
			body.append( $(anRows) );
		}
	
		// Empty table needs a specific class
		$(oSettings.nTableWrapper).toggleClass('dt-empty-footer', $('tr', oSettings.nTFoot).length === 0);
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings], true );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	
	
	/**
	 * Redraw the table - taking account of the various features which are enabled
	 *  @param {object} oSettings dataTables settings object
	 *  @param {boolean} [holdPosition] Keep the current paging position. By default
	 *    the paging is reset to the first page
	 *  @memberof DataTable#oApi
	 */
	function _fnReDraw( settings, holdPosition, recompute )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if (recompute === undefined || recompute === true) {
			if ( sort ) {
				_fnSort( settings );
			}
	
			if ( filter ) {
				_fnFilterComplete( settings, settings.oPreviousSearch );
			}
			else {
				// No filtering, so we want to just use the display master
				settings.aiDisplay = settings.aiDisplayMaster.slice();
			}
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings._drawHold = false;
	}
	
	
	/*
	 * Table is empty - create a row with an empty message in it
	 */
	function _emptyRow ( settings ) {
		var oLang = settings.oLanguage;
		var zero = oLang.sZeroRecords;
		var dataSrc = _fnDataSource( settings );
	
		if (
			(settings.iDraw < 1 && dataSrc === 'ssp') ||
			(settings.iDraw <= 1 && dataSrc === 'ajax')
		) {
			zero = oLang.sLoadingRecords;
		}
		else if ( oLang.sEmptyTable && settings.fnRecordsTotal() === 0 )
		{
			zero = oLang.sEmptyTable;
		}
	
		return $( '<tr/>' )
			.append( $('<td />', {
				'colSpan': _fnVisbleColumns( settings ),
				'class':   settings.oClasses.empty.row
			} ).html( zero ) )[0];
	}
	
	
	/**
	 * Convert a `layout` object given by a user to the object structure needed
	 * for the renderer. This is done twice, once for above and once for below
	 * the table. Ordering must also be considered.
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} layout Layout object to convert
	 * @param {string} side `top` or `bottom`
	 * @returns Converted array structure - one item for each row.
	 */
	function _layoutArray ( settings, layout, side )
	{
		var groups = {};
	
		// Combine into like groups (e.g. `top`, `top2`, etc)
		$.each( layout, function ( pos, val ) {
			if (val === null) {
				return;
			}
	
			var splitPos = pos.replace(/([A-Z])/g, ' $1').split(' ');
	
			if ( ! groups[ splitPos[0] ] ) {
				groups[ splitPos[0] ] = {};
			}
	
			var align = splitPos.length === 1 ?
				'full' :
				splitPos[1].toLowerCase();
			var group = groups[ splitPos[0] ];
			var groupRun = function (contents, innerVal) {
				// If it is an object, then there can be multiple features contained in it
				if ( $.isPlainObject( innerVal ) ) {
					Object.keys(innerVal).map(function (key) {
						contents.push( {
							feature: key,
							opts: innerVal[key]
						});
					});
				}
				else {
					contents.push(innerVal);
				}
			}
	
			// Transform to an object with a contents property
			if (! group[align] || ! group[align].contents) {
				group[align] = { contents: [] };
			}
	
			// Allow for an array or just a single object
			if ( Array.isArray(val)) {
				for (var i=0 ; i<val.length ; i++) {
					groupRun(group[align].contents, val[i]);
				}
			}
			else {
				groupRun(group[ align ].contents, val);
			}
	
			// And make contents an array
			if ( ! Array.isArray( group[ align ].contents ) ) {
				group[ align ].contents = [ group[ align ].contents ];
			}
		} );
	
		var filtered = Object.keys(groups)
			.map( function ( pos ) {
				// Filter to only the side we need
				if ( pos.indexOf(side) !== 0 ) {
					return null;
				}
	
				return {
					name: pos,
					val: groups[pos]
				};
			} )
			.filter( function (item) {
				return item !== null;
			});
	
		// Order by item identifier
		filtered.sort( function ( a, b ) {
			var order1 = a.name.replace(/[^0-9]/g, '') * 1;
			var order2 = b.name.replace(/[^0-9]/g, '') * 1;
	
			return order2 - order1;
		} );
		
		if ( side === 'bottom' ) {
			filtered.reverse();
		}
	
		// Split into rows
		var rows = [];
		for ( var i=0, ien=filtered.length ; i<ien ; i++ ) {
			if (  filtered[i].val.full ) {
				rows.push( { full: filtered[i].val.full } );
				_layoutResolve( settings, rows[ rows.length - 1 ] );
	
				delete filtered[i].val.full;
			}
	
			if ( Object.keys(filtered[i].val).length ) {
				rows.push( filtered[i].val );
				_layoutResolve( settings, rows[ rows.length - 1 ] );
			}
		}
	
		return rows;
	}
	
	
	/**
	 * Convert the contents of a row's layout object to nodes that can be inserted
	 * into the document by a renderer. Execute functions, look up plug-ins, etc.
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} row Layout object for this row
	 */
	function _layoutResolve( settings, row ) {
		var getFeature = function (feature, opts) {
			if ( ! _ext.features[ feature ] ) {
				_fnLog( settings, 0, 'Unknown feature: '+ feature );
			}
	
			return _ext.features[ feature ].apply( this, [settings, opts] );
		};
	
		var resolve = function ( item ) {
			var line = row[ item ].contents;
	
			for ( var i=0, ien=line.length ; i<ien ; i++ ) {
				if ( ! line[i] ) {
					continue;
				}
				else if ( typeof line[i] === 'string' ) {
					line[i] = getFeature( line[i], null );
				}
				else if ( $.isPlainObject(line[i]) ) {
					// If it's an object, it just has feature and opts properties from
					// the transform in _layoutArray
					line[i] = getFeature(line[i].feature, line[i].opts);
				}
				else if ( typeof line[i].node === 'function' ) {
					line[i] = line[i].node( settings );
				}
				else if ( typeof line[i] === 'function' ) {
					var inst = line[i]( settings );
	
					line[i] = typeof inst.node === 'function' ?
						inst.node() :
						inst;
				}
			}
		};
	
		$.each( row, function ( key ) {
			resolve( key );
		} );
	}
	
	
	/**
	 * Add the options to the page HTML for the table
	 *  @param {object} settings DataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddOptionsHtml ( settings )
	{
		var classes = settings.oClasses;
		var table = $(settings.nTable);
	
		// Wrapper div around everything DataTables controls
		var insert = $('<div/>')
			.attr({
				id:      settings.sTableId+'_wrapper',
				'class': classes.container
			})
			.insertBefore(table);
	
		settings.nTableWrapper = insert[0];
	
		if (settings.sDom) {
			// Legacy
			_fnLayoutDom(settings, settings.sDom, insert);
		}
		else {
			var top = _layoutArray( settings, settings.layout, 'top' );
			var bottom = _layoutArray( settings, settings.layout, 'bottom' );
			var renderer = _fnRenderer( settings, 'layout' );
		
			// Everything above - the renderer will actually insert the contents into the document
			top.forEach(function (item) {
				renderer( settings, insert, item );
			});
	
			// The table - always the center of attention
			renderer( settings, insert, {
				full: {
					table: true,
					contents: [ _fnFeatureHtmlTable(settings) ]
				}
			} );
	
			// Everything below
			bottom.forEach(function (item) {
				renderer( settings, insert, item );
			});
		}
	
		// Processing floats on top, so it isn't an inserted feature
		_processingHtml( settings );
	}
	
	/**
	 * Draw the table with the legacy DOM property
	 * @param {*} settings DT settings object
	 * @param {*} dom DOM string
	 * @param {*} insert Insert point
	 */
	function _fnLayoutDom( settings, dom, insert )
	{
		var parts = dom.match(/(".*?")|('.*?')|./g);
		var featureNode, option, newNode, next, attr;
	
		for ( var i=0 ; i<parts.length ; i++ ) {
			featureNode = null;
			option = parts[i];
	
			if ( option == '<' ) {
				// New container div
				newNode = $('<div/>');
	
				// Check to see if we should append an id and/or a class name to the container
				next = parts[i+1];
	
				if ( next[0] == "'" || next[0] == '"' ) {
					attr = next.replace(/['"]/g, '');
	
					var id = '', className;
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( attr.indexOf('.') != -1 ) {
						var split = attr.split('.');
	
						id = split[0];
						className = split[1];
					}
					else if ( attr[0] == "#" ) {
						id = attr;
					}
					else {
						className = attr;
					}
	
					newNode
						.attr('id', id.substring(1))
						.addClass(className);
	
					i++; // Move along the position array
				}
	
				insert.append( newNode );
				insert = newNode;
			}
			else if ( option == '>' ) {
				// End container div
				insert = insert.parent();
			}
			else if ( option == 't' ) {
				// Table
				featureNode = _fnFeatureHtmlTable( settings );
			}
			else
			{
				DataTable.ext.feature.forEach(function(feature) {
					if ( option == feature.cFeature ) {
						featureNode = feature.fnInit( settings );
					}
				});
			}
	
			// Add to the display
			if ( featureNode ) {
				insert.append( featureNode );
			}
		}
	}
	
	
	/**
	 * Use the DOM source to create up an array of header cells. The idea here is to
	 * create a layout grid (array) of rows x columns, which contains a reference
	 * to the cell that that point in the grid (regardless of col/rowspan), such that
	 * any column / row could be removed and the new grid constructed
	 *  @param {node} thead The header/footer element for the table
	 *  @returns {array} Calculated layout array
	 *  @memberof DataTable#oApi
	 */
	function _fnDetectHeader ( settings, thead, write )
	{
		var columns = settings.aoColumns;
		var rows = $(thead).children('tr');
		var row, cell;
		var i, k, l, iLen, shifted, column, colspan, rowspan;
		var isHeader = thead && thead.nodeName.toLowerCase() === 'thead';
		var layout = [];
		var unique;
		var shift = function ( a, i, j ) {
			var k = a[i];
			while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		// We know how many rows there are in the layout - so prep it
		for ( i=0, iLen=rows.length ; i<iLen ; i++ ) {
			layout.push( [] );
		}
	
		for ( i=0, iLen=rows.length ; i<iLen ; i++ ) {
			row = rows[i];
			column = 0;
	
			// For every cell in the row..
			cell = row.firstChild;
			while ( cell ) {
				if (
					cell.nodeName.toUpperCase() == 'TD' ||
					cell.nodeName.toUpperCase() == 'TH'
				) {
					var cols = [];
	
					// Get the col and rowspan attributes from the DOM and sanitise them
					colspan = cell.getAttribute('colspan') * 1;
					rowspan = cell.getAttribute('rowspan') * 1;
					colspan = (!colspan || colspan===0 || colspan===1) ? 1 : colspan;
					rowspan = (!rowspan || rowspan===0 || rowspan===1) ? 1 : rowspan;
	
					// There might be colspan cells already in this row, so shift our target
					// accordingly
					shifted = shift( layout, i, column );
	
					// Cache calculation for unique columns
					unique = colspan === 1 ?
						true :
						false;
					
					// Perform header setup
					if ( write ) {
						if (unique) {
							// Allow column options to be set from HTML attributes
							_fnColumnOptions( settings, shifted, $(cell).data() );
							
							// Get the width for the column. This can be defined from the
							// width attribute, style attribute or `columns.width` option
							var columnDef = columns[shifted];
							var width = cell.getAttribute('width') || null;
							var t = cell.style.width.match(/width:\s*(\d+[pxem%]+)/);
							if ( t ) {
								width = t[1];
							}
	
							columnDef.sWidthOrig = columnDef.sWidth || width;
	
							if (isHeader) {
								// Column title handling - can be user set, or read from the DOM
								// This happens before the render, so the original is still in place
								if ( columnDef.sTitle !== null && ! columnDef.autoTitle ) {
									cell.innerHTML = columnDef.sTitle;
								}
	
								if (! columnDef.sTitle && unique) {
									columnDef.sTitle = _stripHtml(cell.innerHTML);
									columnDef.autoTitle = true;
								}
							}
							else {
								// Footer specific operations
								if (columnDef.footer) {
									cell.innerHTML = columnDef.footer;
								}
							}
	
							// Fall back to the aria-label attribute on the table header if no ariaTitle is
							// provided.
							if (! columnDef.ariaTitle) {
								columnDef.ariaTitle = $(cell).attr("aria-label") || columnDef.sTitle;
							}
	
							// Column specific class names
							if ( columnDef.className ) {
								$(cell).addClass( columnDef.className );
							}
						}
	
						// Wrap the column title so we can write to it in future
						if ( $('span.dt-column-title', cell).length === 0) {
							$('<span>')
								.addClass('dt-column-title')
								.append(cell.childNodes)
								.appendTo(cell);
						}
	
						if ( isHeader && $('span.dt-column-order', cell).length === 0) {
							$('<span>')
								.addClass('dt-column-order')
								.appendTo(cell);
						}
					}
	
					// If there is col / rowspan, copy the information into the layout grid
					for ( l=0 ; l<colspan ; l++ ) {
						for ( k=0 ; k<rowspan ; k++ ) {
							layout[i+k][shifted+l] = {
								cell: cell,
								unique: unique
							};
	
							layout[i+k].row = row;
						}
	
						cols.push( shifted+l );
					}
	
					// Assign an attribute so spanning cells can still be identified
					// as belonging to a column
					cell.setAttribute('data-dt-column', _unique(cols).join(','));
				}
	
				cell = cell.nextSibling;
			}
		}
	
		return layout;
	}
	
	/**
	 * Set the start position for draw
	 *  @param {object} oSettings dataTables settings object
	 */
	function _fnStart( oSettings )
	{
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var iInitDisplayStart = oSettings.iInitDisplayStart;
	
		// Check and see if we have an initial draw position from state saving
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	}
	
	/**
	 * Create an Ajax call based on the table's settings, taking into account that
	 * parameters can have multiple forms, and backwards compatibility.
	 *
	 * @param {object} oSettings dataTables settings object
	 * @param {array} data Data to send to the server, required by
	 *     DataTables - may be augmented by developer callbacks
	 * @param {function} fn Callback function to run when data is obtained
	 */
	function _fnBuildAjax( oSettings, data, fn )
	{
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			var status = oSettings.jqXHR
				? oSettings.jqXHR.status
				: null;
	
			if ( json === null || (typeof status === 'number' && status == 204 ) ) {
				json = {};
				_fnAjaxDataSrc( oSettings, json, [] );
			}
	
			var error = json.error || json.sError;
			if ( error ) {
				_fnLog( oSettings, 0, error );
			}
	
			oSettings.json = json;
	
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR], true );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"url": typeof ajax === 'string' ?
				ajax :
				'',
			"data": data,
			"success": callback,
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR], true );
	
				if ( ret.indexOf(true) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// If `ajax` option is an object, extend and override our default base
		if ( $.isPlainObject( ajax ) ) {
			$.extend( baseAjax, ajax )
		}
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data, baseAjax], true );
	
		if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else if (ajax.url === '') {
			// No url, so don't load any data. Just apply an empty data array
			// to the object for the callback.
			var empty = {};
	
			DataTable.util.set(ajax.dataSrc)(empty, []);
			callback(empty);
		}
		else {
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( baseAjax );
	
			// Restore for next time around
			if ( ajaxData ) {
				ajax.data = ajaxData;
			}
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		settings.iDraw++;
		_fnProcessingDisplay( settings, true );
	
		_fnBuildAjax(
			settings,
			_fnAjaxParameters( settings ),
			function(json) {
				_fnAjaxUpdateDraw( settings, json );
			}
		);
	}
	
	
	/**
	 * Build up the parameters in an object needed for a server-side processing
	 * request.
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {bool} block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			colData = function ( idx, prop ) {
				return typeof columns[idx][prop] === 'function' ?
					'function' :
					columns[idx][prop];
			};
	
		return {
			draw: settings.iDraw,
			columns: columns.map( function ( column, i ) {
				return {
					data: colData(i, 'mData'),
					name: column.sName,
					searchable: column.bSearchable,
					orderable: column.bSortable,
					search: {
						value: preColSearch[i].search,
						regex: preColSearch[i].regex,
						fixed: Object.keys(column.searchFixed).map( function(name) {
							return {
								name: name,
								term: column.searchFixed[name].toString()
							}
						})
					}
				};
			} ),
			order: _fnSortFlatten( settings ).map( function ( val ) {
				return {
					column: val.col,
					dir: val.dir,
					name: colData(val.col, 'sName')
				};
			} ),
			start: settings._iDisplayStart,
			length: features.bPaginate ?
				settings._iDisplayLength :
				-1,
			search: {
				value: preSearch.search,
				regex: preSearch.regex,
				fixed: Object.keys(settings.searchFixed).map( function(name) {
					return {
						name: name,
						term: settings.searchFixed[name].toString()
					}
				})
			}
		};
	}
	
	
	/**
	 * Data the data from the server (nuking the old) and redraw the table
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} json json data return from the server.
	 *  @param {string} json.sEcho Tracking flag for DataTables to match requests
	 *  @param {int} json.iTotalRecords Number of records in the data set, not accounting for filtering
	 *  @param {int} json.iTotalDisplayRecords Number of records in the data set, accounting for filtering
	 *  @param {array} json.aaData The data to display on this page
	 *  @param {string} [json.sColumns] Column ordering (sName, comma separated)
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdateDraw ( settings, json )
	{
		var data = _fnAjaxDataSrc(settings, json);
		var draw = _fnAjaxDataSrcParam(settings, 'draw', json);
		var recordsTotal = _fnAjaxDataSrcParam(settings, 'recordsTotal', json);
		var recordsFiltered = _fnAjaxDataSrcParam(settings, 'recordsFiltered', json);
	
		if ( draw !== undefined ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		// No data in returned object, so rather than an array, we show an empty table
		if ( ! data ) {
			data = [];
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		_fnDraw( settings, true );
		_fnInitComplete( settings );
		_fnProcessingDisplay( settings, false );
	}
	
	
	/**
	 * Get the data from the JSON data source to use for drawing a table. Using
	 * `_fnGetObjectDataFn` allows the data to be sourced from a property of the
	 * source object, or from a processing function.
	 *  @param {object} settings dataTables settings object
	 *  @param  {object} json Data source object / array from the server
	 *  @return {array} Array of data to use
	 */
	function _fnAjaxDataSrc ( settings, json, write )
	{
		var dataProp = 'data';
	
		if ($.isPlainObject( settings.ajax ) && settings.ajax.dataSrc !== undefined) {
			// Could in inside a `dataSrc` object, or not!
			var dataSrc = settings.ajax.dataSrc;
	
			// string, function and object are valid types
			if (typeof dataSrc === 'string' || typeof dataSrc === 'function') {
				dataProp = dataSrc;
			}
			else if (dataSrc.data !== undefined) {
				dataProp = dataSrc.data;
			}
		}
	
		if ( ! write ) {
			if ( dataProp === 'data' ) {
				// If the default, then we still want to support the old style, and safely ignore
				// it if possible
				return json.aaData || json[dataProp];
			}
	
			return dataProp !== "" ?
				_fnGetObjectDataFn( dataProp )( json ) :
				json;
		}
		
		// set
		_fnSetObjectDataFn( dataProp )( json, write );
	}
	
	/**
	 * Very similar to _fnAjaxDataSrc, but for the other SSP properties
	 * @param {*} settings DataTables settings object
	 * @param {*} param Target parameter
	 * @param {*} json JSON data
	 * @returns Resolved value
	 */
	function _fnAjaxDataSrcParam (settings, param, json) {
		var dataSrc = $.isPlainObject( settings.ajax )
			? settings.ajax.dataSrc
			: null;
	
		if (dataSrc && dataSrc[param]) {
			// Get from custom location
			return _fnGetObjectDataFn( dataSrc[param] )( json );
		}
	
		// else - Default behaviour
		var old = '';
	
		// Legacy support
		if (param === 'draw') {
			old = 'sEcho';
		}
		else if (param === 'recordsTotal') {
			old = 'iTotalRecords';
		}
		else if (param === 'recordsFiltered') {
			old = 'iTotalDisplayRecords';
		}
	
		return json[old] !== undefined
			? json[old]
			: json[param];
	}
	
	
	/**
	 * Filter the table using both the global filter and column based filtering
	 *  @param {object} settings dataTables settings object
	 *  @param {object} input search information
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterComplete ( settings, input )
	{
		var columnsSearch = settings.aoPreSearchCols;
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo As per sort - can this be moved into an event handler?
		_fnColumnTypes( settings );
	
		// In server-side processing all filtering is done by the server, so no point hanging around here
		if ( _fnDataSource( settings ) != 'ssp' )
		{
			// Check if any of the rows were invalidated
			_fnFilterData( settings );
	
			// Start from the full data set
			settings.aiDisplay = settings.aiDisplayMaster.slice();
	
			// Global filter first
			_fnFilter( settings.aiDisplay, settings, input.search, input );
	
			$.each(settings.searchFixed, function (name, term) {
				_fnFilter(settings.aiDisplay, settings, term, {});
			});
	
			// Then individual column filters
			for ( var i=0 ; i<columnsSearch.length ; i++ )
			{
				var col = columnsSearch[i];
	
				_fnFilter(
					settings.aiDisplay,
					settings,
					col.search,
					col,
					i
				);
	
				$.each(settings.aoColumns[i].searchFixed, function (name, term) {
					_fnFilter(settings.aiDisplay, settings, term, {}, i);
				});
			}
	
			// And finally global filtering
			_fnFilterCustom( settings );
		}
	
		// Tell the draw function we have been filtering
		settings.bFiltered = true;
	
		_fnCallbackFire( settings, null, 'search', [settings] );
	}
	
	
	/**
	 * Apply custom filtering functions
	 * 
	 * This is legacy now that we have named functions, but it is widely used
	 * from 1.x, so it is not yet deprecated.
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			displayRows.push.apply(displayRows, rows);
		}
	}
	
	
	/**
	 * Filter the data table based on user input and draw the table
	 */
	function _fnFilter( searchRows, settings, input, options, column )
	{
		if ( input === '' ) {
			return;
		}
	
		var i = 0;
		var matched = [];
	
		// Search term can be a function, regex or string - if a string we apply our
		// smart filtering regex (assuming the options require that)
		var searchFunc = typeof input === 'function' ? input : null;
		var rpSearch = input instanceof RegExp
			? input
			: searchFunc
				? null
				: _fnFilterCreateSearch( input, options );
	
		// Then for each row, does the test pass. If not, lop the row from the array
		for (i=0 ; i<searchRows.length ; i++) {
			var row = settings.aoData[ searchRows[i] ];
			var data = column === undefined
				? row._sFilterRow
				: row._aFilterData[ column ];
	
			if ( (searchFunc && searchFunc(data, row._aData, searchRows[i], column)) || (rpSearch && rpSearch.test(data)) ) {
				matched.push(searchRows[i]);
			}
		}
	
		// Mutate the searchRows array
		searchRows.length = matched.length;
	
		for (i=0 ; i<matched.length ; i++) {
			searchRows[i] = matched[i];
		}
	}
	
	
	/**
	 * Build a regular expression object suitable for searching a table
	 *  @param {string} sSearch string to search for
	 *  @param {bool} bRegex treat as a regular expression or not
	 *  @param {bool} bSmart perform smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @returns {RegExp} constructed object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCreateSearch( search, inOpts )
	{
		var not = [];
		var options = $.extend({}, {
			boundary: false,
			caseInsensitive: true,
			exact: false,
			regex: false,
			smart: true
		}, inOpts);
	
		if (typeof search !== 'string') {
			search = search.toString();
		}
	
		// Remove diacritics if normalize is set up to do so
		search = _normalize(search);
	
		if (options.exact) {
			return new RegExp(
				'^'+_fnEscapeRegex(search)+'$',
				options.caseInsensitive ? 'i' : ''
			);
		}
	
		search = options.regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( options.smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. And a negative look around for
			 * finding rows which don't contain a given string.
			 * 
			 * So this is the sort of thing we want to generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var parts = search.match( /!?["\u201C][^"\u201D]+["\u201D]|[^ ]+/g ) || [''];
			var a = parts.map( function ( word ) {
				var negative = false;
				var m;
	
				// Determine if it is a "does not include"
				if ( word.charAt(0) === '!' ) {
					negative = true;
					word = word.substring(1);
				}
	
				// Strip the quotes from around matched phrases
				if ( word.charAt(0) === '"' ) {
					m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
				else if ( word.charAt(0) === '\u201C' ) {
					// Smart quote match (iPhone users)
					m = word.match( /^\u201C(.*)\u201D$/ );
					word = m ? m[1] : word;
				}
	
				// For our "not" case, we need to modify the string that is
				// allowed to match at the end of the expression.
				if (negative) {
					if (word.length > 1) {
						not.push('(?!'+word+')');
					}
	
					word = '';
				}
	
				return word.replace(/"/g, '');
			} );
	
			var match = not.length
				? not.join('')
				: '';
	
			var boundary = options.boundary
				? '\\b'
				: '';
	
			search = '^(?=.*?'+boundary+a.join( ')(?=.*?'+boundary )+')('+match+'.)*$';
		}
	
		return new RegExp( search, options.caseInsensitive ? 'i' : '' );
	}
	
	
	/**
	 * Escape a string such that it can be used in a regular expression
	 *  @param {string} sVal string to escape
	 *  @returns {string} escaped string
	 *  @memberof DataTable#oApi
	 */
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var column;
		var j, jen, filterData, cellData, row;
		var wasInvalidated = false;
	
		for ( var rowIdx=0 ; rowIdx<data.length ; rowIdx++ ) {
			if (! data[rowIdx]) {
				continue;
			}
	
			row = data[rowIdx];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, rowIdx, j, 'filter' );
	
						// Search in DataTables is string based
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					// If it looks like there is an HTML entity in the string,
					// attempt to decode it so sorting works as expected. Note that
					// we could use a single line of jQuery to do this, but the DOM
					// method used here is much faster https://jsperf.com/html-decode
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n\u2028]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitialise ( settings )
	{
		var i, iAjaxStart=settings.iInitDisplayStart;
	
		/* Ensure that the table data is fully initialised */
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		/* Build and draw the header / footer for the table */
		_fnBuildHead( settings, 'header' );
		_fnBuildHead( settings, 'footer' );
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		// Enable features
		_fnAddOptionsHtml( settings );
		_fnSortInit( settings );
	
		_colGroup( settings );
	
		/* Okay to show that something is going on now */
		_fnProcessingDisplay( settings, true );
	
		_fnCallbackFire( settings, null, 'preInit', [settings], true );
	
		// If there is default sorting required - let's do it. The sort function
		// will do the drawing for us. Otherwise we draw the table regardless of the
		// Ajax source - this allows the table to look initialised for Ajax sourcing
		// data (show 'loading' message possibly)
		_fnReDraw( settings );
	
		var dataSrc = _fnDataSource( settings );
	
		// Server-side processing init complete is done by _fnAjaxUpdateDraw
		if ( dataSrc != 'ssp' ) {
			// if there is an ajax source load the data
			if ( dataSrc == 'ajax' ) {
				_fnBuildAjax( settings, {}, function(json) {
					var aData = _fnAjaxDataSrc( settings, json );
	
					// Got the data - add it to the table
					for ( i=0 ; i<aData.length ; i++ ) {
						_fnAddData( settings, aData[i] );
					}
	
					// Reset the init display for cookie saving. We've already done
					// a filter, and therefore cleared it before. So we need to make
					// it appear 'fresh'
					settings.iInitDisplayStart = iAjaxStart;
	
					_fnReDraw( settings );
					_fnProcessingDisplay( settings, false );
					_fnInitComplete( settings );
				}, settings );
			}
			else {
				_fnInitComplete( settings );
				_fnProcessingDisplay( settings, false );
			}
		}
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitComplete ( settings )
	{
		if (settings._bInitComplete) {
			return;
		}
	
		var args = [settings, settings.json];
	
		settings._bInitComplete = true;
	
		// Table is fully set up and we have data, so calculate the
		// column widths
		_fnAdjustColumnSizing( settings );
	
		_fnCallbackFire( settings, null, 'plugin-init', args, true );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', args, true );
	}
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}
	
	/**
	 * Alter the display settings to change the page
	 *  @param {object} settings DataTables settings object
	 *  @param {string|int} action Paging action to take: "first", "previous",
	 *    "next" or "last" or page number to jump to (integer)
	 *  @param [bool] redraw Automatically draw the update or not
	 *  @returns {bool} true page has changed, false - no change
	 *  @memberof DataTable#oApi
	 */
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
				start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else if ( action === 'ellipsis' )
		{
			return;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		_fnCallbackFire( settings, null, changed ? 'page' : 'page-nc', [settings] );
	
		if ( changed && redraw ) {
			_fnDraw( settings );
		}
	
		return changed;
	}
	
	
	/**
	 * Generate the node required for the processing node
	 *  @param {object} settings DataTables settings object
	 */
	function _processingHtml ( settings )
	{
		var table = settings.nTable;
		var scrolling = settings.oScroll.sX !== '' || settings.oScroll.sY !== '';
	
		if ( settings.oFeatures.bProcessing ) {
			var n = $('<div/>', {
					'id': settings.sTableId + '_processing',
					'class': settings.oClasses.processing.container,
					'role': 'status'
				} )
				.html( settings.oLanguage.sProcessing )
				.append('<div><div></div><div></div><div></div><div></div></div>');
	
			// Different positioning depending on if scrolling is enabled or not
			if (scrolling) {
				n.prependTo( $('div.dt-scroll', settings.nTableWrapper) );
			}
			else {
				n.insertBefore( table );
			}
	
			$(table).on( 'processing.dt.DT', function (e, s, show) {
				n.css( 'display', show ? 'block' : 'none' );
			} );
		}
	}
	
	
	/**
	 * Display or hide the processing indicator
	 *  @param {object} settings DataTables settings object
	 *  @param {bool} show Show the processing indicator (true) or not (false)
	 */
	function _fnProcessingDisplay ( settings, show )
	{
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	/**
	 * Add any control elements for the table - specifically scrolling
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Node to add to the DOM
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses.scrolling;
		var caption = settings.captionNode;
		var captionSide = caption ? caption._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		/*
		 * The HTML structure that we want to generate in this function is:
		 *  div - scroller
		 *    div - scroll head
		 *      div - scroll head inner
		 *        table - scroll head table
		 *          thead - thead
		 *    div - scroll body
		 *      table - table (master table)
		 *        thead - thead clone for sizing
		 *        tbody - tbody
		 *    div - scroll foot
		 *      div - scroll foot inner
		 *        table - scroll foot table
		 *          tfoot - tfoot
		 */
		var scroller = $( _div, { 'class': classes.container } )
			.append(
				$(_div, { 'class': classes.header.self } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.header.inner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.body } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.footer.self } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.footer.inner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		$(scrollBody).on( 'scroll.DT', function () {
			var scrollLeft = this.scrollLeft;
	
			scrollHead.scrollLeft = scrollLeft;
	
			if ( footer ) {
				scrollFoot.scrollLeft = scrollLeft;
			}
		} );
	
		// When focus is put on the header cells, we might need to scroll the body
		$('th, td', scrollHead).on('focus', function () {
			var scrollLeft = scrollHead.scrollLeft;
	
			scrollBody.scrollLeft = scrollLeft;
	
			if ( footer ) {
				scrollBody.scrollLeft = scrollLeft;
			}
		});
	
		$(scrollBody).css('max-height', scrollY);
		if (! scroll.bCollapse) {
			$(scrollBody).css('height', scrollY);
		}
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push(_fnScrollDraw);
	
		return scroller[0];
	}
	
	
	
	/**
	 * Update the header, footer and body tables for resizing - i.e. column
	 * alignment.
	 *
	 * Welcome to the most horrible function DataTables. The process that this
	 * function follows is basically:
	 *   1. Re-create the table inside the scrolling div
	 *   2. Correct colgroup > col values if needed
	 *   3. Copy colgroup > col over to header and footer
	 *   4. Clean up
	 *
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderInner = divHeader.children('div'),
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			footer         = settings.nTFoot && $('th, td', settings.nTFoot).length ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			headerCopy, footerCopy;
	
		// If the scrollbar visibility has changed from the last draw, we need to
		// adjust the column sizes as the table width will have changed to account
		// for the scrollbar
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		// 1. Re-create the table inside the scrolling div
		// Remove the old minimised thead and tfoot elements in the inner table
		table.children('thead, tfoot').remove();
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerCopy.find('th, td').removeAttr('tabindex');
		headerCopy.find('[id]').removeAttr('id');
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerCopy.find('[id]').removeAttr('id');
		}
	
		// 2. Correct colgroup > col values if needed
		// It is possible that the cell sizes are smaller than the content, so we need to
		// correct colgroup>col for such cases. This can happen if the auto width detection
		// uses a cell which has a longer string, but isn't the widest! For example 
		// "Chief Executive Officer (CEO)" is the longest string in the demo, but
		// "Systems Administrator" is actually the widest string since it doesn't collapse.
		// Note the use of translating into a column index to get the `col` element. This
		// is because of Responsive which might remove `col` elements, knocking the alignment
		// of the indexes out.
		if (settings.aiDisplay.length) {
			// Get the column sizes from the first row in the table
			var colSizes = table.children('tbody').eq(0).children('tr').eq(0).children('th, td').map(function (vis) {
				return {
					idx: _fnVisibleToColumnIndex(settings, vis),
					width: $(this).outerWidth()
				}
			});
	
			// Check against what the colgroup > col is set to and correct if needed
			for (var i=0 ; i<colSizes.length ; i++) {
				var colEl = settings.aoColumns[ colSizes[i].idx ].colEl[0];
				var colWidth = colEl.style.width.replace('px', '');
	
				if (colWidth !== colSizes[i].width) {
					colEl.style.width = colSizes[i].width + 'px';
				}
			}
		}
	
		// 3. Copy the colgroup over to the header and footer
		divHeaderTable
			.find('colgroup')
			.remove();
	
		divHeaderTable.append(settings.colgroup.clone());
	
		if ( footer ) {
			divFooterTable
				.find('colgroup')
				.remove();
	
			divFooterTable.append(settings.colgroup.clone());
		}
	
		// "Hide" the header and footer that we used for the sizing. We need to keep
		// the content of the cell so that the width applied to the header and body
		// both match, but we want to hide it completely.
		$('th, td', headerCopy).each(function () {
			$(this.childNodes).wrapAll('<div class="dt-scroll-sizing">');
		});
	
		if ( footer ) {
			$('th, td', footerCopy).each(function () {
				$(this.childNodes).wrapAll('<div class="dt-scroll-sizing">');
			});
		}
	
		// 4. Clean up
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var isScrolling = Math.floor(table.height()) > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var paddingSide = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
	
		// Set the width's of the header and footer tables
		var outerWidth = table.outerWidth();
	
		divHeaderTable.css('width', _fnStringToCss( outerWidth ));
		divHeaderInner
			.css('width', _fnStringToCss( outerWidth ))
			.css(paddingSide, isScrolling ? barWidth+"px" : "0px");
	
		if ( footer ) {
			divFooterTable.css('width', _fnStringToCss( outerWidth ));
			divFooterInner
				.css('width', _fnStringToCss( outerWidth ))
				.css(paddingSide, isScrolling ? barWidth+"px" : "0px");
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').prependTo(table);
	
		// Adjust the position of the header in case we loose the y-scrollbar
		divBody.trigger('scroll');
	
		// If sorting or filtering has occurred, jump the scrolling back to the top
		// only if we aren't holding the position
		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	/**
	 * Calculate the width of columns for the table
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnCalculateColumnWidths ( settings )
	{
		// Not interested in doing column width calculation if auto-width is disabled
		if (! settings.oFeatures.bAutoWidth) {
			return;
		}
	
		var
			table = settings.nTable,
			columns = settings.aoColumns,
			scroll = settings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			visibleColumns = _fnGetColumns( settings, 'bVisible' ),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			i, column, columnIdx;
	
		var styleWidth = table.style.width;
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		// Let plug-ins know that we are doing a recalc, in case they have changed any of the
		// visible columns their own way (e.g. Responsive uses display:none).
		_fnCallbackFire(
			settings,
			null,
			'column-calc',
			{visible: visibleColumns},
			false
		);
	
		// Construct a single row, worst case, table with the widest
		// node in the data, assign any user defined widths, then insert it into
		// the DOM and allow the browser to do all the hard work of calculating
		// table widths
		var tmpTable = $(table.cloneNode())
			.css( 'visibility', 'hidden' )
			.removeAttr( 'id' );
	
		// Clean up the table body
		tmpTable.append('<tbody>')
		var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );
	
		// Clone the table header and footer - we can't use the header / footer
		// from the cloned table, since if scrolling is active, the table's
		// real header and footer are contained in different table tags
		tmpTable
			.append( $(settings.nTHead).clone() )
			.append( $(settings.nTFoot).clone() );
	
		// Remove any assigned widths from the footer (from scrolling)
		tmpTable.find('tfoot th, tfoot td').css('width', '');
	
		// Apply custom sizing to the cloned header
		tmpTable.find('thead th, thead td').each( function () {
			// Get the `width` from the header layout
			var width = _fnColumnsSumWidth( settings, this, true, false );
	
			if ( width ) {
				this.style.width = width;
	
				// For scrollX we need to force the column width otherwise the
				// browser will collapse it. If this width is smaller than the
				// width the column requires, then it will have no effect
				if ( scrollX ) {
					$( this ).append( $('<div/>').css( {
						width: width,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
			else {
				this.style.width = '';
			}
		} );
	
		// Find the widest piece of data for each column and put it into the table
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			columnIdx = visibleColumns[i];
			column = columns[ columnIdx ];
	
			var longest = _fnGetMaxLenString(settings, columnIdx);
			var autoClass = _ext.type.className[column.sType];
			var text = longest + column.sContentPadding;
			var insert = longest.indexOf('<') === -1
				? document.createTextNode(text)
				: text
			
			$('<td/>')
				.addClass(autoClass)
				.addClass(column.sClass)
				.append(insert)
				.appendTo(tr);
		}
	
		// Tidy the temporary table - remove name attributes so there aren't
		// duplicated in the dom (radio elements for example)
		$('[name]', tmpTable).removeAttr('name');
	
		// Table has been built, attach to the document so we can work with it.
		// A holding element is used, positioned at the top of the container
		// with minimal height, so it has no effect on if the container scrolls
		// or not. Otherwise it might trigger scrolling when it actually isn't
		// needed
		var holder = $('<div/>').css( scrollX || scrollY ?
				{
					position: 'absolute',
					top: 0,
					left: 0,
					height: 1,
					right: 0,
					overflow: 'hidden'
				} :
				{}
			)
			.append( tmpTable )
			.appendTo( tableContainer );
	
		// When scrolling (X or Y) we want to set the width of the table as 
		// appropriate. However, when not scrolling leave the table width as it
		// is. This results in slightly different, but I think correct behaviour
		if ( scrollX && scrollXInner ) {
			tmpTable.width( scrollXInner );
		}
		else if ( scrollX ) {
			tmpTable.css( 'width', 'auto' );
			tmpTable.removeAttr('width');
	
			// If there is no width attribute or style, then allow the table to
			// collapse
			if ( tmpTable.width() < tableContainer.clientWidth && tableWidthAttr ) {
				tmpTable.width( tableContainer.clientWidth );
			}
		}
		else if ( scrollY ) {
			tmpTable.width( tableContainer.clientWidth );
		}
		else if ( tableWidthAttr ) {
			tmpTable.width( tableWidthAttr );
		}
	
		// Get the width of each column in the constructed table
		var total = 0;
		var bodyCells = tmpTable.find('tbody tr').eq(0).children();
	
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			// Use getBounding for sub-pixel accuracy, which we then want to round up!
			var bounding = bodyCells[i].getBoundingClientRect().width;
	
			// Total is tracked to remove any sub-pixel errors as the outerWidth
			// of the table might not equal the total given here
			total += bounding;
	
			// Width for each column to use
			columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding );
		}
	
		table.style.width = _fnStringToCss( total );
	
		// Finished with the table - ditch it
		holder.remove();
	
		// If there is a width attr, we want to attach an event listener which
		// allows the table sizing to automatically adjust when the window is
		// resized. Use the width attr rather than CSS, since we can't know if the
		// CSS is a relative value or absolute - DOM read is always px.
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! settings._reszEvt ) {
			var bindResize = function () {
				$(window).on('resize.DT-'+settings.sInstance, DataTable.util.throttle( function () {
					if (! settings.bDestroying) {
						_fnAdjustColumnSizing( settings );
					}
				} ) );
			};
	
			bindResize();
	
			settings._reszEvt = true;
		}
	}
	
	
	/**
	 * Get the maximum strlen for each data column
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {string} string of the max length
	 *  @memberof DataTable#oApi
	 */
	function _fnGetMaxLenString( settings, colIdx )
	{
		var column = settings.aoColumns[colIdx];
	
		if (! column.maxLenString) {
			var s, max='', maxLen = -1;
		
			for ( var i=0, ien=settings.aiDisplayMaster.length ; i<ien ; i++ ) {
				var rowIdx = settings.aiDisplayMaster[i];
				var data = _fnGetRowDisplay(settings, rowIdx)[colIdx];
	
				var cellString = data && typeof data === 'object' && data.nodeType
					? data.innerHTML
					: data+'';
	
				// Remove id / name attributes from elements so they
				// don't interfere with existing elements
				cellString = cellString
					.replace(/id=".*?"/g, '')
					.replace(/name=".*?"/g, '');
	
				s = _stripHtml(cellString)
					.replace( /&nbsp;/g, ' ' );
		
				if ( s.length > maxLen ) {
					// We want the HTML in the string, but the length that
					// is important is the stripped string
					max = cellString;
					maxLen = s.length;
				}
			}
	
			column.maxLenString = max;
		}
	
		return column.maxLenString;
	}
	
	
	/**
	 * Append a CSS unit (only if required) to a string
	 *  @param {string} value to css-ify
	 *  @returns {string} value with css unit
	 *  @memberof DataTable#oApi
	 */
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	/**
	 * Re-insert the `col` elements for current visibility
	 *
	 * @param {*} settings DT settings
	 */
	function _colGroup( settings ) {
		var cols = settings.aoColumns;
	
		settings.colgroup.empty();
	
		for (i=0 ; i<cols.length ; i++) {
			if (cols[i].bVisible) {
				settings.colgroup.append(cols[i].colEl);
			}
		}
	}
	
	
	function _fnSortInit( settings ) {
		var target = settings.nTHead;
		var headerRows = target.querySelectorAll('tr');
		var legacyTop = settings.bSortCellsTop;
		var notSelector = ':not([data-dt-order="disable"]):not([data-dt-order="icon-only"])';
		
		// Legacy support for `orderCellsTop`
		if (legacyTop === true) {
			target = headerRows[0];
		}
		else if (legacyTop === false) {
			target = headerRows[ headerRows.length - 1 ];
		}
	
		_fnSortAttachListener(
			settings,
			target,
			target === settings.nTHead
				? 'tr'+notSelector+' th'+notSelector+', tr'+notSelector+' td'+notSelector
				: 'th'+notSelector+', td'+notSelector
		);
	
		// Need to resolve the user input array into our internal structure
		var order = [];
		_fnSortResolve( settings, order, settings.aaSorting );
	
		settings.aaSorting = order;
	}
	
	
	function _fnSortAttachListener(settings, node, selector, column, callback) {
		_fnBindAction( node, selector, function (e) {
			var run = false;
			var columns = column === undefined
				? _fnColumnsFromHeader( e.target )
				: [column];
	
			if ( columns.length ) {
				for ( var i=0, ien=columns.length ; i<ien ; i++ ) {
					var ret = _fnSortAdd( settings, columns[i], i, e.shiftKey );
	
					if (ret !== false) {
						run = true;
					}					
	
					// If the first entry is no sort, then subsequent
					// sort columns are ignored
					if (settings.aaSorting.length === 1 && settings.aaSorting[0][1] === '') {
						break;
					}
				}
	
				if (run) {
					_fnProcessingDisplay( settings, true );
	
					// Allow the processing display to show
					setTimeout( function () {
						_fnSort( settings );
						_fnSortDisplay( settings, settings.aiDisplay );
	
						// Sort processing done - redraw has its own processing display
						_fnProcessingDisplay( settings, false );
	
						_fnReDraw( settings, false, false );
	
						if (callback) {
							callback();
						}
					}, 0);
				}
			}
		} );
	}
	
	/**
	 * Sort the display array to match the master's order
	 * @param {*} settings
	 */
	function _fnSortDisplay(settings, display) {
		if (display.length < 2) {
			return;
		}
	
		var master = settings.aiDisplayMaster;
		var masterMap = {};
		var map = {};
		var i;
	
		// Rather than needing an `indexOf` on master array, we can create a map
		for (i=0 ; i<master.length ; i++) {
			masterMap[master[i]] = i;
		}
	
		// And then cache what would be the indexOf fom the display
		for (i=0 ; i<display.length ; i++) {
			map[display[i]] = masterMap[display[i]];
		}
	
		display.sort(function(a, b){
			// Short version of this function is simply `master.indexOf(a) - master.indexOf(b);`
			return map[a] - map[b];
		});
	}
	
	
	function _fnSortResolve (settings, nestedSort, sort) {
		var push = function ( a ) {
			if ($.isPlainObject(a)) {
				if (a.idx !== undefined) {
					// Index based ordering
					nestedSort.push([a.idx, a.dir]);
				}
				else if (a.name) {
					// Name based ordering
					var cols = _pluck( settings.aoColumns, 'sName');
					var idx = cols.indexOf(a.name);
	
					if (idx !== -1) {
						nestedSort.push([idx, a.dir]);
					}
				}
			}
			else {
				// Plain column index and direction pair
				nestedSort.push(a);
			}
		};
	
		if ( $.isPlainObject(sort) ) {
			// Object
			push(sort);
		}
		else if ( sort.length && typeof sort[0] === 'number' ) {
			// 1D array
			push(sort);
		}
		else if ( sort.length ) {
			// 2D array
			for (var z=0; z<sort.length; z++) {
				push(sort[z]); // Object or array
			}
		}
	}
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, k, kLen,
			aSort = [],
			extSort = DataTable.ext.type.order,
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [];
		
		if ( ! settings.oFeatures.bSort ) {
			return aSort;
		}
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( Array.isArray( fixed ) ) {
			_fnSortResolve( settings, nestedSort, fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			_fnSortResolve( settings, nestedSort, fixed.pre );
		}
	
		_fnSortResolve( settings, nestedSort, settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			_fnSortResolve( settings, nestedSort, fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
	
			if ( aoColumns[ srcCol ] ) {
				aDataSort = aoColumns[ srcCol ].aDataSort;
	
				for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
				{
					iCol = aDataSort[k];
					sType = aoColumns[ iCol ].sType || 'string';
	
					if ( nestedSort[i]._idx === undefined ) {
						nestedSort[i]._idx = aoColumns[iCol].asSorting.indexOf(nestedSort[i][1]);
					}
	
					if ( nestedSort[i][1] ) {
						aSort.push( {
							src:       srcCol,
							col:       iCol,
							dir:       nestedSort[i][1],
							index:     nestedSort[i]._idx,
							type:      sType,
							formatter: extSort[ sType+"-pre" ],
							sorter:    extSort[ sType+"-"+nestedSort[i][1] ]
						} );
					}
				}
			}
		}
	
		return aSort;
	}
	
	/**
	 * Change the order of the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSort ( oSettings, col, dir )
	{
		var
			i, ien, iLen,
			aiOrig = [],
			extSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo Can this be moved into a 'data-ready' handler which is called when
		//   data is going to be used in the table?
		_fnColumnTypes( oSettings );
	
		// Allow a specific column to be sorted, which will _not_ alter the display
		// master
		if (col !== undefined) {
			var srcCol = oSettings.aoColumns[col];
			aSort = [{
				src:       col,
				col:       col,
				dir:       dir,
				index:     0,
				type:      srcCol.sType,
				formatter: extSort[ srcCol.sType+"-pre" ],
				sorter:    extSort[ srcCol.sType+"-"+dir ]
			}];
			displayMaster = displayMaster.slice();
		}
		else {
			aSort = _fnSortFlatten( oSettings );
		}
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Reset the initial positions on each pass so we get a stable sort
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ i ] = i;
			}
	
			// If the first sort is desc, then reverse the array to preserve original
			// order, just in reverse
			if (aSort.length && aSort[0].dir === 'desc') {
				aiOrig.reverse();
			}
	
			/* Do the sort - here we want multi-column sorting based on a given data source (column)
			 * and sorting function (from oSort) in a certain direction. It's reasonably complex to
			 * follow on it's own, but this is what we want (example two column sorting):
			 *  fnLocalSorting = function(a,b){
			 *    var test;
			 *    test = oSort['string-asc']('data11', 'data12');
			 *      if (test !== 0)
			 *        return test;
			 *    test = oSort['numeric-desc']('data21', 'data22');
			 *    if (test !== 0)
			 *      return test;
			 *    return oSort['numeric-asc']( aiOrig[a], aiOrig[b] );
			 *  }
			 * Basically we have a test for each sorting column, if the data in that column is equal,
			 * test the next column. If all columns match, then we use a numeric sort on the row
			 * positions in the original data array to provide a stable sort.
			 */
			displayMaster.sort( function ( a, b ) {
				var
					x, y, k, test, sort,
					len=aSort.length,
					dataA = aoData[a]._aSortData,
					dataB = aoData[b]._aSortData;
	
				for ( k=0 ; k<len ; k++ ) {
					sort = aSort[k];
	
					// Data, which may have already been through a `-pre` function
					x = dataA[ sort.col ];
					y = dataB[ sort.col ];
	
					if (sort.sorter) {
						// If there is a custom sorter (`-asc` or `-desc`) for this
						// data type, use it
						test = sort.sorter(x, y);
	
						if ( test !== 0 ) {
							return test;
						}
					}
					else {
						// Otherwise, use generic sorting
						test = x<y ? -1 : x>y ? 1 : 0;
	
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
				}
	
				x = aiOrig[a];
				y = aiOrig[b];
	
				return x<y ? -1 : x>y ? 1 : 0;
			} );
		}
		else if ( aSort.length === 0 ) {
			// Apply index order
			displayMaster.sort(function (x, y) {
				return x<y ? -1 : x>y ? 1 : 0;
			});
		}
	
		if (col === undefined) {
			// Tell the draw function that we have sorted the data
			oSettings.bSorted = true;
	
			_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort] );
		}
	
		return displayMaster;
	}
	
	
	/**
	 * Function to run on user sort request
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {int} addIndex Counter
	 *  @param {boolean} [shift=false] Shift click add
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortAdd ( settings, colIdx, addIndex, shift )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = asSorting.indexOf(a[1]);
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		if ( ! col.bSortable ) {
			return false;
		}
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( (shift || addIndex) && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = _pluck(sorting, '0').indexOf(colIdx);
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else if (shift) {
				// No sort on this column yet, being added by shift click
				// add it as itself
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
			else {
				// No sort on this column yet, being added from a colspan
				// so add with same direction as first column
				sorting.push( [ colIdx, sorting[0][1], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	}
	
	
	/**
	 * Set the sorting classes on table's body, Note: it is safe to call this function
	 * when bSort and bSortClasses are false
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.order.position;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	
	// Get the data to sort a column, be it from cache, fresh (populating the
	// cache), or from a sort formatter
	function _fnSortData( settings, colIdx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ colIdx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, colIdx,
				_fnColumnIndexToVisible( settings, colIdx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
		var data = settings.aoData;
	
		for ( var rowIdx=0 ; rowIdx<data.length ; rowIdx++ ) {
			// Sparse array
			if (! data[rowIdx]) {
				continue;
			}
	
			row = data[rowIdx];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[colIdx] || customSort ) {
				cellData = customSort ?
					customData[rowIdx] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, rowIdx, colIdx, 'sort' );
	
				row._aSortData[ colIdx ] = formatter ?
					formatter( cellData, settings ) :
					cellData;
			}
		}
	}
	
	
	/**
	 * State information for a table
	 *
	 * @param {*} settings
	 * @returns State object
	 */
	function _fnSaveState ( settings )
	{
		if (settings._bLoadingState) {
			return;
		}
	
		/* Store the interesting variables */
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   $.extend( true, [], settings.aaSorting ),
			search:  $.extend({}, settings.oPreviousSearch),
			columns: settings.aoColumns.map( function ( col, i ) {
				return {
					visible: col.bVisible,
					search: $.extend({}, settings.aoPreSearchCols[i])
				};
			} )
		};
	
		settings.oSavedState = state;
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
		
		if ( settings.oFeatures.bStateSave && !settings.bDestroying )
		{
			settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
		}	
	}
	
	
	/**
	 * Attempt to load a saved table state
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oInit DataTables init object so we can override settings
	 *  @param {function} callback Callback to execute when the state has been loaded
	 *  @memberof DataTable#oApi
	 */
	function _fnLoadState ( settings, init, callback )
	{
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var loaded = function(state) {
			_fnImplementState(settings, state, callback);
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			_fnImplementState( settings, state, callback );
		}
		// otherwise, wait for the loaded callback to be executed
	
		return true;
	}
	
	function _fnImplementState ( settings, s, callback) {
		var i, ien;
		var columns = settings.aoColumns;
		settings._bLoadingState = true;
	
		// When StateRestore was introduced the state could now be implemented at any time
		// Not just initialisation. To do this an api instance is required in some places
		var api = settings._bInitComplete ? new DataTable.Api(settings) : null;
	
		if ( ! s || ! s.time ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Reject old data
		var duration = settings.iStateDuration;
		if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Allow custom and plug-in manipulation functions to alter the saved data set and
		// cancelling of loading by returning false
		var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
		if ( abStateLoad.indexOf(false) !== -1 ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Number of columns have changed - all bets are off, no restore of settings
		if ( s.columns && columns.length !== s.columns.length ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Store the saved state so it might be accessed at any time
		settings.oLoadedState = $.extend( true, {}, s );
	
		// This is needed for ColReorder, which has to happen first to allow all
		// the stored indexes to be usable. It is not publicly documented.
		_fnCallbackFire( settings, null, 'stateLoadInit', [settings, s], true );
	
		// Page Length
		if ( s.length !== undefined ) {
			// If already initialised just set the value directly so that the select element is also updated
			if (api) {
				api.page.len(s.length)
			}
			else {
				settings._iDisplayLength   = s.length;
			}
		}
	
		// Restore key features - todo - for 1.11 this needs to be done by
		// subscribed events
		if ( s.start !== undefined ) {
			if(api === null) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			else {
				_fnPageChange(settings, s.start/settings._iDisplayLength);
			}
		}
	
		// Order
		if ( s.order !== undefined ) {
			settings.aaSorting = [];
			$.each( s.order, function ( i, col ) {
				settings.aaSorting.push( col[0] >= columns.length ?
					[ 0, col[1] ] :
					col
				);
			} );
		}
	
		// Search
		if ( s.search !== undefined ) {
			$.extend( settings.oPreviousSearch, s.search );
		}
	
		// Columns
		if ( s.columns ) {
			for ( i=0, ien=s.columns.length ; i<ien ; i++ ) {
				var col = s.columns[i];
	
				// Visibility
				if ( col.visible !== undefined ) {
					// If the api is defined, the table has been initialised so we need to use it rather than internal settings
					if (api) {
						// Don't redraw the columns on every iteration of this loop, we will do this at the end instead
						api.column(i).visible(col.visible, false);
					}
					else {
						columns[i].bVisible = col.visible;
					}
				}
	
				// Search
				if ( col.search !== undefined ) {
					$.extend( settings.aoPreSearchCols[i], col.search );
				}
			}
			
			// If the api is defined then we need to adjust the columns once the visibility has been changed
			if (api) {
				api.columns.adjust();
			}
		}
	
		settings._bLoadingState = false;
		_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
		callback();
	}
	
	/**
	 * Log an error message
	 *  @param {object} settings dataTables settings object
	 *  @param {int} level log error messages, or display them to the user
	 *  @param {string} msg error message
	 *  @param {int} tn Technical note id to get more information about the error.
	 *  @memberof DataTable#oApi
	 */
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'https://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'dt-error', [ settings, tn, msg ], true );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	
	/**
	 * See if a property is defined on one object, if so assign it to the other object
	 *  @param {object} ret target object
	 *  @param {object} src source object
	 *  @param {string} name property
	 *  @param {string} [mappedName] name to map too - optional, name used if not given
	 *  @memberof DataTable#oApi
	 */
	function _fnMap( ret, src, name, mappedName )
	{
		if ( Array.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( Array.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}
	
	
	/**
	 * Extend objects - very similar to jQuery.extend, but deep copy objects, and
	 * shallow copy arrays. The reason we need to do this, is that we don't want to
	 * deep copy array init values (such as aaSorting) since the dev wouldn't be
	 * able to override them, but we do want to deep copy arrays.
	 *  @param {object} out Object to extend
	 *  @param {object} extender Object from which the properties will be applied to
	 *      out
	 *  @param {boolean} breakRefs If true, then arrays will be sliced to take an
	 *      independent copy with the exception of the `data` or `aaData` parameters
	 *      if they are present. This is so you can pass in a collection to
	 *      DataTables and have that used as your data source without breaking the
	 *      references
	 *  @returns {object} out Reference, just for convenience - out === the return.
	 *  @memberof DataTable#oApi
	 *  @todo This doesn't take account of arrays inside the deep copied objects.
	 */
	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( Object.prototype.hasOwnProperty.call(extender, prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && Array.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	
	/**
	 * Bind an event handers to allow a click or return key to activate the callback.
	 * This is good for accessibility since a return on the keyboard will have the
	 * same effect as a click, if the element has focus.
	 *  @param {element} n Element to bind the action to
	 *  @param {object|string} selector Selector (for delegated events) or data object
	 *   to pass to the triggered function
	 *  @param {function} fn Callback function for when the event is triggered
	 *  @memberof DataTable#oApi
	 */
	function _fnBindAction( n, selector, fn )
	{
		$(n)
			.on( 'click.DT', selector, function (e) {
				fn(e);
			} )
			.on( 'keypress.DT', selector, function (e){
				if ( e.which === 13 ) {
					e.preventDefault();
					fn(e);
				}
			} )
			.on( 'selectstart.DT', selector, function () {
				// Don't want a double click resulting in text selection
				return false;
			} );
	}
	
	
	/**
	 * Register a callback function. Easily allows a callback function to be added to
	 * an array store of callback functions that can then all be called together.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} store Name of the array storage for the callbacks in oSettings
	 *  @param {function} fn Function to be called back
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackReg( settings, store, fn )
	{
		if ( fn ) {
			settings[store].push(fn);
		}
	}
	
	
	/**
	 * Fire callback functions and trigger events. Note that the loop over the
	 * callback array store is done backwards! Further note that you do not want to
	 * fire off triggers in time sensitive applications (for example cell creation)
	 * as its slow.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} callbackArr Name of the array storage for the callbacks in
	 *      oSettings
	 *  @param {string} eventName Name of the jQuery custom event to trigger. If
	 *      null no trigger is fired
	 *  @param {array} args Array of arguments to pass to the callback function /
	 *      trigger
	 *  @param {boolean} [bubbles] True if the event should bubble
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackFire( settings, callbackArr, eventName, args, bubbles )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = settings[callbackArr].slice().reverse().map( function (val) {
				return val.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null) {
			var e = $.Event( eventName+'.dt' );
			var table = $(settings.nTable);
			
			// Expose the DataTables API on the event object for easy access
			e.dt = settings.api;
	
			table[bubbles ?  'trigger' : 'triggerHandler']( e, args );
	
			// If not yet attached to the document, trigger the event
			// on the body directly to sort of simulate the bubble
			if (bubbles && table.parents('body').length === 0) {
				$('body').trigger( e, args );
			}
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}
	
	
	/**
	 * Detect the data source being used for the table. Used to simplify the code
	 * a little (ajax) and to make it compress a little smaller.
	 *
	 *  @param {object} settings dataTables settings object
	 *  @returns {string} Data source
	 *  @memberof DataTable#oApi
	 */
	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax ) {
			return 'ajax';
		}
		return 'dom';
	}
	
	/**
	 * Common replacement for language strings
	 *
	 * @param {*} settings DT settings object
	 * @param {*} str String with values to replace
	 * @param {*} entries Plural number for _ENTRIES_ - can be undefined
	 * @returns String
	 */
	function _fnMacros ( settings, str, entries )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is
		// used only internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			max        = settings.fnRecordsTotal(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, max ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) ).
			replace(/_ENTRIES_/g, settings.api.i18n('entries', '', entries) ).
			replace(/_ENTRIES-MAX_/g, settings.api.i18n('entries', '', max) ).
			replace(/_ENTRIES-TOTAL_/g, settings.api.i18n('entries', '', vis) );
	}
	
	
	
	/**
	 * Computed structure of the DataTables API, defined by the options passed to
	 * `DataTable.Api.register()` when building the API.
	 *
	 * The structure is built in order to speed creation and extension of the Api
	 * objects since the extensions are effectively pre-parsed.
	 *
	 * The array is an array of objects with the following structure, where this
	 * base array represents the Api prototype base:
	 *
	 *     [
	 *       {
	 *         name:      'data'                -- string   - Property name
	 *         val:       function () {},       -- function - Api method (or undefined if just an object
	 *         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	 *         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	 *       },
	 *       {
	 *         name:     'row'
	 *         val:       {},
	 *         methodExt: [ ... ],
	 *         propExt:   [
	 *           {
	 *             name:      'data'
	 *             val:       function () {},
	 *             methodExt: [ ... ],
	 *             propExt:   [ ... ]
	 *           },
	 *           ...
	 *         ]
	 *       }
	 *     ]
	 *
	 * @type {Array}
	 * @ignore
	 */
	var __apiStruct = [];
	
	
	/**
	 * `Array.prototype` reference.
	 *
	 * @type object
	 * @ignore
	 */
	var __arrayProto = Array.prototype;
	
	
	/**
	 * Abstraction for `context` parameter of the `Api` constructor to allow it to
	 * take several different forms for ease of use.
	 *
	 * Each of the input parameter types will be converted to a DataTables settings
	 * object where possible.
	 *
	 * @param  {string|node|jQuery|object} mixed DataTable identifier. Can be one
	 *   of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 *   * `DataTables.Api` - API instance
	 * @return {array|null} Matching DataTables settings objects. `null` or
	 *   `undefined` is returned if no matching DataTable is found.
	 * @ignore
	 */
	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = _pluck(settings, 'nTable');
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oFeatures ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = tables.indexOf(mixed);
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed).get();
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed.get();
		}
	
		if ( jq ) {
			return settings.filter(function (v, idx) {
				return jq.includes(tables[idx]);
			});
		}
	};
	
	
	/**
	 * DataTables API class - used to control and interface with  one or more
	 * DataTables enhanced tables.
	 *
	 * The API class is heavily based on jQuery, presenting a chainable interface
	 * that you can use to interact with tables. Each instance of the API class has
	 * a "context" - i.e. the tables that it will operate on. This could be a single
	 * table, all tables on a page or a sub-set thereof.
	 *
	 * Additionally the API is designed to allow you to easily work with the data in
	 * the tables, retrieving and manipulating it as required. This is done by
	 * presenting the API class as an array like interface. The contents of the
	 * array depend upon the actions requested by each method (for example
	 * `rows().nodes()` will return an array of nodes, while `rows().data()` will
	 * return an array of objects or arrays depending upon your table's
	 * configuration). The API object has a number of array like methods (`push`,
	 * `pop`, `reverse` etc) as well as additional helper methods (`each`, `pluck`,
	 * `unique` etc) to assist your working with the data held in a table.
	 *
	 * Most methods (those which return an Api instance) are chainable, which means
	 * the return from a method call also has all of the methods available that the
	 * top level object had. For example, these two calls are equivalent:
	 *
	 *     // Not chained
	 *     api.row.add( {...} );
	 *     api.draw();
	 *
	 *     // Chained
	 *     api.row.add( {...} ).draw();
	 *
	 * @class DataTable.Api
	 * @param {array|object|string|jQuery} context DataTable identifier. This is
	 *   used to define which DataTables enhanced tables this API will operate on.
	 *   Can be one of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 * @param {array} [data] Data to initialise the Api instance with.
	 *
	 * @example
	 *   // Direct initialisation during DataTables construction
	 *   var api = $('#example').DataTable();
	 *
	 * @example
	 *   // Initialisation using a DataTables jQuery object
	 *   var api = $('#example').dataTable().api();
	 *
	 * @example
	 *   // Initialisation as a constructor
	 *   var api = new DataTable.Api( 'table.dataTable' );
	 */
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings.push.apply( settings, a );
			}
		};
	
		if ( Array.isArray( context ) ) {
			for ( var i=0, ien=context.length ; i<ien ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = settings.length > 1
			? _unique( settings )
			: settings;
	
		// Initial data
		if ( data ) {
			this.push.apply(this, data);
		}
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
		context: [], // array of table settings objects
	
		count: function ()
		{
			return this.flatten().length;
		},
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
		filter: function ( fn )
		{
			var a = __arrayProto.filter.call( this, fn, this );
	
			return new _Api( this.context, a );
		},
	
		flatten: function ()
		{
			var a = [];
	
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
		get: function ( idx )
		{
			return this[ idx ];
		},
	
		join:    __arrayProto.join,
	
		includes: function ( find ) {
			return this.indexOf( find ) === -1 ? false : true;
		},
	
		indexOf: __arrayProto.indexOf,
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'every' || type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
		lastIndexOf: __arrayProto.lastIndexOf,
	
		length:  0,
	
		map: function ( fn )
		{
			var a = __arrayProto.map.call( this, fn, this );
	
			return new _Api( this.context, a );
		},
	
		pluck: function ( prop )
		{
			var fn = DataTable.util.get(prop);
	
			return this.map( function ( el ) {
				return fn(el);
			} );
		},
	
		pop:     __arrayProto.pop,
	
		push:    __arrayProto.push,
	
		reduce: __arrayProto.reduce,
	
		reduceRight: __arrayProto.reduceRight,
	
		reverse: __arrayProto.reverse,
	
		// Object with rows, columns and opts
		selector: null,
	
		shift:   __arrayProto.shift,
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
		sort:    __arrayProto.sort,
	
		splice:  __arrayProto.splice,
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
		to$: function ()
		{
			return $( this );
		},
	
		toJQuery: function ()
		{
			return $( this );
		},
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this.toArray()) );
		},
	
		unshift: __arrayProto.unshift
	} );
	
	
	function _api_scope( scope, fn, struc ) {
		return function () {
			var ret = fn.apply( scope || this, arguments );
	
			// Method extension
			_Api.extend( ret, ret, struc.methodExt );
			return ret;
		};
	}
	
	function _api_find( src, name ) {
		for ( var i=0, ien=src.length ; i<ien ; i++ ) {
			if ( src[i].name === name ) {
				return src[i];
			}
		}
		return null;
	}
	
	window.__apiStruct = __apiStruct;
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			struct;
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			if (struct.name === '__proto__') {
				continue;
			}
	
			// Value
			obj[ struct.name ] = struct.type === 'function' ?
				_api_scope( scope, struct.val, struct ) :
				struct.type === 'object' ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	//     [
	//       {
	//         name:      'data'                -- string   - Property name
	//         val:       function () {},       -- function - Api method (or undefined if just an object
	//         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	//         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	//       },
	//       {
	//         name:     'row'
	//         val:       {},
	//         methodExt: [ ... ],
	//         propExt:   [
	//           {
	//             name:      'data'
	//             val:       function () {},
	//             methodExt: [ ... ],
	//             propExt:   [ ... ]
	//           },
	//           ...
	//         ]
	//       }
	//     ]
	
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( Array.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = _api_find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   [],
					type:      'object'
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
				src.type = typeof val === 'function' ?
					'function' :
					$.isPlainObject( val ) ?
						'object' :
						'other';
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					Array.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	
	/**
	 * Selector for HTML tables. Apply the given selector to the give array of
	 * DataTables settings objects.
	 *
	 * @param {string|integer} [selector] jQuery selector string or integer
	 * @param  {array} Array of DataTables settings objects to be filtered
	 * @return {array}
	 * @ignore
	 */
	var __table_selector = function ( selector, a )
	{
		if ( Array.isArray(selector) ) {
			var result = [];
	
			selector.forEach(function (sel) {
				var inner = __table_selector(sel, a);
	
				result.push.apply(result, inner);
			});
	
			return result.filter( function (item) {
				return item;
			});
		}
	
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = a.map( function (el) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function () {
				// Need to translate back from the table node to the settings
				var idx = nodes.indexOf(this);
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	
	/**
	 * Context selector for the API's context (i.e. the tables the API instance
	 * refers to.
	 *
	 * @name    DataTable.Api#tables
	 * @param {string|integer} [selector] Selector to pick which tables the iterator
	 *   should operate on. If not given, all tables in the current context are
	 *   used. This can be given as a jQuery selector (for example `':gt(0)'`) to
	 *   select multiple tables or as an integer to select a single table.
	 * @returns {DataTable.Api} Returns a new API instance if a selector is given.
	 */
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector !== undefined && selector !== null ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	// Common methods, combined to reduce size
	[
		['nodes', 'node', 'nTable'],
		['body', 'body', 'nTBody'],
		['header', 'header', 'nTHead'],
		['footer', 'footer', 'nTFoot'],
	].forEach(function (item) {
		_api_registerPlural(
			'tables().' + item[0] + '()',
			'table().' + item[1] + '()' ,
			function () {
				return this.iterator( 'table', function ( ctx ) {
					return ctx[item[2]];
				}, 1 );
			}
		);
	});
	
	// Structure methods
	[
		['header', 'aoHeader'],
		['footer', 'aoFooter'],
	].forEach(function (item) {
		_api_register( 'table().' + item[0] + '.structure()' , function (selector) {
			var indexes = this.columns(selector).indexes().flatten();
			var ctx = this.context[0];
			
			return _fnHeaderLayout(ctx, ctx[item[1]], indexes);
		} );
	})
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	_api_register( 'tables().every()', function ( fn ) {
		var that = this;
	
		return this.iterator('table', function (s, i) {
			fn.call(that.table(i), i);
		});
	});
	
	_api_register( 'caption()', function ( value, side ) {
		var context = this.context;
	
		// Getter - return existing node's content
		if ( value === undefined ) {
			var caption = context[0].captionNode;
	
			return caption && context.length ?
				caption.innerHTML : 
				null;
		}
	
		return this.iterator( 'table', function ( ctx ) {
			var table = $(ctx.nTable);
			var caption = $(ctx.captionNode);
			var container = $(ctx.nTableWrapper);
	
			// Create the node if it doesn't exist yet
			if ( ! caption.length ) {
				caption = $('<caption/>').html( value );
				ctx.captionNode = caption[0];
	
				// If side isn't set, we need to insert into the document to let the
				// CSS decide so we can read it back, otherwise there is no way to
				// know if the CSS would put it top or bottom for scrolling
				if (! side) {
					table.prepend(caption);
	
					side = caption.css('caption-side');
				}
			}
	
			caption.html( value );
	
			if ( side ) {
				caption.css( 'caption-side', side );
				caption[0]._captionSide = side;
			}
	
			if (container.find('div.dataTables_scroll').length) {
				var selector = (side === 'top' ? 'Head' : 'Foot');
	
				container.find('div.dataTables_scroll'+ selector +' table').prepend(caption);
			}
			else {
				table.prepend(caption);
			}
		}, 1 );
	} );
	
	_api_register( 'caption.node()', function () {
		var ctx = this.context;
	
		return ctx.length ? ctx[0].captionNode : null;
	} );
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	
	/**
	 * Get the current page index.
	 *
	 * @return {integer} Current page index (zero based)
	 *//**
	 * Set the current page.
	 *
	 * Note that if you attempt to show a page which does not exist, DataTables will
	 * not throw an error, but rather reset the paging.
	 *
	 * @param {integer|string} action The paging action to take. This can be one of:
	 *  * `integer` - The page index to jump to
	 *  * `string` - An action to take:
	 *    * `first` - Jump to first page.
	 *    * `next` - Jump to the next page
	 *    * `previous` - Jump to previous page
	 *    * `last` - Jump to the last page.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	
	/**
	 * Paging information for the first table in the current context.
	 *
	 * If you require paging information for another table, use the `table()` method
	 * with a suitable selector.
	 *
	 * @return {object} Object with the following properties set:
	 *  * `page` - Current page index (zero based - i.e. the first page is `0`)
	 *  * `pages` - Total number of pages
	 *  * `start` - Display index for the first record shown on the current page
	 *  * `end` - Display index for the last record shown on the current page
	 *  * `length` - Display length (number of records). Note that generally `start
	 *    + length = end`, but this is not always true, for example if there are
	 *    only 2 records to show on the final page, with a length of 10.
	 *  * `recordsTotal` - Full data set length
	 *  * `recordsDisplay` - Data set length once the current filtering criterion
	 *    are applied.
	 */
	_api_register( 'page.info()', function () {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	
	/**
	 * Get the current page length.
	 *
	 * @return {integer} Current page length. Note `-1` indicates that all records
	 *   are to be shown.
	 *//**
	 * Set the current page length.
	 *
	 * @param {integer} Page length to set. Use `-1` to show all records.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, {}, function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnInitComplete( settings );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	/**
	 * Get the JSON response from the last Ajax request that DataTables made to the
	 * server. Note that this returns the JSON from the first table in the current
	 * context.
	 *
	 * @return {object} JSON received from the server.
	 */
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Get the data submitted in the last Ajax request
	 */
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Reload tables from the Ajax data source. Note that this function will
	 * automatically re-draw the table when the remote data has been loaded.
	 *
	 * @param {boolean} [reset=true] Reset (default) or hold the current paging
	 *   position. A full re-sort and re-filter is performed when this method is
	 *   called, which is why the pagination reset is the default action.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	
	/**
	 * Get the current Ajax URL. Note that this returns the URL from the first
	 * table in the current context.
	 *
	 * @return {string} Current Ajax source URL
	 *//**
	 * Set the Ajax URL. Note that this will set the URL for all tables in the
	 * current context.
	 *
	 * @param {string} url URL to set.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return $.isPlainObject( ctx.ajax ) ?
				ctx.ajax.url :
				ctx.ajax;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
		} );
	} );
	
	
	/**
	 * Load data from the newly set Ajax URL. Note that this method is only
	 * available when `ajax.url()` is used to set a URL. Additionally, this method
	 * has the same effect as calling `ajax.reload()` but is provided for
	 * convenience when setting a new URL. Like `ajax.reload()` it will
	 * automatically redraw the table once the remote data has been loaded.
	 *
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			a, i, ien, j, jen,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			// Only split on simple strings - complex expressions will be jQuery selectors
			a = selector[i] && selector[i].split && ! selector[i].match(/[[(:]/) ?
				selector[i].split(',') :
				[ selector[i] ];
	
			for ( j=0, jen=a.length ; j<jen ; j++ ) {
				res = selectFn( typeof a[j] === 'string' ? (a[j]).trim() : a[j] );
	
				// Remove empty items
				res = res.filter( function (item) {
					return item !== null && item !== undefined;
				});
	
				if ( res && res.length ) {
					out = out.concat( res );
				}
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	// Reduce the API instance to the first item found
	var _selector_first = function ( old )
	{
		let inst = new _Api(old.context[0]);
	
		// Use a push rather than passing to the constructor, since it will
		// merge arrays down automatically, which isn't what is wanted here
		if (old.length) {
			inst.push( old[0] );
		}
	
		inst.selector = old.selector;
	
		// Limit to a single row / column / cell
		if (inst.length && inst[0].length > 1) {
			inst[0].splice(1);
		}
	
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
	
		if ( page == 'current' ) {
			// Current page implies that order=current and filter=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				displayMaster.forEach(function (item) {
					if (! Object.prototype.hasOwnProperty.call(displayFilteredMap, item)) {
						a.push(item);
					}
				});
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if (! settings.aoData[i]) {
					continue;
				}
	
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = displayFiltered.indexOf(i);
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
		else if ( typeof order === 'number' ) {
			// Order the rows by the given column
			var ordered = _fnSort(settings, order, 'asc');
	
			if (search === 'none') {
				a = ordered;
			}
			else { // applied | removed
				for (i=0; i<ordered.length; i++) {
					tmp = displayFiltered.indexOf(ordered[i]);
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( ordered[i] );
					}
				}
			}
		}
	
		return a;
	};
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Rows
	 *
	 * {}          - no selector - use all available rows
	 * {integer}   - row aoData index
	 * {node}      - TR node
	 * {string}    - jQuery selector to apply to the TR elements
	 * {array}     - jQuery array of nodes, or simply an array of TR nodes
	 *
	 */
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && rows.indexOf(selInt) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return rows.map( function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel.parentNode ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			// ID selector. Want to always be able to select rows by id, regardless
			// of if the tr element has been created or not, so can't rely upon
			// jQuery here - hence a custom implementation. This does not match
			// Sizzle's fast selector or HTML4 - in HTML5 the ID can be anything,
			// but to select it using a CSS selector engine (like Sizzle or
			// querySelect) it would need to need to be escaped for some characters.
			// DataTables simplifies this for row selectors since you can select
			// only a row. A # indicates an id any anything that follows is the id -
			// unescaped.
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);
	
			// Selector - jQuery selector string, array of nodes or jQuery object/
			// As jQuery's .filter() allows jQuery objects to be passed in filter,
			// it also allows arrays, so this will cope with all three options
			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		var matched = _selector_run( 'row', selector, run, settings, opts );
	
		if (opts.order === 'current' || opts.order === 'applied') {
			_fnSortDisplay(settings, matched);
		}
	
		return matched;
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		this.iterator( 'row', function ( settings, row ) {
			var data = settings.aoData;
			var rowData = data[ row ];
	
			// Delete from the display arrays
			var idx = settings.aiDisplayMaster.indexOf(row);
			if (idx !== -1) {
				settings.aiDisplayMaster.splice(idx, 1);
			}
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
	
			data[row] = null;
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		modRows.push.apply(modRows, newRows);
	
		return modRows;
	} );
	
	
	
	
	
	/**
	 *
	 */
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length && this[0].length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( Array.isArray( data ) && row.nTr && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		if (ctx.length && this.length && this[0].length) {
			var row = ctx[0].aoData[ this[0] ];
	
			if (row && row.nTr) {
				return row.nTr;
			}
		}
	
		return null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	$(document).on('plugin-init.dt', function (e, context) {
		var api = new _Api( context );
	
		api.on( 'stateSaveParams.DT', function ( e, settings, d ) {
			// This could be more compact with the API, but it is a lot faster as a simple
			// internal loop
			var idFn = settings.rowIdFn;
			var rows = settings.aiDisplayMaster;
			var ids = [];
	
			for (var i=0 ; i<rows.length ; i++) {
				var rowIdx = rows[i];
				var data = settings.aoData[rowIdx];
	
				if (data._detailsShow) {
					ids.push( '#' + idFn(data._aData) );
				}
			}
	
			d.childRows = ids;
		});
	
		// For future state loads (e.g. with StateRestore)
		api.on( 'stateLoaded.DT', function (e, settings, state) {
			__details_state_load( api, state );
		});
	
		// And the initial load state
		__details_state_load( api, api.state.loaded() );
	});
	
	var __details_state_load = function (api, state)
	{
		if ( state && state.childRows ) {
			api
				.rows( state.childRows.map(function (id) {
					// Escape any `:` characters from the row id. Accounts for
					// already escaped characters.
					return id.replace(/([^:\\]*(?:\\.[^:\\]*)*):/g, "$1\\:");
				}) )
				.every( function () {
					_fnCallbackFire( api.settings()[0], null, 'requestChild', [ this ] )
				});
		}
	}
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( Array.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				r.setAttribute( 'data-dt-row', row.idx );
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td></td></tr>')
					.attr( 'data-dt-row', row.idx )
					.addClass( k );
				
				$('td', created)
					.addClass( k )
					.html( r )[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	// Make state saving of child row details async to allow them to be batch processed
	var __details_state = DataTable.util.throttle(
		function (ctx) {
			_fnSaveState( ctx[0] )
		},
		500
	);
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
				$( row.nTr ).removeClass( 'dt-hasChild' );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
					$( row.nTr ).addClass( 'dt-hasChild' );
				}
				else {
					row._details.detach();
					$( row.nTr ).removeClass( 'dt-hasChild' );
				}
	
				_fnCallbackFire( ctx[0], null, 'childRow', [ show, api.row( api[0] ) ] )
	
				__details_events( ctx[0] );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-sizing'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row && row._details ) {
						row._details.each(function () {
							var el = $(this).children('td');
	
							if (el.length == 1) {
								el.attr('colspan', visible);
							}
						});
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i] && data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length && ctx[0].aoData[ this[0] ]
				? ctx[0].aoData[ this[0] ]._details
				: undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length && ctx[0].aoData[ this[0] ] ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Columns
	 *
	 * {integer}           - column index (>=0 count from left, <0 count from right)
	 * "{integer}:visIdx"  - visible column index (i.e. translate to column index)  (>=0 count from left, <0 count from right)
	 * "{integer}:visible" - alias for {integer}:visIdx  (>=0 count from left, <0 count from right)
	 * "{string}:name"     - column name
	 * "{string}"          - jQuery selector on column header nodes
	 *
	 */
	
	// can be an array of these items, comma separated list, or an array of comma
	// separated lists
	
	var __re_column_selector = /^([^:]+)?:(name|title|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows, type ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column, type ) );
		}
		return a;
	};
	
	
	var __column_header = function ( settings, column, row ) {
		var header = settings.aoHeader;
		var target = row !== undefined
			? row
			: settings.bSortCellsTop // legacy support
				? 0
				: header.length - 1;
	
		return header[target][column].cell;
	};
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names = _pluck( columns, 'sName' ),
			titles = _pluck( columns, 'sTitle' ),
			cells = DataTable.util.get('[].[].cell')(settings.aoHeader),
			nodes = _unique( _flatten([], cells) );
		
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return columns.map(function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							__column_header( settings, idx )
						) ? idx : null;
				});
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						if (match[1]) {
							var idx = parseInt( match[1], 10 );
							// Visible index given, convert to column index
							if ( idx < 0 ) {
								// Counting from the right
								var visColumns = columns.map( function (col,i) {
									return col.bVisible ? i : null;
								} );
								return [ visColumns[ visColumns.length + idx ] ];
							}
							// Counting from the left
							return [ _fnVisibleToColumnIndex( settings, idx ) ];
						}
						
						// `:visible` on its own
						return columns.map( function (col, i) {
							return col.bVisible ? i : null;
						} );
	
					case 'name':
						// match by name. `names` is column index complete and in order
						return names.map( function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					case 'title':
						// match by column title
						return titles.map( function (title, i) {
							return title === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return _fnColumnsFromHeader( this ); // `nodes` is column index complete and in order
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		return _selector_run( 'column', selector, run, settings, opts );
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return false;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = _pluck(cols, 'bVisible').indexOf(true, column+1);
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				if (data[i]) {
					tr = data[i].nTr;
					cells = data[i].anCells;
	
					if ( tr ) {
						// insertBefore can act like appendChild if 2nd arg is null
						tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
					}
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
	
		_colGroup(settings);
		
		return true;
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( row ) {
		return this.iterator( 'column', function (settings, column) {
			return __column_header(settings, column, row);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( row ) {
		return this.iterator( 'column', function ( settings, column ) {
			var footer = settings.aoFooter;
	
			if (! footer.length) {
				return null;
			}
	
			return settings.aoFooter[row !== undefined ? row : 0][column].cell;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().render()', 'column().render()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return __columnData( settings, column, i, j, rows, type );
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().init()', 'column().init()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column];
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().titles()', 'column().title()', function (title, row) {
		return this.iterator( 'column', function ( settings, column ) {
			// Argument shifting
			if (typeof title === 'number') {
				row = title;
				title = undefined;
			}
	
			var span = $('span.dt-column-title', this.column(column).header(row));
	
			if (title !== undefined) {
				span.html(title);
				return this;
			}
	
			return span.html();
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().types()', 'column().type()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			var type = settings.aoColumns[column].sType;
	
			// If the type was invalidated, then resolve it. This actually does
			// all columns at the moment. Would only happen once if getting all
			// column's data types.
			if (! type) {
				_fnColumnTypes(settings);
			}
	
			return type;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var that = this;
		var changed = [];
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			
			if (__setColumnVis( settings, column, vis )) {
				changed.push(column);
			}
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			this.iterator( 'table', function ( settings ) {
				// Redraw the header after changes
				_fnDrawHead( settings, settings.aoHeader );
				_fnDrawHead( settings, settings.aoFooter );
		
				// Update colspan for no records display. Child rows and extensions will use their own
				// listeners to do this - only need to update the empty table item here
				if ( ! settings.aiDisplay.length ) {
					$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
				}
		
				_fnSaveState( settings );
	
				// Second loop once the first is done for events
				that.iterator( 'column', function ( settings, column ) {
					if (changed.includes(column)) {
						_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
					}
				} );
	
				if ( changed.length && (calc === undefined || calc) ) {
					that.columns.adjust();
				}
			});
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().widths()', 'column().width()', function () {
		// Injects a fake row into the table for just a moment so the widths can
		// be read, regardless of colspan in the header and rows being present in
		// the body
		var columns = this.columns(':visible').count();
		var row = $('<tr>').html('<td>' + Array(columns).join('</td><td>') + '</td>');
	
		$(this.table().body()).append(row);
	
		var widths = row.children().map(function () {
			return $(this).outerWidth();
		});
	
		row.remove();
		
		return this.iterator( 'column', function ( settings, column ) {
			var visIdx = _fnColumnIndexToVisible( settings, column );
	
			return visIdx !== null ? widths[visIdx] : 0;
		}, 1);
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $(_flatten( [], cells ));
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && rows.indexOf(s.row) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
					};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// The default built in options need to apply to row and columns
		var internalOpts = opts ? {
			page: opts.page,
			order: opts.order,
			search: opts.search
		} : {};
	
		// Row + column selector
		var columns = this.columns( columnSelector, internalOpts );
		var rows = this.rows( rowSelector, internalOpts );
		var i, ien, j, jen;
	
		var cellsNoOpts = this.iterator( 'table', function ( settings, idx ) {
			var a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
	
			return a;
		}, 1 );
	
		// There is currently only one extension which uses a cell selector extension
		// It is a _major_ performance drag to run this if it isn't needed, so this is
		// an extension specific check at the moment
		var cells = opts && opts.selected ?
			this.cells( cellsNoOpts, opts ) :
			cellsNoOpts;
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	
	
	/**
	 * Get current ordering (sorting) that has been applied to the table.
	 *
	 * @returns {array} 2D array containing the sorting information for the first
	 *   table in the current context. Each element in the parent array represents
	 *   a column being sorted upon (i.e. multi-sorting with two columns would have
	 *   2 inner arrays). The inner arrays may have 2 or 3 elements. The first is
	 *   the column index that the sorting condition applies to, the second is the
	 *   direction of the sort (`desc` or `asc`) and, optionally, the third is the
	 *   index of the sorting order from the `column.sorting` initialisation array.
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {integer} order Column index to sort upon.
	 * @param {string} direction Direction of the sort to be applied (`asc` or `desc`)
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 1D array of sorting information to be applied.
	 * @param {array} [...] Optional additional sorting conditions
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 2D array of sorting information to be applied.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
		var args = Array.prototype.slice.call( arguments );
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( args.length > 1 ) {
			// Arguments passed in (list of 1D arrays)
			order = args;
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSorting = Array.isArray(order) ? order.slice() : order;
		} );
	} );
	
	
	/**
	 * Attach a sort listener to an element for a given column
	 *
	 * @param {node|jQuery|string} node Identifier for the element(s) to attach the
	 *   listener to. This can take the form of a single DOM node, a jQuery
	 *   collection of nodes or a jQuery selector which will identify the node(s).
	 * @param {integer} column the column that a click on this node will sort on
	 * @param {function} [callback] callback function when sort is run
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener(settings, node, {}, column, callback);
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return Array.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		if ( ! dir ) {
			return this.iterator( 'column', function ( settings, idx ) {
				var sort = _fnSortFlatten( settings );
	
				for ( var i=0, ien=sort.length ; i<ien ; i++ ) {
					if ( sort[i].col === idx ) {
						return sort[i].dir;
					}
				}
	
				return null;
			}, 1 );
		}
		else {
			return this.iterator( 'table', function ( settings, i ) {
				settings.aaSorting = that[i].map( function (col) {
					return [ col, dir ];
				} );
			} );
		}
	} );
	
	_api_registerPlural('columns().orderable()', 'column().orderable()', function ( directions ) {
		return this.iterator( 'column', function ( settings, idx ) {
			var col = settings.aoColumns[idx];
	
			return directions ?
				col.asSorting :
				col.bSortable;
		}, 1 );
	} );
	
	
	_api_register( 'processing()', function ( show ) {
		return this.iterator( 'table', function ( ctx ) {
			_fnProcessingDisplay( ctx, show );
		} );
	} );
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.search :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			if (typeof regex === 'object') {
				// New style options to pass to the search builder
				_fnFilterComplete( settings, $.extend( settings.oPreviousSearch, regex, {
					search: input
				} ) );
			}
			else {
				// Compat for the old options
				_fnFilterComplete( settings, $.extend( settings.oPreviousSearch, {
					search: input,
					regex:  regex === null ? false : regex,
					smart:  smart === null ? true  : smart,
					caseInsensitive: caseInsen === null ? true : caseInsen
				} ) );
			}
		} );
	} );
	
	_api_register( 'search.fixed()', function ( name, search ) {
		var ret = this.iterator( true, 'table', function ( settings ) {
			var fixed = settings.searchFixed;
	
			if (! name) {
				return Object.keys(fixed)
			}
			else if (search === undefined) {
				return fixed[name];
			}
			else if (search === null) {
				delete fixed[name];
			}
			else {
				fixed[name] = search;
			}
	
			return this;
		} );
	
		return name !== undefined && search === undefined
			? ret[0]
			: ret;
	} );
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].search;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				if (typeof regex === 'object') {
					// New style options to pass to the search builder
					$.extend( preSearch[ column ], regex, {
						search: input
					} );
				}
				else {
					// Old style (with not all options available)
					$.extend( preSearch[ column ], {
						search: input,
						regex:  regex === null ? false : regex,
						smart:  smart === null ? true  : smart,
						caseInsensitive: caseInsen === null ? true : caseInsen
					} );
				}
	
				_fnFilterComplete( settings, settings.oPreviousSearch );
			} );
		}
	);
	
	_api_register([
			'columns().search.fixed()',
			'column().search.fixed()'
		],
		function ( name, search ) {
			var ret = this.iterator( true, 'column', function ( settings, colIdx ) {
				var fixed = settings.aoColumns[colIdx].searchFixed;
	
				if (! name) {
					return Object.keys(fixed)
				}
				else if (search === undefined) {
					return fixed[name];
				}
				else if (search === null) {
					delete fixed[name];
				}
				else {
					fixed[name] = search;
				}
	
				return this;
			} );
	
			return name !== undefined && search === undefined
				? ret[0]
				: ret;
		}
	);
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function ( set, ignoreTime ) {
		// getter
		if ( ! set ) {
			return this.context.length ?
				this.context[0].oSavedState :
				null;
		}
	
		var setMutate = $.extend( true, {}, set );
	
		// setter
		return this.iterator( 'table', function ( settings ) {
			if ( ignoreTime !== false ) {
				setMutate.time = +new Date() + 100;
			}
	
			_fnImplementState( settings, setMutate, function(){} );
		} );
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	/**
	 * Set the jQuery or window object to be used by DataTables
	 *
	 * @param {*} module Library / container object
	 * @param {string} [type] Library or container type `lib`, `win` or `datetime`.
	 *   If not provided, automatic detection is attempted.
	 */
	DataTable.use = function (module, type) {
		if (type === 'lib' || module.fn) {
			$ = module;
		}
		else if (type == 'win' || module.document) {
			window = module;
			document = module.document;
		}
		else if (type === 'datetime' || module.type === 'DateTime') {
			DataTable.DateTime = module;
		}
	}
	
	/**
	 * CommonJS factory function pass through. This will check if the arguments
	 * given are a window object or a jQuery object. If so they are set
	 * accordingly.
	 * @param {*} root Window
	 * @param {*} jq jQUery
	 * @returns {boolean} Indicator
	 */
	DataTable.factory = function (root, jq) {
		var is = false;
	
		// Test if the first parameter is a window object
		if (root && root.document) {
			window = root;
			document = root.document;
		}
	
		// Test if the second parameter is a jQuery object
		if (jq && jq.fn && jq.fn.jquery) {
			$ = jq;
			is = true;
		}
	
		return is;
	}
	
	/**
	 * Provide a common method for plug-ins to check the version of DataTables being
	 * used, in order to ensure compatibility.
	 *
	 *  @param {string} version Version string to check for, in the format "X.Y.Z".
	 *    Note that the formats "X" and "X.Y" are also acceptable.
	 *  @param {string} [version2=current DataTables version] As above, but optional.
	 *   If not given the current DataTables version will be used.
	 *  @returns {boolean} true if this version of DataTables is greater or equal to
	 *    the required version, or false if this version of DataTales is not
	 *    suitable
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    alert( $.fn.dataTable.versionCheck( '1.9.0' ) );
	 */
	DataTable.versionCheck = function( version, version2 )
	{
		var aThis = version2 ?
			version2.split('.') :
			DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	/**
	 * Check if a `<table>` node is a DataTable table already or not.
	 *
	 *  @param {node|jquery|string} table Table node, jQuery object or jQuery
	 *      selector for the table to test. Note that if more than more than one
	 *      table is passed on, only the first will be checked
	 *  @returns {boolean} true the table given is a DataTable, or false otherwise
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    if ( ! $.fn.DataTable.isDataTable( '#example' ) ) {
	 *      $('#example').dataTable();
	 *    }
	 */
	DataTable.isDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	
	/**
	 * Get all DataTable tables that have been initialised - optionally you can
	 * select to get only currently visible tables.
	 *
	 *  @param {boolean} [visible=false] Flag to indicate if you want all (default)
	 *    or visible tables only.
	 *  @returns {array} Array of `table` nodes (not DataTable instances) which are
	 *    DataTables
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    $.each( $.fn.dataTable.tables(true), function () {
	 *      $(table).DataTable().columns.adjust();
	 *    } );
	 */
	DataTable.tables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = DataTable.settings
			.filter( function (o) {
				return !visible || (visible && $(o.nTable).is(':visible')) 
					? true
					: false;
			} )
			.map( function (o) {
				return o.nTable;
			});
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	/**
	 * Convert from camel case parameters to Hungarian notation. This is made public
	 * for the extensions to provide the same ability as DataTables core to accept
	 * either the 1.9 style Hungarian notation, or the 1.10+ style camelCase
	 * parameters.
	 *
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 */
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	
	/**
	 *
	 */
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = args[0].split( /\s/ ).map( function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'error()', function (msg) {
		return this.iterator( 'table', function ( settings ) {
			_fnLog( settings, 0, msg );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'trigger()', function ( name, args, bubbles ) {
		return this.iterator( 'table', function ( settings ) {
			return _fnCallbackFire( settings, null, name, args, bubbles );
		} ).flatten();
	} );
	
	
	_api_register( 'ready()', function ( fn ) {
		var ctx = this.context;
	
		// Get status of first table
		if (! fn) {
			return ctx.length
				? (ctx[0]._bInitComplete || false)
				: null;
		}
	
		// Function to run either once the table becomes ready or
		// immediately if it is already ready.
		return this.tables().every(function () {
			if (this.context[0]._bInitComplete) {
				fn.call(this);
			}
			else {
				this.on('init', function () {
					fn.call(this);
				});
			}
		} );
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = settings.aoData.map( function (r) { return r ? r.nTr : null; } );
			var orderClasses = classes.order;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings], true );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			settings.colgroup.remove();
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$('th, td', thead)
				.removeClass(
					orderClasses.canAsc + ' ' +
					orderClasses.canDesc + ' ' +
					orderClasses.isAsc + ' ' +
					orderClasses.isDesc
				)
				.css('width', '');
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			var orig = settings.nTableWrapper.parentNode;
			var insertBefore = settings.nTableWrapper.nextSibling;
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, insertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.table );
			}
	
			/* Remove the settings object from the settings array */
			var idx = DataTable.settings.indexOf(settings);
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
			var inst;
			var counter = 0;
	
			return this.iterator( 'every', function ( settings, selectedIdx, tableIdx ) {
				inst = api[ type ](selectedIdx, opts);
	
				if (type === 'cell') {
					fn.call(inst, inst[0][0].row, inst[0][0].column, tableIdx, counter);
				}
				else {
					fn.call(inst, selectedIdx, tableIdx, counter);
				}
	
				counter++;
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( $.isPlainObject( resolved ) ) {
			resolved = plural !== undefined && resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return typeof resolved === 'string'
			? resolved.replace( '%d', plural ) // nb: plural might be undefined,
			: resolved;
	} );
	
	/**
	 * Version string for plug-ins to check compatibility. Allowed format is
	 * `a.b.c-d` where: a:int, b:int, c:int, d:string(dev|beta|alpha). `d` is used
	 * only for non-release builds. See https://semver.org/ for more information.
	 *  @member
	 *  @type string
	 *  @default Version number
	 */
	DataTable.version = "2.0.8";
	
	/**
	 * Private data store, containing all of the settings objects that are
	 * created for the tables on a given page.
	 *
	 * Note that the `DataTable.settings` object is aliased to
	 * `jQuery.fn.dataTableExt` through which it may be accessed and
	 * manipulated, or `jQuery.fn.dataTable.settings`.
	 *  @member
	 *  @type array
	 *  @default []
	 *  @private
	 */
	DataTable.settings = [];
	
	/**
	 * Object models container, for the various models that DataTables has
	 * available to it. These models define the objects that are used to hold
	 * the active state and configuration of the table.
	 *  @namespace
	 */
	DataTable.models = {};
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * search information for the global filter and individual column filters.
	 *  @namespace
	 */
	DataTable.models.oSearch = {
		/**
		 * Flag to indicate if the filtering should be case insensitive or not
		 */
		"caseInsensitive": true,
	
		/**
		 * Applied search term
		 */
		"search": "",
	
		/**
		 * Flag to indicate if the search term should be interpreted as a
		 * regular expression (true) or not (false) and therefore and special
		 * regex characters escaped.
		 */
		"regex": false,
	
		/**
		 * Flag to indicate if DataTables is to use its smart filtering or not.
		 */
		"smart": true,
	
		/**
		 * Flag to indicate if DataTables should only trigger a search when
		 * the return key is pressed.
		 */
		"return": false
	};
	
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * each individual row. This is the object format used for the settings
	 * aoData array.
	 *  @namespace
	 */
	DataTable.models.oRow = {
		/**
		 * TR element for the row
		 */
		"nTr": null,
	
		/**
		 * Array of TD elements for each row. This is null until the row has been
		 * created.
		 */
		"anCells": null,
	
		/**
		 * Data object from the original data source for the row. This is either
		 * an array if using the traditional form of DataTables, or an object if
		 * using mData options. The exact type will depend on the passed in
		 * data from the data source, or will be an array if using DOM a data
		 * source.
		 */
		"_aData": [],
	
		/**
		 * Sorting data cache - this array is ostensibly the same length as the
		 * number of columns (although each index is generated only as it is
		 * needed), and holds the data that is used for sorting each column in the
		 * row. We do this cache generation at the start of the sort in order that
		 * the formatting of the sort data need be done only once for each cell
		 * per sort. This array should not be read from or written to by anything
		 * other than the master sorting methods.
		 */
		"_aSortData": null,
	
		/**
		 * Per cell filtering data cache. As per the sort data cache, used to
		 * increase the performance of the filtering in DataTables
		 */
		"_aFilterData": null,
	
		/**
		 * Filtering data cache. This is the same as the cell filtering cache, but
		 * in this case a string rather than an array. This is easily computed with
		 * a join on `_aFilterData`, but is provided as a cache so the join isn't
		 * needed on every search (memory traded for performance)
		 */
		"_sFilterRow": null,
	
		/**
		 * Denote if the original data source was from the DOM, or the data source
		 * object. This is used for invalidating data, so DataTables can
		 * automatically read data from the original source, unless uninstructed
		 * otherwise.
		 */
		"src": null,
	
		/**
		 * Index in the aoData array. This saves an indexOf lookup when we have the
		 * object, but want to know the index
		 */
		"idx": -1,
	
		/**
		 * Cached display value
		 */
		displayData: null
	};
	
	
	/**
	 * Template object for the column information object in DataTables. This object
	 * is held in the settings aoColumns array and contains all the information that
	 * DataTables needs about each individual column.
	 *
	 * Note that this object is related to {@link DataTable.defaults.column}
	 * but this one is the internal data store for DataTables's cache of columns.
	 * It should NOT be manipulated outside of DataTables. Any configuration should
	 * be done through the initialisation options.
	 *  @namespace
	 */
	DataTable.models.oColumn = {
		/**
		 * Column index.
		 */
		"idx": null,
	
		/**
		 * A list of the columns that sorting should occur on when this column
		 * is sorted. That this property is an array allows multi-column sorting
		 * to be defined for a column (for example first name / last name columns
		 * would benefit from this). The values are integers pointing to the
		 * columns to be sorted on (typically it will be a single integer pointing
		 * at itself, but that doesn't need to be the case).
		 */
		"aDataSort": null,
	
		/**
		 * Define the sorting directions that are applied to the column, in sequence
		 * as the column is repeatedly sorted upon - i.e. the first value is used
		 * as the sorting direction when the column if first sorted (clicked on).
		 * Sort it again (click again) and it will move on to the next index.
		 * Repeat until loop.
		 */
		"asSorting": null,
	
		/**
		 * Flag to indicate if the column is searchable, and thus should be included
		 * in the filtering or not.
		 */
		"bSearchable": null,
	
		/**
		 * Flag to indicate if the column is sortable or not.
		 */
		"bSortable": null,
	
		/**
		 * Flag to indicate if the column is currently visible in the table or not
		 */
		"bVisible": null,
	
		/**
		 * Store for manual type assignment using the `column.type` option. This
		 * is held in store so we can manipulate the column's `sType` property.
		 */
		"_sManualType": null,
	
		/**
		 * Flag to indicate if HTML5 data attributes should be used as the data
		 * source for filtering or sorting. True is either are.
		 */
		"_bAttrSrc": false,
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 */
		"fnCreatedCell": null,
	
		/**
		 * Function to get data from a cell in a column. You should <b>never</b>
		 * access data directly through _aData internally in DataTables - always use
		 * the method attached to this property. It allows mData to function as
		 * required. This function is automatically assigned by the column
		 * initialisation method
		 */
		"fnGetData": null,
	
		/**
		 * Function to set data for a cell in the column. You should <b>never</b>
		 * set the data directly to _aData internally in DataTables - always use
		 * this method. It allows mData to function as required. This function
		 * is automatically assigned by the column initialisation method
		 */
		"fnSetData": null,
	
		/**
		 * Property to read the value for the cells in the column from the data
		 * source array / object. If null, then the default content is used, if a
		 * function is given then the return from the function is used.
		 */
		"mData": null,
	
		/**
		 * Partner property to mData which is used (only when defined) to get
		 * the data - i.e. it is basically the same as mData, but without the
		 * 'set' option, and also the data fed to it is the result from mData.
		 * This is the rendering method to match the data method of mData.
		 */
		"mRender": null,
	
		/**
		 * The class to apply to all TD elements in the table's TBODY for the column
		 */
		"sClass": null,
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 */
		"sContentPadding": null,
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because mData
		 * is set to null, or because the data source itself is null).
		 */
		"sDefaultContent": null,
	
		/**
		 * Name for the column, allowing reference to the column by name as well as
		 * by index (needs a lookup to work by name).
		 */
		"sName": null,
	
		/**
		 * Custom sorting data type - defines which of the available plug-ins in
		 * afnSortData the custom sorting will use - if any is defined.
		 */
		"sSortDataType": 'std',
	
		/**
		 * Class to be applied to the header element when sorting on this column
		 */
		"sSortingClass": null,
	
		/**
		 * Title of the column - what is seen in the TH element (nTh).
		 */
		"sTitle": null,
	
		/**
		 * Column sorting and filtering type
		 */
		"sType": null,
	
		/**
		 * Width of the column
		 */
		"sWidth": null,
	
		/**
		 * Width of the column when it was first "encountered"
		 */
		"sWidthOrig": null,
	
		/** Cached string which is the longest in the column */
		maxLenString: null,
	
		/**
		 * Store for named searches
		 */
		searchFixed: null
	};
	
	
	/*
	 * Developer note: The properties of the object below are given in Hungarian
	 * notation, that was used as the interface for DataTables prior to v1.10, however
	 * from v1.10 onwards the primary interface is camel case. In order to avoid
	 * breaking backwards compatibility utterly with this change, the Hungarian
	 * version is still, internally the primary interface, but is is not documented
	 * - hence the @name tags in each doc comment. This allows a Javascript function
	 * to create a map from Hungarian notation to camel case (going the other direction
	 * would require each property to be listed, which would add around 3K to the size
	 * of DataTables, while this method is about a 0.5K hit).
	 *
	 * Ultimately this does pave the way for Hungarian notation to be dropped
	 * completely, but that is a massive amount of work and will break current
	 * installs (therefore is on-hold until v2).
	 */
	
	/**
	 * Initialisation options that can be given to DataTables at initialisation
	 * time.
	 *  @namespace
	 */
	DataTable.defaults = {
		/**
		 * An array of data to use for the table, passed in at initialisation which
		 * will be used in preference to any data which is already in the DOM. This is
		 * particularly useful for constructing tables purely in Javascript, for
		 * example with a custom Ajax call.
		 */
		"aaData": null,
	
	
		/**
		 * If ordering is enabled, then DataTables will perform a first pass sort on
		 * initialisation. You can define which column(s) the sort is performed
		 * upon, and the sorting direction, with this variable. The `sorting` array
		 * should contain an array for each column to be sorted initially containing
		 * the column's index and a direction string ('asc' or 'desc').
		 */
		"aaSorting": [[0,'asc']],
	
	
		/**
		 * This parameter is basically identical to the `sorting` parameter, but
		 * cannot be overridden by user interaction with the table. What this means
		 * is that you could have a column (visible or hidden) which the sorting
		 * will always be forced on first - any sorting after that (from the user)
		 * will then be performed as required. This can be useful for grouping rows
		 * together.
		 */
		"aaSortingFixed": [],
	
	
		/**
		 * DataTables can be instructed to load data to display in the table from a
		 * Ajax source. This option defines how that Ajax call is made and where to.
		 *
		 * The `ajax` property has three different modes of operation, depending on
		 * how it is defined. These are:
		 *
		 * * `string` - Set the URL from where the data should be loaded from.
		 * * `object` - Define properties for `jQuery.ajax`.
		 * * `function` - Custom data get function
		 *
		 * `string`
		 * --------
		 *
		 * As a string, the `ajax` property simply defines the URL from which
		 * DataTables will load data.
		 *
		 * `object`
		 * --------
		 *
		 * As an object, the parameters in the object are passed to
		 * [jQuery.ajax](https://api.jquery.com/jQuery.ajax/) allowing fine control
		 * of the Ajax request. DataTables has a number of default parameters which
		 * you can override using this option. Please refer to the jQuery
		 * documentation for a full description of the options available, although
		 * the following parameters provide additional options in DataTables or
		 * require special consideration:
		 *
		 * * `data` - As with jQuery, `data` can be provided as an object, but it
		 *   can also be used as a function to manipulate the data DataTables sends
		 *   to the server. The function takes a single parameter, an object of
		 *   parameters with the values that DataTables has readied for sending. An
		 *   object may be returned which will be merged into the DataTables
		 *   defaults, or you can add the items to the object that was passed in and
		 *   not return anything from the function. This supersedes `fnServerParams`
		 *   from DataTables 1.9-.
		 *
		 * * `dataSrc` - By default DataTables will look for the property `data` (or
		 *   `aaData` for compatibility with DataTables 1.9-) when obtaining data
		 *   from an Ajax source or for server-side processing - this parameter
		 *   allows that property to be changed. You can use Javascript dotted
		 *   object notation to get a data source for multiple levels of nesting, or
		 *   it my be used as a function. As a function it takes a single parameter,
		 *   the JSON returned from the server, which can be manipulated as
		 *   required, with the returned value being that used by DataTables as the
		 *   data source for the table.
		 *
		 * * `success` - Should not be overridden it is used internally in
		 *   DataTables. To manipulate / transform the data returned by the server
		 *   use `ajax.dataSrc`, or use `ajax` as a function (see below).
		 *
		 * `function`
		 * ----------
		 *
		 * As a function, making the Ajax call is left up to yourself allowing
		 * complete control of the Ajax request. Indeed, if desired, a method other
		 * than Ajax could be used to obtain the required data, such as Web storage
		 * or an AIR database.
		 *
		 * The function is given four parameters and no return is required. The
		 * parameters are:
		 *
		 * 1. _object_ - Data to send to the server
		 * 2. _function_ - Callback function that must be executed when the required
		 *    data has been obtained. That data should be passed into the callback
		 *    as the only parameter
		 * 3. _object_ - DataTables settings object for the table
		 */
		"ajax": null,
	
	
		/**
		 * This parameter allows you to readily specify the entries in the length drop
		 * down menu that DataTables shows when pagination is enabled. It can be
		 * either a 1D array of options which will be used for both the displayed
		 * option and the value, or a 2D array which will use the array in the first
		 * position as the value, and the array in the second position as the
		 * displayed options (useful for language strings such as 'All').
		 *
		 * Note that the `pageLength` property will be automatically set to the
		 * first value given in this array, unless `pageLength` is also provided.
		 */
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
	
		/**
		 * The `columns` option in the initialisation parameter allows you to define
		 * details about the way individual columns behave. For a full list of
		 * column options that can be set, please see
		 * {@link DataTable.defaults.column}. Note that if you use `columns` to
		 * define your columns, you must have an entry in the array for every single
		 * column that you have in your table (these can be null if you don't which
		 * to specify any options).
		 */
		"aoColumns": null,
	
		/**
		 * Very similar to `columns`, `columnDefs` allows you to target a specific
		 * column, multiple columns, or all columns, using the `targets` property of
		 * each object in the array. This allows great flexibility when creating
		 * tables, as the `columnDefs` arrays can be of any length, targeting the
		 * columns you specifically want. `columnDefs` may use any of the column
		 * options available: {@link DataTable.defaults.column}, but it _must_
		 * have `targets` defined in each object in the array. Values in the `targets`
		 * array may be:
		 *   <ul>
		 *     <li>a string - class name will be matched on the TH for the column</li>
		 *     <li>0 or a positive integer - column index counting from the left</li>
		 *     <li>a negative integer - column index counting from the right</li>
		 *     <li>the string "_all" - all columns (i.e. assign a default)</li>
		 *   </ul>
		 */
		"aoColumnDefs": null,
	
	
		/**
		 * Basically the same as `search`, this parameter defines the individual column
		 * filtering state at initialisation time. The array must be of the same size
		 * as the number of columns, and each element be an object with the parameters
		 * `search` and `escapeRegex` (the latter is optional). 'null' is also
		 * accepted and the default will be used.
		 */
		"aoSearchCols": [],
	
	
		/**
		 * Enable or disable automatic column width calculation. This can be disabled
		 * as an optimisation (it takes some time to calculate the widths) if the
		 * tables widths are passed in using `columns`.
		 */
		"bAutoWidth": true,
	
	
		/**
		 * Deferred rendering can provide DataTables with a huge speed boost when you
		 * are using an Ajax or JS data source for the table. This option, when set to
		 * true, will cause DataTables to defer the creation of the table elements for
		 * each row until they are needed for a draw - saving a significant amount of
		 * time.
		 */
		"bDeferRender": true,
	
	
		/**
		 * Replace a DataTable which matches the given selector and replace it with
		 * one which has the properties of the new initialisation object passed. If no
		 * table matches the selector, then the new DataTable will be constructed as
		 * per normal.
		 */
		"bDestroy": false,
	
	
		/**
		 * Enable or disable filtering of data. Filtering in DataTables is "smart" in
		 * that it allows the end user to input multiple words (space separated) and
		 * will match a row containing those words, even if not in the order that was
		 * specified (this allow matching across multiple columns). Note that if you
		 * wish to use filtering in DataTables this must remain 'true' - to remove the
		 * default filtering input box and retain filtering abilities, please use
		 * {@link DataTable.defaults.dom}.
		 */
		"bFilter": true,
	
		/**
		 * Used only for compatiblity with DT1
		 * @deprecated
		 */
		"bInfo": true,
	
		/**
		 * Used only for compatiblity with DT1
		 * @deprecated
		 */
		"bLengthChange": true,
	
		/**
		 * Enable or disable pagination.
		 */
		"bPaginate": true,
	
	
		/**
		 * Enable or disable the display of a 'processing' indicator when the table is
		 * being processed (e.g. a sort). This is particularly useful for tables with
		 * large amounts of data where it can take a noticeable amount of time to sort
		 * the entries.
		 */
		"bProcessing": false,
	
	
		/**
		 * Retrieve the DataTables object for the given selector. Note that if the
		 * table has already been initialised, this parameter will cause DataTables
		 * to simply return the object that has already been set up - it will not take
		 * account of any changes you might have made to the initialisation object
		 * passed to DataTables (setting this parameter to true is an acknowledgement
		 * that you understand this). `destroy` can be used to reinitialise a table if
		 * you need.
		 */
		"bRetrieve": false,
	
	
		/**
		 * When vertical (y) scrolling is enabled, DataTables will force the height of
		 * the table's viewport to the given height at all times (useful for layout).
		 * However, this can look odd when filtering data down to a small data set,
		 * and the footer is left "floating" further down. This parameter (when
		 * enabled) will cause DataTables to collapse the table's viewport down when
		 * the result set will fit within the given Y height.
		 */
		"bScrollCollapse": false,
	
	
		/**
		 * Configure DataTables to use server-side processing. Note that the
		 * `ajax` parameter must also be given in order to give DataTables a
		 * source to obtain the required data for each draw.
		 */
		"bServerSide": false,
	
	
		/**
		 * Enable or disable sorting of columns. Sorting of individual columns can be
		 * disabled by the `sortable` option for each column.
		 */
		"bSort": true,
	
	
		/**
		 * Enable or display DataTables' ability to sort multiple columns at the
		 * same time (activated by shift-click by the user).
		 */
		"bSortMulti": true,
	
	
		/**
		 * Allows control over whether DataTables should use the top (true) unique
		 * cell that is found for a single column, or the bottom (false - default).
		 * This is useful when using complex headers.
		 */
		"bSortCellsTop": null,
	
	
		/**
		 * Enable or disable the addition of the classes `sorting\_1`, `sorting\_2` and
		 * `sorting\_3` to the columns which are currently being sorted on. This is
		 * presented as a feature switch as it can increase processing time (while
		 * classes are removed and added) so for large data sets you might want to
		 * turn this off.
		 */
		"bSortClasses": true,
	
	
		/**
		 * Enable or disable state saving. When enabled HTML5 `localStorage` will be
		 * used to save table display information such as pagination information,
		 * display length, filtering and sorting. As such when the end user reloads
		 * the page the display display will match what thy had previously set up.
		 */
		"bStateSave": false,
	
	
		/**
		 * This function is called when a TR element is created (and all TD child
		 * elements have been inserted), or registered if using a DOM source, allowing
		 * manipulation of the TR element (adding classes etc).
		 */
		"fnCreatedRow": null,
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify any aspect you want about the created DOM.
		 */
		"fnDrawCallback": null,
	
	
		/**
		 * Identical to fnHeaderCallback() but for the table footer this function
		 * allows you to modify the table footer on every 'draw' event.
		 */
		"fnFooterCallback": null,
	
	
		/**
		 * When rendering large numbers in the information element for the table
		 * (i.e. "Showing 1 to 10 of 57 entries") DataTables will render large numbers
		 * to have a comma separator for the 'thousands' units (e.g. 1 million is
		 * rendered as "1,000,000") to help readability for the end user. This
		 * function will override the default method DataTables uses.
		 */
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify the header row. This can be used to calculate and
		 * display useful information about the table.
		 */
		"fnHeaderCallback": null,
	
	
		/**
		 * The information element can be used to convey information about the current
		 * state of the table. Although the internationalisation options presented by
		 * DataTables are quite capable of dealing with most customisations, there may
		 * be times where you wish to customise the string further. This callback
		 * allows you to do exactly that.
		 */
		"fnInfoCallback": null,
	
	
		/**
		 * Called when the table has been initialised. Normally DataTables will
		 * initialise sequentially and there will be no need for this function,
		 * however, this does not hold true when using external language information
		 * since that is obtained using an async XHR call.
		 */
		"fnInitComplete": null,
	
	
		/**
		 * Called at the very start of each table draw and can be used to cancel the
		 * draw by returning false, any other return (including undefined) results in
		 * the full draw occurring).
		 */
		"fnPreDrawCallback": null,
	
	
		/**
		 * This function allows you to 'post process' each row after it have been
		 * generated for each table draw, but before it is rendered on screen. This
		 * function might be used for setting the row class name etc.
		 */
		"fnRowCallback": null,
	
	
		/**
		 * Load the table state. With this function you can define from where, and how, the
		 * state of a table is loaded. By default DataTables will load from `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 */
		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {
				return {};
			}
		},
	
	
		/**
		 * Callback which allows modification of the saved state prior to loading that state.
		 * This callback is called when the table is loading state from the stored data, but
		 * prior to the settings object being modified by the saved state. Note that for
		 * plug-in authors, you should use the `stateLoadParams` event to load parameters for
		 * a plug-in.
		 */
		"fnStateLoadParams": null,
	
	
		/**
		 * Callback that is called when the state has been loaded from the state saving method
		 * and the DataTables settings object has been modified as a result of the loaded state.
		 */
		"fnStateLoaded": null,
	
	
		/**
		 * Save the table state. This function allows you to define where and how the state
		 * information for the table is stored By default DataTables will use `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 */
		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {
				// noop
			}
		},
	
	
		/**
		 * Callback which allows modification of the state to be saved. Called when the table
		 * has changed state a new state save is required. This method allows modification of
		 * the state saving object prior to actually doing the save, including addition or
		 * other state properties or modification. Note that for plug-in authors, you should
		 * use the `stateSaveParams` event to save parameters for a plug-in.
		 */
		"fnStateSaveParams": null,
	
	
		/**
		 * Duration for which the saved state information is considered valid. After this period
		 * has elapsed the state will be returned to the default.
		 * Value is given in seconds.
		 */
		"iStateDuration": 7200,
	
	
		/**
		 * Number of rows to display on a single page when using pagination. If
		 * feature enabled (`lengthChange`) then the end user will be able to override
		 * this to a custom setting using a pop-up menu.
		 */
		"iDisplayLength": 10,
	
	
		/**
		 * Define the starting point for data display when using DataTables with
		 * pagination. Note that this parameter is the number of records, rather than
		 * the page number, so if you have 10 records per page and want to start on
		 * the third page, it should be "20".
		 */
		"iDisplayStart": 0,
	
	
		/**
		 * By default DataTables allows keyboard navigation of the table (sorting, paging,
		 * and filtering) by adding a `tabindex` attribute to the required elements. This
		 * allows you to tab through the controls and press the enter key to activate them.
		 * The tabindex is default 0, meaning that the tab follows the flow of the document.
		 * You can overrule this using this parameter if you wish. Use a value of -1 to
		 * disable built-in keyboard navigation.
		 */
		"iTabIndex": 0,
	
	
		/**
		 * Classes that DataTables assigns to the various components and features
		 * that it adds to the HTML table. This allows classes to be configured
		 * during initialisation in addition to through the static
		 * {@link DataTable.ext.oStdClasses} object).
		 */
		"oClasses": {},
	
	
		/**
		 * All strings that DataTables uses in the user interface that it creates
		 * are defined in this object, allowing you to modified them individually or
		 * completely replace them all as required.
		 */
		"oLanguage": {
			/**
			 * Strings that are used for WAI-ARIA labels and controls only (these are not
			 * actually visible on the page, but will be read by screenreaders, and thus
			 * must be internationalised as well).
			 */
			"oAria": {
				/**
				 * ARIA label that is added to the table headers when the column may be sorted
				 */
				"orderable": ": Activate to sort",
	
				/**
				 * ARIA label that is added to the table headers when the column is currently being sorted
				 */
				"orderableReverse": ": Activate to invert sorting",
	
				/**
				 * ARIA label that is added to the table headers when the column is currently being 
				 * sorted and next step is to remove sorting
				 */
				"orderableRemove": ": Activate to remove sorting",
	
				paginate: {
					first: 'First',
					last: 'Last',
					next: 'Next',
					previous: 'Previous'
				}
			},
	
			/**
			 * Pagination string used by DataTables for the built-in pagination
			 * control types.
			 */
			"oPaginate": {
				/**
				 * Label and character for first page button («)
				 */
				"sFirst": "\u00AB",
	
				/**
				 * Last page button (»)
				 */
				"sLast": "\u00BB",
	
				/**
				 * Next page button (›)
				 */
				"sNext": "\u203A",
	
				/**
				 * Previous page button (‹)
				 */
				"sPrevious": "\u2039",
			},
	
			/**
			 * Plural object for the data type the table is showing
			 */
			entries: {
				_: "entries",
				1: "entry"
			},
	
			/**
			 * This string is shown in preference to `zeroRecords` when the table is
			 * empty of data (regardless of filtering). Note that this is an optional
			 * parameter - if it is not given, the value of `zeroRecords` will be used
			 * instead (either the default or given value).
			 */
			"sEmptyTable": "No data available in table",
	
	
			/**
			 * This string gives information to the end user about the information
			 * that is current on display on the page. The following tokens can be
			 * used in the string and will be dynamically replaced as the table
			 * display updates. This tokens can be placed anywhere in the string, or
			 * removed as needed by the language requires:
			 *
			 * * `\_START\_` - Display index of the first record on the current page
			 * * `\_END\_` - Display index of the last record on the current page
			 * * `\_TOTAL\_` - Number of records in the table after filtering
			 * * `\_MAX\_` - Number of records in the table without filtering
			 * * `\_PAGE\_` - Current page number
			 * * `\_PAGES\_` - Total number of pages of data in the table
			 */
			"sInfo": "Showing _START_ to _END_ of _TOTAL_ _ENTRIES-TOTAL_",
	
	
			/**
			 * Display information string for when the table is empty. Typically the
			 * format of this string should match `info`.
			 */
			"sInfoEmpty": "Showing 0 to 0 of 0 _ENTRIES-TOTAL_",
	
	
			/**
			 * When a user filters the information in a table, this string is appended
			 * to the information (`info`) to give an idea of how strong the filtering
			 * is. The variable _MAX_ is dynamically updated.
			 */
			"sInfoFiltered": "(filtered from _MAX_ total _ENTRIES-MAX_)",
	
	
			/**
			 * If can be useful to append extra information to the info string at times,
			 * and this variable does exactly that. This information will be appended to
			 * the `info` (`infoEmpty` and `infoFiltered` in whatever combination they are
			 * being used) at all times.
			 */
			"sInfoPostFix": "",
	
	
			/**
			 * This decimal place operator is a little different from the other
			 * language options since DataTables doesn't output floating point
			 * numbers, so it won't ever use this for display of a number. Rather,
			 * what this parameter does is modify the sort methods of the table so
			 * that numbers which are in a format which has a character other than
			 * a period (`.`) as a decimal place will be sorted numerically.
			 *
			 * Note that numbers with different decimal places cannot be shown in
			 * the same table and still be sortable, the table must be consistent.
			 * However, multiple different tables on the page can use different
			 * decimal place characters.
			 */
			"sDecimal": "",
	
	
			/**
			 * DataTables has a build in number formatter (`formatNumber`) which is
			 * used to format large numbers that are used in the table information.
			 * By default a comma is used, but this can be trivially changed to any
			 * character you wish with this parameter.
			 */
			"sThousands": ",",
	
	
			/**
			 * Detail the action that will be taken when the drop down menu for the
			 * pagination length option is changed. The '_MENU_' variable is replaced
			 * with a default select list of 10, 25, 50 and 100, and can be replaced
			 * with a custom select box if required.
			 */
			"sLengthMenu": "_MENU_ _ENTRIES_ per page",
	
	
			/**
			 * When using Ajax sourced data and during the first draw when DataTables is
			 * gathering the data, this message is shown in an empty row in the table to
			 * indicate to the end user the the data is being loaded. Note that this
			 * parameter is not used when loading data by server-side processing, just
			 * Ajax sourced data with client-side processing.
			 */
			"sLoadingRecords": "Loading...",
	
	
			/**
			 * Text which is displayed when the table is processing a user action
			 * (usually a sort command or similar).
			 */
			"sProcessing": "",
	
	
			/**
			 * Details the actions that will be taken when the user types into the
			 * filtering input text box. The variable "_INPUT_", if used in the string,
			 * is replaced with the HTML text box for the filtering input allowing
			 * control over where it appears in the string. If "_INPUT_" is not given
			 * then the input box is appended to the string automatically.
			 */
			"sSearch": "Search:",
	
	
			/**
			 * Assign a `placeholder` attribute to the search `input` element
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.searchPlaceholder
			 */
			"sSearchPlaceholder": "",
	
	
			/**
			 * All of the language information can be stored in a file on the
			 * server-side, which DataTables will look up if this parameter is passed.
			 * It must store the URL of the language file, which is in a JSON format,
			 * and the object has the same properties as the oLanguage object in the
			 * initialiser object (i.e. the above parameters). Please refer to one of
			 * the example language files to see how this works in action.
			 */
			"sUrl": "",
	
	
			/**
			 * Text shown inside the table records when the is no information to be
			 * displayed after filtering. `emptyTable` is shown when there is simply no
			 * information in the table at all (regardless of filtering).
			 */
			"sZeroRecords": "No matching records found"
		},
	
	
		/**
		 * This parameter allows you to have define the global filtering state at
		 * initialisation time. As an object the `search` parameter must be
		 * defined, but all other parameters are optional. When `regex` is true,
		 * the search string will be treated as a regular expression, when false
		 * (default) it will be treated as a straight string. When `smart`
		 * DataTables will use it's smart filtering methods (to word match at
		 * any point in the data), when false this will not be done.
		 */
		"oSearch": $.extend( {}, DataTable.models.oSearch ),
	
	
		/**
		 * Table and control layout. This replaces the legacy `dom` option.
		 */
		layout: {
			topStart: 'pageLength',
			topEnd: 'search',
			bottomStart: 'info',
			bottomEnd: 'paging'
		},
	
	
		/**
		 * Legacy DOM layout option
		 */
		"sDom": null,
	
	
		/**
		 * Search delay option. This will throttle full table searches that use the
		 * DataTables provided search input element (it does not effect calls to
		 * `dt-api search()`, providing a delay before the search is made.
		 */
		"searchDelay": null,
	
	
		/**
		 * DataTables features six different built-in options for the buttons to
		 * display for pagination control:
		 *
		 * * `numbers` - Page number buttons only
		 * * `simple` - 'Previous' and 'Next' buttons only
		 * * 'simple_numbers` - 'Previous' and 'Next' buttons, plus page numbers
		 * * `full` - 'First', 'Previous', 'Next' and 'Last' buttons
		 * * `full_numbers` - 'First', 'Previous', 'Next' and 'Last' buttons, plus page numbers
		 * * `first_last_numbers` - 'First' and 'Last' buttons, plus page numbers
		 */
		"sPaginationType": "full_numbers",
	
	
		/**
		 * Enable horizontal scrolling. When a table is too wide to fit into a
		 * certain layout, or you have a large number of columns in the table, you
		 * can enable x-scrolling to show the table in a viewport, which can be
		 * scrolled. This property can be `true` which will allow the table to
		 * scroll horizontally when needed, or any CSS unit, or a number (in which
		 * case it will be treated as a pixel measurement). Setting as simply `true`
		 * is recommended.
		 */
		"sScrollX": "",
	
	
		/**
		 * This property can be used to force a DataTable to use more width than it
		 * might otherwise do when x-scrolling is enabled. For example if you have a
		 * table which requires to be well spaced, this parameter is useful for
		 * "over-sizing" the table, and thus forcing scrolling. This property can by
		 * any CSS unit, or a number (in which case it will be treated as a pixel
		 * measurement).
		 */
		"sScrollXInner": "",
	
	
		/**
		 * Enable vertical scrolling. Vertical scrolling will constrain the DataTable
		 * to the given height, and enable scrolling for any data which overflows the
		 * current viewport. This can be used as an alternative to paging to display
		 * a lot of data in a small area (although paging and scrolling can both be
		 * enabled at the same time). This property can be any CSS unit, or a number
		 * (in which case it will be treated as a pixel measurement).
		 */
		"sScrollY": "",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * Set the HTTP method that is used to make the Ajax call for server-side
		 * processing or Ajax sourced data.
		 */
		"sServerMethod": "GET",
	
	
		/**
		 * DataTables makes use of renderers when displaying HTML elements for
		 * a table. These renderers can be added or modified by plug-ins to
		 * generate suitable mark-up for a site. For example the Bootstrap
		 * integration plug-in for DataTables uses a paging button renderer to
		 * display pagination buttons in the mark-up required by Bootstrap.
		 *
		 * For further information about the renderers available see
		 * DataTable.ext.renderer
		 */
		"renderer": null,
	
	
		/**
		 * Set the data property name that DataTables should use to get a row's id
		 * to set as the `id` property in the node.
		 */
		"rowId": "DT_RowId",
	
	
		/**
		 * Caption value
		 */
		"caption": null
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	
	/*
	 * Developer note - See note in model.defaults.js about the use of Hungarian
	 * notation and camel case.
	 */
	
	/**
	 * Column options that can be given to DataTables at initialisation time.
	 *  @namespace
	 */
	DataTable.defaults.column = {
		/**
		 * Define which column(s) an order will occur on for this column. This
		 * allows a column's ordering to take multiple columns into account when
		 * doing a sort or use the data from a different column. For example first
		 * name / last name columns make sense to do a multi-column sort over the
		 * two columns.
		 */
		"aDataSort": null,
		"iDataSort": -1,
	
		ariaTitle: '',
	
	
		/**
		 * You can control the default ordering direction, and even alter the
		 * behaviour of the sort handler (i.e. only allow ascending ordering etc)
		 * using this parameter.
		 */
		"asSorting": [ 'asc', 'desc', '' ],
	
	
		/**
		 * Enable or disable filtering on the data in this column.
		 */
		"bSearchable": true,
	
	
		/**
		 * Enable or disable ordering on this column.
		 */
		"bSortable": true,
	
	
		/**
		 * Enable or disable the display of this column.
		 */
		"bVisible": true,
	
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 */
		"fnCreatedCell": null,
	
	
		/**
		 * This property can be used to read data from any data source property,
		 * including deeply nested objects / properties. `data` can be given in a
		 * number of different ways which effect its behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object. Note that
		 *      function notation is recommended for use in `render` rather than
		 *      `data` as it is much simpler to use as a renderer.
		 * * `null` - use the original data source for the row rather than plucking
		 *   data directly from it. This action has effects on two other
		 *   initialisation options:
		 *    * `defaultContent` - When null is given as the `data` option and
		 *      `defaultContent` is specified for the column, the value defined by
		 *      `defaultContent` will be used for the cell.
		 *    * `render` - When null is used for the `data` option and the `render`
		 *      option is specified for the column, the whole data source for the
		 *      row is used for the renderer.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * `{array|object}` The data source for the row
		 *      * `{string}` The type call data requested - this will be 'set' when
		 *        setting data or 'filter', 'display', 'type', 'sort' or undefined
		 *        when gathering data. Note that when `undefined` is given for the
		 *        type DataTables expects to get the raw data for the object back<
		 *      * `{*}` Data to set when the second parameter is 'set'.
		 *    * Return:
		 *      * The return value from the function is not required when 'set' is
		 *        the type of call, but otherwise the return is what will be used
		 *        for the data requested.
		 *
		 * Note that `data` is a getter and setter option. If you just require
		 * formatting of data for output, you will likely want to use `render` which
		 * is simply a getter and thus simpler to use.
		 *
		 * Note that prior to DataTables 1.9.2 `data` was called `mDataProp`. The
		 * name change reflects the flexibility of this property and is consistent
		 * with the naming of mRender. If 'mDataProp' is given, then it will still
		 * be used by DataTables, as it automatically maps the old name to the new
		 * if required.
		 */
		"mData": null,
	
	
		/**
		 * This property is the rendering partner to `data` and it is suggested that
		 * when you want to manipulate data for display (including filtering,
		 * sorting etc) without altering the underlying data for the table, use this
		 * property. `render` can be considered to be the the read only companion to
		 * `data` which is read / write (then as such more complex). Like `data`
		 * this option can be given in a number of different ways to effect its
		 * behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object.
		 * * `object` - use different data for the different data types requested by
		 *   DataTables ('filter', 'display', 'type' or 'sort'). The property names
		 *   of the object is the data type the property refers to and the value can
		 *   defined using an integer, string or function using the same rules as
		 *   `render` normally does. Note that an `_` option _must_ be specified.
		 *   This is the default value to use if you haven't specified a value for
		 *   the data type requested by DataTables.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * {array|object} The data source for the row (based on `data`)
		 *      * {string} The type call data requested - this will be 'filter',
		 *        'display', 'type' or 'sort'.
		 *      * {array|object} The full data source for the row (not based on
		 *        `data`)
		 *    * Return:
		 *      * The return value from the function is what will be used for the
		 *        data requested.
		 */
		"mRender": null,
	
	
		/**
		 * Change the cell type created for the column - either TD cells or TH cells. This
		 * can be useful as TH cells have semantic meaning in the table body, allowing them
		 * to act as a header for a row (you may wish to add scope='row' to the TH elements).
		 */
		"sCellType": "td",
	
	
		/**
		 * Class to give to each cell in this column.
		 */
		"sClass": "",
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 * Generally you shouldn't need this!
		 */
		"sContentPadding": "",
	
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because `data`
		 * is set to null, or because the data source itself is null).
		 */
		"sDefaultContent": null,
	
	
		/**
		 * This parameter is only used in DataTables' server-side processing. It can
		 * be exceptionally useful to know what columns are being displayed on the
		 * client side, and to map these to database fields. When defined, the names
		 * also allow DataTables to reorder information from the server if it comes
		 * back in an unexpected order (i.e. if you switch your columns around on the
		 * client-side, your server-side code does not also need updating).
		 */
		"sName": "",
	
	
		/**
		 * Defines a data source type for the ordering which can be used to read
		 * real-time information from the table (updating the internally cached
		 * version) prior to ordering. This allows ordering to occur on user
		 * editable elements such as form inputs.
		 */
		"sSortDataType": "std",
	
	
		/**
		 * The title of this column.
		 */
		"sTitle": null,
	
	
		/**
		 * The type allows you to specify how the data for this column will be
		 * ordered. Four types (string, numeric, date and html (which will strip
		 * HTML tags before ordering)) are currently available. Note that only date
		 * formats understood by Javascript's Date() object will be accepted as type
		 * date. For example: "Mar 26, 2008 5:03 PM". May take the values: 'string',
		 * 'numeric', 'date' or 'html' (by default). Further types can be adding
		 * through plug-ins.
		 */
		"sType": null,
	
	
		/**
		 * Defining the width of the column, this parameter may take any CSS value
		 * (3em, 20px etc). DataTables applies 'smart' widths to columns which have not
		 * been given a specific width through this interface ensuring that the table
		 * remains readable.
		 */
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	
	
	/**
	 * DataTables settings object - this holds all the information needed for a
	 * given table, including configuration, data and current application of the
	 * table options. DataTables does not have a single instance for each DataTable
	 * with the settings attached to that instance, but rather instances of the
	 * DataTable "class" are created on-the-fly as needed (typically by a
	 * $().dataTable() call) and the settings object is then applied to that
	 * instance.
	 *
	 * Note that this object is related to {@link DataTable.defaults} but this
	 * one is the internal data store for DataTables's cache of columns. It should
	 * NOT be manipulated outside of DataTables. Any configuration should be done
	 * through the initialisation options.
	 */
	DataTable.models.oSettings = {
		/**
		 * Primary features of DataTables and their enablement state.
		 */
		"oFeatures": {
	
			/**
			 * Flag to say if DataTables should automatically try to calculate the
			 * optimum table and columns widths (true) or not (false).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bAutoWidth": null,
	
			/**
			 * Delay the creation of TR and TD elements until they are actually
			 * needed by a driven page draw. This can give a significant speed
			 * increase for Ajax source and Javascript source data, but makes no
			 * difference at all for DOM and server-side processing tables.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bDeferRender": null,
	
			/**
			 * Enable filtering on the table or not. Note that if this is disabled
			 * then there is no filtering at all on the table, including fnFilter.
			 * To just remove the filtering input use sDom and remove the 'f' option.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bFilter": null,
	
			/**
			 * Used only for compatiblity with DT1
			 * @deprecated
			 */
			"bInfo": true,
	
			/**
			 * Used only for compatiblity with DT1
			 * @deprecated
			 */
			"bLengthChange": true,
	
			/**
			 * Pagination enabled or not. Note that if this is disabled then length
			 * changing must also be disabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bPaginate": null,
	
			/**
			 * Processing indicator enable flag whenever DataTables is enacting a
			 * user request - typically an Ajax request for server-side processing.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bProcessing": null,
	
			/**
			 * Server-side processing enabled flag - when enabled DataTables will
			 * get all data from the server for every draw - there is no filtering,
			 * sorting or paging done on the client-side.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bServerSide": null,
	
			/**
			 * Sorting enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSort": null,
	
			/**
			 * Multi-column sorting
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSortMulti": null,
	
			/**
			 * Apply a class to the columns which are being sorted to provide a
			 * visual highlight or not. This can slow things down when enabled since
			 * there is a lot of DOM interaction.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSortClasses": null,
	
			/**
			 * State saving enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bStateSave": null
		},
	
	
		/**
		 * Scrolling settings for a table.
		 */
		"oScroll": {
			/**
			 * When the table is shorter in height than sScrollY, collapse the
			 * table container down to the height of the table (when true).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bCollapse": null,
	
			/**
			 * Width of the scrollbar for the web-browser's platform. Calculated
			 * during table initialisation.
			 */
			"iBarWidth": 0,
	
			/**
			 * Viewport width for horizontal scrolling. Horizontal scrolling is
			 * disabled if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"sX": null,
	
			/**
			 * Width to expand the table to when using x-scrolling. Typically you
			 * should not need to use this.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @deprecated
			 */
			"sXInner": null,
	
			/**
			 * Viewport height for vertical scrolling. Vertical scrolling is disabled
			 * if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"sY": null
		},
	
		/**
		 * Language information for the table.
		 */
		"oLanguage": {
			/**
			 * Information callback function. See
			 * {@link DataTable.defaults.fnInfoCallback}
			 */
			"fnInfoCallback": null
		},
	
		/**
		 * Browser support parameters
		 */
		"oBrowser": {
			/**
			 * Determine if the vertical scrollbar is on the right or left of the
			 * scrolling container - needed for rtl language layout, although not
			 * all browsers move the scrollbar (Safari).
			 */
			"bScrollbarLeft": false,
	
			/**
			 * Browser scrollbar width
			 */
			"barWidth": 0
		},
	
	
		"ajax": null,
	
	
		/**
		 * Array referencing the nodes which are used for the features. The
		 * parameters of this object match what is allowed by sDom - i.e.
		 *   <ul>
		 *     <li>'l' - Length changing</li>
		 *     <li>'f' - Filtering input</li>
		 *     <li>'t' - The table!</li>
		 *     <li>'i' - Information</li>
		 *     <li>'p' - Pagination</li>
		 *     <li>'r' - pRocessing</li>
		 *   </ul>
		 */
		"aanFeatures": [],
	
		/**
		 * Store data information - see {@link DataTable.models.oRow} for detailed
		 * information.
		 */
		"aoData": [],
	
		/**
		 * Array of indexes which are in the current display (after filtering etc)
		 */
		"aiDisplay": [],
	
		/**
		 * Array of indexes for display - no filtering
		 */
		"aiDisplayMaster": [],
	
		/**
		 * Map of row ids to data indexes
		 */
		"aIds": {},
	
		/**
		 * Store information about each column that is in use
		 */
		"aoColumns": [],
	
		/**
		 * Store information about the table's header
		 */
		"aoHeader": [],
	
		/**
		 * Store information about the table's footer
		 */
		"aoFooter": [],
	
		/**
		 * Store the applied global search information in case we want to force a
		 * research or compare the old search to a new one.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"oPreviousSearch": {},
	
		/**
		 * Store for named searches
		 */
		searchFixed: {},
	
		/**
		 * Store the applied search for each column - see
		 * {@link DataTable.models.oSearch} for the format that is used for the
		 * filtering information for each column.
		 */
		"aoPreSearchCols": [],
	
		/**
		 * Sorting that is applied to the table. Note that the inner arrays are
		 * used in the following manner:
		 * <ul>
		 *   <li>Index 0 - column number</li>
		 *   <li>Index 1 - current sorting direction</li>
		 * </ul>
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aaSorting": null,
	
		/**
		 * Sorting that is always applied to the table (i.e. prefixed in front of
		 * aaSorting).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aaSortingFixed": [],
	
		/**
		 * If restoring a table - we should restore its width
		 */
		"sDestroyWidth": 0,
	
		/**
		 * Callback functions array for every time a row is inserted (i.e. on a draw).
		 */
		"aoRowCallback": [],
	
		/**
		 * Callback functions for the header on each draw.
		 */
		"aoHeaderCallback": [],
	
		/**
		 * Callback function for the footer on each draw.
		 */
		"aoFooterCallback": [],
	
		/**
		 * Array of callback functions for draw callback functions
		 */
		"aoDrawCallback": [],
	
		/**
		 * Array of callback functions for row created function
		 */
		"aoRowCreatedCallback": [],
	
		/**
		 * Callback functions for just before the table is redrawn. A return of
		 * false will be used to cancel the draw.
		 */
		"aoPreDrawCallback": [],
	
		/**
		 * Callback functions for when the table has been initialised.
		 */
		"aoInitComplete": [],
	
	
		/**
		 * Callbacks for modifying the settings to be stored for state saving, prior to
		 * saving state.
		 */
		"aoStateSaveParams": [],
	
		/**
		 * Callbacks for modifying the settings that have been stored for state saving
		 * prior to using the stored values to restore the state.
		 */
		"aoStateLoadParams": [],
	
		/**
		 * Callbacks for operating on the settings object once the saved state has been
		 * loaded
		 */
		"aoStateLoaded": [],
	
		/**
		 * Cache the table ID for quick access
		 */
		"sTableId": "",
	
		/**
		 * The TABLE node for the main table
		 */
		"nTable": null,
	
		/**
		 * Permanent ref to the thead element
		 */
		"nTHead": null,
	
		/**
		 * Permanent ref to the tfoot element - if it exists
		 */
		"nTFoot": null,
	
		/**
		 * Permanent ref to the tbody element
		 */
		"nTBody": null,
	
		/**
		 * Cache the wrapper node (contains all DataTables controlled elements)
		 */
		"nTableWrapper": null,
	
		/**
		 * Indicate if all required information has been read in
		 */
		"bInitialised": false,
	
		/**
		 * Information about open rows. Each object in the array has the parameters
		 * 'nTr' and 'nParent'
		 */
		"aoOpenRows": [],
	
		/**
		 * Dictate the positioning of DataTables' control elements - see
		 * {@link DataTable.model.oInit.sDom}.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sDom": null,
	
		/**
		 * Search delay (in mS)
		 */
		"searchDelay": null,
	
		/**
		 * Which type of pagination should be used.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sPaginationType": "two_button",
	
		/**
		 * Number of paging controls on the page. Only used for backwards compatibility
		 */
		pagingControls: 0,
	
		/**
		 * The state duration (for `stateSave`) in seconds.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"iStateDuration": 0,
	
		/**
		 * Array of callback functions for state saving. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the JSON string to save that has been thus far created. Returns
		 *       a JSON string to be inserted into a json object
		 *       (i.e. '"param": [ 0, 1, 2]')</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 */
		"aoStateSave": [],
	
		/**
		 * Array of callback functions for state loading. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the object stored. May return false to cancel state loading</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 */
		"aoStateLoad": [],
	
		/**
		 * State that was saved. Useful for back reference
		 */
		"oSavedState": null,
	
		/**
		 * State that was loaded. Useful for back reference
		 */
		"oLoadedState": null,
	
		/**
		 * Note if draw should be blocked while getting data
		 */
		"bAjaxDataGet": true,
	
		/**
		 * The last jQuery XHR object that was used for server-side data gathering.
		 * This can be used for working with the XHR information in one of the
		 * callbacks
		 */
		"jqXHR": null,
	
		/**
		 * JSON returned from the server in the last Ajax request
		 */
		"json": undefined,
	
		/**
		 * Data submitted as part of the last Ajax request
		 */
		"oAjaxData": undefined,
	
		/**
		 * Send the XHR HTTP method - GET or POST (could be PUT or DELETE if
		 * required).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sServerMethod": null,
	
		/**
		 * Format numbers for display.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"fnFormatNumber": null,
	
		/**
		 * List of options that can be used for the user selectable length menu.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aLengthMenu": null,
	
		/**
		 * Counter for the draws that the table does. Also used as a tracker for
		 * server-side processing
		 */
		"iDraw": 0,
	
		/**
		 * Indicate if a redraw is being done - useful for Ajax
		 */
		"bDrawing": false,
	
		/**
		 * Draw index (iDraw) of the last error when parsing the returned data
		 */
		"iDrawError": -1,
	
		/**
		 * Paging display length
		 */
		"_iDisplayLength": 10,
	
		/**
		 * Paging start point - aiDisplay index
		 */
		"_iDisplayStart": 0,
	
		/**
		 * Server-side processing - number of records in the result set
		 * (i.e. before filtering), Use fnRecordsTotal rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 */
		"_iRecordsTotal": 0,
	
		/**
		 * Server-side processing - number of records in the current display set
		 * (i.e. after filtering). Use fnRecordsDisplay rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 */
		"_iRecordsDisplay": 0,
	
		/**
		 * The classes to use for the table
		 */
		"oClasses": {},
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if filtering has been done in the draw. Deprecated in favour of
		 * events.
		 *  @deprecated
		 */
		"bFiltered": false,
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if sorting has been done in the draw. Deprecated in favour of
		 * events.
		 *  @deprecated
		 */
		"bSorted": false,
	
		/**
		 * Indicate that if multiple rows are in the header and there is more than
		 * one unique cell per column, if the top one (true) or bottom one (false)
		 * should be used for sorting / title by DataTables.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"bSortCellsTop": null,
	
		/**
		 * Initialisation object that is used for the table
		 */
		"oInit": null,
	
		/**
		 * Destroy callback functions - for plug-ins to attach themselves to the
		 * destroy so they can clean up markup and events.
		 */
		"aoDestroyCallback": [],
	
	
		/**
		 * Get the number of records in the current record set, before filtering
		 */
		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		/**
		 * The DataTables object for this table
		 */
		"oInstance": null,
	
		/**
		 * Unique identifier for each instance of the DataTables object. If there
		 * is an ID on the table node, then it takes that value, otherwise an
		 * incrementing internal counter is used.
		 */
		"sInstance": null,
	
		/**
		 * tabindex attribute value that is added to DataTables control elements, allowing
		 * keyboard navigation of the table and its controls.
		 */
		"iTabIndex": 0,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollHead": null,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollFoot": null,
	
		/**
		 * Last applied sort
		 */
		"aLastSort": [],
	
		/**
		 * Stored plug-in instances
		 */
		"oPlugins": {},
	
		/**
		 * Function used to get a row's id from the row's data
		 */
		"rowIdFn": null,
	
		/**
		 * Data location where to store a row's id
		 */
		"rowId": null,
	
		caption: '',
	
		captionNode: null,
	
		colgroup: null
	};
	
	/**
	 * Extension object for DataTables that is used to provide all extension
	 * options.
	 *
	 * Note that the `DataTable.ext` object is available through
	 * `jQuery.fn.dataTable.ext` where it may be accessed and manipulated. It is
	 * also aliased to `jQuery.fn.dataTableExt` for historic reasons.
	 *  @namespace
	 *  @extends DataTable.models.ext
	 */
	
	
	var extPagination = DataTable.ext.pager;
	
	// Paging buttons configuration
	$.extend( extPagination, {
		simple: function () {
			return [ 'previous', 'next' ];
		},
	
		full: function () {
			return [  'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function () {
			return [ 'numbers' ];
		},
	
		simple_numbers: function () {
			return [ 'previous', 'numbers', 'next' ];
		},
	
		full_numbers: function () {
			return [ 'first', 'previous', 'numbers', 'next', 'last' ];
		},
		
		first_last: function () {
			return ['first', 'last'];
		},
		
		first_last_numbers: function () {
			return ['first', 'numbers', 'last'];
		},
	
		// For testing and plug-ins to use
		_numbers: _pagingNumbers,
	
		// Number of number buttons - legacy, use `numbers` option for paging feature
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pagingButton: {
			_: function (settings, buttonType, content, active, disabled) {
				var classes = settings.oClasses.paging;
				var btnClasses = [classes.button];
				var btn;
	
				if (active) {
					btnClasses.push(classes.active);
				}
	
				if (disabled) {
					btnClasses.push(classes.disabled)
				}
	
				if (buttonType === 'ellipsis') {
					btn = $('<span class="ellipsis"></span>').html(content)[0];
				}
				else {
					btn = $('<button>', {
						class: btnClasses.join(' '),
						role: 'link',
						type: 'button'
					}).html(content);
				}
	
				return {
					display: btn,
					clicker: btn
				}
			}
		},
	
		pagingContainer: {
			_: function (settings, buttons) {
				// No wrapping element - just append directly to the host
				return buttons;
			}
		}
	} );
	
	// Common function to remove new lines, strip HTML and diacritic control
	var _filterString = function (stripHtml, normalize) {
		return function (str) {
			if (_empty(str) || typeof str !== 'string') {
				return str;
			}
	
			str = str.replace( _re_new_lines, " " );
	
			if (stripHtml) {
				str = _stripHtml(str);
			}
	
			if (normalize) {
				str = _normalize(str, false);
			}
	
			return str;
		};
	}
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	function __mldFnName(name) {
		return name.replace(/[\W]/g, '_')
	}
	
	// Common logic for moment, luxon or a date action
	function __mld( dt, momentFn, luxonFn, dateFn, arg1 ) {
		if (window.moment) {
			return dt[momentFn]( arg1 );
		}
		else if (window.luxon) {
			return dt[luxonFn]( arg1 );
		}
		
		return dateFn ? dt[dateFn]( arg1 ) : dt;
	}
	
	
	var __mlWarning = false;
	function __mldObj (d, format, locale) {
		var dt;
	
		if (window.moment) {
			dt = window.moment.utc( d, format, locale, true );
	
			if (! dt.isValid()) {
				return null;
			}
		}
		else if (window.luxon) {
			dt = format && typeof d === 'string'
				? window.luxon.DateTime.fromFormat( d, format )
				: window.luxon.DateTime.fromISO( d );
	
			if (! dt.isValid) {
				return null;
			}
	
			dt.setLocale(locale);
		}
		else if (! format) {
			// No format given, must be ISO
			dt = new Date(d);
		}
		else {
			if (! __mlWarning) {
				alert('DataTables warning: Formatted date without Moment.js or Luxon - https://datatables.net/tn/17');
			}
	
			__mlWarning = true;
		}
	
		return dt;
	}
	
	// Wrapper for date, datetime and time which all operate the same way with the exception of
	// the output string for auto locale support
	function __mlHelper (localeString) {
		return function ( from, to, locale, def ) {
			// Luxon and Moment support
			// Argument shifting
			if ( arguments.length === 0 ) {
				locale = 'en';
				to = null; // means toLocaleString
				from = null; // means iso8601
			}
			else if ( arguments.length === 1 ) {
				locale = 'en';
				to = from;
				from = null;
			}
			else if ( arguments.length === 2 ) {
				locale = to;
				to = from;
				from = null;
			}
	
			var typeName = 'datetime' + (to ? '-' + __mldFnName(to) : '');
	
			// Add type detection and sorting specific to this date format - we need to be able to identify
			// date type columns as such, rather than as numbers in extensions. Hence the need for this.
			if (! DataTable.ext.type.order[typeName]) {
				DataTable.type(typeName, {
					detect: function (d) {
						// The renderer will give the value to type detect as the type!
						return d === typeName ? typeName : false;
					},
					order: {
						pre: function (d) {
							// The renderer gives us Moment, Luxon or Date obects for the sorting, all of which have a
							// `valueOf` which gives milliseconds epoch
							return d.valueOf();
						}
					},
					className: 'dt-right'
				});
			}
		
			return function ( d, type ) {
				// Allow for a default value
				if (d === null || d === undefined) {
					if (def === '--now') {
						// We treat everything as UTC further down, so no changes are
						// made, as such need to get the local date / time as if it were
						// UTC
						var local = new Date();
						d = new Date( Date.UTC(
							local.getFullYear(), local.getMonth(), local.getDate(),
							local.getHours(), local.getMinutes(), local.getSeconds()
						) );
					}
					else {
						d = '';
					}
				}
	
				if (type === 'type') {
					// Typing uses the type name for fast matching
					return typeName;
				}
	
				if (d === '') {
					return type !== 'sort'
						? ''
						: __mldObj('0000-01-01 00:00:00', null, locale);
				}
	
				// Shortcut. If `from` and `to` are the same, we are using the renderer to
				// format for ordering, not display - its already in the display format.
				if ( to !== null && from === to && type !== 'sort' && type !== 'type' && ! (d instanceof Date) ) {
					return d;
				}
	
				var dt = __mldObj(d, from, locale);
	
				if (dt === null) {
					return d;
				}
	
				if (type === 'sort') {
					return dt;
				}
				
				var formatted = to === null
					? __mld(dt, 'toDate', 'toJSDate', '')[localeString]()
					: __mld(dt, 'format', 'toFormat', 'toISOString', to);
	
				// XSS protection
				return type === 'display' ?
					_escapeHtml( formatted ) :
					formatted;
			};
		}
	}
	
	// Based on locale, determine standard number formatting
	// Fallback for legacy browsers is US English
	var __thousands = ',';
	var __decimal = '.';
	
	if (window.Intl !== undefined) {
		try {
			var num = new Intl.NumberFormat().formatToParts(100000.1);
		
			for (var i=0 ; i<num.length ; i++) {
				if (num[i].type === 'group') {
					__thousands = num[i].value;
				}
				else if (num[i].type === 'decimal') {
					__decimal = num[i].value;
				}
			}
		}
		catch (e) {
			// noop
		}
	}
	
	// Formatted date time detection - use by declaring the formats you are going to use
	DataTable.datetime = function ( format, locale ) {
		var typeName = 'datetime-detect-' + __mldFnName(format);
	
		if (! locale) {
			locale = 'en';
		}
	
		if (! DataTable.ext.type.order[typeName]) {
			DataTable.type(typeName, {
				detect: function (d) {
					var dt = __mldObj(d, format, locale);
					return d === '' || dt ? typeName : false;
				},
				order: {
					pre: function (d) {
						return __mldObj(d, format, locale) || 0;
					}
				},
				className: 'dt-right'
			});
		}
	}
	
	/**
	 * Helpers for `columns.render`.
	 *
	 * The options defined here can be used with the `columns.render` initialisation
	 * option to provide a display renderer. The following functions are defined:
	 *
	 * * `moment` - Uses the MomentJS library to convert from a given format into another.
	 * This renderer has three overloads:
	 *   * 1 parameter:
	 *     * `string` - Format to convert to (assumes input is ISO8601 and locale is `en`)
	 *   * 2 parameters:
	 *     * `string` - Format to convert from
	 *     * `string` - Format to convert to. Assumes `en` locale
	 *   * 3 parameters:
	 *     * `string` - Format to convert from
	 *     * `string` - Format to convert to
	 *     * `string` - Locale
	 * * `number` - Will format numeric data (defined by `columns.data`) for
	 *   display, retaining the original unformatted data for sorting and filtering.
	 *   It takes 5 parameters:
	 *   * `string` - Thousands grouping separator
	 *   * `string` - Decimal point indicator
	 *   * `integer` - Number of decimal points to show
	 *   * `string` (optional) - Prefix.
	 *   * `string` (optional) - Postfix (/suffix).
	 * * `text` - Escape HTML to help prevent XSS attacks. It has no optional
	 *   parameters.
	 *
	 * @example
	 *   // Column definition using the number renderer
	 *   {
	 *     data: "salary",
	 *     render: $.fn.dataTable.render.number( '\'', '.', 0, '$' )
	 *   }
	 *
	 * @namespace
	 */
	DataTable.render = {
		date: __mlHelper('toLocaleDateString'),
		datetime: __mlHelper('toLocaleString'),
		time: __mlHelper('toLocaleTimeString'),
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			// Auto locale detection
			if (thousands === null || thousands === undefined) {
				thousands = __thousands;
			}
	
			if (decimal === null || decimal === undefined) {
				decimal = __decimal;
			}
	
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					if (d === '' || d === null) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
					var abs = Math.abs(flo);
	
					// Scientific notation for large and small numbers
					if (abs >= 100000000000 || (abs < 0.0001 && abs !== 0) ) {
						var exp = flo.toExponential(precision).split(/e\+?/);
						return exp[0] + ' x 10<sup>' + exp[1] + '</sup>';
					}
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return _escapeHtml( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					// If zero, then can't have a negative prefix
					if (intPart === 0 && parseFloat(floatPart) === 0) {
						negative = '';
					}
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: _escapeHtml,
				filter: _escapeHtml
			};
		}
	};
	
	
	var _extTypes = DataTable.ext.type;
	
	// Get / set type
	DataTable.type = function (name, prop, val) {
		if (! prop) {
			return {
				className: _extTypes.className[name],
				detect: _extTypes.detect.find(function (fn) {
					return fn.name === name;
				}),
				order: {
					pre: _extTypes.order[name + '-pre'],
					asc: _extTypes.order[name + '-asc'],
					desc: _extTypes.order[name + '-desc']
				},
				render: _extTypes.render[name],
				search: _extTypes.search[name]
			};
		}
	
		var setProp = function(prop, propVal) {
			_extTypes[prop][name] = propVal;
		};
		var setDetect = function (fn) {
			// Wrap to allow the function to return `true` rather than
			// specifying the type name.
			var cb = function (d, s) {
				var ret = fn(d, s);
	
				return ret === true
					? name
					: ret;
			};
			Object.defineProperty(cb, "name", {value: name});
	
			var idx = _extTypes.detect.findIndex(function (fn) {
				return fn.name === name;
			});
	
			if (idx === -1) {
				_extTypes.detect.unshift(cb);
			}
			else {
				_extTypes.detect.splice(idx, 1, cb);
			}
		};
		var setOrder = function (obj) {
			_extTypes.order[name + '-pre'] = obj.pre; // can be undefined
			_extTypes.order[name + '-asc'] = obj.asc; // can be undefined
			_extTypes.order[name + '-desc'] = obj.desc; // can be undefined
		};
	
		// prop is optional
		if (val === undefined) {
			val = prop;
			prop = null;
		}
	
		if (prop === 'className') {
			setProp('className', val);
		}
		else if (prop === 'detect') {
			setDetect(val);
		}
		else if (prop === 'order') {
			setOrder(val);
		}
		else if (prop === 'render') {
			setProp('render', val);
		}
		else if (prop === 'search') {
			setProp('search', val);
		}
		else if (! prop) {
			if (val.className) {
				setProp('className', val.className);
			}
	
			if (val.detect !== undefined) {
				setDetect(val.detect);
			}
	
			if (val.order) {
				setOrder(val.order);
			}
	
			if (val.render !== undefined) {
				setProp('render', val.render);
			}
	
			if (val.search !== undefined) {
				setProp('search', val.search);
			}
		}
	}
	
	// Get a list of types
	DataTable.types = function () {
		return _extTypes.detect.map(function (fn) {
			return fn.name;
		});
	};
	
	//
	// Built in data types
	//
	
	DataTable.type('string', {
		detect: function () {
			return 'string';
		},
		order: {
			pre: function ( a ) {
				// This is a little complex, but faster than always calling toString,
				// http://jsperf.com/tostring-v-check
				return _empty(a) ?
					'' :
					typeof a === 'string' ?
						a.toLowerCase() :
						! a.toString ?
							'' :
							a.toString();
			}
		},
		search: _filterString(false, true)
	});
	
	
	DataTable.type('html', {
		detect: function ( d ) {
			return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1) ?
				'html' : null;
		},
		order: {
			pre: function ( a ) {
				return _empty(a) ?
					'' :
					a.replace ?
						_stripHtml(a).trim().toLowerCase() :
						a+'';
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('date', {
		className: 'dt-type-date',
		detect: function ( d )
		{
			// V8 tries _very_ hard to make a string passed into `Date.parse()`
			// valid, so we need to use a regex to restrict date formats. Use a
			// plug-in for anything other than ISO8601 style strings
			if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
				return null;
			}
			var parsed = Date.parse(d);
			return (parsed !== null && !isNaN(parsed)) || _empty(d) ? 'date' : null;
		},
		order: {
			pre: function ( d ) {
				var ts = Date.parse( d );
				return isNaN(ts) ? -Infinity : ts;
			}
		}
	});
	
	
	DataTable.type('html-num-fmt', {
		className: 'dt-type-numeric',
		detect: function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal, true ) ? 'html-num-fmt' : null;
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_html, _re_formatted_numeric );
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('html-num', {
		className: 'dt-type-numeric',
		detect: function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal ) ? 'html-num' : null;
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_html );
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('num-fmt', {
		className: 'dt-type-numeric',
		detect: function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal, true ) ? 'num-fmt' : null;
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_formatted_numeric );
			}
		}
	});
	
	
	DataTable.type('num', {
		className: 'dt-type-numeric',
		detect: function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal ) ? 'num' : null;
		},
		order: {
			pre: function (d, s) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp );
			}
		}
	});
	
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
		
		var type = typeof d;
	
		if (type === 'number' || type === 'bigint') {
			return d;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	$.extend( true, DataTable.ext.renderer, {
		footer: {
			_: function ( settings, cell, classes ) {
				cell.addClass(classes.tfoot.cell);
			}
		},
	
		header: {
			_: function ( settings, cell, classes ) {
				cell.addClass(classes.thead.cell);
	
				if (! settings.oFeatures.bSort) {
					cell.addClass(classes.order.none);
				}
	
				var legacyTop = settings.bSortCellsTop;
				var headerRows = cell.closest('thead').find('tr');
				var rowIdx = cell.parent().index();
	
				// Conditions to not apply the ordering icons
				if (
					// Cells and rows which have the attribute to disable the icons
					cell.attr('data-dt-order') === 'disable' ||
					cell.parent().attr('data-dt-order') === 'disable' ||
	
					// Legacy support for `orderCellsTop`. If it is set, then cells
					// which are not in the top or bottom row of the header (depending
					// on the value) do not get the sorting classes applied to them
					(legacyTop === true && rowIdx !== 0) ||
					(legacyTop === false && rowIdx !== headerRows.length - 1)
				) {
					return;
				}
	
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var orderClasses = classes.order;
					var columns = ctx.api.columns( cell );
					var col = settings.aoColumns[columns.flatten()[0]];
					var orderable = columns.orderable().includes(true);
					var ariaType = '';
					var indexes = columns.indexes();
					var sortDirs = columns.orderable(true).flatten();
					var orderedColumns = ',' + sorting.map( function (val) {
						return val.col;
					} ).join(',') + ',';
	
					cell
						.removeClass(
							orderClasses.isAsc +' '+
							orderClasses.isDesc
						)
						.toggleClass( orderClasses.none, ! orderable )
						.toggleClass( orderClasses.canAsc, orderable && sortDirs.includes('asc') )
						.toggleClass( orderClasses.canDesc, orderable && sortDirs.includes('desc') );
					
					var sortIdx = orderedColumns.indexOf( ',' + indexes.toArray().join(',') + ',' );
	
					if ( sortIdx !== -1 ) {
						// Get the ordering direction for the columns under this cell
						// Note that it is possible for a cell to be asc and desc sorting
						// (column spanning cells)
						var orderDirs = columns.order();
	
						cell.addClass(
							orderDirs.includes('asc') ? orderClasses.isAsc : '' +
							orderDirs.includes('desc') ? orderClasses.isDesc : ''
						);
					}
	
					// The ARIA spec says that only one column should be marked with aria-sort
					if ( sortIdx === 0 ) {
						var firstSort = sorting[0];
						var sortOrder = col.asSorting;
	
						cell.attr('aria-sort', firstSort.dir === 'asc' ? 'ascending' : 'descending');
	
						// Determine if the next click will remove sorting or change the sort
						ariaType = ! sortOrder[firstSort.index + 1] ? 'Remove' : 'Reverse';
					}
					else {
						cell.removeAttr('aria-sort');
					}
	
					cell.attr('aria-label', orderable
						? col.ariaTitle + ctx.api.i18n('oAria.orderable' + ariaType)
						: col.ariaTitle
					);
	
					if (orderable) {
						cell.find('.dt-column-title').attr('role', 'button');
						cell.attr('tabindex', 0)
					}
				} );
			}
		},
	
		layout: {
			_: function ( settings, container, items ) {
				var row = $('<div/>')
					.addClass('dt-layout-row')
					.appendTo( container );
	
				$.each( items, function (key, val) {
					var klass = ! val.table ?
						'dt-'+key+' ' :
						'';
	
					if (val.table) {
						row.addClass('dt-layout-table');
					}
	
					$('<div/>')
						.attr({
							id: val.id || null,
							"class": 'dt-layout-cell '+klass+(val.className || '')
						})
						.append( val.contents )
						.appendTo( row );
				} );
			}
		}
	} );
	
	
	DataTable.feature = {};
	
	// Third parameter is internal only!
	DataTable.feature.register = function ( name, cb, legacy ) {
		DataTable.ext.features[ name ] = cb;
	
		if (legacy) {
			_ext.feature.push({
				cFeature: legacy,
				fnInit: cb
			});
		}
	};
	
	DataTable.feature.register( 'info', function ( settings, opts ) {
		// For compatibility with the legacy `info` top level option
		if (! settings.oFeatures.bInfo) {
			return null;
		}
	
		var
			lang  = settings.oLanguage,
			tid = settings.sTableId,
			n = $('<div/>', {
				'class': settings.oClasses.info.container,
			} );
	
		opts = $.extend({
			callback: lang.fnInfoCallback,
			empty: lang.sInfoEmpty,
			postfix: lang.sInfoPostFix,
			search: lang.sInfoFiltered,
			text: lang.sInfo,
		}, opts);
	
	
		// Update display on each draw
		settings.aoDrawCallback.push(function (s) {
			_fnUpdateInfo(s, opts, n);
		});
	
		// For the first info display in the table, we add a callback and aria information.
		if (! settings._infoEl) {
			n.attr({
				'aria-live': 'polite',
				id: tid+'_info',
				role: 'status'
			});
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
	
			settings._infoEl = n;
		}
	
		return n;
	}, 'i' );
	
	/**
	 * Update the information elements in the display
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnUpdateInfo ( settings, opts, node )
	{
		var
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total
				? opts.text
				: opts.empty;
	
		if ( total !== max ) {
			// Record set after filtering
			out += ' ' + opts.search;
		}
	
		// Convert the macros
		out += opts.postfix;
		out = _fnMacros( settings, out );
	
		if ( opts.callback ) {
			out = opts.callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		node.html( out );
	
		_fnCallbackFire(settings, null, 'info', [settings, node[0], out]);
	}
	
	var __searchCounter = 0;
	
	// opts
	// - text
	// - placeholder
	DataTable.feature.register( 'search', function ( settings, opts ) {
		// Don't show the input if filtering isn't available on the table
		if (! settings.oFeatures.bFilter) {
			return null;
		}
	
		var classes = settings.oClasses.search;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var input = '<input type="search" class="'+classes.input+'"/>';
	
		opts = $.extend({
			placeholder: language.sSearchPlaceholder,
			text: language.sSearch
		}, opts);
	
		// The _INPUT_ is optional - is appended if not present
		if (opts.text.indexOf('_INPUT_') === -1) {
			opts.text += '_INPUT_';
		}
	
		opts.text = _fnMacros(settings, opts.text);
	
		// We can put the <input> outside of the label if it is at the start or end
		// which helps improve accessability (not all screen readers like implicit
		// for elements).
		var end = opts.text.match(/_INPUT_$/);
		var start = opts.text.match(/^_INPUT_/);
		var removed = opts.text.replace(/_INPUT_/, '');
		var str = '<label>' + opts.text + '</label>';
	
		if (start) {
			str = '_INPUT_<label>' + removed + '</label>';
		}
		else if (end) {
			str = '<label>' + removed + '</label>_INPUT_';
		}
	
		var filter = $('<div>')
			.addClass(classes.container)
			.append(str.replace(/_INPUT_/, input));
	
		// add for and id to label and input
		filter.find('label').attr('for', 'dt-search-' + __searchCounter);
		filter.find('input').attr('id', 'dt-search-' + __searchCounter);
		__searchCounter++;
	
		var searchFn = function(event) {
			var val = this.value;
	
			if(previousSearch.return && event.key !== "Enter") {
				return;
			}
	
			/* Now do the filter */
			if ( val != previousSearch.search ) {
				previousSearch.search = val;
	
				_fnFilterComplete( settings, previousSearch );
	
				// Need to redraw, without resorting
				settings._iDisplayStart = 0;
				_fnDraw( settings );
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.search )
			.attr( 'placeholder', opts.placeholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					DataTable.util.debounce( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'mouseup.DT', function(e) {
				// Edge fix! Edge 17 does not trigger anything other than mouse events when clicking
				// on the clear icon (Edge bug 17584515). This is safe in other browsers as `searchFn`
				// checks the value to see if it has changed. In other browsers it won't have.
				setTimeout( function () {
					searchFn.call(jqFilter[0], e);
				}, 10);
			} )
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s && jqFilter[0] !== document.activeElement ) {
				jqFilter.val( typeof previousSearch.search !== 'function'
					? previousSearch.search
					: ''
				);
			}
		} );
	
		return filter;
	}, 'f' );
	
	// opts
	// - type - button configuration
	// - buttons - number of buttons to show - must be odd
	DataTable.feature.register( 'paging', function ( settings, opts ) {
		// Don't show the paging input if the table doesn't have paging enabled
		if (! settings.oFeatures.bPaginate) {
			return null;
		}
	
		opts = $.extend({
			buttons: DataTable.ext.pager.numbers_length,
			type: settings.sPaginationType,
			boundaryNumbers: true
		}, opts);
	
		// To be removed in 2.1
		if (opts.numbers) {
			opts.buttons = opts.numbers;
		}
	
		var host = $('<div/>').addClass( settings.oClasses.paging.container + ' paging_' + opts.type );
		var draw = function () {
			_pagingDraw(settings, host, opts);
		};
	
		settings.aoDrawCallback.push(draw);
	
		// Responsive redraw of paging control
		$(settings.nTable).on('column-sizing.dt.DT', draw);
	
		return host;
	}, 'p' );
	
	function _pagingDraw(settings, host, opts) {
		if (! settings._bInitComplete) {
			return;
		}
	
		var
			plugin = DataTable.ext.pager[ opts.type ],
			aria = settings.oLanguage.oAria.paginate || {},
			start      = settings._iDisplayStart,
			len        = settings._iDisplayLength,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1,
			page = all ? 0 : Math.ceil( start / len ),
			pages = all ? 1 : Math.ceil( visRecords / len ),
			buttons = plugin()
				.map(function (val) {
					return val === 'numbers'
						? _pagingNumbers(page, pages, opts.buttons, opts.boundaryNumbers)
						: val;
				})
				.flat();
	
		var buttonEls = [];
	
		for (var i=0 ; i<buttons.length ; i++) {
			var button = buttons[i];
	
			var btnInfo = _pagingButtonInfo(settings, button, page, pages);
			var btn = _fnRenderer( settings, 'pagingButton' )(
				settings,
				button,
				btnInfo.display,
				btnInfo.active,
				btnInfo.disabled
			);
	
			// Common attributes
			$(btn.clicker).attr({
				'aria-controls': settings.sTableId,
				'aria-disabled': btnInfo.disabled ? 'true' : null,
				'aria-current': btnInfo.active ? 'page' : null,
				'aria-label': aria[ button ],
				'data-dt-idx': button,
				'tabIndex': btnInfo.disabled ? -1 : settings.iTabIndex,
			});
	
			if (typeof button !== 'number') {
				$(btn.clicker).addClass(button);
			}
	
			_fnBindAction(
				btn.clicker, {action: button}, function(e) {
					e.preventDefault();
	
					_fnPageChange( settings, e.data.action, true );
				}
			);
	
			buttonEls.push(btn.display);
		}
	
		var wrapped = _fnRenderer(settings, 'pagingContainer')(
			settings, buttonEls
		);
	
		var activeEl = host.find(document.activeElement).data('dt-idx');
	
		host.empty().append(wrapped);
	
		if ( activeEl !== undefined ) {
			host.find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
		}
	
		// Responsive - check if the buttons are over two lines based on the
		// height of the buttons and the container.
		if (
			buttonEls.length && // any buttons
			opts.numbers > 1 && // prevent infinite
			$(host).height() >= ($(buttonEls[0]).outerHeight() * 2) - 10
		) {
			_pagingDraw(settings, host, $.extend({}, opts, { numbers: opts.numbers - 2 }));
		}
	}
	
	/**
	 * Get properties for a button based on the current paging state of the table
	 *
	 * @param {*} settings DT settings object
	 * @param {*} button The button type in question
	 * @param {*} page Table's current page
	 * @param {*} pages Number of pages
	 * @returns Info object
	 */
	function _pagingButtonInfo(settings, button, page, pages) {
		var lang = settings.oLanguage.oPaginate;
		var o = {
			display: '',
			active: false,
			disabled: false
		};
	
		switch ( button ) {
			case 'ellipsis':
				o.display = '&#x2026;';
				o.disabled = true;
				break;
	
			case 'first':
				o.display = lang.sFirst;
	
				if (page === 0) {
					o.disabled = true;
				}
				break;
	
			case 'previous':
				o.display = lang.sPrevious;
	
				if ( page === 0 ) {
					o.disabled = true;
				}
				break;
	
			case 'next':
				o.display = lang.sNext;
	
				if ( pages === 0 || page === pages-1 ) {
					o.disabled = true;
				}
				break;
	
			case 'last':
				o.display = lang.sLast;
	
				if ( pages === 0 || page === pages-1 ) {
					o.disabled = true;
				}
				break;
	
			default:
				if ( typeof button === 'number' ) {
					o.display = settings.fnFormatNumber( button + 1 );
					
					if (page === button) {
						o.active = true;
					}
				}
				break;
		}
	
		return o;
	}
	
	/**
	 * Compute what number buttons to show in the paging control
	 *
	 * @param {*} page Current page
	 * @param {*} pages Total number of pages
	 * @param {*} buttons Target number of number buttons
	 * @param {boolean} addFirstLast Indicate if page 1 and end should be included
	 * @returns Buttons to show
	 */
	function _pagingNumbers ( page, pages, buttons, addFirstLast ) {
		var
			numbers = [],
			half = Math.floor(buttons / 2),
			before = addFirstLast ? 2 : 1,
			after = addFirstLast ? 1 : 0;
	
		if ( pages <= buttons ) {
			numbers = _range(0, pages);
		}
		else if (buttons === 1) {
			// Single button - current page only
			numbers = [page];
		}
		else if (buttons === 3) {
			// Special logic for just three buttons
			if (page <= 1) {
				numbers = [0, 1, 'ellipsis'];
			}
			else if (page >= pages - 2) {
				numbers = _range(pages-2, pages);
				numbers.unshift('ellipsis');
			}
			else {
				numbers = ['ellipsis', page, 'ellipsis'];
			}
		}
		else if ( page <= half ) {
			numbers = _range(0, buttons-before);
			numbers.push('ellipsis');
	
			if (addFirstLast) {
				numbers.push(pages-1);
			}
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range(pages-(buttons-before), pages);
			numbers.unshift('ellipsis');
	
			if (addFirstLast) {
				numbers.unshift(0);
			}
		}
		else {
			numbers = _range(page-half+before, page+half-after);
			numbers.push('ellipsis');
			numbers.unshift('ellipsis');
	
			if (addFirstLast) {
				numbers.push(pages-1);
				numbers.unshift(0);
			}
		}
	
		return numbers;
	}
	
	var __lengthCounter = 0;
	
	// opts
	// - menu
	// - text
	DataTable.feature.register( 'pageLength', function ( settings, opts ) {
		var features = settings.oFeatures;
	
		// For compatibility with the legacy `pageLength` top level option
		if (! features.bPaginate || ! features.bLengthChange) {
			return null;
		}
	
		opts = $.extend({
			menu: settings.aLengthMenu,
			text: settings.oLanguage.sLengthMenu
		}, opts);
	
		var
			classes  = settings.oClasses.length,
			tableId  = settings.sTableId,
			menu     = opts.menu,
			lengths  = [],
			language = [],
			i;
	
		// Options can be given in a number of ways
		if (Array.isArray( menu[0] )) {
			// Old 1.x style - 2D array
			lengths = menu[0];
			language = menu[1];
		}
		else {
			for ( i=0 ; i<menu.length ; i++ ) {
				// An object with different label and value
				if ($.isPlainObject(menu[i])) {
					lengths.push(menu[i].value);
					language.push(menu[i].label);
				}
				else {
					// Or just a number to display and use
					lengths.push(menu[i]);
					language.push(menu[i]);
				}
			}
		}
	
		// We can put the <select> outside of the label if it is at the start or
		// end which helps improve accessability (not all screen readers like
		// implicit for elements).
		var end = opts.text.match(/_MENU_$/);
		var start = opts.text.match(/^_MENU_/);
		var removed = opts.text.replace(/_MENU_/, '');
		var str = '<label>' + opts.text + '</label>';
	
		if (start) {
			str = '_MENU_<label>' + removed + '</label>';
		}
		else if (end) {
			str = '<label>' + removed + '</label>_MENU_';
		}
	
		// Wrapper element - use a span as a holder for where the select will go
		var div = $('<div/>')
			.addClass( classes.container )
			.append(
				str.replace( '_MENU_', '<span></span>' )
			);
	
		// Save text node content for macro updating
		var textNodes = [];
		div.find('label')[0].childNodes.forEach(function (el) {
			if (el.nodeType === Node.TEXT_NODE) {
				textNodes.push({
					el: el,
					text: el.textContent
				});
			}
		})
	
		// Update the label text in case it has an entries value
		var updateEntries = function (len) {
			textNodes.forEach(function (node) {
				node.el.textContent = _fnMacros(settings, node.text, len);
			});
		}
	
		// Next, the select itself, along with the options
		var select = $('<select/>', {
			'name':          tableId+'_length',
			'aria-controls': tableId,
			'class':         classes.select
		} );
	
		for ( i=0 ; i<lengths.length ; i++ ) {
			select[0][ i ] = new Option(
				typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i],
				lengths[i]
			);
		}
	
		// add for and id to label and input
		div.find('label').attr('for', 'dt-length-' + __lengthCounter);
		select.attr('id', 'dt-length-' + __lengthCounter);
		__lengthCounter++;
	
		// Swap in the select list
		div.find('span').replaceWith(select);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function() {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
	
				// Resolve plurals in the text for the new length
				updateEntries(len);
			}
		} );
	
		updateEntries(settings._iDisplayLength);
	
		return div;
	}, 'l' );
	
	// jQuery access
	$.fn.dataTable = DataTable;
	
	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;
	
	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;
	
	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};
	
	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );

	return DataTable;
}));

/*! DataTables Bootstrap 5 integration
 * © SpryMedia Ltd - datatables.net/license
 */

(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		var jq = require('jquery');
		var cjsRequires = function (root, $) {
			if ( ! $.fn.dataTable ) {
				require('datatables.net')(root, $);
			}
		};

		if (typeof window === 'undefined') {
			module.exports = function (root, $) {
				if ( ! root ) {
					// CommonJS environments without a window global must pass a
					// root. This will give an error otherwise
					root = window;
				}

				if ( ! $ ) {
					$ = jq( root );
				}

				cjsRequires( root, $ );
				return factory( $, root, root.document );
			};
		}
		else {
			cjsRequires( window, jq );
			module.exports = factory( jq, window, window.document );
		}
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document ) {
'use strict';
var DataTable = $.fn.dataTable;



/**
 * DataTables integration for Bootstrap 5.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See https://datatables.net/manual/styling/bootstrap
 * for further information.
 */

/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( true, DataTable.ext.classes, {
	container: "dt-container dt-bootstrap5",
	search: {
		input: "form-control form-control-sm"
	},
	length: {
		select: "form-select form-select-sm"
	},
	processing: {
		container: "dt-processing card"
	}
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pagingButton.bootstrap = function (settings, buttonType, content, active, disabled) {
	var btnClasses = ['dt-paging-button', 'page-item'];

	if (active) {
		btnClasses.push('active');
	}

	if (disabled) {
		btnClasses.push('disabled')
	}

	var li = $('<li>').addClass(btnClasses.join(' '));
	var a = $('<a>', {
		'href': disabled ? null : '#',
		'class': 'page-link'
	})
		.html(content)
		.appendTo(li);

	return {
		display: li,
		clicker: a
	};
};

DataTable.ext.renderer.pagingContainer.bootstrap = function (settings, buttonEls) {
	return $('<ul/>').addClass('pagination').append(buttonEls);
};

DataTable.ext.renderer.layout.bootstrap = function ( settings, container, items ) {
	var row = $( '<div/>', {
			"class": items.full ?
				'row mt-2 justify-content-md-center' :
				'row mt-2 justify-content-between'
		} )
		.appendTo( container );

	$.each( items, function (key, val) {
		var klass;

		// Apply start / end (left / right when ltr) margins
		if (val.table) {
			klass = 'col-12';
		}
		else if (key === 'start') {
			klass = 'col-md-auto me-auto';
		}
		else if (key === 'end') {
			klass = 'col-md-auto ms-auto';
		}
		else {
			klass = 'col-md';
		}

		$( '<div/>', {
				id: val.id || null,
				"class": klass + ' ' + (val.className || '')
			} )
			.append( val.contents )
			.appendTo( row );
	} );
};


return DataTable;
}));

"use strict";

//
// Datatables.net Initialization
//

// Set Defaults

var defaults = {
	"language": {		
		"info": "Showing _START_ to _END_ of _TOTAL_ records",
    "infoEmpty": "Showing no records",
		"lengthMenu": "_MENU_",
		"processing": '<span class="spinner-border w-15px h-15px text-muted align-middle me-2"></span> <span class="text-gray-600">Loading...</span>',
		"paginate": {
			"first": '<i class="kt-outline kt-double-left"></i>',
			"last": '<i class="kt-outline kt-double-right"></i>',
			"next": '<i class="next"></i>',
			"previous": '<i class="previous"></i>'
		}
	}
};

$.extend(true, $.fn.dataTable.defaults, defaults);

/*! DataTables Bootstrap 4 integration
 * ©2011-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 4 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	pagingType: 'simple_numbers',
	dom: 
		"<'table-responsive'tr>" +
		"<'row'" + 
		"<'col-sm-12 col-md-5 d-flex align-items-center justify-content-center justify-content-md-start dt-toolbar'li>" + 
		"<'col-sm-12 col-md-7 d-flex align-items-center justify-content-center justify-content-md-end'p>" +
		">",	
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm form-control-solid",
	sLengthSelect: "form-select form-select-sm form-select-solid",
	sProcessing:   "dataTables_processing",
	sPageButton:   "paginate_button page-item",
	search: {
		input: "form-control form-control-solid form-control-sm"
	},
	length: {
		select: "form-select form-select-solid form-select-sm"
	}
} );

/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {	
				attach( container, button );
			} else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame.
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

/*!

JSZip v3.10.1 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/main/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/main/LICENSE
*/

!function(e){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define([],e);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=e()}}(function(){return function s(a,o,h){function u(r,e){if(!o[r]){if(!a[r]){var t="function"==typeof require&&require;if(!e&&t)return t(r,!0);if(l)return l(r,!0);var n=new Error("Cannot find module '"+r+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[r]={exports:{}};a[r][0].call(i.exports,function(e){var t=a[r][1][e];return u(t||e)},i,i.exports,s,a,o,h)}return o[r].exports}for(var l="function"==typeof require&&require,e=0;e<h.length;e++)u(h[e]);return u}({1:[function(e,t,r){"use strict";var d=e("./utils"),c=e("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(e){for(var t,r,n,i,s,a,o,h=[],u=0,l=e.length,f=l,c="string"!==d.getTypeOf(e);u<e.length;)f=l-u,n=c?(t=e[u++],r=u<l?e[u++]:0,u<l?e[u++]:0):(t=e.charCodeAt(u++),r=u<l?e.charCodeAt(u++):0,u<l?e.charCodeAt(u++):0),i=t>>2,s=(3&t)<<4|r>>4,a=1<f?(15&r)<<2|n>>6:64,o=2<f?63&n:64,h.push(p.charAt(i)+p.charAt(s)+p.charAt(a)+p.charAt(o));return h.join("")},r.decode=function(e){var t,r,n,i,s,a,o=0,h=0,u="data:";if(e.substr(0,u.length)===u)throw new Error("Invalid base64 input, it looks like a data url.");var l,f=3*(e=e.replace(/[^A-Za-z0-9+/=]/g,"")).length/4;if(e.charAt(e.length-1)===p.charAt(64)&&f--,e.charAt(e.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(l=c.uint8array?new Uint8Array(0|f):new Array(0|f);o<e.length;)t=p.indexOf(e.charAt(o++))<<2|(i=p.indexOf(e.charAt(o++)))>>4,r=(15&i)<<4|(s=p.indexOf(e.charAt(o++)))>>2,n=(3&s)<<6|(a=p.indexOf(e.charAt(o++))),l[h++]=t,64!==s&&(l[h++]=r),64!==a&&(l[h++]=n);return l}},{"./support":30,"./utils":32}],2:[function(e,t,r){"use strict";var n=e("./external"),i=e("./stream/DataWorker"),s=e("./stream/Crc32Probe"),a=e("./stream/DataLengthProbe");function o(e,t,r,n,i){this.compressedSize=e,this.uncompressedSize=t,this.crc32=r,this.compression=n,this.compressedContent=i}o.prototype={getContentWorker:function(){var e=new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),t=this;return e.on("end",function(){if(this.streamInfo.data_length!==t.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),e},getCompressedWorker:function(){return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(e,t,r){return e.pipe(new s).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression",t)},t.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(e,t,r){"use strict";var n=e("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(){return new n("STORE compression")},uncompressWorker:function(){return new n("STORE decompression")}},r.DEFLATE=e("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(e,t,r){"use strict";var n=e("./utils");var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t){return void 0!==e&&e.length?"string"!==n.getTypeOf(e)?function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}(0|t,e,e.length,0):function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t.charCodeAt(a))];return-1^e}(0|t,e,e.length,0):0}},{"./utils":32}],5:[function(e,t,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(e,t,r){"use strict";var n=null;n="undefined"!=typeof Promise?Promise:e("lie"),t.exports={Promise:n}},{lie:37}],7:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,i=e("pako"),s=e("./utils"),a=e("./stream/GenericWorker"),o=n?"uint8array":"array";function h(e,t){a.call(this,"FlateWorker/"+e),this._pako=null,this._pakoAction=e,this._pakoOptions=t,this.meta={}}r.magic="\b\0",s.inherits(h,a),h.prototype.processChunk=function(e){this.meta=e.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,e.data),!1)},h.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},h.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},h.prototype._createPako=function(){this._pako=new i[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var t=this;this._pako.onData=function(e){t.push({data:e,meta:t.meta})}},r.compressWorker=function(e){return new h("Deflate",e)},r.uncompressWorker=function(){return new h("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(e,t,r){"use strict";function A(e,t){var r,n="";for(r=0;r<t;r++)n+=String.fromCharCode(255&e),e>>>=8;return n}function n(e,t,r,n,i,s){var a,o,h=e.file,u=e.compression,l=s!==O.utf8encode,f=I.transformTo("string",s(h.name)),c=I.transformTo("string",O.utf8encode(h.name)),d=h.comment,p=I.transformTo("string",s(d)),m=I.transformTo("string",O.utf8encode(d)),_=c.length!==h.name.length,g=m.length!==d.length,b="",v="",y="",w=h.dir,k=h.date,x={crc32:0,compressedSize:0,uncompressedSize:0};t&&!r||(x.crc32=e.crc32,x.compressedSize=e.compressedSize,x.uncompressedSize=e.uncompressedSize);var S=0;t&&(S|=8),l||!_&&!g||(S|=2048);var z=0,C=0;w&&(z|=16),"UNIX"===i?(C=798,z|=function(e,t){var r=e;return e||(r=t?16893:33204),(65535&r)<<16}(h.unixPermissions,w)):(C=20,z|=function(e){return 63&(e||0)}(h.dosPermissions)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v=A(1,1)+A(B(f),4)+c,b+="up"+A(v.length,2)+v),g&&(y=A(1,1)+A(B(p),4)+m,b+="uc"+A(y.length,2)+y);var E="";return E+="\n\0",E+=A(S,2),E+=u.magic,E+=A(a,2),E+=A(o,2),E+=A(x.crc32,4),E+=A(x.compressedSize,4),E+=A(x.uncompressedSize,4),E+=A(f.length,2),E+=A(b.length,2),{fileRecord:R.LOCAL_FILE_HEADER+E+f+b,dirRecord:R.CENTRAL_FILE_HEADER+A(C,2)+E+A(p.length,2)+"\0\0\0\0"+A(z,4)+A(n,4)+f+b+p}}var I=e("../utils"),i=e("../stream/GenericWorker"),O=e("../utf8"),B=e("../crc32"),R=e("../signature");function s(e,t,r,n){i.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=t,this.zipPlatform=r,this.encodeFileName=n,this.streamFiles=e,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}I.inherits(s,i),s.prototype.push=function(e){var t=e.meta.percent||0,r=this.entriesCount,n=this._sources.length;this.accumulate?this.contentBuffer.push(e):(this.bytesWritten+=e.data.length,i.prototype.push.call(this,{data:e.data,meta:{currentFile:this.currentFile,percent:r?(t+100*(r-n-1))/r:100}}))},s.prototype.openedSource=function(e){this.currentSourceOffset=this.bytesWritten,this.currentFile=e.file.name;var t=this.streamFiles&&!e.file.dir;if(t){var r=n(e,t,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},s.prototype.closedSource=function(e){this.accumulate=!1;var t=this.streamFiles&&!e.file.dir,r=n(e,t,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(r.dirRecord),t)this.push({data:function(e){return R.DATA_DESCRIPTOR+A(e.crc32,4)+A(e.compressedSize,4)+A(e.uncompressedSize,4)}(e),meta:{percent:100}});else for(this.push({data:r.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},s.prototype.flush=function(){for(var e=this.bytesWritten,t=0;t<this.dirRecords.length;t++)this.push({data:this.dirRecords[t],meta:{percent:100}});var r=this.bytesWritten-e,n=function(e,t,r,n,i){var s=I.transformTo("string",i(n));return R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+A(e,2)+A(e,2)+A(t,4)+A(r,4)+A(s.length,2)+s}(this.dirRecords.length,r,e,this.zipComment,this.encodeFileName);this.push({data:n,meta:{percent:100}})},s.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},s.prototype.registerPrevious=function(e){this._sources.push(e);var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.closedSource(t.previous.streamInfo),t._sources.length?t.prepareNextSource():t.end()}),e.on("error",function(e){t.error(e)}),this},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},s.prototype.error=function(e){var t=this._sources;if(!i.prototype.error.call(this,e))return!1;for(var r=0;r<t.length;r++)try{t[r].error(e)}catch(e){}return!0},s.prototype.lock=function(){i.prototype.lock.call(this);for(var e=this._sources,t=0;t<e.length;t++)e[t].lock()},t.exports=s},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(e,t,r){"use strict";var u=e("../compressions"),n=e("./ZipFileWorker");r.generateWorker=function(e,a,t){var o=new n(a.streamFiles,t,a.platform,a.encodeFileName),h=0;try{e.forEach(function(e,t){h++;var r=function(e,t){var r=e||t,n=u[r];if(!n)throw new Error(r+" is not a valid compression method !");return n}(t.options.compression,a.compression),n=t.options.compressionOptions||a.compressionOptions||{},i=t.dir,s=t.date;t._compressWorker(r,n).withStreamInfo("file",{name:e,dir:i,date:s,comment:t.comment||"",unixPermissions:t.unixPermissions,dosPermissions:t.dosPermissions}).pipe(o)}),o.entriesCount=h}catch(e){o.error(e)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(e,t,r){"use strict";function n(){if(!(this instanceof n))return new n;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files=Object.create(null),this.comment=null,this.root="",this.clone=function(){var e=new n;for(var t in this)"function"!=typeof this[t]&&(e[t]=this[t]);return e}}(n.prototype=e("./object")).loadAsync=e("./load"),n.support=e("./support"),n.defaults=e("./defaults"),n.version="3.10.1",n.loadAsync=function(e,t){return(new n).loadAsync(e,t)},n.external=e("./external"),t.exports=n},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(e,t,r){"use strict";var u=e("./utils"),i=e("./external"),n=e("./utf8"),s=e("./zipEntries"),a=e("./stream/Crc32Probe"),l=e("./nodejsUtils");function f(n){return new i.Promise(function(e,t){var r=n.decompressed.getContentWorker().pipe(new a);r.on("error",function(e){t(e)}).on("end",function(){r.streamInfo.crc32!==n.decompressed.crc32?t(new Error("Corrupted zip : CRC32 mismatch")):e()}).resume()})}t.exports=function(e,o){var h=this;return o=u.extend(o||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:n.utf8decode}),l.isNode&&l.isStream(e)?i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):u.prepareContent("the loaded zip file",e,!0,o.optimizedBinaryString,o.base64).then(function(e){var t=new s(o);return t.load(e),t}).then(function(e){var t=[i.Promise.resolve(e)],r=e.files;if(o.checkCRC32)for(var n=0;n<r.length;n++)t.push(f(r[n]));return i.Promise.all(t)}).then(function(e){for(var t=e.shift(),r=t.files,n=0;n<r.length;n++){var i=r[n],s=i.fileNameStr,a=u.resolve(i.fileNameStr);h.file(a,i.decompressed,{binary:!0,optimizedBinaryString:!0,date:i.date,dir:i.dir,comment:i.fileCommentStr.length?i.fileCommentStr:null,unixPermissions:i.unixPermissions,dosPermissions:i.dosPermissions,createFolders:o.createFolders}),i.dir||(h.file(a).unsafeOriginalName=s)}return t.zipComment.length&&(h.comment=t.zipComment),h})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../stream/GenericWorker");function s(e,t){i.call(this,"Nodejs stream input adapter for "+e),this._upstreamEnded=!1,this._bindStream(t)}n.inherits(s,i),s.prototype._bindStream=function(e){var t=this;(this._stream=e).pause(),e.on("data",function(e){t.push({data:e,meta:{percent:0}})}).on("error",function(e){t.isPaused?this.generatedError=e:t.error(e)}).on("end",function(){t.isPaused?t._upstreamEnded=!0:t.end()})},s.prototype.pause=function(){return!!i.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},t.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(e,t,r){"use strict";var i=e("readable-stream").Readable;function n(e,t,r){i.call(this,t),this._helper=e;var n=this;e.on("data",function(e,t){n.push(e)||n._helper.pause(),r&&r(t)}).on("error",function(e){n.emit("error",e)}).on("end",function(){n.push(null)})}e("../utils").inherits(n,i),n.prototype._read=function(){this._helper.resume()},t.exports=n},{"../utils":32,"readable-stream":16}],14:[function(e,t,r){"use strict";t.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(e,t){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(e,t);if("number"==typeof e)throw new Error('The "data" argument must not be a number');return new Buffer(e,t)},allocBuffer:function(e){if(Buffer.alloc)return Buffer.alloc(e);var t=new Buffer(e);return t.fill(0),t},isBuffer:function(e){return Buffer.isBuffer(e)},isStream:function(e){return e&&"function"==typeof e.on&&"function"==typeof e.pause&&"function"==typeof e.resume}}},{}],15:[function(e,t,r){"use strict";function s(e,t,r){var n,i=u.getTypeOf(t),s=u.extend(r||{},f);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(e=g(e)),s.createFolders&&(n=_(e))&&b.call(this,n,!0);var a="string"===i&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!a),(t instanceof c&&0===t.uncompressedSize||s.dir||!t||0===t.length)&&(s.base64=!1,s.binary=!0,t="",s.compression="STORE",i="string");var o=null;o=t instanceof c||t instanceof l?t:p.isNode&&p.isStream(t)?new m(e,t):u.prepareContent(e,t,s.binary,s.optimizedBinaryString,s.base64);var h=new d(e,o,s);this.files[e]=h}var i=e("./utf8"),u=e("./utils"),l=e("./stream/GenericWorker"),a=e("./stream/StreamHelper"),f=e("./defaults"),c=e("./compressedObject"),d=e("./zipObject"),o=e("./generate"),p=e("./nodejsUtils"),m=e("./nodejs/NodejsStreamInputAdapter"),_=function(e){"/"===e.slice(-1)&&(e=e.substring(0,e.length-1));var t=e.lastIndexOf("/");return 0<t?e.substring(0,t):""},g=function(e){return"/"!==e.slice(-1)&&(e+="/"),e},b=function(e,t){return t=void 0!==t?t:f.createFolders,e=g(e),this.files[e]||s.call(this,e,null,{dir:!0,createFolders:t}),this.files[e]};function h(e){return"[object RegExp]"===Object.prototype.toString.call(e)}var n={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(e){var t,r,n;for(t in this.files)n=this.files[t],(r=t.slice(this.root.length,t.length))&&t.slice(0,this.root.length)===this.root&&e(r,n)},filter:function(r){var n=[];return this.forEach(function(e,t){r(e,t)&&n.push(t)}),n},file:function(e,t,r){if(1!==arguments.length)return e=this.root+e,s.call(this,e,t,r),this;if(h(e)){var n=e;return this.filter(function(e,t){return!t.dir&&n.test(e)})}var i=this.files[this.root+e];return i&&!i.dir?i:null},folder:function(r){if(!r)return this;if(h(r))return this.filter(function(e,t){return t.dir&&r.test(e)});var e=this.root+r,t=b.call(this,e),n=this.clone();return n.root=t.name,n},remove:function(r){r=this.root+r;var e=this.files[r];if(e||("/"!==r.slice(-1)&&(r+="/"),e=this.files[r]),e&&!e.dir)delete this.files[r];else for(var t=this.filter(function(e,t){return t.name.slice(0,r.length)===r}),n=0;n<t.length;n++)delete this.files[t[n].name];return this},generate:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(e){var t,r={};try{if((r=u.extend(e||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:i.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");u.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var n=r.comment||this.comment||"";t=o.generateWorker(this,r,n)}catch(e){(t=new l("error")).error(e)}return new a(t,r.type||"string",r.mimeType)},generateAsync:function(e,t){return this.generateInternalStream(e).accumulate(t)},generateNodeStream:function(e,t){return(e=e||{}).type||(e.type="nodebuffer"),this.generateInternalStream(e).toNodejsStream(t)}};t.exports=n},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(e,t,r){"use strict";t.exports=e("stream")},{stream:void 0}],17:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e);for(var t=0;t<this.data.length;t++)e[t]=255&e[t]}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data[this.zero+e]},i.prototype.lastIndexOfSignature=function(e){for(var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===t&&this.data[s+1]===r&&this.data[s+2]===n&&this.data[s+3]===i)return s-this.zero;return-1},i.prototype.readAndCheckSignature=function(e){var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.readData(4);return t===s[0]&&r===s[1]&&n===s[2]&&i===s[3]},i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return[];var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],18:[function(e,t,r){"use strict";var n=e("../utils");function i(e){this.data=e,this.length=e.length,this.index=0,this.zero=0}i.prototype={checkOffset:function(e){this.checkIndex(this.index+e)},checkIndex:function(e){if(this.length<this.zero+e||e<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+e+"). Corrupted zip ?")},setIndex:function(e){this.checkIndex(e),this.index=e},skip:function(e){this.setIndex(this.index+e)},byteAt:function(){},readInt:function(e){var t,r=0;for(this.checkOffset(e),t=this.index+e-1;t>=this.index;t--)r=(r<<8)+this.byteAt(t);return this.index+=e,r},readString:function(e){return n.transformTo("string",this.readData(e))},readData:function(){},lastIndexOfSignature:function(){},readAndCheckSignature:function(){},readDate:function(){var e=this.readInt(4);return new Date(Date.UTC(1980+(e>>25&127),(e>>21&15)-1,e>>16&31,e>>11&31,e>>5&63,(31&e)<<1))}},t.exports=i},{"../utils":32}],19:[function(e,t,r){"use strict";var n=e("./Uint8ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data.charCodeAt(this.zero+e)},i.prototype.lastIndexOfSignature=function(e){return this.data.lastIndexOf(e)-this.zero},i.prototype.readAndCheckSignature=function(e){return e===this.readData(4)},i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],21:[function(e,t,r){"use strict";var n=e("./ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return new Uint8Array(0);var t=this.data.subarray(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./ArrayReader":17}],22:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../support"),s=e("./ArrayReader"),a=e("./StringReader"),o=e("./NodeBufferReader"),h=e("./Uint8ArrayReader");t.exports=function(e){var t=n.getTypeOf(e);return n.checkSupport(t),"string"!==t||i.uint8array?"nodebuffer"===t?new o(e):i.uint8array?new h(n.transformTo("uint8array",e)):new s(n.transformTo("array",e)):new a(e)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(e,t,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../utils");function s(e){n.call(this,"ConvertWorker to "+e),this.destType=e}i.inherits(s,n),s.prototype.processChunk=function(e){this.push({data:i.transformTo(this.destType,e.data),meta:e.meta})},t.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../crc32");function s(){n.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}e("../utils").inherits(s,n),s.prototype.processChunk=function(e){this.streamInfo.crc32=i(e.data,this.streamInfo.crc32||0),this.push(e)},t.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataLengthProbe for "+e),this.propName=e,this.withStreamInfo(e,0)}n.inherits(s,i),s.prototype.processChunk=function(e){if(e){var t=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=t+e.data.length}i.prototype.processChunk.call(this,e)},t.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataWorker");var t=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,e.then(function(e){t.dataIsReady=!0,t.data=e,t.max=e&&e.length||0,t.type=n.getTypeOf(e),t.isPaused||t._tickAndRepeat()},function(e){t.error(e)})}n.inherits(s,i),s.prototype.cleanUp=function(){i.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,n.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(n.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var e=null,t=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":e=this.data.substring(this.index,t);break;case"uint8array":e=this.data.subarray(this.index,t);break;case"array":case"nodebuffer":e=this.data.slice(this.index,t)}return this.index=t,this.push({data:e,meta:{percent:this.max?this.index/this.max*100:0}})},t.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(e,t,r){"use strict";function n(e){this.name=e||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}n.prototype={push:function(e){this.emit("data",e)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(e){this.emit("error",e)}return!0},error:function(e){return!this.isFinished&&(this.isPaused?this.generatedError=e:(this.isFinished=!0,this.emit("error",e),this.previous&&this.previous.error(e),this.cleanUp()),!0)},on:function(e,t){return this._listeners[e].push(t),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(e,t){if(this._listeners[e])for(var r=0;r<this._listeners[e].length;r++)this._listeners[e][r].call(this,t)},pipe:function(e){return e.registerPrevious(this)},registerPrevious:function(e){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=e.streamInfo,this.mergeStreamInfo(),this.previous=e;var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.end()}),e.on("error",function(e){t.error(e)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var e=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),e=!0),this.previous&&this.previous.resume(),!e},flush:function(){},processChunk:function(e){this.push(e)},withStreamInfo:function(e,t){return this.extraStreamInfo[e]=t,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var e in this.extraStreamInfo)Object.prototype.hasOwnProperty.call(this.extraStreamInfo,e)&&(this.streamInfo[e]=this.extraStreamInfo[e])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var e="Worker "+this.name;return this.previous?this.previous+" -> "+e:e}},t.exports=n},{}],29:[function(e,t,r){"use strict";var h=e("../utils"),i=e("./ConvertWorker"),s=e("./GenericWorker"),u=e("../base64"),n=e("../support"),a=e("../external"),o=null;if(n.nodestream)try{o=e("../nodejs/NodejsStreamOutputAdapter")}catch(e){}function l(e,o){return new a.Promise(function(t,r){var n=[],i=e._internalType,s=e._outputType,a=e._mimeType;e.on("data",function(e,t){n.push(e),o&&o(t)}).on("error",function(e){n=[],r(e)}).on("end",function(){try{var e=function(e,t,r){switch(e){case"blob":return h.newBlob(h.transformTo("arraybuffer",t),r);case"base64":return u.encode(t);default:return h.transformTo(e,t)}}(s,function(e,t){var r,n=0,i=null,s=0;for(r=0;r<t.length;r++)s+=t[r].length;switch(e){case"string":return t.join("");case"array":return Array.prototype.concat.apply([],t);case"uint8array":for(i=new Uint8Array(s),r=0;r<t.length;r++)i.set(t[r],n),n+=t[r].length;return i;case"nodebuffer":return Buffer.concat(t);default:throw new Error("concat : unsupported type '"+e+"'")}}(i,n),a);t(e)}catch(e){r(e)}n=[]}).resume()})}function f(e,t,r){var n=t;switch(t){case"blob":case"arraybuffer":n="uint8array";break;case"base64":n="string"}try{this._internalType=n,this._outputType=t,this._mimeType=r,h.checkSupport(n),this._worker=e.pipe(new i(n)),e.lock()}catch(e){this._worker=new s("error"),this._worker.error(e)}}f.prototype={accumulate:function(e){return l(this,e)},on:function(e,t){var r=this;return"data"===e?this._worker.on(e,function(e){t.call(r,e.data,e.meta)}):this._worker.on(e,function(){h.delay(t,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(e){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},e)}},t.exports=f},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(e,t,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var n=new ArrayBuffer(0);try{r.blob=0===new Blob([n],{type:"application/zip"}).size}catch(e){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);i.append(n),r.blob=0===i.getBlob("application/zip").size}catch(e){r.blob=!1}}}try{r.nodestream=!!e("readable-stream").Readable}catch(e){r.nodestream=!1}},{"readable-stream":16}],31:[function(e,t,s){"use strict";for(var o=e("./utils"),h=e("./support"),r=e("./nodejsUtils"),n=e("./stream/GenericWorker"),u=new Array(256),i=0;i<256;i++)u[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;u[254]=u[254]=1;function a(){n.call(this,"utf-8 decode"),this.leftOver=null}function l(){n.call(this,"utf-8 encode")}s.utf8encode=function(e){return h.nodebuffer?r.newBufferFrom(e,"utf-8"):function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=h.uint8array?new Uint8Array(o):new Array(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t}(e)},s.utf8decode=function(e){return h.nodebuffer?o.transformTo("nodebuffer",e).toString("utf-8"):function(e){var t,r,n,i,s=e.length,a=new Array(2*s);for(t=r=0;t<s;)if((n=e[t++])<128)a[r++]=n;else if(4<(i=u[n]))a[r++]=65533,t+=i-1;else{for(n&=2===i?31:3===i?15:7;1<i&&t<s;)n=n<<6|63&e[t++],i--;1<i?a[r++]=65533:n<65536?a[r++]=n:(n-=65536,a[r++]=55296|n>>10&1023,a[r++]=56320|1023&n)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(e=o.transformTo(h.uint8array?"uint8array":"array",e))},o.inherits(a,n),a.prototype.processChunk=function(e){var t=o.transformTo(h.uint8array?"uint8array":"array",e.data);if(this.leftOver&&this.leftOver.length){if(h.uint8array){var r=t;(t=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),t.set(r,this.leftOver.length)}else t=this.leftOver.concat(t);this.leftOver=null}var n=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}(t),i=t;n!==t.length&&(h.uint8array?(i=t.subarray(0,n),this.leftOver=t.subarray(n,t.length)):(i=t.slice(0,n),this.leftOver=t.slice(n,t.length))),this.push({data:s.utf8decode(i),meta:e.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(l,n),l.prototype.processChunk=function(e){this.push({data:s.utf8encode(e.data),meta:e.meta})},s.Utf8EncodeWorker=l},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(e,t,a){"use strict";var o=e("./support"),h=e("./base64"),r=e("./nodejsUtils"),u=e("./external");function n(e){return e}function l(e,t){for(var r=0;r<e.length;++r)t[r]=255&e.charCodeAt(r);return t}e("setimmediate"),a.newBlob=function(t,r){a.checkSupport("blob");try{return new Blob([t],{type:r})}catch(e){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return n.append(t),n.getBlob(r)}catch(e){throw new Error("Bug : can't construct the Blob.")}}};var i={stringifyByChunk:function(e,t,r){var n=[],i=0,s=e.length;if(s<=r)return String.fromCharCode.apply(null,e);for(;i<s;)"array"===t||"nodebuffer"===t?n.push(String.fromCharCode.apply(null,e.slice(i,Math.min(i+r,s)))):n.push(String.fromCharCode.apply(null,e.subarray(i,Math.min(i+r,s)))),i+=r;return n.join("")},stringifyByChar:function(e){for(var t="",r=0;r<e.length;r++)t+=String.fromCharCode(e[r]);return t},applyCanBeUsed:{uint8array:function(){try{return o.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(e){return!1}}(),nodebuffer:function(){try{return o.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(e){return!1}}()}};function s(e){var t=65536,r=a.getTypeOf(e),n=!0;if("uint8array"===r?n=i.applyCanBeUsed.uint8array:"nodebuffer"===r&&(n=i.applyCanBeUsed.nodebuffer),n)for(;1<t;)try{return i.stringifyByChunk(e,r,t)}catch(e){t=Math.floor(t/2)}return i.stringifyByChar(e)}function f(e,t){for(var r=0;r<e.length;r++)t[r]=e[r];return t}a.applyFromCharCode=s;var c={};c.string={string:n,array:function(e){return l(e,new Array(e.length))},arraybuffer:function(e){return c.string.uint8array(e).buffer},uint8array:function(e){return l(e,new Uint8Array(e.length))},nodebuffer:function(e){return l(e,r.allocBuffer(e.length))}},c.array={string:s,array:n,arraybuffer:function(e){return new Uint8Array(e).buffer},uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(e)}},c.arraybuffer={string:function(e){return s(new Uint8Array(e))},array:function(e){return f(new Uint8Array(e),new Array(e.byteLength))},arraybuffer:n,uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(new Uint8Array(e))}},c.uint8array={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return e.buffer},uint8array:n,nodebuffer:function(e){return r.newBufferFrom(e)}},c.nodebuffer={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return c.nodebuffer.uint8array(e).buffer},uint8array:function(e){return f(e,new Uint8Array(e.length))},nodebuffer:n},a.transformTo=function(e,t){if(t=t||"",!e)return t;a.checkSupport(e);var r=a.getTypeOf(t);return c[r][e](t)},a.resolve=function(e){for(var t=e.split("/"),r=[],n=0;n<t.length;n++){var i=t[n];"."===i||""===i&&0!==n&&n!==t.length-1||(".."===i?r.pop():r.push(i))}return r.join("/")},a.getTypeOf=function(e){return"string"==typeof e?"string":"[object Array]"===Object.prototype.toString.call(e)?"array":o.nodebuffer&&r.isBuffer(e)?"nodebuffer":o.uint8array&&e instanceof Uint8Array?"uint8array":o.arraybuffer&&e instanceof ArrayBuffer?"arraybuffer":void 0},a.checkSupport=function(e){if(!o[e.toLowerCase()])throw new Error(e+" is not supported by this platform")},a.MAX_VALUE_16BITS=65535,a.MAX_VALUE_32BITS=-1,a.pretty=function(e){var t,r,n="";for(r=0;r<(e||"").length;r++)n+="\\x"+((t=e.charCodeAt(r))<16?"0":"")+t.toString(16).toUpperCase();return n},a.delay=function(e,t,r){setImmediate(function(){e.apply(r||null,t||[])})},a.inherits=function(e,t){function r(){}r.prototype=t.prototype,e.prototype=new r},a.extend=function(){var e,t,r={};for(e=0;e<arguments.length;e++)for(t in arguments[e])Object.prototype.hasOwnProperty.call(arguments[e],t)&&void 0===r[t]&&(r[t]=arguments[e][t]);return r},a.prepareContent=function(r,e,n,i,s){return u.Promise.resolve(e).then(function(n){return o.blob&&(n instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(n)))&&"undefined"!=typeof FileReader?new u.Promise(function(t,r){var e=new FileReader;e.onload=function(e){t(e.target.result)},e.onerror=function(e){r(e.target.error)},e.readAsArrayBuffer(n)}):n}).then(function(e){var t=a.getTypeOf(e);return t?("arraybuffer"===t?e=a.transformTo("uint8array",e):"string"===t&&(s?e=h.decode(e):n&&!0!==i&&(e=function(e){return l(e,o.uint8array?new Uint8Array(e.length):new Array(e.length))}(e))),e):u.Promise.reject(new Error("Can't read the data of '"+r+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,setimmediate:54}],33:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),i=e("./utils"),s=e("./signature"),a=e("./zipEntry"),o=e("./support");function h(e){this.files=[],this.loadOptions=e}h.prototype={checkSignature:function(e){if(!this.reader.readAndCheckSignature(e)){this.reader.index-=4;var t=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+i.pretty(t)+", expected "+i.pretty(e)+")")}},isSignature:function(e,t){var r=this.reader.index;this.reader.setIndex(e);var n=this.reader.readString(4)===t;return this.reader.setIndex(r),n},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var e=this.reader.readData(this.zipCommentLength),t=o.uint8array?"uint8array":"array",r=i.transformTo(t,e);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var e,t,r,n=this.zip64EndOfCentralSize-44;0<n;)e=this.reader.readInt(2),t=this.reader.readInt(4),r=this.reader.readData(t),this.zip64ExtensibleData[e]={id:e,length:t,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var e,t;for(e=0;e<this.files.length;e++)t=this.files[e],this.reader.setIndex(t.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),t.readLocalPart(this.reader),t.handleUTF8(),t.processAttributes()},readCentralDir:function(){var e;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(e=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(e);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var e=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(e<0)throw!this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html"):new Error("Corrupted zip: can't find end of central directory");this.reader.setIndex(e);var t=e;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===i.MAX_VALUE_16BITS||this.diskWithCentralDirStart===i.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===i.MAX_VALUE_16BITS||this.centralDirRecords===i.MAX_VALUE_16BITS||this.centralDirSize===i.MAX_VALUE_32BITS||this.centralDirOffset===i.MAX_VALUE_32BITS){if(this.zip64=!0,(e=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(e),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var n=t-r;if(0<n)this.isSignature(t,s.CENTRAL_FILE_HEADER)||(this.reader.zero=n);else if(n<0)throw new Error("Corrupted zip: missing "+Math.abs(n)+" bytes.")},prepareReader:function(e){this.reader=n(e)},load:function(e){this.prepareReader(e),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},t.exports=h},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utils":32,"./zipEntry":34}],34:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),s=e("./utils"),i=e("./compressedObject"),a=e("./crc32"),o=e("./utf8"),h=e("./compressions"),u=e("./support");function l(e,t){this.options=e,this.loadOptions=t}l.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(e){var t,r;if(e.skip(22),this.fileNameLength=e.readInt(2),r=e.readInt(2),this.fileName=e.readData(this.fileNameLength),e.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(t=function(e){for(var t in h)if(Object.prototype.hasOwnProperty.call(h,t)&&h[t].magic===e)return h[t];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new i(this.compressedSize,this.uncompressedSize,this.crc32,t,e.readData(this.compressedSize))},readCentralPart:function(e){this.versionMadeBy=e.readInt(2),e.skip(2),this.bitFlag=e.readInt(2),this.compressionMethod=e.readString(2),this.date=e.readDate(),this.crc32=e.readInt(4),this.compressedSize=e.readInt(4),this.uncompressedSize=e.readInt(4);var t=e.readInt(2);if(this.extraFieldsLength=e.readInt(2),this.fileCommentLength=e.readInt(2),this.diskNumberStart=e.readInt(2),this.internalFileAttributes=e.readInt(2),this.externalFileAttributes=e.readInt(4),this.localHeaderOffset=e.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");e.skip(t),this.readExtraFields(e),this.parseZIP64ExtraField(e),this.fileComment=e.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var e=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==e&&(this.dosPermissions=63&this.externalFileAttributes),3==e&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(){if(this.extraFields[1]){var e=n(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=e.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=e.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=e.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=e.readInt(4))}},readExtraFields:function(e){var t,r,n,i=e.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});e.index+4<i;)t=e.readInt(2),r=e.readInt(2),n=e.readData(r),this.extraFields[t]={id:t,length:r,value:n};e.setIndex(i)},handleUTF8:function(){var e=u.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var t=this.findExtraFieldUnicodePath();if(null!==t)this.fileNameStr=t;else{var r=s.transformTo(e,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var n=this.findExtraFieldUnicodeComment();if(null!==n)this.fileCommentStr=n;else{var i=s.transformTo(e,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(i)}}},findExtraFieldUnicodePath:function(){var e=this.extraFields[28789];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileName)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null},findExtraFieldUnicodeComment:function(){var e=this.extraFields[25461];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileComment)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null}},t.exports=l},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(e,t,r){"use strict";function n(e,t,r){this.name=e,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=t,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=e("./stream/StreamHelper"),i=e("./stream/DataWorker"),a=e("./utf8"),o=e("./compressedObject"),h=e("./stream/GenericWorker");n.prototype={internalStream:function(e){var t=null,r="string";try{if(!e)throw new Error("No output type specified.");var n="string"===(r=e.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),t=this._decompressWorker();var i=!this._dataBinary;i&&!n&&(t=t.pipe(new a.Utf8EncodeWorker)),!i&&n&&(t=t.pipe(new a.Utf8DecodeWorker))}catch(e){(t=new h("error")).error(e)}return new s(t,r,"")},async:function(e,t){return this.internalStream(e).accumulate(t)},nodeStream:function(e,t){return this.internalStream(e||"nodebuffer").toNodejsStream(t)},_compressWorker:function(e,t){if(this._data instanceof o&&this._data.compression.magic===e.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,e,t)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof h?this._data:new i(this._data)}};for(var u=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],l=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},f=0;f<u.length;f++)n.prototype[u[f]]=l;t.exports=n},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(e,l,t){(function(t){"use strict";var r,n,e=t.MutationObserver||t.WebKitMutationObserver;if(e){var i=0,s=new e(u),a=t.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=i=++i%2}}else if(t.setImmediate||void 0===t.MessageChannel)r="document"in t&&"onreadystatechange"in t.document.createElement("script")?function(){var e=t.document.createElement("script");e.onreadystatechange=function(){u(),e.onreadystatechange=null,e.parentNode.removeChild(e),e=null},t.document.documentElement.appendChild(e)}:function(){setTimeout(u,0)};else{var o=new t.MessageChannel;o.port1.onmessage=u,r=function(){o.port2.postMessage(0)}}var h=[];function u(){var e,t;n=!0;for(var r=h.length;r;){for(t=h,h=[],e=-1;++e<r;)t[e]();r=h.length}n=!1}l.exports=function(e){1!==h.push(e)||n||r()}}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(e,t,r){"use strict";var i=e("immediate");function u(){}var l={},s=["REJECTED"],a=["FULFILLED"],n=["PENDING"];function o(e){if("function"!=typeof e)throw new TypeError("resolver must be a function");this.state=n,this.queue=[],this.outcome=void 0,e!==u&&d(this,e)}function h(e,t,r){this.promise=e,"function"==typeof t&&(this.onFulfilled=t,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function f(t,r,n){i(function(){var e;try{e=r(n)}catch(e){return l.reject(t,e)}e===t?l.reject(t,new TypeError("Cannot resolve promise with itself")):l.resolve(t,e)})}function c(e){var t=e&&e.then;if(e&&("object"==typeof e||"function"==typeof e)&&"function"==typeof t)return function(){t.apply(e,arguments)}}function d(t,e){var r=!1;function n(e){r||(r=!0,l.reject(t,e))}function i(e){r||(r=!0,l.resolve(t,e))}var s=p(function(){e(i,n)});"error"===s.status&&n(s.value)}function p(e,t){var r={};try{r.value=e(t),r.status="success"}catch(e){r.status="error",r.value=e}return r}(t.exports=o).prototype.finally=function(t){if("function"!=typeof t)return this;var r=this.constructor;return this.then(function(e){return r.resolve(t()).then(function(){return e})},function(e){return r.resolve(t()).then(function(){throw e})})},o.prototype.catch=function(e){return this.then(null,e)},o.prototype.then=function(e,t){if("function"!=typeof e&&this.state===a||"function"!=typeof t&&this.state===s)return this;var r=new this.constructor(u);this.state!==n?f(r,this.state===a?e:t,this.outcome):this.queue.push(new h(r,e,t));return r},h.prototype.callFulfilled=function(e){l.resolve(this.promise,e)},h.prototype.otherCallFulfilled=function(e){f(this.promise,this.onFulfilled,e)},h.prototype.callRejected=function(e){l.reject(this.promise,e)},h.prototype.otherCallRejected=function(e){f(this.promise,this.onRejected,e)},l.resolve=function(e,t){var r=p(c,t);if("error"===r.status)return l.reject(e,r.value);var n=r.value;if(n)d(e,n);else{e.state=a,e.outcome=t;for(var i=-1,s=e.queue.length;++i<s;)e.queue[i].callFulfilled(t)}return e},l.reject=function(e,t){e.state=s,e.outcome=t;for(var r=-1,n=e.queue.length;++r<n;)e.queue[r].callRejected(t);return e},o.resolve=function(e){if(e instanceof this)return e;return l.resolve(new this(u),e)},o.reject=function(e){var t=new this(u);return l.reject(t,e)},o.all=function(e){var r=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var n=e.length,i=!1;if(!n)return this.resolve([]);var s=new Array(n),a=0,t=-1,o=new this(u);for(;++t<n;)h(e[t],t);return o;function h(e,t){r.resolve(e).then(function(e){s[t]=e,++a!==n||i||(i=!0,l.resolve(o,s))},function(e){i||(i=!0,l.reject(o,e))})}},o.race=function(e){var t=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var r=e.length,n=!1;if(!r)return this.resolve([]);var i=-1,s=new this(u);for(;++i<r;)a=e[i],t.resolve(a).then(function(e){n||(n=!0,l.resolve(s,e))},function(e){n||(n=!0,l.reject(s,e))});var a;return s}},{immediate:36}],38:[function(e,t,r){"use strict";var n={};(0,e("./lib/utils/common").assign)(n,e("./lib/deflate"),e("./lib/inflate"),e("./lib/zlib/constants")),t.exports=n},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(e,t,r){"use strict";var a=e("./zlib/deflate"),o=e("./utils/common"),h=e("./utils/strings"),i=e("./zlib/messages"),s=e("./zlib/zstream"),u=Object.prototype.toString,l=0,f=-1,c=0,d=8;function p(e){if(!(this instanceof p))return new p(e);this.options=o.assign({level:f,method:d,chunkSize:16384,windowBits:15,memLevel:8,strategy:c,to:""},e||{});var t=this.options;t.raw&&0<t.windowBits?t.windowBits=-t.windowBits:t.gzip&&0<t.windowBits&&t.windowBits<16&&(t.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,t.level,t.method,t.windowBits,t.memLevel,t.strategy);if(r!==l)throw new Error(i[r]);if(t.header&&a.deflateSetHeader(this.strm,t.header),t.dictionary){var n;if(n="string"==typeof t.dictionary?h.string2buf(t.dictionary):"[object ArrayBuffer]"===u.call(t.dictionary)?new Uint8Array(t.dictionary):t.dictionary,(r=a.deflateSetDictionary(this.strm,n))!==l)throw new Error(i[r]);this._dict_set=!0}}function n(e,t){var r=new p(t);if(r.push(e,!0),r.err)throw r.msg||i[r.err];return r.result}p.prototype.push=function(e,t){var r,n,i=this.strm,s=this.options.chunkSize;if(this.ended)return!1;n=t===~~t?t:!0===t?4:0,"string"==typeof e?i.input=h.string2buf(e):"[object ArrayBuffer]"===u.call(e)?i.input=new Uint8Array(e):i.input=e,i.next_in=0,i.avail_in=i.input.length;do{if(0===i.avail_out&&(i.output=new o.Buf8(s),i.next_out=0,i.avail_out=s),1!==(r=a.deflate(i,n))&&r!==l)return this.onEnd(r),!(this.ended=!0);0!==i.avail_out&&(0!==i.avail_in||4!==n&&2!==n)||("string"===this.options.to?this.onData(h.buf2binstring(o.shrinkBuf(i.output,i.next_out))):this.onData(o.shrinkBuf(i.output,i.next_out)))}while((0<i.avail_in||0===i.avail_out)&&1!==r);return 4===n?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===l):2!==n||(this.onEnd(l),!(i.avail_out=0))},p.prototype.onData=function(e){this.chunks.push(e)},p.prototype.onEnd=function(e){e===l&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Deflate=p,r.deflate=n,r.deflateRaw=function(e,t){return(t=t||{}).raw=!0,n(e,t)},r.gzip=function(e,t){return(t=t||{}).gzip=!0,n(e,t)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(e,t,r){"use strict";var c=e("./zlib/inflate"),d=e("./utils/common"),p=e("./utils/strings"),m=e("./zlib/constants"),n=e("./zlib/messages"),i=e("./zlib/zstream"),s=e("./zlib/gzheader"),_=Object.prototype.toString;function a(e){if(!(this instanceof a))return new a(e);this.options=d.assign({chunkSize:16384,windowBits:0,to:""},e||{});var t=this.options;t.raw&&0<=t.windowBits&&t.windowBits<16&&(t.windowBits=-t.windowBits,0===t.windowBits&&(t.windowBits=-15)),!(0<=t.windowBits&&t.windowBits<16)||e&&e.windowBits||(t.windowBits+=32),15<t.windowBits&&t.windowBits<48&&0==(15&t.windowBits)&&(t.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new i,this.strm.avail_out=0;var r=c.inflateInit2(this.strm,t.windowBits);if(r!==m.Z_OK)throw new Error(n[r]);this.header=new s,c.inflateGetHeader(this.strm,this.header)}function o(e,t){var r=new a(t);if(r.push(e,!0),r.err)throw r.msg||n[r.err];return r.result}a.prototype.push=function(e,t){var r,n,i,s,a,o,h=this.strm,u=this.options.chunkSize,l=this.options.dictionary,f=!1;if(this.ended)return!1;n=t===~~t?t:!0===t?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof e?h.input=p.binstring2buf(e):"[object ArrayBuffer]"===_.call(e)?h.input=new Uint8Array(e):h.input=e,h.next_in=0,h.avail_in=h.input.length;do{if(0===h.avail_out&&(h.output=new d.Buf8(u),h.next_out=0,h.avail_out=u),(r=c.inflate(h,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&l&&(o="string"==typeof l?p.string2buf(l):"[object ArrayBuffer]"===_.call(l)?new Uint8Array(l):l,r=c.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===f&&(r=m.Z_OK,f=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);h.next_out&&(0!==h.avail_out&&r!==m.Z_STREAM_END&&(0!==h.avail_in||n!==m.Z_FINISH&&n!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(i=p.utf8border(h.output,h.next_out),s=h.next_out-i,a=p.buf2string(h.output,i),h.next_out=s,h.avail_out=u-s,s&&d.arraySet(h.output,h.output,i,s,0),this.onData(a)):this.onData(d.shrinkBuf(h.output,h.next_out)))),0===h.avail_in&&0===h.avail_out&&(f=!0)}while((0<h.avail_in||0===h.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(n=m.Z_FINISH),n===m.Z_FINISH?(r=c.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):n!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(h.avail_out=0))},a.prototype.onData=function(e){this.chunks.push(e)},a.prototype.onEnd=function(e){e===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=d.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(e,t){return(t=t||{}).raw=!0,o(e,t)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(e){for(var t=Array.prototype.slice.call(arguments,1);t.length;){var r=t.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var n in r)r.hasOwnProperty(n)&&(e[n]=r[n])}}return e},r.shrinkBuf=function(e,t){return e.length===t?e:e.subarray?e.subarray(0,t):(e.length=t,e)};var i={arraySet:function(e,t,r,n,i){if(t.subarray&&e.subarray)e.set(t.subarray(r,r+n),i);else for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){var t,r,n,i,s,a;for(t=n=0,r=e.length;t<r;t++)n+=e[t].length;for(a=new Uint8Array(n),t=i=0,r=e.length;t<r;t++)s=e[t],a.set(s,i),i+=s.length;return a}},s={arraySet:function(e,t,r,n,i){for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){return[].concat.apply([],e)}};r.setTyped=function(e){e?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,i)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(n)},{}],42:[function(e,t,r){"use strict";var h=e("./common"),i=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(e){i=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(e){s=!1}for(var u=new h.Buf8(256),n=0;n<256;n++)u[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;function l(e,t){if(t<65537&&(e.subarray&&s||!e.subarray&&i))return String.fromCharCode.apply(null,h.shrinkBuf(e,t));for(var r="",n=0;n<t;n++)r+=String.fromCharCode(e[n]);return r}u[254]=u[254]=1,r.string2buf=function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=new h.Buf8(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t},r.buf2binstring=function(e){return l(e,e.length)},r.binstring2buf=function(e){for(var t=new h.Buf8(e.length),r=0,n=t.length;r<n;r++)t[r]=e.charCodeAt(r);return t},r.buf2string=function(e,t){var r,n,i,s,a=t||e.length,o=new Array(2*a);for(r=n=0;r<a;)if((i=e[r++])<128)o[n++]=i;else if(4<(s=u[i]))o[n++]=65533,r+=s-1;else{for(i&=2===s?31:3===s?15:7;1<s&&r<a;)i=i<<6|63&e[r++],s--;1<s?o[n++]=65533:i<65536?o[n++]=i:(i-=65536,o[n++]=55296|i>>10&1023,o[n++]=56320|1023&i)}return l(o,n)},r.utf8border=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}},{"./common":41}],43:[function(e,t,r){"use strict";t.exports=function(e,t,r,n){for(var i=65535&e|0,s=e>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(i=i+t[n++]|0)|0,--a;);i%=65521,s%=65521}return i|s<<16|0}},{}],44:[function(e,t,r){"use strict";t.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(e,t,r){"use strict";var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}},{}],46:[function(e,t,r){"use strict";var h,c=e("../utils/common"),u=e("./trees"),d=e("./adler32"),p=e("./crc32"),n=e("./messages"),l=0,f=4,m=0,_=-2,g=-1,b=4,i=2,v=8,y=9,s=286,a=30,o=19,w=2*s+1,k=15,x=3,S=258,z=S+x+1,C=42,E=113,A=1,I=2,O=3,B=4;function R(e,t){return e.msg=n[t],t}function T(e){return(e<<1)-(4<e?9:0)}function D(e){for(var t=e.length;0<=--t;)e[t]=0}function F(e){var t=e.state,r=t.pending;r>e.avail_out&&(r=e.avail_out),0!==r&&(c.arraySet(e.output,t.pending_buf,t.pending_out,r,e.next_out),e.next_out+=r,t.pending_out+=r,e.total_out+=r,e.avail_out-=r,t.pending-=r,0===t.pending&&(t.pending_out=0))}function N(e,t){u._tr_flush_block(e,0<=e.block_start?e.block_start:-1,e.strstart-e.block_start,t),e.block_start=e.strstart,F(e.strm)}function U(e,t){e.pending_buf[e.pending++]=t}function P(e,t){e.pending_buf[e.pending++]=t>>>8&255,e.pending_buf[e.pending++]=255&t}function L(e,t){var r,n,i=e.max_chain_length,s=e.strstart,a=e.prev_length,o=e.nice_match,h=e.strstart>e.w_size-z?e.strstart-(e.w_size-z):0,u=e.window,l=e.w_mask,f=e.prev,c=e.strstart+S,d=u[s+a-1],p=u[s+a];e.prev_length>=e.good_match&&(i>>=2),o>e.lookahead&&(o=e.lookahead);do{if(u[(r=t)+a]===p&&u[r+a-1]===d&&u[r]===u[s]&&u[++r]===u[s+1]){s+=2,r++;do{}while(u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&s<c);if(n=S-(c-s),s=c-S,a<n){if(e.match_start=t,o<=(a=n))break;d=u[s+a-1],p=u[s+a]}}}while((t=f[t&l])>h&&0!=--i);return a<=e.lookahead?a:e.lookahead}function j(e){var t,r,n,i,s,a,o,h,u,l,f=e.w_size;do{if(i=e.window_size-e.lookahead-e.strstart,e.strstart>=f+(f-z)){for(c.arraySet(e.window,e.window,f,f,0),e.match_start-=f,e.strstart-=f,e.block_start-=f,t=r=e.hash_size;n=e.head[--t],e.head[t]=f<=n?n-f:0,--r;);for(t=r=f;n=e.prev[--t],e.prev[t]=f<=n?n-f:0,--r;);i+=f}if(0===e.strm.avail_in)break;if(a=e.strm,o=e.window,h=e.strstart+e.lookahead,u=i,l=void 0,l=a.avail_in,u<l&&(l=u),r=0===l?0:(a.avail_in-=l,c.arraySet(o,a.input,a.next_in,l,h),1===a.state.wrap?a.adler=d(a.adler,o,l,h):2===a.state.wrap&&(a.adler=p(a.adler,o,l,h)),a.next_in+=l,a.total_in+=l,l),e.lookahead+=r,e.lookahead+e.insert>=x)for(s=e.strstart-e.insert,e.ins_h=e.window[s],e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+1])&e.hash_mask;e.insert&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+x-1])&e.hash_mask,e.prev[s&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=s,s++,e.insert--,!(e.lookahead+e.insert<x)););}while(e.lookahead<z&&0!==e.strm.avail_in)}function Z(e,t){for(var r,n;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!==r&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r)),e.match_length>=x)if(n=u._tr_tally(e,e.strstart-e.match_start,e.match_length-x),e.lookahead-=e.match_length,e.match_length<=e.max_lazy_match&&e.lookahead>=x){for(e.match_length--;e.strstart++,e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart,0!=--e.match_length;);e.strstart++}else e.strstart+=e.match_length,e.match_length=0,e.ins_h=e.window[e.strstart],e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+1])&e.hash_mask;else n=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++;if(n&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function W(e,t){for(var r,n,i;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),e.prev_length=e.match_length,e.prev_match=e.match_start,e.match_length=x-1,0!==r&&e.prev_length<e.max_lazy_match&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r),e.match_length<=5&&(1===e.strategy||e.match_length===x&&4096<e.strstart-e.match_start)&&(e.match_length=x-1)),e.prev_length>=x&&e.match_length<=e.prev_length){for(i=e.strstart+e.lookahead-x,n=u._tr_tally(e,e.strstart-1-e.prev_match,e.prev_length-x),e.lookahead-=e.prev_length-1,e.prev_length-=2;++e.strstart<=i&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!=--e.prev_length;);if(e.match_available=0,e.match_length=x-1,e.strstart++,n&&(N(e,!1),0===e.strm.avail_out))return A}else if(e.match_available){if((n=u._tr_tally(e,0,e.window[e.strstart-1]))&&N(e,!1),e.strstart++,e.lookahead--,0===e.strm.avail_out)return A}else e.match_available=1,e.strstart++,e.lookahead--}return e.match_available&&(n=u._tr_tally(e,0,e.window[e.strstart-1]),e.match_available=0),e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function M(e,t,r,n,i){this.good_length=e,this.max_lazy=t,this.nice_length=r,this.max_chain=n,this.func=i}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=v,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new c.Buf16(2*w),this.dyn_dtree=new c.Buf16(2*(2*a+1)),this.bl_tree=new c.Buf16(2*(2*o+1)),D(this.dyn_ltree),D(this.dyn_dtree),D(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new c.Buf16(k+1),this.heap=new c.Buf16(2*s+1),D(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new c.Buf16(2*s+1),D(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function G(e){var t;return e&&e.state?(e.total_in=e.total_out=0,e.data_type=i,(t=e.state).pending=0,t.pending_out=0,t.wrap<0&&(t.wrap=-t.wrap),t.status=t.wrap?C:E,e.adler=2===t.wrap?0:1,t.last_flush=l,u._tr_init(t),m):R(e,_)}function K(e){var t=G(e);return t===m&&function(e){e.window_size=2*e.w_size,D(e.head),e.max_lazy_match=h[e.level].max_lazy,e.good_match=h[e.level].good_length,e.nice_match=h[e.level].nice_length,e.max_chain_length=h[e.level].max_chain,e.strstart=0,e.block_start=0,e.lookahead=0,e.insert=0,e.match_length=e.prev_length=x-1,e.match_available=0,e.ins_h=0}(e.state),t}function Y(e,t,r,n,i,s){if(!e)return _;var a=1;if(t===g&&(t=6),n<0?(a=0,n=-n):15<n&&(a=2,n-=16),i<1||y<i||r!==v||n<8||15<n||t<0||9<t||s<0||b<s)return R(e,_);8===n&&(n=9);var o=new H;return(e.state=o).strm=e,o.wrap=a,o.gzhead=null,o.w_bits=n,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=i+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+x-1)/x),o.window=new c.Buf8(2*o.w_size),o.head=new c.Buf16(o.hash_size),o.prev=new c.Buf16(o.w_size),o.lit_bufsize=1<<i+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new c.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=t,o.strategy=s,o.method=r,K(e)}h=[new M(0,0,0,0,function(e,t){var r=65535;for(r>e.pending_buf_size-5&&(r=e.pending_buf_size-5);;){if(e.lookahead<=1){if(j(e),0===e.lookahead&&t===l)return A;if(0===e.lookahead)break}e.strstart+=e.lookahead,e.lookahead=0;var n=e.block_start+r;if((0===e.strstart||e.strstart>=n)&&(e.lookahead=e.strstart-n,e.strstart=n,N(e,!1),0===e.strm.avail_out))return A;if(e.strstart-e.block_start>=e.w_size-z&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):(e.strstart>e.block_start&&(N(e,!1),e.strm.avail_out),A)}),new M(4,4,8,4,Z),new M(4,5,16,8,Z),new M(4,6,32,32,Z),new M(4,4,16,16,W),new M(8,16,32,32,W),new M(8,16,128,128,W),new M(8,32,128,256,W),new M(32,128,258,1024,W),new M(32,258,258,4096,W)],r.deflateInit=function(e,t){return Y(e,t,v,15,8,0)},r.deflateInit2=Y,r.deflateReset=K,r.deflateResetKeep=G,r.deflateSetHeader=function(e,t){return e&&e.state?2!==e.state.wrap?_:(e.state.gzhead=t,m):_},r.deflate=function(e,t){var r,n,i,s;if(!e||!e.state||5<t||t<0)return e?R(e,_):_;if(n=e.state,!e.output||!e.input&&0!==e.avail_in||666===n.status&&t!==f)return R(e,0===e.avail_out?-5:_);if(n.strm=e,r=n.last_flush,n.last_flush=t,n.status===C)if(2===n.wrap)e.adler=0,U(n,31),U(n,139),U(n,8),n.gzhead?(U(n,(n.gzhead.text?1:0)+(n.gzhead.hcrc?2:0)+(n.gzhead.extra?4:0)+(n.gzhead.name?8:0)+(n.gzhead.comment?16:0)),U(n,255&n.gzhead.time),U(n,n.gzhead.time>>8&255),U(n,n.gzhead.time>>16&255),U(n,n.gzhead.time>>24&255),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,255&n.gzhead.os),n.gzhead.extra&&n.gzhead.extra.length&&(U(n,255&n.gzhead.extra.length),U(n,n.gzhead.extra.length>>8&255)),n.gzhead.hcrc&&(e.adler=p(e.adler,n.pending_buf,n.pending,0)),n.gzindex=0,n.status=69):(U(n,0),U(n,0),U(n,0),U(n,0),U(n,0),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,3),n.status=E);else{var a=v+(n.w_bits-8<<4)<<8;a|=(2<=n.strategy||n.level<2?0:n.level<6?1:6===n.level?2:3)<<6,0!==n.strstart&&(a|=32),a+=31-a%31,n.status=E,P(n,a),0!==n.strstart&&(P(n,e.adler>>>16),P(n,65535&e.adler)),e.adler=1}if(69===n.status)if(n.gzhead.extra){for(i=n.pending;n.gzindex<(65535&n.gzhead.extra.length)&&(n.pending!==n.pending_buf_size||(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending!==n.pending_buf_size));)U(n,255&n.gzhead.extra[n.gzindex]),n.gzindex++;n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),n.gzindex===n.gzhead.extra.length&&(n.gzindex=0,n.status=73)}else n.status=73;if(73===n.status)if(n.gzhead.name){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.name.length?255&n.gzhead.name.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.gzindex=0,n.status=91)}else n.status=91;if(91===n.status)if(n.gzhead.comment){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.comment.length?255&n.gzhead.comment.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.status=103)}else n.status=103;if(103===n.status&&(n.gzhead.hcrc?(n.pending+2>n.pending_buf_size&&F(e),n.pending+2<=n.pending_buf_size&&(U(n,255&e.adler),U(n,e.adler>>8&255),e.adler=0,n.status=E)):n.status=E),0!==n.pending){if(F(e),0===e.avail_out)return n.last_flush=-1,m}else if(0===e.avail_in&&T(t)<=T(r)&&t!==f)return R(e,-5);if(666===n.status&&0!==e.avail_in)return R(e,-5);if(0!==e.avail_in||0!==n.lookahead||t!==l&&666!==n.status){var o=2===n.strategy?function(e,t){for(var r;;){if(0===e.lookahead&&(j(e),0===e.lookahead)){if(t===l)return A;break}if(e.match_length=0,r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++,r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):3===n.strategy?function(e,t){for(var r,n,i,s,a=e.window;;){if(e.lookahead<=S){if(j(e),e.lookahead<=S&&t===l)return A;if(0===e.lookahead)break}if(e.match_length=0,e.lookahead>=x&&0<e.strstart&&(n=a[i=e.strstart-1])===a[++i]&&n===a[++i]&&n===a[++i]){s=e.strstart+S;do{}while(n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&i<s);e.match_length=S-(s-i),e.match_length>e.lookahead&&(e.match_length=e.lookahead)}if(e.match_length>=x?(r=u._tr_tally(e,1,e.match_length-x),e.lookahead-=e.match_length,e.strstart+=e.match_length,e.match_length=0):(r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++),r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):h[n.level].func(n,t);if(o!==O&&o!==B||(n.status=666),o===A||o===O)return 0===e.avail_out&&(n.last_flush=-1),m;if(o===I&&(1===t?u._tr_align(n):5!==t&&(u._tr_stored_block(n,0,0,!1),3===t&&(D(n.head),0===n.lookahead&&(n.strstart=0,n.block_start=0,n.insert=0))),F(e),0===e.avail_out))return n.last_flush=-1,m}return t!==f?m:n.wrap<=0?1:(2===n.wrap?(U(n,255&e.adler),U(n,e.adler>>8&255),U(n,e.adler>>16&255),U(n,e.adler>>24&255),U(n,255&e.total_in),U(n,e.total_in>>8&255),U(n,e.total_in>>16&255),U(n,e.total_in>>24&255)):(P(n,e.adler>>>16),P(n,65535&e.adler)),F(e),0<n.wrap&&(n.wrap=-n.wrap),0!==n.pending?m:1)},r.deflateEnd=function(e){var t;return e&&e.state?(t=e.state.status)!==C&&69!==t&&73!==t&&91!==t&&103!==t&&t!==E&&666!==t?R(e,_):(e.state=null,t===E?R(e,-3):m):_},r.deflateSetDictionary=function(e,t){var r,n,i,s,a,o,h,u,l=t.length;if(!e||!e.state)return _;if(2===(s=(r=e.state).wrap)||1===s&&r.status!==C||r.lookahead)return _;for(1===s&&(e.adler=d(e.adler,t,l,0)),r.wrap=0,l>=r.w_size&&(0===s&&(D(r.head),r.strstart=0,r.block_start=0,r.insert=0),u=new c.Buf8(r.w_size),c.arraySet(u,t,l-r.w_size,r.w_size,0),t=u,l=r.w_size),a=e.avail_in,o=e.next_in,h=e.input,e.avail_in=l,e.next_in=0,e.input=t,j(r);r.lookahead>=x;){for(n=r.strstart,i=r.lookahead-(x-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[n+x-1])&r.hash_mask,r.prev[n&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=n,n++,--i;);r.strstart=n,r.lookahead=x-1,j(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=x-1,r.match_available=0,e.next_in=o,e.input=h,e.avail_in=a,r.wrap=s,m},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(e,t,r){"use strict";t.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(e,t,r){"use strict";t.exports=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C;r=e.state,n=e.next_in,z=e.input,i=n+(e.avail_in-5),s=e.next_out,C=e.output,a=s-(t-e.avail_out),o=s+(e.avail_out-257),h=r.dmax,u=r.wsize,l=r.whave,f=r.wnext,c=r.window,d=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,b=(1<<r.distbits)-1;e:do{p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=m[d&g];t:for(;;){if(d>>>=y=v>>>24,p-=y,0===(y=v>>>16&255))C[s++]=65535&v;else{if(!(16&y)){if(0==(64&y)){v=m[(65535&v)+(d&(1<<y)-1)];continue t}if(32&y){r.mode=12;break e}e.msg="invalid literal/length code",r.mode=30;break e}w=65535&v,(y&=15)&&(p<y&&(d+=z[n++]<<p,p+=8),w+=d&(1<<y)-1,d>>>=y,p-=y),p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=_[d&b];r:for(;;){if(d>>>=y=v>>>24,p-=y,!(16&(y=v>>>16&255))){if(0==(64&y)){v=_[(65535&v)+(d&(1<<y)-1)];continue r}e.msg="invalid distance code",r.mode=30;break e}if(k=65535&v,p<(y&=15)&&(d+=z[n++]<<p,(p+=8)<y&&(d+=z[n++]<<p,p+=8)),h<(k+=d&(1<<y)-1)){e.msg="invalid distance too far back",r.mode=30;break e}if(d>>>=y,p-=y,(y=s-a)<k){if(l<(y=k-y)&&r.sane){e.msg="invalid distance too far back",r.mode=30;break e}if(S=c,(x=0)===f){if(x+=u-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}}else if(f<y){if(x+=u+f-y,(y-=f)<w){for(w-=y;C[s++]=c[x++],--y;);if(x=0,f<w){for(w-=y=f;C[s++]=c[x++],--y;);x=s-k,S=C}}}else if(x+=f-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}for(;2<w;)C[s++]=S[x++],C[s++]=S[x++],C[s++]=S[x++],w-=3;w&&(C[s++]=S[x++],1<w&&(C[s++]=S[x++]))}else{for(x=s-k;C[s++]=C[x++],C[s++]=C[x++],C[s++]=C[x++],2<(w-=3););w&&(C[s++]=C[x++],1<w&&(C[s++]=C[x++]))}break}}break}}while(n<i&&s<o);n-=w=p>>3,d&=(1<<(p-=w<<3))-1,e.next_in=n,e.next_out=s,e.avail_in=n<i?i-n+5:5-(n-i),e.avail_out=s<o?o-s+257:257-(s-o),r.hold=d,r.bits=p}},{}],49:[function(e,t,r){"use strict";var I=e("../utils/common"),O=e("./adler32"),B=e("./crc32"),R=e("./inffast"),T=e("./inftrees"),D=1,F=2,N=0,U=-2,P=1,n=852,i=592;function L(e){return(e>>>24&255)+(e>>>8&65280)+((65280&e)<<8)+((255&e)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(e){var t;return e&&e.state?(t=e.state,e.total_in=e.total_out=t.total=0,e.msg="",t.wrap&&(e.adler=1&t.wrap),t.mode=P,t.last=0,t.havedict=0,t.dmax=32768,t.head=null,t.hold=0,t.bits=0,t.lencode=t.lendyn=new I.Buf32(n),t.distcode=t.distdyn=new I.Buf32(i),t.sane=1,t.back=-1,N):U}function o(e){var t;return e&&e.state?((t=e.state).wsize=0,t.whave=0,t.wnext=0,a(e)):U}function h(e,t){var r,n;return e&&e.state?(n=e.state,t<0?(r=0,t=-t):(r=1+(t>>4),t<48&&(t&=15)),t&&(t<8||15<t)?U:(null!==n.window&&n.wbits!==t&&(n.window=null),n.wrap=r,n.wbits=t,o(e))):U}function u(e,t){var r,n;return e?(n=new s,(e.state=n).window=null,(r=h(e,t))!==N&&(e.state=null),r):U}var l,f,c=!0;function j(e){if(c){var t;for(l=new I.Buf32(512),f=new I.Buf32(32),t=0;t<144;)e.lens[t++]=8;for(;t<256;)e.lens[t++]=9;for(;t<280;)e.lens[t++]=7;for(;t<288;)e.lens[t++]=8;for(T(D,e.lens,0,288,l,0,e.work,{bits:9}),t=0;t<32;)e.lens[t++]=5;T(F,e.lens,0,32,f,0,e.work,{bits:5}),c=!1}e.lencode=l,e.lenbits=9,e.distcode=f,e.distbits=5}function Z(e,t,r,n){var i,s=e.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),n>=s.wsize?(I.arraySet(s.window,t,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(n<(i=s.wsize-s.wnext)&&(i=n),I.arraySet(s.window,t,r-n,i,s.wnext),(n-=i)?(I.arraySet(s.window,t,r-n,n,0),s.wnext=n,s.whave=s.wsize):(s.wnext+=i,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=i))),0}r.inflateReset=o,r.inflateReset2=h,r.inflateResetKeep=a,r.inflateInit=function(e){return u(e,15)},r.inflateInit2=u,r.inflate=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C=0,E=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!e||!e.state||!e.output||!e.input&&0!==e.avail_in)return U;12===(r=e.state).mode&&(r.mode=13),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,f=o,c=h,x=N;e:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(2&r.wrap&&35615===u){E[r.check=0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0),l=u=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&u)<<8)+(u>>8))%31){e.msg="incorrect header check",r.mode=30;break}if(8!=(15&u)){e.msg="unknown compression method",r.mode=30;break}if(l-=4,k=8+(15&(u>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){e.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,e.adler=r.check=1,r.mode=512&u?10:12,l=u=0;break;case 2:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.flags=u,8!=(255&r.flags)){e.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){e.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=u>>8&1),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=3;case 3:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.time=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,E[2]=u>>>16&255,E[3]=u>>>24&255,r.check=B(r.check,E,4,0)),l=u=0,r.mode=4;case 4:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.xflags=255&u,r.head.os=u>>8),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=5;case 5:if(1024&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length=u,r.head&&(r.head.extra_len=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(d=r.length)&&(d=o),d&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,n,s,d,k)),512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,r.length-=d),r.length))break e;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(65535&r.check)){e.msg="header crc mismatch",r.mode=30;break}l=u=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),e.adler=r.check=0,r.mode=12;break;case 10:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}e.adler=r.check=L(u),l=u=0,r.mode=11;case 11:if(0===r.havedict)return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,2;e.adler=r.check=1,r.mode=12;case 12:if(5===t||6===t)break e;case 13:if(r.last){u>>>=7&l,l-=7&l,r.mode=27;break}for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}switch(r.last=1&u,l-=1,3&(u>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==t)break;u>>>=2,l-=2;break e;case 2:r.mode=17;break;case 3:e.msg="invalid block type",r.mode=30}u>>>=2,l-=2;break;case 14:for(u>>>=7&l,l-=7&l;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if((65535&u)!=(u>>>16^65535)){e.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&u,l=u=0,r.mode=15,6===t)break e;case 15:r.mode=16;case 16:if(d=r.length){if(o<d&&(d=o),h<d&&(d=h),0===d)break e;I.arraySet(i,n,s,d,a),o-=d,s+=d,h-=d,a+=d,r.length-=d;break}r.mode=12;break;case 17:for(;l<14;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.nlen=257+(31&u),u>>>=5,l-=5,r.ndist=1+(31&u),u>>>=5,l-=5,r.ncode=4+(15&u),u>>>=4,l-=4,286<r.nlen||30<r.ndist){e.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.lens[A[r.have++]]=7&u,u>>>=3,l-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=T(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(b<16)u>>>=_,l-=_,r.lens[r.have++]=b;else{if(16===b){for(z=_+2;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u>>>=_,l-=_,0===r.have){e.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],d=3+(3&u),u>>>=2,l-=2}else if(17===b){for(z=_+3;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=3+(7&(u>>>=_)),u>>>=3,l-=3}else{for(z=_+7;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=11+(127&(u>>>=_)),u>>>=7,l-=7}if(r.have+d>r.nlen+r.ndist){e.msg="invalid bit length repeat",r.mode=30;break}for(;d--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){e.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=T(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=T(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){e.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===t)break e;case 20:r.mode=21;case 21:if(6<=o&&258<=h){e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,R(e,c),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(g&&0==(240&g)){for(v=_,y=g,w=b;g=(C=r.lencode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,r.length=b,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){e.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(C=r.distcode[u&(1<<r.distbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(0==(240&g)){for(v=_,y=g,w=b;g=(C=r.distcode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,64&g){e.msg="invalid distance code",r.mode=30;break}r.offset=b,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.offset+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){e.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===h)break e;if(d=c-h,r.offset>d){if((d=r.offset-d)>r.whave&&r.sane){e.msg="invalid distance too far back",r.mode=30;break}p=d>r.wnext?(d-=r.wnext,r.wsize-d):r.wnext-d,d>r.length&&(d=r.length),m=r.window}else m=i,p=a-r.offset,d=r.length;for(h<d&&(d=h),h-=d,r.length-=d;i[a++]=m[p++],--d;);0===r.length&&(r.mode=21);break;case 26:if(0===h)break e;i[a++]=r.length,h--,r.mode=21;break;case 27:if(r.wrap){for(;l<32;){if(0===o)break e;o--,u|=n[s++]<<l,l+=8}if(c-=h,e.total_out+=c,r.total+=c,c&&(e.adler=r.check=r.flags?B(r.check,i,c,a-c):O(r.check,i,c,a-c)),c=h,(r.flags?u:L(u))!==r.check){e.msg="incorrect data check",r.mode=30;break}l=u=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(4294967295&r.total)){e.msg="incorrect length check",r.mode=30;break}l=u=0}r.mode=29;case 29:x=1;break e;case 30:x=-3;break e;case 31:return-4;case 32:default:return U}return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,(r.wsize||c!==e.avail_out&&r.mode<30&&(r.mode<27||4!==t))&&Z(e,e.output,e.next_out,c-e.avail_out)?(r.mode=31,-4):(f-=e.avail_in,c-=e.avail_out,e.total_in+=f,e.total_out+=c,r.total+=c,r.wrap&&c&&(e.adler=r.check=r.flags?B(r.check,i,c,e.next_out-c):O(r.check,i,c,e.next_out-c)),e.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==f&&0===c||4===t)&&x===N&&(x=-5),x)},r.inflateEnd=function(e){if(!e||!e.state)return U;var t=e.state;return t.window&&(t.window=null),e.state=null,N},r.inflateGetHeader=function(e,t){var r;return e&&e.state?0==(2&(r=e.state).wrap)?U:((r.head=t).done=!1,N):U},r.inflateSetDictionary=function(e,t){var r,n=t.length;return e&&e.state?0!==(r=e.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,t,n,0)!==r.check?-3:Z(e,t,n,n)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(e,t,r){"use strict";var D=e("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];t.exports=function(e,t,r,n,i,s,a,o){var h,u,l,f,c,d,p,m,_,g=o.bits,b=0,v=0,y=0,w=0,k=0,x=0,S=0,z=0,C=0,E=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),R=null,T=0;for(b=0;b<=15;b++)O[b]=0;for(v=0;v<n;v++)O[t[r+v]]++;for(k=g,w=15;1<=w&&0===O[w];w--);if(w<k&&(k=w),0===w)return i[s++]=20971520,i[s++]=20971520,o.bits=1,0;for(y=1;y<w&&0===O[y];y++);for(k<y&&(k=y),b=z=1;b<=15;b++)if(z<<=1,(z-=O[b])<0)return-1;if(0<z&&(0===e||1!==w))return-1;for(B[1]=0,b=1;b<15;b++)B[b+1]=B[b]+O[b];for(v=0;v<n;v++)0!==t[r+v]&&(a[B[t[r+v]]++]=v);if(d=0===e?(A=R=a,19):1===e?(A=F,I-=257,R=N,T-=257,256):(A=U,R=P,-1),b=y,c=s,S=v=E=0,l=-1,f=(C=1<<(x=k))-1,1===e&&852<C||2===e&&592<C)return 1;for(;;){for(p=b-S,_=a[v]<d?(m=0,a[v]):a[v]>d?(m=R[T+a[v]],A[I+a[v]]):(m=96,0),h=1<<b-S,y=u=1<<x;i[c+(E>>S)+(u-=h)]=p<<24|m<<16|_|0,0!==u;);for(h=1<<b-1;E&h;)h>>=1;if(0!==h?(E&=h-1,E+=h):E=0,v++,0==--O[b]){if(b===w)break;b=t[r+a[v]]}if(k<b&&(E&f)!==l){for(0===S&&(S=k),c+=y,z=1<<(x=b-S);x+S<w&&!((z-=O[x+S])<=0);)x++,z<<=1;if(C+=1<<x,1===e&&852<C||2===e&&592<C)return 1;i[l=E&f]=k<<24|x<<16|c-s|0}}return 0!==E&&(i[c+E]=b-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(e,t,r){"use strict";t.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(e,t,r){"use strict";var i=e("../utils/common"),o=0,h=1;function n(e){for(var t=e.length;0<=--t;)e[t]=0}var s=0,a=29,u=256,l=u+1+a,f=30,c=19,_=2*l+1,g=15,d=16,p=7,m=256,b=16,v=17,y=18,w=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],k=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],x=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],z=new Array(2*(l+2));n(z);var C=new Array(2*f);n(C);var E=new Array(512);n(E);var A=new Array(256);n(A);var I=new Array(a);n(I);var O,B,R,T=new Array(f);function D(e,t,r,n,i){this.static_tree=e,this.extra_bits=t,this.extra_base=r,this.elems=n,this.max_length=i,this.has_stree=e&&e.length}function F(e,t){this.dyn_tree=e,this.max_code=0,this.stat_desc=t}function N(e){return e<256?E[e]:E[256+(e>>>7)]}function U(e,t){e.pending_buf[e.pending++]=255&t,e.pending_buf[e.pending++]=t>>>8&255}function P(e,t,r){e.bi_valid>d-r?(e.bi_buf|=t<<e.bi_valid&65535,U(e,e.bi_buf),e.bi_buf=t>>d-e.bi_valid,e.bi_valid+=r-d):(e.bi_buf|=t<<e.bi_valid&65535,e.bi_valid+=r)}function L(e,t,r){P(e,r[2*t],r[2*t+1])}function j(e,t){for(var r=0;r|=1&e,e>>>=1,r<<=1,0<--t;);return r>>>1}function Z(e,t,r){var n,i,s=new Array(g+1),a=0;for(n=1;n<=g;n++)s[n]=a=a+r[n-1]<<1;for(i=0;i<=t;i++){var o=e[2*i+1];0!==o&&(e[2*i]=j(s[o]++,o))}}function W(e){var t;for(t=0;t<l;t++)e.dyn_ltree[2*t]=0;for(t=0;t<f;t++)e.dyn_dtree[2*t]=0;for(t=0;t<c;t++)e.bl_tree[2*t]=0;e.dyn_ltree[2*m]=1,e.opt_len=e.static_len=0,e.last_lit=e.matches=0}function M(e){8<e.bi_valid?U(e,e.bi_buf):0<e.bi_valid&&(e.pending_buf[e.pending++]=e.bi_buf),e.bi_buf=0,e.bi_valid=0}function H(e,t,r,n){var i=2*t,s=2*r;return e[i]<e[s]||e[i]===e[s]&&n[t]<=n[r]}function G(e,t,r){for(var n=e.heap[r],i=r<<1;i<=e.heap_len&&(i<e.heap_len&&H(t,e.heap[i+1],e.heap[i],e.depth)&&i++,!H(t,n,e.heap[i],e.depth));)e.heap[r]=e.heap[i],r=i,i<<=1;e.heap[r]=n}function K(e,t,r){var n,i,s,a,o=0;if(0!==e.last_lit)for(;n=e.pending_buf[e.d_buf+2*o]<<8|e.pending_buf[e.d_buf+2*o+1],i=e.pending_buf[e.l_buf+o],o++,0===n?L(e,i,t):(L(e,(s=A[i])+u+1,t),0!==(a=w[s])&&P(e,i-=I[s],a),L(e,s=N(--n),r),0!==(a=k[s])&&P(e,n-=T[s],a)),o<e.last_lit;);L(e,m,t)}function Y(e,t){var r,n,i,s=t.dyn_tree,a=t.stat_desc.static_tree,o=t.stat_desc.has_stree,h=t.stat_desc.elems,u=-1;for(e.heap_len=0,e.heap_max=_,r=0;r<h;r++)0!==s[2*r]?(e.heap[++e.heap_len]=u=r,e.depth[r]=0):s[2*r+1]=0;for(;e.heap_len<2;)s[2*(i=e.heap[++e.heap_len]=u<2?++u:0)]=1,e.depth[i]=0,e.opt_len--,o&&(e.static_len-=a[2*i+1]);for(t.max_code=u,r=e.heap_len>>1;1<=r;r--)G(e,s,r);for(i=h;r=e.heap[1],e.heap[1]=e.heap[e.heap_len--],G(e,s,1),n=e.heap[1],e.heap[--e.heap_max]=r,e.heap[--e.heap_max]=n,s[2*i]=s[2*r]+s[2*n],e.depth[i]=(e.depth[r]>=e.depth[n]?e.depth[r]:e.depth[n])+1,s[2*r+1]=s[2*n+1]=i,e.heap[1]=i++,G(e,s,1),2<=e.heap_len;);e.heap[--e.heap_max]=e.heap[1],function(e,t){var r,n,i,s,a,o,h=t.dyn_tree,u=t.max_code,l=t.stat_desc.static_tree,f=t.stat_desc.has_stree,c=t.stat_desc.extra_bits,d=t.stat_desc.extra_base,p=t.stat_desc.max_length,m=0;for(s=0;s<=g;s++)e.bl_count[s]=0;for(h[2*e.heap[e.heap_max]+1]=0,r=e.heap_max+1;r<_;r++)p<(s=h[2*h[2*(n=e.heap[r])+1]+1]+1)&&(s=p,m++),h[2*n+1]=s,u<n||(e.bl_count[s]++,a=0,d<=n&&(a=c[n-d]),o=h[2*n],e.opt_len+=o*(s+a),f&&(e.static_len+=o*(l[2*n+1]+a)));if(0!==m){do{for(s=p-1;0===e.bl_count[s];)s--;e.bl_count[s]--,e.bl_count[s+1]+=2,e.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(n=e.bl_count[s];0!==n;)u<(i=e.heap[--r])||(h[2*i+1]!==s&&(e.opt_len+=(s-h[2*i+1])*h[2*i],h[2*i+1]=s),n--)}}(e,t),Z(s,u,e.bl_count)}function X(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),t[2*(r+1)+1]=65535,n=0;n<=r;n++)i=a,a=t[2*(n+1)+1],++o<h&&i===a||(o<u?e.bl_tree[2*i]+=o:0!==i?(i!==s&&e.bl_tree[2*i]++,e.bl_tree[2*b]++):o<=10?e.bl_tree[2*v]++:e.bl_tree[2*y]++,s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4))}function V(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),n=0;n<=r;n++)if(i=a,a=t[2*(n+1)+1],!(++o<h&&i===a)){if(o<u)for(;L(e,i,e.bl_tree),0!=--o;);else 0!==i?(i!==s&&(L(e,i,e.bl_tree),o--),L(e,b,e.bl_tree),P(e,o-3,2)):o<=10?(L(e,v,e.bl_tree),P(e,o-3,3)):(L(e,y,e.bl_tree),P(e,o-11,7));s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4)}}n(T);var q=!1;function J(e,t,r,n){P(e,(s<<1)+(n?1:0),3),function(e,t,r,n){M(e),n&&(U(e,r),U(e,~r)),i.arraySet(e.pending_buf,e.window,t,r,e.pending),e.pending+=r}(e,t,r,!0)}r._tr_init=function(e){q||(function(){var e,t,r,n,i,s=new Array(g+1);for(n=r=0;n<a-1;n++)for(I[n]=r,e=0;e<1<<w[n];e++)A[r++]=n;for(A[r-1]=n,n=i=0;n<16;n++)for(T[n]=i,e=0;e<1<<k[n];e++)E[i++]=n;for(i>>=7;n<f;n++)for(T[n]=i<<7,e=0;e<1<<k[n]-7;e++)E[256+i++]=n;for(t=0;t<=g;t++)s[t]=0;for(e=0;e<=143;)z[2*e+1]=8,e++,s[8]++;for(;e<=255;)z[2*e+1]=9,e++,s[9]++;for(;e<=279;)z[2*e+1]=7,e++,s[7]++;for(;e<=287;)z[2*e+1]=8,e++,s[8]++;for(Z(z,l+1,s),e=0;e<f;e++)C[2*e+1]=5,C[2*e]=j(e,5);O=new D(z,w,u+1,l,g),B=new D(C,k,0,f,g),R=new D(new Array(0),x,0,c,p)}(),q=!0),e.l_desc=new F(e.dyn_ltree,O),e.d_desc=new F(e.dyn_dtree,B),e.bl_desc=new F(e.bl_tree,R),e.bi_buf=0,e.bi_valid=0,W(e)},r._tr_stored_block=J,r._tr_flush_block=function(e,t,r,n){var i,s,a=0;0<e.level?(2===e.strm.data_type&&(e.strm.data_type=function(e){var t,r=4093624447;for(t=0;t<=31;t++,r>>>=1)if(1&r&&0!==e.dyn_ltree[2*t])return o;if(0!==e.dyn_ltree[18]||0!==e.dyn_ltree[20]||0!==e.dyn_ltree[26])return h;for(t=32;t<u;t++)if(0!==e.dyn_ltree[2*t])return h;return o}(e)),Y(e,e.l_desc),Y(e,e.d_desc),a=function(e){var t;for(X(e,e.dyn_ltree,e.l_desc.max_code),X(e,e.dyn_dtree,e.d_desc.max_code),Y(e,e.bl_desc),t=c-1;3<=t&&0===e.bl_tree[2*S[t]+1];t--);return e.opt_len+=3*(t+1)+5+5+4,t}(e),i=e.opt_len+3+7>>>3,(s=e.static_len+3+7>>>3)<=i&&(i=s)):i=s=r+5,r+4<=i&&-1!==t?J(e,t,r,n):4===e.strategy||s===i?(P(e,2+(n?1:0),3),K(e,z,C)):(P(e,4+(n?1:0),3),function(e,t,r,n){var i;for(P(e,t-257,5),P(e,r-1,5),P(e,n-4,4),i=0;i<n;i++)P(e,e.bl_tree[2*S[i]+1],3);V(e,e.dyn_ltree,t-1),V(e,e.dyn_dtree,r-1)}(e,e.l_desc.max_code+1,e.d_desc.max_code+1,a+1),K(e,e.dyn_ltree,e.dyn_dtree)),W(e),n&&M(e)},r._tr_tally=function(e,t,r){return e.pending_buf[e.d_buf+2*e.last_lit]=t>>>8&255,e.pending_buf[e.d_buf+2*e.last_lit+1]=255&t,e.pending_buf[e.l_buf+e.last_lit]=255&r,e.last_lit++,0===t?e.dyn_ltree[2*r]++:(e.matches++,t--,e.dyn_ltree[2*(A[r]+u+1)]++,e.dyn_dtree[2*N(t)]++),e.last_lit===e.lit_bufsize-1},r._tr_align=function(e){P(e,2,3),L(e,m,z),function(e){16===e.bi_valid?(U(e,e.bi_buf),e.bi_buf=0,e.bi_valid=0):8<=e.bi_valid&&(e.pending_buf[e.pending++]=255&e.bi_buf,e.bi_buf>>=8,e.bi_valid-=8)}(e)}},{"../utils/common":41}],53:[function(e,t,r){"use strict";t.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(e,t,r){(function(e){!function(r,n){"use strict";if(!r.setImmediate){var i,s,t,a,o=1,h={},u=!1,l=r.document,e=Object.getPrototypeOf&&Object.getPrototypeOf(r);e=e&&e.setTimeout?e:r,i="[object process]"==={}.toString.call(r.process)?function(e){process.nextTick(function(){c(e)})}:function(){if(r.postMessage&&!r.importScripts){var e=!0,t=r.onmessage;return r.onmessage=function(){e=!1},r.postMessage("","*"),r.onmessage=t,e}}()?(a="setImmediate$"+Math.random()+"$",r.addEventListener?r.addEventListener("message",d,!1):r.attachEvent("onmessage",d),function(e){r.postMessage(a+e,"*")}):r.MessageChannel?((t=new MessageChannel).port1.onmessage=function(e){c(e.data)},function(e){t.port2.postMessage(e)}):l&&"onreadystatechange"in l.createElement("script")?(s=l.documentElement,function(e){var t=l.createElement("script");t.onreadystatechange=function(){c(e),t.onreadystatechange=null,s.removeChild(t),t=null},s.appendChild(t)}):function(e){setTimeout(c,0,e)},e.setImmediate=function(e){"function"!=typeof e&&(e=new Function(""+e));for(var t=new Array(arguments.length-1),r=0;r<t.length;r++)t[r]=arguments[r+1];var n={callback:e,args:t};return h[o]=n,i(o),o++},e.clearImmediate=f}function f(e){delete h[e]}function c(e){if(u)setTimeout(c,0,e);else{var t=h[e];if(t){u=!0;try{!function(e){var t=e.callback,r=e.args;switch(r.length){case 0:t();break;case 1:t(r[0]);break;case 2:t(r[0],r[1]);break;case 3:t(r[0],r[1],r[2]);break;default:t.apply(n,r)}}(t)}finally{f(e),u=!1}}}}function d(e){e.source===r&&"string"==typeof e.data&&0===e.data.indexOf(a)&&c(+e.data.slice(a.length))}}("undefined"==typeof self?void 0===e?this:e:self)}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[10])(10)});
/*! pdfmake v0.2.10, @license MIT, @link http://pdfmake.org */
!function webpackUniversalModuleDefinition(r,i){if("object"==typeof exports&&"object"==typeof module)module.exports=i();else if("function"==typeof define&&define.amd)define([],i);else{var o=i();for(var u in o)("object"==typeof exports?exports:r)[u]=o[u]}}("undefined"!=typeof self?self:this,(function(){return function(){var r={41783:function(r,i){"use strict";i.OP=0,i.CL=1,i.CP=2,i.QU=3,i.GL=4,i.NS=5,i.EX=6,i.SY=7,i.IS=8,i.PR=9,i.PO=10,i.NU=11,i.AL=12,i.HL=13,i.ID=14,i.IN=15,i.HY=16,i.BA=17,i.BB=18,i.B2=19,i.ZW=20,i.CM=21,i.WJ=22,i.H2=23,i.H3=24,i.JL=25,i.JV=26,i.JT=27,i.RI=28,i.EB=29,i.EM=30,i.ZWJ=31,i.CB=32,i.AI=33,i.BK=34,i.CJ=35,i.CR=36,i.LF=37,i.NL=38,i.SA=39,i.SG=40,i.SP=41,i.XX=42},98282:function(r,i){"use strict";i.DI_BRK=0,i.IN_BRK=1,i.CI_BRK=2,i.CP_BRK=3,i.PR_BRK=4,i.pairTable=[[4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,3,4,4,4,4,4,4,4,4,4,4,4],[0,4,4,1,1,4,4,4,4,1,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,4,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[4,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,0,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,1,1,1,1,0,0,4,2,4,1,1,1,1,1,0,1,1,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,1,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,4,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,1,1,1,1,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,1,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,1,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,0,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0]]},57013:function(r,i,o){"use strict";o(18756),i.EncodeStream=o(16176),i.DecodeStream=o(78984),i.Array=o(38637),i.LazyArray=o(88687),i.Bitfield=o(72959),i.Boolean=o(64888),i.Buffer=o(77324),i.Enum=o(71499),i.Optional=o(72526),i.Reserved=o(10298),i.String=o(36291),i.Struct=o(2731),i.VersionedStruct=o(48084);var u=o(76949),p=o(63787),g=o(41545);Object.assign(i,u,p,g)},38637:function(r,i,o){"use strict";function _createForOfIteratorHelperLoose(r,i){var o="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(o)return(o=o.call(r)).next.bind(o);if(Array.isArray(r)||(o=function _unsupportedIterableToArray(r,i){if(!r)return;if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);"Object"===o&&r.constructor&&(o=r.constructor.name);if("Map"===o||"Set"===o)return Array.from(r);if("Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o))return _arrayLikeToArray(r,i)}(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0;return function(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}o(65292),o(73844),o(69330),o(11765),o(81755),o(20731),o(24863),o(14032),o(61726),o(57114),o(58281),o(94712);var u=o(63787).Number,p=o(76949),g=function(){function ArrayT(r,i,o){void 0===o&&(o="count"),this.type=r,this.length=i,this.lengthType=o}var r=ArrayT.prototype;return r.decode=function decode(r,i){var o,g=r.pos,y=[],w=i;if(null!=this.length&&(o=p.resolveLength(this.length,r,i)),this.length instanceof u&&(Object.defineProperties(y,{parent:{value:i},_startOffset:{value:g},_currentOffset:{value:0,writable:!0},_length:{value:o}}),w=y),null==o||"bytes"===this.lengthType)for(var _=null!=o?r.pos+o:(null!=i?i._length:void 0)?i._startOffset+i._length:r.length;r.pos<_;)y.push(this.type.decode(r,w));else for(var x=0,k=o;x<k;x++)y.push(this.type.decode(r,w));return y},r.size=function size(r,i){if(!r)return this.type.size(null,i)*p.resolveLength(this.length,null,i);var size=0;this.length instanceof u&&(size+=this.length.size(),i={parent:i});for(var o,g=_createForOfIteratorHelperLoose(r);!(o=g()).done;){var y=o.value;size+=this.type.size(y,i)}return size},r.encode=function encode(r,i,o){var p=o;this.length instanceof u&&((p={pointers:[],startOffset:r.pos,parent:o}).pointerOffset=r.pos+this.size(i,p),this.length.encode(r,i.length));for(var g,y=_createForOfIteratorHelperLoose(i);!(g=y()).done;){var w=g.value;this.type.encode(r,w,p)}if(this.length instanceof u)for(var _=0;_<p.pointers.length;){var x=p.pointers[_++];x.type.encode(r,x.val)}},ArrayT}();r.exports=g},72959:function(r,i,o){"use strict";o(74516);var u=function(){function Bitfield(r,i){void 0===i&&(i=[]),this.type=r,this.flags=i}var r=Bitfield.prototype;return r.decode=function decode(r){for(var i=this.type.decode(r),o={},u=0;u<this.flags.length;u++){var p=this.flags[u];null!=p&&(o[p]=!!(i&1<<u))}return o},r.size=function size(){return this.type.size()},r.encode=function encode(r,i){for(var o=0,u=0;u<this.flags.length;u++){var p=this.flags[u];null!=p&&i[p]&&(o|=1<<u)}return this.type.encode(r,o)},Bitfield}();r.exports=u},64888:function(r){"use strict";var i=function(){function BooleanT(r){this.type=r}var r=BooleanT.prototype;return r.decode=function decode(r,i){return!!this.type.decode(r,i)},r.size=function size(r,i){return this.type.size(r,i)},r.encode=function encode(r,i,o){return this.type.encode(r,+i,o)},BooleanT}();r.exports=i},77324:function(r,i,o){"use strict";var u=o(76949),p=o(63787).Number,g=function(){function BufferT(r){this.length=r}var r=BufferT.prototype;return r.decode=function decode(r,i){var o=u.resolveLength(this.length,r,i);return r.readBuffer(o)},r.size=function size(r,i){return r?r.length:u.resolveLength(this.length,null,i)},r.encode=function encode(r,i,o){return this.length instanceof p&&this.length.encode(r,i.length),r.writeBuffer(i)},BufferT}();r.exports=g},78984:function(r,i,o){"use strict";var u,p=o(50621).Buffer;o(20731),o(14032),o(61726),o(57114),o(46467);try{u=o(54171)}catch(r){}var g=function(){function DecodeStream(r){this.buffer=r,this.pos=0,this.length=this.buffer.length}var r=DecodeStream.prototype;return r.readString=function readString(r,i){switch(void 0===i&&(i="ascii"),i){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.buffer.toString(i,this.pos,this.pos+=r);case"utf16be":for(var o=p.from(this.readBuffer(r)),g=0,y=o.length-1;g<y;g+=2){var w=o[g];o[g]=o[g+1],o[g+1]=w}return o.toString("utf16le");default:if(o=this.readBuffer(r),u)try{return u.decode(o,i)}catch(r){}return o}},r.readBuffer=function readBuffer(r){return this.buffer.slice(this.pos,this.pos+=r)},r.readUInt24BE=function readUInt24BE(){return(this.readUInt16BE()<<8)+this.readUInt8()},r.readUInt24LE=function readUInt24LE(){return this.readUInt16LE()+(this.readUInt8()<<16)},r.readInt24BE=function readInt24BE(){return(this.readInt16BE()<<8)+this.readUInt8()},r.readInt24LE=function readInt24LE(){return this.readUInt16LE()+(this.readInt8()<<16)},DecodeStream}();g.TYPES={UInt8:1,UInt16:2,UInt24:3,UInt32:4,Int8:1,Int16:2,Int24:3,Int32:4,Float:4,Double:8};var y=function _loop(r){if("read"===r.slice(0,4)){var i=g.TYPES[r.replace(/read|[BL]E/g,"")];g.prototype[r]=function(){var o=this.buffer[r](this.pos);return this.pos+=i,o}}};for(var w in p.prototype)y(w);r.exports=g},16176:function(r,i,o){"use strict";var u,p=o(50621).Buffer;function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(41584),o(20731),o(61726),o(46467);var g=o(9760),y=o(78984);try{u=o(54171)}catch(r){}var w=function(r){function EncodeStream(i){var o;return void 0===i&&(i=65536),(o=r.apply(this,arguments)||this).buffer=p.alloc(i),o.bufferOffset=0,o.pos=0,o}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(EncodeStream,r);var i=EncodeStream.prototype;return i._read=function _read(){},i.ensure=function ensure(r){if(this.bufferOffset+r>this.buffer.length)return this.flush()},i.flush=function flush(){if(this.bufferOffset>0)return this.push(p.from(this.buffer.slice(0,this.bufferOffset))),this.bufferOffset=0},i.writeBuffer=function writeBuffer(r){return this.flush(),this.push(r),this.pos+=r.length},i.writeString=function writeString(r,i){switch(void 0===i&&(i="ascii"),i){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.writeBuffer(p.from(r,i));case"utf16be":for(var o=p.from(r,"utf16le"),g=0,y=o.length-1;g<y;g+=2){var w=o[g];o[g]=o[g+1],o[g+1]=w}return this.writeBuffer(o);default:if(u)return this.writeBuffer(u.encode(r,i));throw new Error("Install iconv-lite to enable additional string encodings.")}},i.writeUInt24BE=function writeUInt24BE(r){return this.ensure(3),this.buffer[this.bufferOffset++]=r>>>16&255,this.buffer[this.bufferOffset++]=r>>>8&255,this.buffer[this.bufferOffset++]=255&r,this.pos+=3},i.writeUInt24LE=function writeUInt24LE(r){return this.ensure(3),this.buffer[this.bufferOffset++]=255&r,this.buffer[this.bufferOffset++]=r>>>8&255,this.buffer[this.bufferOffset++]=r>>>16&255,this.pos+=3},i.writeInt24BE=function writeInt24BE(r){return r>=0?this.writeUInt24BE(r):this.writeUInt24BE(r+16777215+1)},i.writeInt24LE=function writeInt24LE(r){return r>=0?this.writeUInt24LE(r):this.writeUInt24LE(r+16777215+1)},i.fill=function fill(r,i){if(i<this.buffer.length)return this.ensure(i),this.buffer.fill(r,this.bufferOffset,this.bufferOffset+i),this.bufferOffset+=i,this.pos+=i;var o=p.alloc(i);return o.fill(r),this.writeBuffer(o)},i.end=function end(){return this.flush(),this.push(null)},EncodeStream}(g.Readable),_=function _loop(r){if("write"===r.slice(0,5)){var i=+y.TYPES[r.replace(/write|[BL]E/g,"")];w.prototype[r]=function(o){return this.ensure(i),this.buffer[r](o,this.bufferOffset),this.bufferOffset+=i,this.pos+=i}}};for(var x in p.prototype)_(x);r.exports=w},71499:function(r){"use strict";var i=function(){function Enum(r,i){void 0===i&&(i=[]),this.type=r,this.options=i}var r=Enum.prototype;return r.decode=function decode(r){var i=this.type.decode(r);return this.options[i]||i},r.size=function size(){return this.type.size()},r.encode=function encode(r,i){var o=this.options.indexOf(i);if(-1===o)throw new Error("Unknown option in enum: "+i);return this.type.encode(r,o)},Enum}();r.exports=i},88687:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(14032),o(68067);var u=o(38637),p=o(63787).Number,g=o(76949),y=o(7187).inspect,w=function(r){function LazyArrayT(){return r.apply(this,arguments)||this}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(LazyArrayT,r);var i=LazyArrayT.prototype;return i.decode=function decode(r,i){var o=r.pos,u=g.resolveLength(this.length,r,i);this.length instanceof p&&(i={parent:i,_startOffset:o,_currentOffset:0,_length:u});var y=new _(this.type,u,r,i);return r.pos+=u*this.type.size(null,i),y},i.size=function size(i,o){return i instanceof _&&(i=i.toArray()),r.prototype.size.call(this,i,o)},i.encode=function encode(i,o,u){return o instanceof _&&(o=o.toArray()),r.prototype.encode.call(this,i,o,u)},LazyArrayT}(u),_=function(){function LazyArray(r,i,o,u){this.type=r,this.length=i,this.stream=o,this.ctx=u,this.base=this.stream.pos,this.items=[]}var r=LazyArray.prototype;return r.get=function get(r){if(!(r<0||r>=this.length)){if(null==this.items[r]){var i=this.stream.pos;this.stream.pos=this.base+this.type.size(null,this.ctx)*r,this.items[r]=this.type.decode(this.stream,this.ctx),this.stream.pos=i}return this.items[r]}},r.toArray=function toArray(){for(var r=[],i=0,o=this.length;i<o;i++)r.push(this.get(i));return r},r.inspect=function inspect(){return y(this.toArray())},LazyArray}();r.exports=w},63787:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}var u=o(78984),p=function(){function NumberT(r,i){void 0===i&&(i="BE"),this.type=r,this.endian=i,this.fn=this.type,"8"!==this.type[this.type.length-1]&&(this.fn+=this.endian)}var r=NumberT.prototype;return r.size=function size(){return u.TYPES[this.type]},r.decode=function decode(r){return r["read"+this.fn]()},r.encode=function encode(r,i){return r["write"+this.fn](i)},NumberT}();i.Number=p,i.uint8=new p("UInt8"),i.uint16be=i.uint16=new p("UInt16","BE"),i.uint16le=new p("UInt16","LE"),i.uint24be=i.uint24=new p("UInt24","BE"),i.uint24le=new p("UInt24","LE"),i.uint32be=i.uint32=new p("UInt32","BE"),i.uint32le=new p("UInt32","LE"),i.int8=new p("Int8"),i.int16be=i.int16=new p("Int16","BE"),i.int16le=new p("Int16","LE"),i.int24be=i.int24=new p("Int24","BE"),i.int24le=new p("Int24","LE"),i.int32be=i.int32=new p("Int32","BE"),i.int32le=new p("Int32","LE"),i.floatbe=i.float=new p("Float","BE"),i.floatle=new p("Float","LE"),i.doublebe=i.double=new p("Double","BE"),i.doublele=new p("Double","LE");var g=function(r){function Fixed(i,o,u){var p;return void 0===u&&(u=i>>1),(p=r.call(this,"Int"+i,o)||this)._point=1<<u,p}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(Fixed,r);var i=Fixed.prototype;return i.decode=function decode(i){return r.prototype.decode.call(this,i)/this._point},i.encode=function encode(i,o){return r.prototype.encode.call(this,i,o*this._point|0)},Fixed}(p);i.Fixed=g,i.fixed16be=i.fixed16=new g(16,"BE"),i.fixed16le=new g(16,"LE"),i.fixed32be=i.fixed32=new g(32,"BE"),i.fixed32le=new g(32,"LE")},72526:function(r){"use strict";var i=function(){function Optional(r,i){void 0===i&&(i=!0),this.type=r,this.condition=i}var r=Optional.prototype;return r.decode=function decode(r,i){var o=this.condition;if("function"==typeof o&&(o=o.call(i,i)),o)return this.type.decode(r,i)},r.size=function size(r,i){var o=this.condition;return"function"==typeof o&&(o=o.call(i,i)),o?this.type.size(r,i):0},r.encode=function encode(r,i,o){var u=this.condition;if("function"==typeof u&&(u=u.call(o,o)),u)return this.type.encode(r,i,o)},Optional}();r.exports=i},41545:function(r,i,o){"use strict";var u=o(76949),p=function(){function Pointer(r,i,o){if(void 0===o&&(o={}),this.offsetType=r,this.type=i,this.options=o,"void"===this.type&&(this.type=null),null==this.options.type&&(this.options.type="local"),null==this.options.allowNull&&(this.options.allowNull=!0),null==this.options.nullValue&&(this.options.nullValue=0),null==this.options.lazy&&(this.options.lazy=!1),this.options.relativeTo){if("function"!=typeof this.options.relativeTo)throw new Error("relativeTo option must be a function");this.relativeToGetter=o.relativeTo}}var r=Pointer.prototype;return r.decode=function decode(r,i){var o,p=this,g=this.offsetType.decode(r,i);if(g===this.options.nullValue&&this.options.allowNull)return null;switch(this.options.type){case"local":o=i._startOffset;break;case"immediate":o=r.pos-this.offsetType.size();break;case"parent":o=i.parent._startOffset;break;default:for(var y=i;y.parent;)y=y.parent;o=y._startOffset||0}this.options.relativeTo&&(o+=this.relativeToGetter(i));var w=g+o;if(null!=this.type){var _=null,x=function decodeValue(){if(null!=_)return _;var o=r.pos;return r.pos=w,_=p.type.decode(r,i),r.pos=o,_};return this.options.lazy?new u.PropertyDescriptor({get:x}):x()}return w},r.size=function size(r,i){var o=i;switch(this.options.type){case"local":case"immediate":break;case"parent":i=i.parent;break;default:for(;i.parent;)i=i.parent}var u=this.type;if(null==u){if(!(r instanceof g))throw new Error("Must be a VoidPointer");u=r.type,r=r.value}return r&&i&&(i.pointerSize+=u.size(r,o)),this.offsetType.size()},r.encode=function encode(r,i,o){var u,p=o;if(null!=i){switch(this.options.type){case"local":u=o.startOffset;break;case"immediate":u=r.pos+this.offsetType.size(i,p);break;case"parent":u=(o=o.parent).startOffset;break;default:for(u=0;o.parent;)o=o.parent}this.options.relativeTo&&(u+=this.relativeToGetter(p.val)),this.offsetType.encode(r,o.pointerOffset-u);var y=this.type;if(null==y){if(!(i instanceof g))throw new Error("Must be a VoidPointer");y=i.type,i=i.value}return o.pointers.push({type:y,val:i,parent:p}),o.pointerOffset+=y.size(i,p)}this.offsetType.encode(r,this.options.nullValue)},Pointer}(),g=function VoidPointer(r,i){this.type=r,this.value=i};i.Pointer=p,i.VoidPointer=g},10298:function(r,i,o){"use strict";o(41584);var u=o(76949),p=function(){function Reserved(r,i){void 0===i&&(i=1),this.type=r,this.count=i}var r=Reserved.prototype;return r.decode=function decode(r,i){r.pos+=this.size(null,i)},r.size=function size(r,i){var o=u.resolveLength(this.count,null,i);return this.type.size()*o},r.encode=function encode(r,i,o){return r.fill(0,this.size(i,o))},Reserved}();r.exports=p},36291:function(r,i,o){"use strict";var u=o(50621).Buffer,p=o(63787).Number,g=o(76949),y=function(){function StringT(r,i){void 0===i&&(i="ascii"),this.length=r,this.encoding=i}var r=StringT.prototype;return r.decode=function decode(r,i){var o,u;if(null!=this.length)o=g.resolveLength(this.length,r,i);else{var p;for(p=r.buffer,o=r.length,u=r.pos;u<o&&0!==p[u];)++u;o=u-r.pos}var y=this.encoding;"function"==typeof y&&(y=y.call(i,i)||"ascii");var w=r.readString(o,y);return null==this.length&&r.pos<r.length&&r.pos++,w},r.size=function size(r,i){if(!r)return g.resolveLength(this.length,null,i);var o=this.encoding;"function"==typeof o&&(o=o.call(null!=i?i.val:void 0,null!=i?i.val:void 0)||"ascii"),"utf16be"===o&&(o="utf16le");var size=u.byteLength(r,o);return this.length instanceof p&&(size+=this.length.size()),null==this.length&&size++,size},r.encode=function encode(r,i,o){var g=this.encoding;if("function"==typeof g&&(g=g.call(null!=o?o.val:void 0,null!=o?o.val:void 0)||"ascii"),this.length instanceof p&&this.length.encode(r,u.byteLength(i,g)),r.writeString(i,g),null==this.length)return r.writeUInt8(0)},StringT}();r.exports=y},2731:function(r,i,o){"use strict";var u=o(76949),p=function(){function Struct(r){void 0===r&&(r={}),this.fields=r}var r=Struct.prototype;return r.decode=function decode(r,i,o){void 0===o&&(o=0);var u=this._setup(r,i,o);return this._parseFields(r,u,this.fields),null!=this.process&&this.process.call(u,r),u},r._setup=function _setup(r,i,o){var u={};return Object.defineProperties(u,{parent:{value:i},_startOffset:{value:r.pos},_currentOffset:{value:0,writable:!0},_length:{value:o}}),u},r._parseFields=function _parseFields(r,i,o){for(var p in o){var g,y=o[p];void 0!==(g="function"==typeof y?y.call(i,i):y.decode(r,i))&&(g instanceof u.PropertyDescriptor?Object.defineProperty(i,p,g):i[p]=g),i._currentOffset=r.pos-i._startOffset}},r.size=function size(r,i,o){null==r&&(r={}),null==o&&(o=!0);var u={parent:i,val:r,pointerSize:0},size=0;for(var p in this.fields){var g=this.fields[p];null!=g.size&&(size+=g.size(r[p],u))}return o&&(size+=u.pointerSize),size},r.encode=function encode(r,i,o){var u;null!=this.preEncode&&this.preEncode.call(i,r);var p={pointers:[],startOffset:r.pos,parent:o,val:i,pointerSize:0};for(var g in p.pointerOffset=r.pos+this.size(i,p,!1),this.fields)null!=(u=this.fields[g]).encode&&u.encode(r,i[g],p);for(var y=0;y<p.pointers.length;){var w=p.pointers[y++];w.type.encode(r,w.val,w.parent)}},Struct}();r.exports=p},48084:function(r,i,o){"use strict";function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(14032);var u=o(2731),p=function(r){function VersionedStruct(i,o){var u;return void 0===o&&(o={}),(u=r.call(this)||this).type=i,u.versions=o,"string"==typeof i&&(u.versionPath=i.split(".")),u}!function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(VersionedStruct,r);var i=VersionedStruct.prototype;return i.decode=function decode(r,i,o){void 0===o&&(o=0);var u=this._setup(r,i,o);"string"==typeof this.type?u.version=function getPath(r,i){return i.reduce((function(r,i){return r&&r[i]}),r)}(i,this.versionPath):u.version=this.type.decode(r),this.versions.header&&this._parseFields(r,u,this.versions.header);var p=this.versions[u.version];if(null==p)throw new Error("Unknown version "+u.version);return p instanceof VersionedStruct?p.decode(r,i):(this._parseFields(r,u,p),null!=this.process&&this.process.call(u,r),u)},i.size=function size(r,i,o){var u,p;if(void 0===o&&(o=!0),!r)throw new Error("Not a fixed size");var g={parent:i,val:r,pointerSize:0},size=0;if("string"!=typeof this.type&&(size+=this.type.size(r.version,g)),this.versions.header)for(u in this.versions.header)null!=(p=this.versions.header[u]).size&&(size+=p.size(r[u],g));var y=this.versions[r.version];if(null==y)throw new Error("Unknown version "+r.version);for(u in y)null!=(p=y[u]).size&&(size+=p.size(r[u],g));return o&&(size+=g.pointerSize),size},i.encode=function encode(r,i,o){var u,p;null!=this.preEncode&&this.preEncode.call(i,r);var g={pointers:[],startOffset:r.pos,parent:o,val:i,pointerSize:0};if(g.pointerOffset=r.pos+this.size(i,g,!1),"string"!=typeof this.type&&this.type.encode(r,i.version),this.versions.header)for(u in this.versions.header)null!=(p=this.versions.header[u]).encode&&p.encode(r,i[u],g);var y=this.versions[i.version];for(u in y)null!=(p=y[u]).encode&&p.encode(r,i[u],g);for(var w=0;w<g.pointers.length;){var _=g.pointers[w++];_.type.encode(r,_.val,_.parent)}},VersionedStruct}(u);r.exports=p},76949:function(r,i,o){"use strict";var u=o(63787).Number;i.resolveLength=function(r,i,o){var p;if("number"==typeof r?p=r:"function"==typeof r?p=r.call(o,o):o&&"string"==typeof r?p=o[r]:i&&r instanceof u&&(p=r.decode(i)),isNaN(p))throw new Error("Not a fixed size");return p};i.PropertyDescriptor=function PropertyDescriptor(r){for(var i in void 0===r&&(r={}),this.enumerable=!0,this.configurable=!0,r){var o=r[i];this[i]=o}}},50621:function(r,i,o){"use strict";function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,_toPropertyKey(u.key),u)}}function _toPropertyKey(r){var i=function _toPrimitive(r,i){if("object"!=typeof r||!r)return r;var o=r[Symbol.toPrimitive];if(void 0!==o){var u=o.call(r,i||"default");if("object"!=typeof u)return u;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===i?String:Number)(r)}(r,"string");return"symbol"==typeof i?i:String(i)}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}o(65292),o(73844),o(5597),o(39081),o(68626),o(41584),o(58028),o(81755),o(20731),o(10720),o(7585),o(24863),o(7283),o(2876),o(14032),o(61726),o(57114),o(47458),o(46467),o(72095),o(56912),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529),o(41863);var u=o(32504),p=o(89029),g="function"==typeof Symbol&&"function"==typeof Symbol.for?Symbol.for("nodejs.util.inspect.custom"):null;i.Buffer=Buffer,i.SlowBuffer=function SlowBuffer(r){+r!=r&&(r=0);return Buffer.alloc(+r)},i.INSPECT_MAX_BYTES=50;var y=2147483647;function createBuffer(r){if(r>y)throw new RangeError('The value "'+r+'" is invalid for option "size"');var i=new Uint8Array(r);return Object.setPrototypeOf(i,Buffer.prototype),i}function Buffer(r,i,o){if("number"==typeof r){if("string"==typeof i)throw new TypeError('The "string" argument must be of type string. Received type number');return allocUnsafe(r)}return from(r,i,o)}function from(r,i,o){if("string"==typeof r)return function fromString(r,i){"string"==typeof i&&""!==i||(i="utf8");if(!Buffer.isEncoding(i))throw new TypeError("Unknown encoding: "+i);var o=0|byteLength(r,i),u=createBuffer(o),p=u.write(r,i);p!==o&&(u=u.slice(0,p));return u}(r,i);if(ArrayBuffer.isView(r))return function fromArrayView(r){if(isInstance(r,Uint8Array)){var i=new Uint8Array(r);return fromArrayBuffer(i.buffer,i.byteOffset,i.byteLength)}return fromArrayLike(r)}(r);if(null==r)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r);if(isInstance(r,ArrayBuffer)||r&&isInstance(r.buffer,ArrayBuffer))return fromArrayBuffer(r,i,o);if("undefined"!=typeof SharedArrayBuffer&&(isInstance(r,SharedArrayBuffer)||r&&isInstance(r.buffer,SharedArrayBuffer)))return fromArrayBuffer(r,i,o);if("number"==typeof r)throw new TypeError('The "value" argument must not be of type number. Received type number');var u=r.valueOf&&r.valueOf();if(null!=u&&u!==r)return Buffer.from(u,i,o);var p=function fromObject(r){if(Buffer.isBuffer(r)){var i=0|checked(r.length),o=createBuffer(i);return 0===o.length||r.copy(o,0,0,i),o}if(void 0!==r.length)return"number"!=typeof r.length||numberIsNaN(r.length)?createBuffer(0):fromArrayLike(r);if("Buffer"===r.type&&Array.isArray(r.data))return fromArrayLike(r.data)}(r);if(p)return p;if("undefined"!=typeof Symbol&&null!=Symbol.toPrimitive&&"function"==typeof r[Symbol.toPrimitive])return Buffer.from(r[Symbol.toPrimitive]("string"),i,o);throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r)}function assertSize(r){if("number"!=typeof r)throw new TypeError('"size" argument must be of type number');if(r<0)throw new RangeError('The value "'+r+'" is invalid for option "size"')}function allocUnsafe(r){return assertSize(r),createBuffer(r<0?0:0|checked(r))}function fromArrayLike(r){for(var i=r.length<0?0:0|checked(r.length),o=createBuffer(i),u=0;u<i;u+=1)o[u]=255&r[u];return o}function fromArrayBuffer(r,i,o){if(i<0||r.byteLength<i)throw new RangeError('"offset" is outside of buffer bounds');if(r.byteLength<i+(o||0))throw new RangeError('"length" is outside of buffer bounds');var u;return u=void 0===i&&void 0===o?new Uint8Array(r):void 0===o?new Uint8Array(r,i):new Uint8Array(r,i,o),Object.setPrototypeOf(u,Buffer.prototype),u}function checked(r){if(r>=y)throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+y.toString(16)+" bytes");return 0|r}function byteLength(r,i){if(Buffer.isBuffer(r))return r.length;if(ArrayBuffer.isView(r)||isInstance(r,ArrayBuffer))return r.byteLength;if("string"!=typeof r)throw new TypeError('The "string" argument must be one of type string, Buffer, or ArrayBuffer. Received type '+typeof r);var o=r.length,u=arguments.length>2&&!0===arguments[2];if(!u&&0===o)return 0;for(var p=!1;;)switch(i){case"ascii":case"latin1":case"binary":return o;case"utf8":case"utf-8":return utf8ToBytes(r).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*o;case"hex":return o>>>1;case"base64":return base64ToBytes(r).length;default:if(p)return u?-1:utf8ToBytes(r).length;i=(""+i).toLowerCase(),p=!0}}function slowToString(r,i,o){var u=!1;if((void 0===i||i<0)&&(i=0),i>this.length)return"";if((void 0===o||o>this.length)&&(o=this.length),o<=0)return"";if((o>>>=0)<=(i>>>=0))return"";for(r||(r="utf8");;)switch(r){case"hex":return hexSlice(this,i,o);case"utf8":case"utf-8":return utf8Slice(this,i,o);case"ascii":return asciiSlice(this,i,o);case"latin1":case"binary":return latin1Slice(this,i,o);case"base64":return base64Slice(this,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return utf16leSlice(this,i,o);default:if(u)throw new TypeError("Unknown encoding: "+r);r=(r+"").toLowerCase(),u=!0}}function swap(r,i,o){var u=r[i];r[i]=r[o],r[o]=u}function bidirectionalIndexOf(r,i,o,u,p){if(0===r.length)return-1;if("string"==typeof o?(u=o,o=0):o>2147483647?o=2147483647:o<-2147483648&&(o=-2147483648),numberIsNaN(o=+o)&&(o=p?0:r.length-1),o<0&&(o=r.length+o),o>=r.length){if(p)return-1;o=r.length-1}else if(o<0){if(!p)return-1;o=0}if("string"==typeof i&&(i=Buffer.from(i,u)),Buffer.isBuffer(i))return 0===i.length?-1:arrayIndexOf(r,i,o,u,p);if("number"==typeof i)return i&=255,"function"==typeof Uint8Array.prototype.indexOf?p?Uint8Array.prototype.indexOf.call(r,i,o):Uint8Array.prototype.lastIndexOf.call(r,i,o):arrayIndexOf(r,[i],o,u,p);throw new TypeError("val must be string, number or Buffer")}function arrayIndexOf(r,i,o,u,p){var g,y=1,w=r.length,_=i.length;if(void 0!==u&&("ucs2"===(u=String(u).toLowerCase())||"ucs-2"===u||"utf16le"===u||"utf-16le"===u)){if(r.length<2||i.length<2)return-1;y=2,w/=2,_/=2,o/=2}function read(r,i){return 1===y?r[i]:r.readUInt16BE(i*y)}if(p){var x=-1;for(g=o;g<w;g++)if(read(r,g)===read(i,-1===x?0:g-x)){if(-1===x&&(x=g),g-x+1===_)return x*y}else-1!==x&&(g-=g-x),x=-1}else for(o+_>w&&(o=w-_),g=o;g>=0;g--){for(var k=!0,P=0;P<_;P++)if(read(r,g+P)!==read(i,P)){k=!1;break}if(k)return g}return-1}function hexWrite(r,i,o,u){o=Number(o)||0;var p=r.length-o;u?(u=Number(u))>p&&(u=p):u=p;var g,y=i.length;for(u>y/2&&(u=y/2),g=0;g<u;++g){var w=parseInt(i.substr(2*g,2),16);if(numberIsNaN(w))return g;r[o+g]=w}return g}function utf8Write(r,i,o,u){return blitBuffer(utf8ToBytes(i,r.length-o),r,o,u)}function asciiWrite(r,i,o,u){return blitBuffer(function asciiToBytes(r){for(var i=[],o=0;o<r.length;++o)i.push(255&r.charCodeAt(o));return i}(i),r,o,u)}function base64Write(r,i,o,u){return blitBuffer(base64ToBytes(i),r,o,u)}function ucs2Write(r,i,o,u){return blitBuffer(function utf16leToBytes(r,i){for(var o,u,p,g=[],y=0;y<r.length&&!((i-=2)<0);++y)u=(o=r.charCodeAt(y))>>8,p=o%256,g.push(p),g.push(u);return g}(i,r.length-o),r,o,u)}function base64Slice(r,i,o){return 0===i&&o===r.length?u.fromByteArray(r):u.fromByteArray(r.slice(i,o))}function utf8Slice(r,i,o){o=Math.min(r.length,o);for(var u=[],p=i;p<o;){var g=r[p],y=null,_=g>239?4:g>223?3:g>191?2:1;if(p+_<=o){var x=void 0,k=void 0,P=void 0,O=void 0;switch(_){case 1:g<128&&(y=g);break;case 2:128==(192&(x=r[p+1]))&&(O=(31&g)<<6|63&x)>127&&(y=O);break;case 3:x=r[p+1],k=r[p+2],128==(192&x)&&128==(192&k)&&(O=(15&g)<<12|(63&x)<<6|63&k)>2047&&(O<55296||O>57343)&&(y=O);break;case 4:x=r[p+1],k=r[p+2],P=r[p+3],128==(192&x)&&128==(192&k)&&128==(192&P)&&(O=(15&g)<<18|(63&x)<<12|(63&k)<<6|63&P)>65535&&O<1114112&&(y=O)}}null===y?(y=65533,_=1):y>65535&&(y-=65536,u.push(y>>>10&1023|55296),y=56320|1023&y),u.push(y),p+=_}return function decodeCodePointsArray(r){var i=r.length;if(i<=w)return String.fromCharCode.apply(String,r);var o="",u=0;for(;u<i;)o+=String.fromCharCode.apply(String,r.slice(u,u+=w));return o}(u)}i.kMaxLength=y,Buffer.TYPED_ARRAY_SUPPORT=function typedArraySupport(){try{var r=new Uint8Array(1),i={foo:function foo(){return 42}};return Object.setPrototypeOf(i,Uint8Array.prototype),Object.setPrototypeOf(r,i),42===r.foo()}catch(r){return!1}}(),!Buffer.TYPED_ARRAY_SUPPORT&&"undefined"!=typeof console&&console.error,Object.defineProperty(Buffer.prototype,"parent",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.buffer}}),Object.defineProperty(Buffer.prototype,"offset",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.byteOffset}}),Buffer.poolSize=8192,Buffer.from=function(r,i,o){return from(r,i,o)},Object.setPrototypeOf(Buffer.prototype,Uint8Array.prototype),Object.setPrototypeOf(Buffer,Uint8Array),Buffer.alloc=function(r,i,o){return function alloc(r,i,o){return assertSize(r),r<=0?createBuffer(r):void 0!==i?"string"==typeof o?createBuffer(r).fill(i,o):createBuffer(r).fill(i):createBuffer(r)}(r,i,o)},Buffer.allocUnsafe=function(r){return allocUnsafe(r)},Buffer.allocUnsafeSlow=function(r){return allocUnsafe(r)},Buffer.isBuffer=function isBuffer(r){return null!=r&&!0===r._isBuffer&&r!==Buffer.prototype},Buffer.compare=function compare(r,i){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),isInstance(i,Uint8Array)&&(i=Buffer.from(i,i.offset,i.byteLength)),!Buffer.isBuffer(r)||!Buffer.isBuffer(i))throw new TypeError('The "buf1", "buf2" arguments must be one of type Buffer or Uint8Array');if(r===i)return 0;for(var o=r.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(r[p]!==i[p]){o=r[p],u=i[p];break}return o<u?-1:u<o?1:0},Buffer.isEncoding=function isEncoding(r){switch(String(r).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},Buffer.concat=function concat(r,i){if(!Array.isArray(r))throw new TypeError('"list" argument must be an Array of Buffers');if(0===r.length)return Buffer.alloc(0);var o;if(void 0===i)for(i=0,o=0;o<r.length;++o)i+=r[o].length;var u=Buffer.allocUnsafe(i),p=0;for(o=0;o<r.length;++o){var g=r[o];if(isInstance(g,Uint8Array))p+g.length>u.length?(Buffer.isBuffer(g)||(g=Buffer.from(g)),g.copy(u,p)):Uint8Array.prototype.set.call(u,g,p);else{if(!Buffer.isBuffer(g))throw new TypeError('"list" argument must be an Array of Buffers');g.copy(u,p)}p+=g.length}return u},Buffer.byteLength=byteLength,Buffer.prototype._isBuffer=!0,Buffer.prototype.swap16=function swap16(){var r=this.length;if(r%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var i=0;i<r;i+=2)swap(this,i,i+1);return this},Buffer.prototype.swap32=function swap32(){var r=this.length;if(r%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var i=0;i<r;i+=4)swap(this,i,i+3),swap(this,i+1,i+2);return this},Buffer.prototype.swap64=function swap64(){var r=this.length;if(r%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var i=0;i<r;i+=8)swap(this,i,i+7),swap(this,i+1,i+6),swap(this,i+2,i+5),swap(this,i+3,i+4);return this},Buffer.prototype.toString=function toString(){var r=this.length;return 0===r?"":0===arguments.length?utf8Slice(this,0,r):slowToString.apply(this,arguments)},Buffer.prototype.toLocaleString=Buffer.prototype.toString,Buffer.prototype.equals=function equals(r){if(!Buffer.isBuffer(r))throw new TypeError("Argument must be a Buffer");return this===r||0===Buffer.compare(this,r)},Buffer.prototype.inspect=function inspect(){var r="",o=i.INSPECT_MAX_BYTES;return r=this.toString("hex",0,o).replace(/(.{2})/g,"$1 ").trim(),this.length>o&&(r+=" ... "),"<Buffer "+r+">"},g&&(Buffer.prototype[g]=Buffer.prototype.inspect),Buffer.prototype.compare=function compare(r,i,o,u,p){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),!Buffer.isBuffer(r))throw new TypeError('The "target" argument must be one of type Buffer or Uint8Array. Received type '+typeof r);if(void 0===i&&(i=0),void 0===o&&(o=r?r.length:0),void 0===u&&(u=0),void 0===p&&(p=this.length),i<0||o>r.length||u<0||p>this.length)throw new RangeError("out of range index");if(u>=p&&i>=o)return 0;if(u>=p)return-1;if(i>=o)return 1;if(this===r)return 0;for(var g=(p>>>=0)-(u>>>=0),y=(o>>>=0)-(i>>>=0),w=Math.min(g,y),_=this.slice(u,p),x=r.slice(i,o),k=0;k<w;++k)if(_[k]!==x[k]){g=_[k],y=x[k];break}return g<y?-1:y<g?1:0},Buffer.prototype.includes=function includes(r,i,o){return-1!==this.indexOf(r,i,o)},Buffer.prototype.indexOf=function indexOf(r,i,o){return bidirectionalIndexOf(this,r,i,o,!0)},Buffer.prototype.lastIndexOf=function lastIndexOf(r,i,o){return bidirectionalIndexOf(this,r,i,o,!1)},Buffer.prototype.write=function write(r,i,o,u){if(void 0===i)u="utf8",o=this.length,i=0;else if(void 0===o&&"string"==typeof i)u=i,o=this.length,i=0;else{if(!isFinite(i))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");i>>>=0,isFinite(o)?(o>>>=0,void 0===u&&(u="utf8")):(u=o,o=void 0)}var p=this.length-i;if((void 0===o||o>p)&&(o=p),r.length>0&&(o<0||i<0)||i>this.length)throw new RangeError("Attempt to write outside buffer bounds");u||(u="utf8");for(var g=!1;;)switch(u){case"hex":return hexWrite(this,r,i,o);case"utf8":case"utf-8":return utf8Write(this,r,i,o);case"ascii":case"latin1":case"binary":return asciiWrite(this,r,i,o);case"base64":return base64Write(this,r,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return ucs2Write(this,r,i,o);default:if(g)throw new TypeError("Unknown encoding: "+u);u=(""+u).toLowerCase(),g=!0}},Buffer.prototype.toJSON=function toJSON(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};var w=4096;function asciiSlice(r,i,o){var u="";o=Math.min(r.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(127&r[p]);return u}function latin1Slice(r,i,o){var u="";o=Math.min(r.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(r[p]);return u}function hexSlice(r,i,o){var u=r.length;(!i||i<0)&&(i=0),(!o||o<0||o>u)&&(o=u);for(var p="",g=i;g<o;++g)p+=k[r[g]];return p}function utf16leSlice(r,i,o){for(var u=r.slice(i,o),p="",g=0;g<u.length-1;g+=2)p+=String.fromCharCode(u[g]+256*u[g+1]);return p}function checkOffset(r,i,o){if(r%1!=0||r<0)throw new RangeError("offset is not uint");if(r+i>o)throw new RangeError("Trying to access beyond buffer length")}function checkInt(r,i,o,u,p,g){if(!Buffer.isBuffer(r))throw new TypeError('"buffer" argument must be a Buffer instance');if(i>p||i<g)throw new RangeError('"value" argument is out of bounds');if(o+u>r.length)throw new RangeError("Index out of range")}function wrtBigUInt64LE(r,i,o,u,p){checkIntBI(i,u,p,r,o,7);var g=Number(i&BigInt(4294967295));r[o++]=g,g>>=8,r[o++]=g,g>>=8,r[o++]=g,g>>=8,r[o++]=g;var y=Number(i>>BigInt(32)&BigInt(4294967295));return r[o++]=y,y>>=8,r[o++]=y,y>>=8,r[o++]=y,y>>=8,r[o++]=y,o}function wrtBigUInt64BE(r,i,o,u,p){checkIntBI(i,u,p,r,o,7);var g=Number(i&BigInt(4294967295));r[o+7]=g,g>>=8,r[o+6]=g,g>>=8,r[o+5]=g,g>>=8,r[o+4]=g;var y=Number(i>>BigInt(32)&BigInt(4294967295));return r[o+3]=y,y>>=8,r[o+2]=y,y>>=8,r[o+1]=y,y>>=8,r[o]=y,o+8}function checkIEEE754(r,i,o,u,p,g){if(o+u>r.length)throw new RangeError("Index out of range");if(o<0)throw new RangeError("Index out of range")}function writeFloat(r,i,o,u,g){return i=+i,o>>>=0,g||checkIEEE754(r,0,o,4),p.write(r,i,o,u,23,4),o+4}function writeDouble(r,i,o,u,g){return i=+i,o>>>=0,g||checkIEEE754(r,0,o,8),p.write(r,i,o,u,52,8),o+8}Buffer.prototype.slice=function slice(r,i){var o=this.length;(r=~~r)<0?(r+=o)<0&&(r=0):r>o&&(r=o),(i=void 0===i?o:~~i)<0?(i+=o)<0&&(i=0):i>o&&(i=o),i<r&&(i=r);var u=this.subarray(r,i);return Object.setPrototypeOf(u,Buffer.prototype),u},Buffer.prototype.readUintLE=Buffer.prototype.readUIntLE=function readUIntLE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r],p=1,g=0;++g<i&&(p*=256);)u+=this[r+g]*p;return u},Buffer.prototype.readUintBE=Buffer.prototype.readUIntBE=function readUIntBE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r+--i],p=1;i>0&&(p*=256);)u+=this[r+--i]*p;return u},Buffer.prototype.readUint8=Buffer.prototype.readUInt8=function readUInt8(r,i){return r>>>=0,i||checkOffset(r,1,this.length),this[r]},Buffer.prototype.readUint16LE=Buffer.prototype.readUInt16LE=function readUInt16LE(r,i){return r>>>=0,i||checkOffset(r,2,this.length),this[r]|this[r+1]<<8},Buffer.prototype.readUint16BE=Buffer.prototype.readUInt16BE=function readUInt16BE(r,i){return r>>>=0,i||checkOffset(r,2,this.length),this[r]<<8|this[r+1]},Buffer.prototype.readUint32LE=Buffer.prototype.readUInt32LE=function readUInt32LE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),(this[r]|this[r+1]<<8|this[r+2]<<16)+16777216*this[r+3]},Buffer.prototype.readUint32BE=Buffer.prototype.readUInt32BE=function readUInt32BE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),16777216*this[r]+(this[r+1]<<16|this[r+2]<<8|this[r+3])},Buffer.prototype.readBigUInt64LE=defineBigIntMethod((function readBigUInt64LE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=i+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24),p=this[++r]+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+o*Math.pow(2,24);return BigInt(u)+(BigInt(p)<<BigInt(32))})),Buffer.prototype.readBigUInt64BE=defineBigIntMethod((function readBigUInt64BE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=i*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r],p=this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+o;return(BigInt(u)<<BigInt(32))+BigInt(p)})),Buffer.prototype.readIntLE=function readIntLE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=this[r],p=1,g=0;++g<i&&(p*=256);)u+=this[r+g]*p;return u>=(p*=128)&&(u-=Math.pow(2,8*i)),u},Buffer.prototype.readIntBE=function readIntBE(r,i,o){r>>>=0,i>>>=0,o||checkOffset(r,i,this.length);for(var u=i,p=1,g=this[r+--u];u>0&&(p*=256);)g+=this[r+--u]*p;return g>=(p*=128)&&(g-=Math.pow(2,8*i)),g},Buffer.prototype.readInt8=function readInt8(r,i){return r>>>=0,i||checkOffset(r,1,this.length),128&this[r]?-1*(255-this[r]+1):this[r]},Buffer.prototype.readInt16LE=function readInt16LE(r,i){r>>>=0,i||checkOffset(r,2,this.length);var o=this[r]|this[r+1]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt16BE=function readInt16BE(r,i){r>>>=0,i||checkOffset(r,2,this.length);var o=this[r+1]|this[r]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt32LE=function readInt32LE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),this[r]|this[r+1]<<8|this[r+2]<<16|this[r+3]<<24},Buffer.prototype.readInt32BE=function readInt32BE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),this[r]<<24|this[r+1]<<16|this[r+2]<<8|this[r+3]},Buffer.prototype.readBigInt64LE=defineBigIntMethod((function readBigInt64LE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=this[r+4]+this[r+5]*Math.pow(2,8)+this[r+6]*Math.pow(2,16)+(o<<24);return(BigInt(u)<<BigInt(32))+BigInt(i+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24))})),Buffer.prototype.readBigInt64BE=defineBigIntMethod((function readBigInt64BE(r){validateNumber(r>>>=0,"offset");var i=this[r],o=this[r+7];void 0!==i&&void 0!==o||boundsError(r,this.length-8);var u=(i<<24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r];return(BigInt(u)<<BigInt(32))+BigInt(this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+o)})),Buffer.prototype.readFloatLE=function readFloatLE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),p.read(this,r,!0,23,4)},Buffer.prototype.readFloatBE=function readFloatBE(r,i){return r>>>=0,i||checkOffset(r,4,this.length),p.read(this,r,!1,23,4)},Buffer.prototype.readDoubleLE=function readDoubleLE(r,i){return r>>>=0,i||checkOffset(r,8,this.length),p.read(this,r,!0,52,8)},Buffer.prototype.readDoubleBE=function readDoubleBE(r,i){return r>>>=0,i||checkOffset(r,8,this.length),p.read(this,r,!1,52,8)},Buffer.prototype.writeUintLE=Buffer.prototype.writeUIntLE=function writeUIntLE(r,i,o,u){(r=+r,i>>>=0,o>>>=0,u)||checkInt(this,r,i,o,Math.pow(2,8*o)-1,0);var p=1,g=0;for(this[i]=255&r;++g<o&&(p*=256);)this[i+g]=r/p&255;return i+o},Buffer.prototype.writeUintBE=Buffer.prototype.writeUIntBE=function writeUIntBE(r,i,o,u){(r=+r,i>>>=0,o>>>=0,u)||checkInt(this,r,i,o,Math.pow(2,8*o)-1,0);var p=o-1,g=1;for(this[i+p]=255&r;--p>=0&&(g*=256);)this[i+p]=r/g&255;return i+o},Buffer.prototype.writeUint8=Buffer.prototype.writeUInt8=function writeUInt8(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,1,255,0),this[i]=255&r,i+1},Buffer.prototype.writeUint16LE=Buffer.prototype.writeUInt16LE=function writeUInt16LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,65535,0),this[i]=255&r,this[i+1]=r>>>8,i+2},Buffer.prototype.writeUint16BE=Buffer.prototype.writeUInt16BE=function writeUInt16BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,65535,0),this[i]=r>>>8,this[i+1]=255&r,i+2},Buffer.prototype.writeUint32LE=Buffer.prototype.writeUInt32LE=function writeUInt32LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,4294967295,0),this[i+3]=r>>>24,this[i+2]=r>>>16,this[i+1]=r>>>8,this[i]=255&r,i+4},Buffer.prototype.writeUint32BE=Buffer.prototype.writeUInt32BE=function writeUInt32BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,4294967295,0),this[i]=r>>>24,this[i+1]=r>>>16,this[i+2]=r>>>8,this[i+3]=255&r,i+4},Buffer.prototype.writeBigUInt64LE=defineBigIntMethod((function writeBigUInt64LE(r,i){return void 0===i&&(i=0),wrtBigUInt64LE(this,r,i,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeBigUInt64BE=defineBigIntMethod((function writeBigUInt64BE(r,i){return void 0===i&&(i=0),wrtBigUInt64BE(this,r,i,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeIntLE=function writeIntLE(r,i,o,u){if(r=+r,i>>>=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,r,i,o,p-1,-p)}var g=0,y=1,w=0;for(this[i]=255&r;++g<o&&(y*=256);)r<0&&0===w&&0!==this[i+g-1]&&(w=1),this[i+g]=(r/y|0)-w&255;return i+o},Buffer.prototype.writeIntBE=function writeIntBE(r,i,o,u){if(r=+r,i>>>=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,r,i,o,p-1,-p)}var g=o-1,y=1,w=0;for(this[i+g]=255&r;--g>=0&&(y*=256);)r<0&&0===w&&0!==this[i+g+1]&&(w=1),this[i+g]=(r/y|0)-w&255;return i+o},Buffer.prototype.writeInt8=function writeInt8(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,1,127,-128),r<0&&(r=255+r+1),this[i]=255&r,i+1},Buffer.prototype.writeInt16LE=function writeInt16LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,32767,-32768),this[i]=255&r,this[i+1]=r>>>8,i+2},Buffer.prototype.writeInt16BE=function writeInt16BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,2,32767,-32768),this[i]=r>>>8,this[i+1]=255&r,i+2},Buffer.prototype.writeInt32LE=function writeInt32LE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,2147483647,-2147483648),this[i]=255&r,this[i+1]=r>>>8,this[i+2]=r>>>16,this[i+3]=r>>>24,i+4},Buffer.prototype.writeInt32BE=function writeInt32BE(r,i,o){return r=+r,i>>>=0,o||checkInt(this,r,i,4,2147483647,-2147483648),r<0&&(r=4294967295+r+1),this[i]=r>>>24,this[i+1]=r>>>16,this[i+2]=r>>>8,this[i+3]=255&r,i+4},Buffer.prototype.writeBigInt64LE=defineBigIntMethod((function writeBigInt64LE(r,i){return void 0===i&&(i=0),wrtBigUInt64LE(this,r,i,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeBigInt64BE=defineBigIntMethod((function writeBigInt64BE(r,i){return void 0===i&&(i=0),wrtBigUInt64BE(this,r,i,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeFloatLE=function writeFloatLE(r,i,o){return writeFloat(this,r,i,!0,o)},Buffer.prototype.writeFloatBE=function writeFloatBE(r,i,o){return writeFloat(this,r,i,!1,o)},Buffer.prototype.writeDoubleLE=function writeDoubleLE(r,i,o){return writeDouble(this,r,i,!0,o)},Buffer.prototype.writeDoubleBE=function writeDoubleBE(r,i,o){return writeDouble(this,r,i,!1,o)},Buffer.prototype.copy=function copy(r,i,o,u){if(!Buffer.isBuffer(r))throw new TypeError("argument should be a Buffer");if(o||(o=0),u||0===u||(u=this.length),i>=r.length&&(i=r.length),i||(i=0),u>0&&u<o&&(u=o),u===o)return 0;if(0===r.length||0===this.length)return 0;if(i<0)throw new RangeError("targetStart out of bounds");if(o<0||o>=this.length)throw new RangeError("Index out of range");if(u<0)throw new RangeError("sourceEnd out of bounds");u>this.length&&(u=this.length),r.length-i<u-o&&(u=r.length-i+o);var p=u-o;return this===r&&"function"==typeof Uint8Array.prototype.copyWithin?this.copyWithin(i,o,u):Uint8Array.prototype.set.call(r,this.subarray(o,u),i),p},Buffer.prototype.fill=function fill(r,i,o,u){if("string"==typeof r){if("string"==typeof i?(u=i,i=0,o=this.length):"string"==typeof o&&(u=o,o=this.length),void 0!==u&&"string"!=typeof u)throw new TypeError("encoding must be a string");if("string"==typeof u&&!Buffer.isEncoding(u))throw new TypeError("Unknown encoding: "+u);if(1===r.length){var p=r.charCodeAt(0);("utf8"===u&&p<128||"latin1"===u)&&(r=p)}}else"number"==typeof r?r&=255:"boolean"==typeof r&&(r=Number(r));if(i<0||this.length<i||this.length<o)throw new RangeError("Out of range index");if(o<=i)return this;var g;if(i>>>=0,o=void 0===o?this.length:o>>>0,r||(r=0),"number"==typeof r)for(g=i;g<o;++g)this[g]=r;else{var y=Buffer.isBuffer(r)?r:Buffer.from(r,u),w=y.length;if(0===w)throw new TypeError('The value "'+r+'" is invalid for argument "value"');for(g=0;g<o-i;++g)this[g+i]=y[g%w]}return this};var _={};function E(r,i,o){_[r]=function(o){function NodeError(){var u;return u=o.call(this)||this,Object.defineProperty(_assertThisInitialized(u),"message",{value:i.apply(_assertThisInitialized(u),arguments),writable:!0,configurable:!0}),u.name=u.name+" ["+r+"]",u.stack,delete u.name,u}return function _inheritsLoose(r,i){r.prototype=Object.create(i.prototype),r.prototype.constructor=r,_setPrototypeOf(r,i)}(NodeError,o),NodeError.prototype.toString=function toString(){return this.name+" ["+r+"]: "+this.message},function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}(NodeError,[{key:"code",get:function get(){return r},set:function set(r){Object.defineProperty(this,"code",{configurable:!0,enumerable:!0,value:r,writable:!0})}}]),NodeError}(o)}function addNumericalSeparator(r){for(var i="",o=r.length,u="-"===r[0]?1:0;o>=u+4;o-=3)i="_"+r.slice(o-3,o)+i;return""+r.slice(0,o)+i}function checkIntBI(r,i,o,u,p,g){if(r>o||r<i){var y,w="bigint"==typeof i?"n":"";throw y=g>3?0===i||i===BigInt(0)?">= 0"+w+" and < 2"+w+" ** "+8*(g+1)+w:">= -(2"+w+" ** "+(8*(g+1)-1)+w+") and < 2 ** "+(8*(g+1)-1)+w:">= "+i+w+" and <= "+o+w,new _.ERR_OUT_OF_RANGE("value",y,r)}!function checkBounds(r,i,o){validateNumber(i,"offset"),void 0!==r[i]&&void 0!==r[i+o]||boundsError(i,r.length-(o+1))}(u,p,g)}function validateNumber(r,i){if("number"!=typeof r)throw new _.ERR_INVALID_ARG_TYPE(i,"number",r)}function boundsError(r,i,o){if(Math.floor(r)!==r)throw validateNumber(r,o),new _.ERR_OUT_OF_RANGE(o||"offset","an integer",r);if(i<0)throw new _.ERR_BUFFER_OUT_OF_BOUNDS;throw new _.ERR_OUT_OF_RANGE(o||"offset",">= "+(o?1:0)+" and <= "+i,r)}E("ERR_BUFFER_OUT_OF_BOUNDS",(function(r){return r?r+" is outside of buffer bounds":"Attempt to access memory outside buffer bounds"}),RangeError),E("ERR_INVALID_ARG_TYPE",(function(r,i){return'The "'+r+'" argument must be of type number. Received type '+typeof i}),TypeError),E("ERR_OUT_OF_RANGE",(function(r,i,o){var u='The value of "'+r+'" is out of range.',p=o;return Number.isInteger(o)&&Math.abs(o)>Math.pow(2,32)?p=addNumericalSeparator(String(o)):"bigint"==typeof o&&(p=String(o),(o>Math.pow(BigInt(2),BigInt(32))||o<-Math.pow(BigInt(2),BigInt(32)))&&(p=addNumericalSeparator(p)),p+="n"),u+=" It must be "+i+". Received "+p}),RangeError);var x=/[^+/0-9A-Za-z-_]/g;function utf8ToBytes(r,i){var o;i=i||1/0;for(var u=r.length,p=null,g=[],y=0;y<u;++y){if((o=r.charCodeAt(y))>55295&&o<57344){if(!p){if(o>56319){(i-=3)>-1&&g.push(239,191,189);continue}if(y+1===u){(i-=3)>-1&&g.push(239,191,189);continue}p=o;continue}if(o<56320){(i-=3)>-1&&g.push(239,191,189),p=o;continue}o=65536+(p-55296<<10|o-56320)}else p&&(i-=3)>-1&&g.push(239,191,189);if(p=null,o<128){if((i-=1)<0)break;g.push(o)}else if(o<2048){if((i-=2)<0)break;g.push(o>>6|192,63&o|128)}else if(o<65536){if((i-=3)<0)break;g.push(o>>12|224,o>>6&63|128,63&o|128)}else{if(!(o<1114112))throw new Error("Invalid code point");if((i-=4)<0)break;g.push(o>>18|240,o>>12&63|128,o>>6&63|128,63&o|128)}}return g}function base64ToBytes(r){return u.toByteArray(function base64clean(r){if((r=(r=r.split("=")[0]).trim().replace(x,"")).length<2)return"";for(;r.length%4!=0;)r+="=";return r}(r))}function blitBuffer(r,i,o,u){var p;for(p=0;p<u&&!(p+o>=i.length||p>=r.length);++p)i[p+o]=r[p];return p}function isInstance(r,i){return r instanceof i||null!=r&&null!=r.constructor&&null!=r.constructor.name&&r.constructor.name===i.name}function numberIsNaN(r){return r!=r}var k=function(){for(var r="0123456789abcdef",i=new Array(256),o=0;o<16;++o)for(var u=16*o,p=0;p<16;++p)i[u+p]=r[o]+r[p];return i}();function defineBigIntMethod(r){return"undefined"==typeof BigInt?BufferBigIntNotDefined:r}function BufferBigIntNotDefined(){throw new Error("BigInt not supported")}},38719:function(r,i,o){"use strict";o(10720),o(14032),r.exports="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof DataView},36597:function(r,i,o){"use strict";var u,p,g,y=o(38719),w=o(15567),_=o(32010),x=o(94578),k=o(24517),P=o(20340),O=o(52564),I=o(68664),B=o(48914),D=o(13711),R=o(95892).f,N=o(70176),W=o(69548),U=o(3840),j=o(38688),G=o(46859),X=_.Int8Array,K=X&&X.prototype,Y=_.Uint8ClampedArray,J=Y&&Y.prototype,$=X&&W(X),ee=K&&W(K),te=Object.prototype,re=_.TypeError,ne=j("toStringTag"),ie=G("TYPED_ARRAY_TAG"),oe=G("TYPED_ARRAY_CONSTRUCTOR"),ae=y&&!!U&&"Opera"!==O(_.opera),se=!1,ce={Int8Array:1,Uint8Array:1,Uint8ClampedArray:1,Int16Array:2,Uint16Array:2,Int32Array:4,Uint32Array:4,Float32Array:4,Float64Array:8},le={BigInt64Array:8,BigUint64Array:8},ue=function isTypedArray(r){if(!k(r))return!1;var i=O(r);return P(ce,i)||P(le,i)};for(u in ce)(g=(p=_[u])&&p.prototype)?B(g,oe,p):ae=!1;for(u in le)(g=(p=_[u])&&p.prototype)&&B(g,oe,p);if((!ae||!x($)||$===Function.prototype)&&($=function TypedArray(){throw re("Incorrect invocation")},ae))for(u in ce)_[u]&&U(_[u],$);if((!ae||!ee||ee===te)&&(ee=$.prototype,ae))for(u in ce)_[u]&&U(_[u].prototype,ee);if(ae&&W(J)!==ee&&U(J,ee),w&&!P(ee,ne))for(u in se=!0,R(ee,ne,{get:function get(){return k(this)?this[ie]:void 0}}),ce)_[u]&&B(_[u],ie,u);r.exports={NATIVE_ARRAY_BUFFER_VIEWS:ae,TYPED_ARRAY_CONSTRUCTOR:oe,TYPED_ARRAY_TAG:se&&ie,aTypedArray:function aTypedArray(r){if(ue(r))return r;throw re("Target is not a typed array")},aTypedArrayConstructor:function aTypedArrayConstructor(r){if(x(r)&&(!U||N($,r)))return r;throw re(I(r)+" is not a typed array constructor")},exportTypedArrayMethod:function exportTypedArrayMethod(r,i,o){if(w){if(o)for(var u in ce){var p=_[u];if(p&&P(p.prototype,r))try{delete p.prototype[r]}catch(r){}}ee[r]&&!o||D(ee,r,o?i:ae&&K[r]||i)}},exportTypedArrayStaticMethod:function exportTypedArrayStaticMethod(r,i,o){var u,p;if(w){if(U){if(o)for(u in ce)if((p=_[u])&&P(p,r))try{delete p[r]}catch(r){}if($[r]&&!o)return;try{return D($,r,o?i:ae&&$[r]||i)}catch(r){}}for(u in ce)!(p=_[u])||p[r]&&!o||D(p,r,i)}},isView:function isView(r){if(!k(r))return!1;var i=O(r);return"DataView"===i||P(ce,i)||P(le,i)},isTypedArray:ue,TypedArray:$,TypedArrayPrototype:ee}},89987:function(r,i,o){"use strict";o(24863);var u=o(32010),p=o(38347),g=o(15567),y=o(38719),w=o(7081),_=o(48914),x=o(15341),k=o(47044),P=o(2868),O=o(26882),I=o(23417),B=o(71265),D=o(64397),R=o(69548),N=o(3840),W=o(6611).f,U=o(95892).f,j=o(72864),G=o(73163),X=o(15216),K=o(70172),Y=w.PROPER,J=w.CONFIGURABLE,$=K.get,ee=K.set,te="ArrayBuffer",re="DataView",ne="prototype",ie="Wrong index",oe=u[te],ae=oe,se=ae&&ae[ne],ce=u[re],le=ce&&ce[ne],ue=Object.prototype,fe=u.Array,he=u.RangeError,de=p(j),pe=p([].reverse),ge=D.pack,ye=D.unpack,ve=function packInt8(r){return[255&r]},me=function packInt16(r){return[255&r,r>>8&255]},be=function packInt32(r){return[255&r,r>>8&255,r>>16&255,r>>24&255]},we=function unpackInt32(r){return r[3]<<24|r[2]<<16|r[1]<<8|r[0]},_e=function packFloat32(r){return ge(r,23,4)},Se=function packFloat64(r){return ge(r,52,8)},xe=function addGetter(r,i){U(r[ne],i,{get:function get(){return $(this)[i]}})},Ce=function get(r,i,o,u){var p=B(o),g=$(r);if(p+i>g.byteLength)throw he(ie);var y=$(g.buffer).bytes,w=p+g.byteOffset,_=G(y,w,w+i);return u?_:pe(_)},ke=function set(r,i,o,u,p,g){var y=B(o),w=$(r);if(y+i>w.byteLength)throw he(ie);for(var _=$(w.buffer).bytes,x=y+w.byteOffset,k=u(+p),P=0;P<i;P++)_[x+P]=k[g?P:i-P-1]};if(y){var Pe=Y&&oe.name!==te;if(k((function(){oe(1)}))&&k((function(){new oe(-1)}))&&!k((function(){return new oe,new oe(1.5),new oe(NaN),Pe&&!J})))Pe&&J&&_(oe,"name",te);else{(ae=function ArrayBuffer(r){return P(this,se),new oe(B(r))})[ne]=se;for(var Ae,Ee=W(oe),Te=0;Ee.length>Te;)(Ae=Ee[Te++])in ae||_(ae,Ae,oe[Ae]);se.constructor=ae}N&&R(le)!==ue&&N(le,ue);var Oe=new ce(new ae(2)),Ie=p(le.setInt8);Oe.setInt8(0,2147483648),Oe.setInt8(1,2147483649),!Oe.getInt8(0)&&Oe.getInt8(1)||x(le,{setInt8:function setInt8(r,i){Ie(this,r,i<<24>>24)},setUint8:function setUint8(r,i){Ie(this,r,i<<24>>24)}},{unsafe:!0})}else se=(ae=function ArrayBuffer(r){P(this,se);var i=B(r);ee(this,{bytes:de(fe(i),0),byteLength:i}),g||(this.byteLength=i)})[ne],le=(ce=function DataView(r,i,o){P(this,le),P(r,se);var u=$(r).byteLength,p=O(i);if(p<0||p>u)throw he("Wrong offset");if(p+(o=void 0===o?u-p:I(o))>u)throw he("Wrong length");ee(this,{buffer:r,byteLength:o,byteOffset:p}),g||(this.buffer=r,this.byteLength=o,this.byteOffset=p)})[ne],g&&(xe(ae,"byteLength"),xe(ce,"buffer"),xe(ce,"byteLength"),xe(ce,"byteOffset")),x(le,{getInt8:function getInt8(r){return Ce(this,1,r)[0]<<24>>24},getUint8:function getUint8(r){return Ce(this,1,r)[0]},getInt16:function getInt16(r){var i=Ce(this,2,r,arguments.length>1?arguments[1]:void 0);return(i[1]<<8|i[0])<<16>>16},getUint16:function getUint16(r){var i=Ce(this,2,r,arguments.length>1?arguments[1]:void 0);return i[1]<<8|i[0]},getInt32:function getInt32(r){return we(Ce(this,4,r,arguments.length>1?arguments[1]:void 0))},getUint32:function getUint32(r){return we(Ce(this,4,r,arguments.length>1?arguments[1]:void 0))>>>0},getFloat32:function getFloat32(r){return ye(Ce(this,4,r,arguments.length>1?arguments[1]:void 0),23)},getFloat64:function getFloat64(r){return ye(Ce(this,8,r,arguments.length>1?arguments[1]:void 0),52)},setInt8:function setInt8(r,i){ke(this,1,r,ve,i)},setUint8:function setUint8(r,i){ke(this,1,r,ve,i)},setInt16:function setInt16(r,i){ke(this,2,r,me,i,arguments.length>2?arguments[2]:void 0)},setUint16:function setUint16(r,i){ke(this,2,r,me,i,arguments.length>2?arguments[2]:void 0)},setInt32:function setInt32(r,i){ke(this,4,r,be,i,arguments.length>2?arguments[2]:void 0)},setUint32:function setUint32(r,i){ke(this,4,r,be,i,arguments.length>2?arguments[2]:void 0)},setFloat32:function setFloat32(r,i){ke(this,4,r,_e,i,arguments.length>2?arguments[2]:void 0)},setFloat64:function setFloat64(r,i){ke(this,8,r,Se,i,arguments.length>2?arguments[2]:void 0)}});X(ae,te),X(ce,re),r.exports={ArrayBuffer:ae,DataView:ce}},10720:function(r,i,o){"use strict";var u=o(56475),p=o(32010),g=o(89987),y=o(51334),w="ArrayBuffer",_=g[w];u({global:!0,forced:p[w]!==_},{ArrayBuffer:_}),y(w)},6993:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib.BlockCipher,o=r.algo,p=[],g=[],y=[],w=[],_=[],x=[],k=[],P=[],O=[],I=[];!function(){for(var r=[],i=0;i<256;i++)r[i]=i<128?i<<1:i<<1^283;var o=0,u=0;for(i=0;i<256;i++){var B=u^u<<1^u<<2^u<<3^u<<4;B=B>>>8^255&B^99,p[o]=B,g[B]=o;var D=r[o],R=r[D],N=r[R],W=257*r[B]^16843008*B;y[o]=W<<24|W>>>8,w[o]=W<<16|W>>>16,_[o]=W<<8|W>>>24,x[o]=W,W=16843009*N^65537*R^257*D^16843008*o,k[B]=W<<24|W>>>8,P[B]=W<<16|W>>>16,O[B]=W<<8|W>>>24,I[B]=W,o?(o=D^r[r[r[N^D]]],u^=r[r[u]]):o=u=1}}();var B=[0,1,2,4,8,16,32,64,128,27,54],D=o.AES=i.extend({_doReset:function _doReset(){if(!this._nRounds||this._keyPriorReset!==this._key){for(var r=this._keyPriorReset=this._key,i=r.words,o=r.sigBytes/4,u=4*((this._nRounds=o+6)+1),g=this._keySchedule=[],y=0;y<u;y++)y<o?g[y]=i[y]:(x=g[y-1],y%o?o>6&&y%o==4&&(x=p[x>>>24]<<24|p[x>>>16&255]<<16|p[x>>>8&255]<<8|p[255&x]):(x=p[(x=x<<8|x>>>24)>>>24]<<24|p[x>>>16&255]<<16|p[x>>>8&255]<<8|p[255&x],x^=B[y/o|0]<<24),g[y]=g[y-o]^x);for(var w=this._invKeySchedule=[],_=0;_<u;_++){if(y=u-_,_%4)var x=g[y];else x=g[y-4];w[_]=_<4||y<=4?x:k[p[x>>>24]]^P[p[x>>>16&255]]^O[p[x>>>8&255]]^I[p[255&x]]}}},encryptBlock:function encryptBlock(r,i){this._doCryptBlock(r,i,this._keySchedule,y,w,_,x,p)},decryptBlock:function decryptBlock(r,i){var o=r[i+1];r[i+1]=r[i+3],r[i+3]=o,this._doCryptBlock(r,i,this._invKeySchedule,k,P,O,I,g),o=r[i+1],r[i+1]=r[i+3],r[i+3]=o},_doCryptBlock:function _doCryptBlock(r,i,o,u,p,g,y,w){for(var _=this._nRounds,x=r[i]^o[0],k=r[i+1]^o[1],P=r[i+2]^o[2],O=r[i+3]^o[3],I=4,B=1;B<_;B++){var D=u[x>>>24]^p[k>>>16&255]^g[P>>>8&255]^y[255&O]^o[I++],R=u[k>>>24]^p[P>>>16&255]^g[O>>>8&255]^y[255&x]^o[I++],N=u[P>>>24]^p[O>>>16&255]^g[x>>>8&255]^y[255&k]^o[I++],W=u[O>>>24]^p[x>>>16&255]^g[k>>>8&255]^y[255&P]^o[I++];x=D,k=R,P=N,O=W}D=(w[x>>>24]<<24|w[k>>>16&255]<<16|w[P>>>8&255]<<8|w[255&O])^o[I++],R=(w[k>>>24]<<24|w[P>>>16&255]<<16|w[O>>>8&255]<<8|w[255&x])^o[I++],N=(w[P>>>24]<<24|w[O>>>16&255]<<16|w[x>>>8&255]<<8|w[255&k])^o[I++],W=(w[O>>>24]<<24|w[x>>>16&255]<<16|w[k>>>8&255]<<8|w[255&P])^o[I++],r[i]=D,r[i+1]=R,r[i+2]=N,r[i+3]=W},keySize:8});r.AES=i._createHelper(D)}(),u.AES)},83122:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib.BlockCipher,o=r.algo,p=16,g=[608135816,2242054355,320440878,57701188,2752067618,698298832,137296536,3964562569,1160258022,953160567,3193202383,887688300,3232508343,3380367581,1065670069,3041331479,2450970073,2306472731],y=[[3509652390,2564797868,805139163,3491422135,3101798381,1780907670,3128725573,4046225305,614570311,3012652279,134345442,2240740374,1667834072,1901547113,2757295779,4103290238,227898511,1921955416,1904987480,2182433518,2069144605,3260701109,2620446009,720527379,3318853667,677414384,3393288472,3101374703,2390351024,1614419982,1822297739,2954791486,3608508353,3174124327,2024746970,1432378464,3864339955,2857741204,1464375394,1676153920,1439316330,715854006,3033291828,289532110,2706671279,2087905683,3018724369,1668267050,732546397,1947742710,3462151702,2609353502,2950085171,1814351708,2050118529,680887927,999245976,1800124847,3300911131,1713906067,1641548236,4213287313,1216130144,1575780402,4018429277,3917837745,3693486850,3949271944,596196993,3549867205,258830323,2213823033,772490370,2760122372,1774776394,2652871518,566650946,4142492826,1728879713,2882767088,1783734482,3629395816,2517608232,2874225571,1861159788,326777828,3124490320,2130389656,2716951837,967770486,1724537150,2185432712,2364442137,1164943284,2105845187,998989502,3765401048,2244026483,1075463327,1455516326,1322494562,910128902,469688178,1117454909,936433444,3490320968,3675253459,1240580251,122909385,2157517691,634681816,4142456567,3825094682,3061402683,2540495037,79693498,3249098678,1084186820,1583128258,426386531,1761308591,1047286709,322548459,995290223,1845252383,2603652396,3431023940,2942221577,3202600964,3727903485,1712269319,422464435,3234572375,1170764815,3523960633,3117677531,1434042557,442511882,3600875718,1076654713,1738483198,4213154764,2393238008,3677496056,1014306527,4251020053,793779912,2902807211,842905082,4246964064,1395751752,1040244610,2656851899,3396308128,445077038,3742853595,3577915638,679411651,2892444358,2354009459,1767581616,3150600392,3791627101,3102740896,284835224,4246832056,1258075500,768725851,2589189241,3069724005,3532540348,1274779536,3789419226,2764799539,1660621633,3471099624,4011903706,913787905,3497959166,737222580,2514213453,2928710040,3937242737,1804850592,3499020752,2949064160,2386320175,2390070455,2415321851,4061277028,2290661394,2416832540,1336762016,1754252060,3520065937,3014181293,791618072,3188594551,3933548030,2332172193,3852520463,3043980520,413987798,3465142937,3030929376,4245938359,2093235073,3534596313,375366246,2157278981,2479649556,555357303,3870105701,2008414854,3344188149,4221384143,3956125452,2067696032,3594591187,2921233993,2428461,544322398,577241275,1471733935,610547355,4027169054,1432588573,1507829418,2025931657,3646575487,545086370,48609733,2200306550,1653985193,298326376,1316178497,3007786442,2064951626,458293330,2589141269,3591329599,3164325604,727753846,2179363840,146436021,1461446943,4069977195,705550613,3059967265,3887724982,4281599278,3313849956,1404054877,2845806497,146425753,1854211946],[1266315497,3048417604,3681880366,3289982499,290971e4,1235738493,2632868024,2414719590,3970600049,1771706367,1449415276,3266420449,422970021,1963543593,2690192192,3826793022,1062508698,1531092325,1804592342,2583117782,2714934279,4024971509,1294809318,4028980673,1289560198,2221992742,1669523910,35572830,157838143,1052438473,1016535060,1802137761,1753167236,1386275462,3080475397,2857371447,1040679964,2145300060,2390574316,1461121720,2956646967,4031777805,4028374788,33600511,2920084762,1018524850,629373528,3691585981,3515945977,2091462646,2486323059,586499841,988145025,935516892,3367335476,2599673255,2839830854,265290510,3972581182,2759138881,3795373465,1005194799,847297441,406762289,1314163512,1332590856,1866599683,4127851711,750260880,613907577,1450815602,3165620655,3734664991,3650291728,3012275730,3704569646,1427272223,778793252,1343938022,2676280711,2052605720,1946737175,3164576444,3914038668,3967478842,3682934266,1661551462,3294938066,4011595847,840292616,3712170807,616741398,312560963,711312465,1351876610,322626781,1910503582,271666773,2175563734,1594956187,70604529,3617834859,1007753275,1495573769,4069517037,2549218298,2663038764,504708206,2263041392,3941167025,2249088522,1514023603,1998579484,1312622330,694541497,2582060303,2151582166,1382467621,776784248,2618340202,3323268794,2497899128,2784771155,503983604,4076293799,907881277,423175695,432175456,1378068232,4145222326,3954048622,3938656102,3820766613,2793130115,2977904593,26017576,3274890735,3194772133,1700274565,1756076034,4006520079,3677328699,720338349,1533947780,354530856,688349552,3973924725,1637815568,332179504,3949051286,53804574,2852348879,3044236432,1282449977,3583942155,3416972820,4006381244,1617046695,2628476075,3002303598,1686838959,431878346,2686675385,1700445008,1080580658,1009431731,832498133,3223435511,2605976345,2271191193,2516031870,1648197032,4164389018,2548247927,300782431,375919233,238389289,3353747414,2531188641,2019080857,1475708069,455242339,2609103871,448939670,3451063019,1395535956,2413381860,1841049896,1491858159,885456874,4264095073,4001119347,1565136089,3898914787,1108368660,540939232,1173283510,2745871338,3681308437,4207628240,3343053890,4016749493,1699691293,1103962373,3625875870,2256883143,3830138730,1031889488,3479347698,1535977030,4236805024,3251091107,2132092099,1774941330,1199868427,1452454533,157007616,2904115357,342012276,595725824,1480756522,206960106,497939518,591360097,863170706,2375253569,3596610801,1814182875,2094937945,3421402208,1082520231,3463918190,2785509508,435703966,3908032597,1641649973,2842273706,3305899714,1510255612,2148256476,2655287854,3276092548,4258621189,236887753,3681803219,274041037,1734335097,3815195456,3317970021,1899903192,1026095262,4050517792,356393447,2410691914,3873677099,3682840055],[3913112168,2491498743,4132185628,2489919796,1091903735,1979897079,3170134830,3567386728,3557303409,857797738,1136121015,1342202287,507115054,2535736646,337727348,3213592640,1301675037,2528481711,1895095763,1721773893,3216771564,62756741,2142006736,835421444,2531993523,1442658625,3659876326,2882144922,676362277,1392781812,170690266,3921047035,1759253602,3611846912,1745797284,664899054,1329594018,3901205900,3045908486,2062866102,2865634940,3543621612,3464012697,1080764994,553557557,3656615353,3996768171,991055499,499776247,1265440854,648242737,3940784050,980351604,3713745714,1749149687,3396870395,4211799374,3640570775,1161844396,3125318951,1431517754,545492359,4268468663,3499529547,1437099964,2702547544,3433638243,2581715763,2787789398,1060185593,1593081372,2418618748,4260947970,69676912,2159744348,86519011,2512459080,3838209314,1220612927,3339683548,133810670,1090789135,1078426020,1569222167,845107691,3583754449,4072456591,1091646820,628848692,1613405280,3757631651,526609435,236106946,48312990,2942717905,3402727701,1797494240,859738849,992217954,4005476642,2243076622,3870952857,3732016268,765654824,3490871365,2511836413,1685915746,3888969200,1414112111,2273134842,3281911079,4080962846,172450625,2569994100,980381355,4109958455,2819808352,2716589560,2568741196,3681446669,3329971472,1835478071,660984891,3704678404,4045999559,3422617507,3040415634,1762651403,1719377915,3470491036,2693910283,3642056355,3138596744,1364962596,2073328063,1983633131,926494387,3423689081,2150032023,4096667949,1749200295,3328846651,309677260,2016342300,1779581495,3079819751,111262694,1274766160,443224088,298511866,1025883608,3806446537,1145181785,168956806,3641502830,3584813610,1689216846,3666258015,3200248200,1692713982,2646376535,4042768518,1618508792,1610833997,3523052358,4130873264,2001055236,3610705100,2202168115,4028541809,2961195399,1006657119,2006996926,3186142756,1430667929,3210227297,1314452623,4074634658,4101304120,2273951170,1399257539,3367210612,3027628629,1190975929,2062231137,2333990788,2221543033,2438960610,1181637006,548689776,2362791313,3372408396,3104550113,3145860560,296247880,1970579870,3078560182,3769228297,1714227617,3291629107,3898220290,166772364,1251581989,493813264,448347421,195405023,2709975567,677966185,3703036547,1463355134,2715995803,1338867538,1343315457,2802222074,2684532164,233230375,2599980071,2000651841,3277868038,1638401717,4028070440,3237316320,6314154,819756386,300326615,590932579,1405279636,3267499572,3150704214,2428286686,3959192993,3461946742,1862657033,1266418056,963775037,2089974820,2263052895,1917689273,448879540,3550394620,3981727096,150775221,3627908307,1303187396,508620638,2975983352,2726630617,1817252668,1876281319,1457606340,908771278,3720792119,3617206836,2455994898,1729034894,1080033504],[976866871,3556439503,2881648439,1522871579,1555064734,1336096578,3548522304,2579274686,3574697629,3205460757,3593280638,3338716283,3079412587,564236357,2993598910,1781952180,1464380207,3163844217,3332601554,1699332808,1393555694,1183702653,3581086237,1288719814,691649499,2847557200,2895455976,3193889540,2717570544,1781354906,1676643554,2592534050,3230253752,1126444790,2770207658,2633158820,2210423226,2615765581,2414155088,3127139286,673620729,2805611233,1269405062,4015350505,3341807571,4149409754,1057255273,2012875353,2162469141,2276492801,2601117357,993977747,3918593370,2654263191,753973209,36408145,2530585658,25011837,3520020182,2088578344,530523599,2918365339,1524020338,1518925132,3760827505,3759777254,1202760957,3985898139,3906192525,674977740,4174734889,2031300136,2019492241,3983892565,4153806404,3822280332,352677332,2297720250,60907813,90501309,3286998549,1016092578,2535922412,2839152426,457141659,509813237,4120667899,652014361,1966332200,2975202805,55981186,2327461051,676427537,3255491064,2882294119,3433927263,1307055953,942726286,933058658,2468411793,3933900994,4215176142,1361170020,2001714738,2830558078,3274259782,1222529897,1679025792,2729314320,3714953764,1770335741,151462246,3013232138,1682292957,1483529935,471910574,1539241949,458788160,3436315007,1807016891,3718408830,978976581,1043663428,3165965781,1927990952,4200891579,2372276910,3208408903,3533431907,1412390302,2931980059,4132332400,1947078029,3881505623,4168226417,2941484381,1077988104,1320477388,886195818,18198404,3786409e3,2509781533,112762804,3463356488,1866414978,891333506,18488651,661792760,1628790961,3885187036,3141171499,876946877,2693282273,1372485963,791857591,2686433993,3759982718,3167212022,3472953795,2716379847,445679433,3561995674,3504004811,3574258232,54117162,3331405415,2381918588,3769707343,4154350007,1140177722,4074052095,668550556,3214352940,367459370,261225585,2610173221,4209349473,3468074219,3265815641,314222801,3066103646,3808782860,282218597,3406013506,3773591054,379116347,1285071038,846784868,2669647154,3771962079,3550491691,2305946142,453669953,1268987020,3317592352,3279303384,3744833421,2610507566,3859509063,266596637,3847019092,517658769,3462560207,3443424879,370717030,4247526661,2224018117,4143653529,4112773975,2788324899,2477274417,1456262402,2901442914,1517677493,1846949527,2295493580,3734397586,2176403920,1280348187,1908823572,3871786941,846861322,1172426758,3287448474,3383383037,1655181056,3139813346,901632758,1897031941,2986607138,3066810236,3447102507,1393639104,373351379,950779232,625454576,3124240540,4148612726,2007998917,544563296,2244738638,2330496472,2058025392,1291430526,424198748,50039436,29584100,3605783033,2429876329,2791104160,1057563949,3255363231,3075367218,3463963227,1469046755,985887462]],w={pbox:[],sbox:[]};function F(r,i){var o=i>>24&255,u=i>>16&255,p=i>>8&255,g=255&i,y=r.sbox[0][o]+r.sbox[1][u];return y^=r.sbox[2][p],y+=r.sbox[3][g]}function BlowFish_Encrypt(r,i,o){for(var u,g=i,y=o,w=0;w<p;++w)u=g^=r.pbox[w],g=y=F(r,g)^y,y=u;return u=g,g=y,y=u,y^=r.pbox[p],{left:g^=r.pbox[p+1],right:y}}function BlowFish_Decrypt(r,i,o){for(var u,g=i,y=o,w=p+1;w>1;--w)u=g^=r.pbox[w],g=y=F(r,g)^y,y=u;return u=g,g=y,y=u,y^=r.pbox[1],{left:g^=r.pbox[0],right:y}}function BlowFishInit(r,i,o){for(var u=0;u<4;u++){r.sbox[u]=[];for(var w=0;w<256;w++)r.sbox[u][w]=y[u][w]}for(var _=0,x=0;x<p+2;x++)r.pbox[x]=g[x]^i[_],++_>=o&&(_=0);for(var k=0,P=0,O=0,I=0;I<p+2;I+=2)k=(O=BlowFish_Encrypt(r,k,P)).left,P=O.right,r.pbox[I]=k,r.pbox[I+1]=P;for(var B=0;B<4;B++)for(var D=0;D<256;D+=2)k=(O=BlowFish_Encrypt(r,k,P)).left,P=O.right,r.sbox[B][D]=k,r.sbox[B][D+1]=P;return!0}var _=o.Blowfish=i.extend({_doReset:function _doReset(){if(this._keyPriorReset!==this._key){var r=this._keyPriorReset=this._key,i=r.words,o=r.sigBytes/4;BlowFishInit(w,i,o)}},encryptBlock:function encryptBlock(r,i){var o=BlowFish_Encrypt(w,r[i],r[i+1]);r[i]=o.left,r[i+1]=o.right},decryptBlock:function decryptBlock(r,i){var o=BlowFish_Decrypt(w,r[i],r[i+1]);r[i]=o.left,r[i+1]=o.right},blockSize:2,keySize:4,ivSize:2});r.Blowfish=i._createHelper(_)}(),u.Blowfish)},28395:function(r,i,o){"use strict";var u;o(39081),o(20731),o(23913),o(14032),o(57114),r.exports=(u=o(34559),o(36572),void(u.lib.Cipher||function(r){var i=u,o=i.lib,p=o.Base,g=o.WordArray,y=o.BufferedBlockAlgorithm,w=i.enc,_=(w.Utf8,w.Base64),x=i.algo.EvpKDF,k=o.Cipher=y.extend({cfg:p.extend(),createEncryptor:function createEncryptor(r,i){return this.create(this._ENC_XFORM_MODE,r,i)},createDecryptor:function createDecryptor(r,i){return this.create(this._DEC_XFORM_MODE,r,i)},init:function init(r,i,o){this.cfg=this.cfg.extend(o),this._xformMode=r,this._key=i,this.reset()},reset:function reset(){y.reset.call(this),this._doReset()},process:function process(r){return this._append(r),this._process()},finalize:function finalize(r){return r&&this._append(r),this._doFinalize()},keySize:4,ivSize:4,_ENC_XFORM_MODE:1,_DEC_XFORM_MODE:2,_createHelper:function(){function selectCipherStrategy(r){return"string"==typeof r?U:N}return function(r){return{encrypt:function encrypt(i,o,u){return selectCipherStrategy(o).encrypt(r,i,o,u)},decrypt:function decrypt(i,o,u){return selectCipherStrategy(o).decrypt(r,i,o,u)}}}}()}),P=(o.StreamCipher=k.extend({_doFinalize:function _doFinalize(){return this._process(!0)},blockSize:1}),i.mode={}),O=o.BlockCipherMode=p.extend({createEncryptor:function createEncryptor(r,i){return this.Encryptor.create(r,i)},createDecryptor:function createDecryptor(r,i){return this.Decryptor.create(r,i)},init:function init(r,i){this._cipher=r,this._iv=i}}),I=P.CBC=function(){var i=O.extend();function xorBlock(i,o,u){var p,g=this._iv;g?(p=g,this._iv=r):p=this._prevBlock;for(var y=0;y<u;y++)i[o+y]^=p[y]}return i.Encryptor=i.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize;xorBlock.call(this,r,i,u),o.encryptBlock(r,i),this._prevBlock=r.slice(i,i+u)}}),i.Decryptor=i.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=r.slice(i,i+u);o.decryptBlock(r,i),xorBlock.call(this,r,i,u),this._prevBlock=p}}),i}(),B=(i.pad={}).Pkcs7={pad:function pad(r,i){for(var o=4*i,u=o-r.sigBytes%o,p=u<<24|u<<16|u<<8|u,y=[],w=0;w<u;w+=4)y.push(p);var _=g.create(y,u);r.concat(_)},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},D=(o.BlockCipher=k.extend({cfg:k.cfg.extend({mode:I,padding:B}),reset:function reset(){var r;k.reset.call(this);var i=this.cfg,o=i.iv,u=i.mode;this._xformMode==this._ENC_XFORM_MODE?r=u.createEncryptor:(r=u.createDecryptor,this._minBufferSize=1),this._mode&&this._mode.__creator==r?this._mode.init(this,o&&o.words):(this._mode=r.call(u,this,o&&o.words),this._mode.__creator=r)},_doProcessBlock:function _doProcessBlock(r,i){this._mode.processBlock(r,i)},_doFinalize:function _doFinalize(){var r,i=this.cfg.padding;return this._xformMode==this._ENC_XFORM_MODE?(i.pad(this._data,this.blockSize),r=this._process(!0)):(r=this._process(!0),i.unpad(r)),r},blockSize:4}),o.CipherParams=p.extend({init:function init(r){this.mixIn(r)},toString:function toString(r){return(r||this.formatter).stringify(this)}})),R=(i.format={}).OpenSSL={stringify:function stringify(r){var i=r.ciphertext,o=r.salt;return(o?g.create([1398893684,1701076831]).concat(o).concat(i):i).toString(_)},parse:function parse(r){var i,o=_.parse(r),u=o.words;return 1398893684==u[0]&&1701076831==u[1]&&(i=g.create(u.slice(2,4)),u.splice(0,4),o.sigBytes-=16),D.create({ciphertext:o,salt:i})}},N=o.SerializableCipher=p.extend({cfg:p.extend({format:R}),encrypt:function encrypt(r,i,o,u){u=this.cfg.extend(u);var p=r.createEncryptor(o,u),g=p.finalize(i),y=p.cfg;return D.create({ciphertext:g,key:o,iv:y.iv,algorithm:r,mode:y.mode,padding:y.padding,blockSize:r.blockSize,formatter:u.format})},decrypt:function decrypt(r,i,o,u){return u=this.cfg.extend(u),i=this._parse(i,u.format),r.createDecryptor(o,u).finalize(i.ciphertext)},_parse:function _parse(r,i){return"string"==typeof r?i.parse(r,this):r}}),W=(i.kdf={}).OpenSSL={execute:function execute(r,i,o,u,p){if(u||(u=g.random(8)),p)y=x.create({keySize:i+o,hasher:p}).compute(r,u);else var y=x.create({keySize:i+o}).compute(r,u);var w=g.create(y.words.slice(i),4*o);return y.sigBytes=4*i,D.create({key:y,iv:w,salt:u})}},U=o.PasswordBasedCipher=N.extend({cfg:N.cfg.extend({kdf:W}),encrypt:function encrypt(r,i,o,u){var p=(u=this.cfg.extend(u)).kdf.execute(o,r.keySize,r.ivSize,u.salt,u.hasher);u.iv=p.iv;var g=N.encrypt.call(this,r,i,p.key,u);return g.mixIn(p),g},decrypt:function decrypt(r,i,o,u){u=this.cfg.extend(u),i=this._parse(i,u.format);var p=u.kdf.execute(o,r.keySize,r.ivSize,i.salt,u.hasher);return u.iv=p.iv,N.decrypt.call(this,r,i,p.key,u)}})}()))},34559:function(r,i,o){"use strict";var u;o(39081),o(81755),o(94845),o(20731),o(23913),o(14032),o(57114),o(59735),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529),o(49109),r.exports=(u=u||function(r,i){var u;if("undefined"!=typeof window&&window.crypto&&(u=window.crypto),"undefined"!=typeof self&&self.crypto&&(u=self.crypto),"undefined"!=typeof globalThis&&globalThis.crypto&&(u=globalThis.crypto),!u&&"undefined"!=typeof window&&window.msCrypto&&(u=window.msCrypto),!u&&void 0!==o.g&&o.g.crypto&&(u=o.g.crypto),!u)try{u=o(50477)}catch(r){}var p=function cryptoSecureRandomInt(){if(u){if("function"==typeof u.getRandomValues)try{return u.getRandomValues(new Uint32Array(1))[0]}catch(r){}if("function"==typeof u.randomBytes)try{return u.randomBytes(4).readInt32LE()}catch(r){}}throw new Error("Native crypto module could not be used to get secure random number.")},g=Object.create||function(){function F(){}return function(r){var i;return F.prototype=r,i=new F,F.prototype=null,i}}(),y={},w=y.lib={},_=w.Base={extend:function extend(r){var i=g(this);return r&&i.mixIn(r),i.hasOwnProperty("init")&&this.init!==i.init||(i.init=function(){i.$super.init.apply(this,arguments)}),i.init.prototype=i,i.$super=this,i},create:function create(){var r=this.extend();return r.init.apply(r,arguments),r},init:function init(){},mixIn:function mixIn(r){for(var i in r)r.hasOwnProperty(i)&&(this[i]=r[i]);r.hasOwnProperty("toString")&&(this.toString=r.toString)},clone:function clone(){return this.init.prototype.extend(this)}},x=w.WordArray=_.extend({init:function init(r,o){r=this.words=r||[],this.sigBytes=o!=i?o:4*r.length},toString:function toString(r){return(r||P).stringify(this)},concat:function concat(r){var i=this.words,o=r.words,u=this.sigBytes,p=r.sigBytes;if(this.clamp(),u%4)for(var g=0;g<p;g++){var y=o[g>>>2]>>>24-g%4*8&255;i[u+g>>>2]|=y<<24-(u+g)%4*8}else for(var w=0;w<p;w+=4)i[u+w>>>2]=o[w>>>2];return this.sigBytes+=p,this},clamp:function clamp(){var i=this.words,o=this.sigBytes;i[o>>>2]&=4294967295<<32-o%4*8,i.length=r.ceil(o/4)},clone:function clone(){var clone=_.clone.call(this);return clone.words=this.words.slice(0),clone},random:function random(r){for(var i=[],o=0;o<r;o+=4)i.push(p());return new x.init(i,r)}}),k=y.enc={},P=k.Hex={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push((g>>>4).toString(16)),u.push((15&g).toString(16))}return u.join("")},parse:function parse(r){for(var i=r.length,o=[],u=0;u<i;u+=2)o[u>>>3]|=parseInt(r.substr(u,2),16)<<24-u%8*4;return new x.init(o,i/2)}},O=k.Latin1={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var i=r.length,o=[],u=0;u<i;u++)o[u>>>2]|=(255&r.charCodeAt(u))<<24-u%4*8;return new x.init(o,i)}},I=k.Utf8={stringify:function stringify(r){try{return decodeURIComponent(escape(O.stringify(r)))}catch(r){throw new Error("Malformed UTF-8 data")}},parse:function parse(r){return O.parse(unescape(encodeURIComponent(r)))}},B=w.BufferedBlockAlgorithm=_.extend({reset:function reset(){this._data=new x.init,this._nDataBytes=0},_append:function _append(r){"string"==typeof r&&(r=I.parse(r)),this._data.concat(r),this._nDataBytes+=r.sigBytes},_process:function _process(i){var o,u=this._data,p=u.words,g=u.sigBytes,y=this.blockSize,w=g/(4*y),_=(w=i?r.ceil(w):r.max((0|w)-this._minBufferSize,0))*y,k=r.min(4*_,g);if(_){for(var P=0;P<_;P+=y)this._doProcessBlock(p,P);o=p.splice(0,_),u.sigBytes-=k}return new x.init(o,k)},clone:function clone(){var clone=_.clone.call(this);return clone._data=this._data.clone(),clone},_minBufferSize:0}),D=(w.Hasher=B.extend({cfg:_.extend(),init:function init(r){this.cfg=this.cfg.extend(r),this.reset()},reset:function reset(){B.reset.call(this),this._doReset()},update:function update(r){return this._append(r),this._process(),this},finalize:function finalize(r){return r&&this._append(r),this._doFinalize()},blockSize:16,_createHelper:function _createHelper(r){return function(i,o){return new r.init(o).finalize(i)}},_createHmacHelper:function _createHmacHelper(r){return function(i,o){return new D.HMAC.init(r,o).finalize(i)}}}),y.algo={});return y}(Math),u)},49300:function(r,i,o){"use strict";var u;o(94845),r.exports=(u=o(34559),function(){var r=u,i=r.lib.WordArray;function parseLoop(r,o,u){for(var p=[],g=0,y=0;y<o;y++)if(y%4){var w=u[r.charCodeAt(y-1)]<<y%4*2|u[r.charCodeAt(y)]>>>6-y%4*2;p[g>>>2]|=w<<24-g%4*8,g++}return i.create(p,g)}r.enc.Base64={stringify:function stringify(r){var i=r.words,o=r.sigBytes,u=this._map;r.clamp();for(var p=[],g=0;g<o;g+=3)for(var y=(i[g>>>2]>>>24-g%4*8&255)<<16|(i[g+1>>>2]>>>24-(g+1)%4*8&255)<<8|i[g+2>>>2]>>>24-(g+2)%4*8&255,w=0;w<4&&g+.75*w<o;w++)p.push(u.charAt(y>>>6*(3-w)&63));var _=u.charAt(64);if(_)for(;p.length%4;)p.push(_);return p.join("")},parse:function parse(r){var i=r.length,o=this._map,u=this._reverseMap;if(!u){u=this._reverseMap=[];for(var p=0;p<o.length;p++)u[o.charCodeAt(p)]=p}var g=o.charAt(64);if(g){var y=r.indexOf(g);-1!==y&&(i=y)}return parseLoop(r,i,u)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="}}(),u.enc.Base64)},375:function(r,i,o){"use strict";var u;o(94845),r.exports=(u=o(34559),function(){var r=u,i=r.lib.WordArray;function parseLoop(r,o,u){for(var p=[],g=0,y=0;y<o;y++)if(y%4){var w=u[r.charCodeAt(y-1)]<<y%4*2|u[r.charCodeAt(y)]>>>6-y%4*2;p[g>>>2]|=w<<24-g%4*8,g++}return i.create(p,g)}r.enc.Base64url={stringify:function stringify(r,i){void 0===i&&(i=!0);var o=r.words,u=r.sigBytes,p=i?this._safe_map:this._map;r.clamp();for(var g=[],y=0;y<u;y+=3)for(var w=(o[y>>>2]>>>24-y%4*8&255)<<16|(o[y+1>>>2]>>>24-(y+1)%4*8&255)<<8|o[y+2>>>2]>>>24-(y+2)%4*8&255,_=0;_<4&&y+.75*_<u;_++)g.push(p.charAt(w>>>6*(3-_)&63));var x=p.charAt(64);if(x)for(;g.length%4;)g.push(x);return g.join("")},parse:function parse(r,i){void 0===i&&(i=!0);var o=r.length,u=i?this._safe_map:this._map,p=this._reverseMap;if(!p){p=this._reverseMap=[];for(var g=0;g<u.length;g++)p[u.charCodeAt(g)]=g}var y=u.charAt(64);if(y){var w=r.indexOf(y);-1!==w&&(o=w)}return parseLoop(r,o,p)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",_safe_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"}}(),u.enc.Base64url)},56217:function(r,i,o){"use strict";var u;o(94845),r.exports=(u=o(34559),function(){var r=u,i=r.lib.WordArray,o=r.enc;function swapEndian(r){return r<<8&4278255360|r>>>8&16711935}o.Utf16=o.Utf16BE={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p+=2){var g=i[p>>>2]>>>16-p%4*8&65535;u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var o=r.length,u=[],p=0;p<o;p++)u[p>>>1]|=r.charCodeAt(p)<<16-p%2*16;return i.create(u,2*o)}},o.Utf16LE={stringify:function stringify(r){for(var i=r.words,o=r.sigBytes,u=[],p=0;p<o;p+=2){var g=swapEndian(i[p>>>2]>>>16-p%4*8&65535);u.push(String.fromCharCode(g))}return u.join("")},parse:function parse(r){for(var o=r.length,u=[],p=0;p<o;p++)u[p>>>1]|=swapEndian(r.charCodeAt(p)<<16-p%2*16);return i.create(u,2*o)}}}(),u.enc.Utf16)},36572:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;o(39081),r.exports=(k=o(34559),o(76289),o(30443),p=(u=k).lib,g=p.Base,y=p.WordArray,w=u.algo,_=w.MD5,x=w.EvpKDF=g.extend({cfg:g.extend({keySize:4,hasher:_,iterations:1}),init:function init(r){this.cfg=this.cfg.extend(r)},compute:function compute(r,i){for(var o,u=this.cfg,p=u.hasher.create(),g=y.create(),w=g.words,_=u.keySize,x=u.iterations;w.length<_;){o&&p.update(o),o=p.update(r).finalize(i),p.reset();for(var k=1;k<x;k++)o=p.finalize(o),p.reset();g.concat(o)}return g.sigBytes=4*_,g}}),u.EvpKDF=function(r,i,o){return x.create(o).compute(r,i)},k.EvpKDF)},68219:function(r,i,o){"use strict";var u,p,g,y;o(14032),o(57114),r.exports=(y=o(34559),o(28395),p=(u=y).lib.CipherParams,g=u.enc.Hex,u.format.Hex={stringify:function stringify(r){return r.ciphertext.toString(g)},parse:function parse(r){var i=g.parse(r);return p.create({ciphertext:i})}},y.format.Hex)},30443:function(r,i,o){"use strict";var u,p,g,y;o(39081),r.exports=(u=o(34559),g=(p=u).lib.Base,y=p.enc.Utf8,void(p.algo.HMAC=g.extend({init:function init(r,i){r=this._hasher=new r.init,"string"==typeof i&&(i=y.parse(i));var o=r.blockSize,u=4*o;i.sigBytes>u&&(i=r.finalize(i)),i.clamp();for(var p=this._oKey=i.clone(),g=this._iKey=i.clone(),w=p.words,_=g.words,x=0;x<o;x++)w[x]^=1549556828,_[x]^=909522486;p.sigBytes=g.sigBytes=u,this.reset()},reset:function reset(){var r=this._hasher;r.reset(),r.update(this._iKey)},update:function update(r){return this._hasher.update(r),this},finalize:function finalize(r){var i=this._hasher,o=i.finalize(r);return i.reset(),i.finalize(this._oKey.clone().concat(o))}})))},66122:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(26478),o(80986),o(56217),o(49300),o(375),o(72342),o(76289),o(26739),o(75230),o(97074),o(371),o(80767),o(76930),o(30443),o(95729),o(36572),o(28395),o(2727),o(59493),o(18494),o(5219),o(96112),o(51583),o(99215),o(43960),o(40681),o(16934),o(68219),o(6993),o(64390),o(76635),o(55380),o(9814),o(83122),u)},80986:function(r,i,o){"use strict";var u;o(81755),o(10720),o(14032),o(3131),o(90868),o(65553),o(75626),o(95756),o(56912),o(58099),o(47969),o(59735),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529),r.exports=(u=o(34559),function(){if("function"==typeof ArrayBuffer){var r=u.lib.WordArray,i=r.init,o=r.init=function(r){if(r instanceof ArrayBuffer&&(r=new Uint8Array(r)),(r instanceof Int8Array||"undefined"!=typeof Uint8ClampedArray&&r instanceof Uint8ClampedArray||r instanceof Int16Array||r instanceof Uint16Array||r instanceof Int32Array||r instanceof Uint32Array||r instanceof Float32Array||r instanceof Float64Array)&&(r=new Uint8Array(r.buffer,r.byteOffset,r.byteLength)),r instanceof Uint8Array){for(var o=r.byteLength,u=[],p=0;p<o;p++)u[p>>>2]|=r[p]<<24-p%4*8;i.call(this,u,o)}else i.apply(this,arguments)};o.prototype=r}}(),u.lib.WordArray)},72342:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[];!function(){for(var i=0;i<64;i++)w[i]=4294967296*r.abs(r.sin(i+1))|0}();var _=y.MD5=g.extend({_doReset:function _doReset(){this._hash=new p.init([1732584193,4023233417,2562383102,271733878])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=0;o<16;o++){var u=i+o,p=r[u];r[u]=16711935&(p<<8|p>>>24)|4278255360&(p<<24|p>>>8)}var g=this._hash.words,y=r[i+0],_=r[i+1],x=r[i+2],k=r[i+3],P=r[i+4],O=r[i+5],I=r[i+6],B=r[i+7],D=r[i+8],R=r[i+9],N=r[i+10],W=r[i+11],U=r[i+12],j=r[i+13],G=r[i+14],X=r[i+15],K=g[0],Y=g[1],J=g[2],$=g[3];K=FF(K,Y,J,$,y,7,w[0]),$=FF($,K,Y,J,_,12,w[1]),J=FF(J,$,K,Y,x,17,w[2]),Y=FF(Y,J,$,K,k,22,w[3]),K=FF(K,Y,J,$,P,7,w[4]),$=FF($,K,Y,J,O,12,w[5]),J=FF(J,$,K,Y,I,17,w[6]),Y=FF(Y,J,$,K,B,22,w[7]),K=FF(K,Y,J,$,D,7,w[8]),$=FF($,K,Y,J,R,12,w[9]),J=FF(J,$,K,Y,N,17,w[10]),Y=FF(Y,J,$,K,W,22,w[11]),K=FF(K,Y,J,$,U,7,w[12]),$=FF($,K,Y,J,j,12,w[13]),J=FF(J,$,K,Y,G,17,w[14]),K=GG(K,Y=FF(Y,J,$,K,X,22,w[15]),J,$,_,5,w[16]),$=GG($,K,Y,J,I,9,w[17]),J=GG(J,$,K,Y,W,14,w[18]),Y=GG(Y,J,$,K,y,20,w[19]),K=GG(K,Y,J,$,O,5,w[20]),$=GG($,K,Y,J,N,9,w[21]),J=GG(J,$,K,Y,X,14,w[22]),Y=GG(Y,J,$,K,P,20,w[23]),K=GG(K,Y,J,$,R,5,w[24]),$=GG($,K,Y,J,G,9,w[25]),J=GG(J,$,K,Y,k,14,w[26]),Y=GG(Y,J,$,K,D,20,w[27]),K=GG(K,Y,J,$,j,5,w[28]),$=GG($,K,Y,J,x,9,w[29]),J=GG(J,$,K,Y,B,14,w[30]),K=HH(K,Y=GG(Y,J,$,K,U,20,w[31]),J,$,O,4,w[32]),$=HH($,K,Y,J,D,11,w[33]),J=HH(J,$,K,Y,W,16,w[34]),Y=HH(Y,J,$,K,G,23,w[35]),K=HH(K,Y,J,$,_,4,w[36]),$=HH($,K,Y,J,P,11,w[37]),J=HH(J,$,K,Y,B,16,w[38]),Y=HH(Y,J,$,K,N,23,w[39]),K=HH(K,Y,J,$,j,4,w[40]),$=HH($,K,Y,J,y,11,w[41]),J=HH(J,$,K,Y,k,16,w[42]),Y=HH(Y,J,$,K,I,23,w[43]),K=HH(K,Y,J,$,R,4,w[44]),$=HH($,K,Y,J,U,11,w[45]),J=HH(J,$,K,Y,X,16,w[46]),K=II(K,Y=HH(Y,J,$,K,x,23,w[47]),J,$,y,6,w[48]),$=II($,K,Y,J,B,10,w[49]),J=II(J,$,K,Y,G,15,w[50]),Y=II(Y,J,$,K,O,21,w[51]),K=II(K,Y,J,$,U,6,w[52]),$=II($,K,Y,J,k,10,w[53]),J=II(J,$,K,Y,N,15,w[54]),Y=II(Y,J,$,K,_,21,w[55]),K=II(K,Y,J,$,D,6,w[56]),$=II($,K,Y,J,X,10,w[57]),J=II(J,$,K,Y,I,15,w[58]),Y=II(Y,J,$,K,j,21,w[59]),K=II(K,Y,J,$,P,6,w[60]),$=II($,K,Y,J,W,10,w[61]),J=II(J,$,K,Y,x,15,w[62]),Y=II(Y,J,$,K,R,21,w[63]),g[0]=g[0]+K|0,g[1]=g[1]+Y|0,g[2]=g[2]+J|0,g[3]=g[3]+$|0},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;o[p>>>5]|=128<<24-p%32;var g=r.floor(u/4294967296),y=u;o[15+(p+64>>>9<<4)]=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8),o[14+(p+64>>>9<<4)]=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),i.sigBytes=4*(o.length+1),this._process();for(var w=this._hash,_=w.words,x=0;x<4;x++){var k=_[x];_[x]=16711935&(k<<8|k>>>24)|4278255360&(k<<24|k>>>8)}return w},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});function FF(r,i,o,u,p,g,y){var w=r+(i&o|~i&u)+p+y;return(w<<g|w>>>32-g)+i}function GG(r,i,o,u,p,g,y){var w=r+(i&u|o&~u)+p+y;return(w<<g|w>>>32-g)+i}function HH(r,i,o,u,p,g,y){var w=r+(i^o^u)+p+y;return(w<<g|w>>>32-g)+i}function II(r,i,o,u,p,g,y){var w=r+(o^(i|~u))+p+y;return(w<<g|w>>>32-g)+i}i.MD5=g._createHelper(_),i.HmacMD5=g._createHmacHelper(_)}(Math),u.MD5)},2727:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),o(28395),u.mode.CFB=function(){var r=u.lib.BlockCipherMode.extend();function generateKeystreamAndEncrypt(r,i,o,u){var p,g=this._iv;g?(p=g.slice(0),this._iv=void 0):p=this._prevBlock,u.encryptBlock(p,0);for(var y=0;y<o;y++)r[i+y]^=p[y]}return r.Encryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize;generateKeystreamAndEncrypt.call(this,r,i,u,o),this._prevBlock=r.slice(i,i+u)}}),r.Decryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=r.slice(i,i+u);generateKeystreamAndEncrypt.call(this,r,i,u,o),this._prevBlock=p}}),r}(),u.mode.CFB)},18494:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),o(28395),u.mode.CTRGladman=function(){var r=u.lib.BlockCipherMode.extend();function incWord(r){if(255&~(r>>24))r+=1<<24;else{var i=r>>16&255,o=r>>8&255,u=255&r;255===i?(i=0,255===o?(o=0,255===u?u=0:++u):++o):++i,r=0,r+=i<<16,r+=o<<8,r+=u}return r}function incCounter(r){return 0===(r[0]=incWord(r[0]))&&(r[1]=incWord(r[1])),r}var i=r.Encryptor=r.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._counter;p&&(g=this._counter=p.slice(0),this._iv=void 0),incCounter(g);var y=g.slice(0);o.encryptBlock(y,0);for(var w=0;w<u;w++)r[i+w]^=y[w]}});return r.Decryptor=i,r}(),u.mode.CTRGladman)},59493:function(r,i,o){"use strict";var u,p,g;o(20731),r.exports=(g=o(34559),o(28395),g.mode.CTR=(u=g.lib.BlockCipherMode.extend(),p=u.Encryptor=u.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._counter;p&&(g=this._counter=p.slice(0),this._iv=void 0);var y=g.slice(0);o.encryptBlock(y,0),g[u-1]=g[u-1]+1|0;for(var w=0;w<u;w++)r[i+w]^=y[w]}}),u.Decryptor=p,u),g.mode.CTR)},96112:function(r,i,o){"use strict";var u,p;r.exports=(p=o(34559),o(28395),p.mode.ECB=((u=p.lib.BlockCipherMode.extend()).Encryptor=u.extend({processBlock:function processBlock(r,i){this._cipher.encryptBlock(r,i)}}),u.Decryptor=u.extend({processBlock:function processBlock(r,i){this._cipher.decryptBlock(r,i)}}),u),p.mode.ECB)},5219:function(r,i,o){"use strict";var u,p,g;o(20731),r.exports=(g=o(34559),o(28395),g.mode.OFB=(u=g.lib.BlockCipherMode.extend(),p=u.Encryptor=u.extend({processBlock:function processBlock(r,i){var o=this._cipher,u=o.blockSize,p=this._iv,g=this._keystream;p&&(g=this._keystream=p.slice(0),this._iv=void 0),o.encryptBlock(g,0);for(var y=0;y<u;y++)r[i+y]^=g[y]}}),u.Decryptor=p,u),g.mode.OFB)},51583:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(28395),u.pad.AnsiX923={pad:function pad(r,i){var o=r.sigBytes,u=4*i,p=u-o%u,g=o+p-1;r.clamp(),r.words[g>>>2]|=p<<24-g%4*8,r.sigBytes+=p},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},u.pad.Ansix923)},99215:function(r,i,o){"use strict";var u;o(39081),r.exports=(u=o(34559),o(28395),u.pad.Iso10126={pad:function pad(r,i){var o=4*i,p=o-r.sigBytes%o;r.concat(u.lib.WordArray.random(p-1)).concat(u.lib.WordArray.create([p<<24],1))},unpad:function unpad(r){var i=255&r.words[r.sigBytes-1>>>2];r.sigBytes-=i}},u.pad.Iso10126)},43960:function(r,i,o){"use strict";var u;o(39081),r.exports=(u=o(34559),o(28395),u.pad.Iso97971={pad:function pad(r,i){r.concat(u.lib.WordArray.create([2147483648],1)),u.pad.ZeroPadding.pad(r,i)},unpad:function unpad(r){u.pad.ZeroPadding.unpad(r),r.sigBytes--}},u.pad.Iso97971)},16934:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(28395),u.pad.NoPadding={pad:function pad(){},unpad:function unpad(){}},u.pad.NoPadding)},40681:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(28395),u.pad.ZeroPadding={pad:function pad(r,i){var o=4*i;r.clamp(),r.sigBytes+=o-(r.sigBytes%o||o)},unpad:function unpad(r){var i=r.words,o=r.sigBytes-1;for(o=r.sigBytes-1;o>=0;o--)if(i[o>>>2]>>>24-o%4*8&255){r.sigBytes=o+1;break}}},u.pad.ZeroPadding)},95729:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k,P;o(39081),r.exports=(P=o(34559),o(26739),o(30443),p=(u=P).lib,g=p.Base,y=p.WordArray,w=u.algo,_=w.SHA256,x=w.HMAC,k=w.PBKDF2=g.extend({cfg:g.extend({keySize:4,hasher:_,iterations:25e4}),init:function init(r){this.cfg=this.cfg.extend(r)},compute:function compute(r,i){for(var o=this.cfg,u=x.create(o.hasher,r),p=y.create(),g=y.create([1]),w=p.words,_=g.words,k=o.keySize,P=o.iterations;w.length<k;){var O=u.update(i).finalize(g);u.reset();for(var I=O.words,B=I.length,D=O,R=1;R<P;R++){D=u.finalize(D),u.reset();for(var N=D.words,W=0;W<B;W++)I[W]^=N[W]}p.concat(O),_[0]++}return p.sigBytes=4*k,p}}),u.PBKDF2=function(r,i,o){return k.create(o).compute(r,i)},P.PBKDF2)},9814:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=[],g=[],y=[],w=o.RabbitLegacy=i.extend({_doReset:function _doReset(){var r=this._key.words,i=this.cfg.iv,o=this._X=[r[0],r[3]<<16|r[2]>>>16,r[1],r[0]<<16|r[3]>>>16,r[2],r[1]<<16|r[0]>>>16,r[3],r[2]<<16|r[1]>>>16],u=this._C=[r[2]<<16|r[2]>>>16,4294901760&r[0]|65535&r[1],r[3]<<16|r[3]>>>16,4294901760&r[1]|65535&r[2],r[0]<<16|r[0]>>>16,4294901760&r[2]|65535&r[3],r[1]<<16|r[1]>>>16,4294901760&r[3]|65535&r[0]];this._b=0;for(var p=0;p<4;p++)nextState.call(this);for(p=0;p<8;p++)u[p]^=o[p+4&7];if(i){var g=i.words,y=g[0],w=g[1],_=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),x=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8),k=_>>>16|4294901760&x,P=x<<16|65535&_;for(u[0]^=_,u[1]^=k,u[2]^=x,u[3]^=P,u[4]^=_,u[5]^=k,u[6]^=x,u[7]^=P,p=0;p<4;p++)nextState.call(this)}},_doProcessBlock:function _doProcessBlock(r,i){var o=this._X;nextState.call(this),p[0]=o[0]^o[5]>>>16^o[3]<<16,p[1]=o[2]^o[7]>>>16^o[5]<<16,p[2]=o[4]^o[1]>>>16^o[7]<<16,p[3]=o[6]^o[3]>>>16^o[1]<<16;for(var u=0;u<4;u++)p[u]=16711935&(p[u]<<8|p[u]>>>24)|4278255360&(p[u]<<24|p[u]>>>8),r[i+u]^=p[u]},blockSize:4,ivSize:2});function nextState(){for(var r=this._X,i=this._C,o=0;o<8;o++)g[o]=i[o];for(i[0]=i[0]+1295307597+this._b|0,i[1]=i[1]+3545052371+(i[0]>>>0<g[0]>>>0?1:0)|0,i[2]=i[2]+886263092+(i[1]>>>0<g[1]>>>0?1:0)|0,i[3]=i[3]+1295307597+(i[2]>>>0<g[2]>>>0?1:0)|0,i[4]=i[4]+3545052371+(i[3]>>>0<g[3]>>>0?1:0)|0,i[5]=i[5]+886263092+(i[4]>>>0<g[4]>>>0?1:0)|0,i[6]=i[6]+1295307597+(i[5]>>>0<g[5]>>>0?1:0)|0,i[7]=i[7]+3545052371+(i[6]>>>0<g[6]>>>0?1:0)|0,this._b=i[7]>>>0<g[7]>>>0?1:0,o=0;o<8;o++){var u=r[o]+i[o],p=65535&u,w=u>>>16,_=((p*p>>>17)+p*w>>>15)+w*w,x=((4294901760&u)*u|0)+((65535&u)*u|0);y[o]=_^x}r[0]=y[0]+(y[7]<<16|y[7]>>>16)+(y[6]<<16|y[6]>>>16)|0,r[1]=y[1]+(y[0]<<8|y[0]>>>24)+y[7]|0,r[2]=y[2]+(y[1]<<16|y[1]>>>16)+(y[0]<<16|y[0]>>>16)|0,r[3]=y[3]+(y[2]<<8|y[2]>>>24)+y[1]|0,r[4]=y[4]+(y[3]<<16|y[3]>>>16)+(y[2]<<16|y[2]>>>16)|0,r[5]=y[5]+(y[4]<<8|y[4]>>>24)+y[3]|0,r[6]=y[6]+(y[5]<<16|y[5]>>>16)+(y[4]<<16|y[4]>>>16)|0,r[7]=y[7]+(y[6]<<8|y[6]>>>24)+y[5]|0}r.RabbitLegacy=i._createHelper(w)}(),u.RabbitLegacy)},55380:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=[],g=[],y=[],w=o.Rabbit=i.extend({_doReset:function _doReset(){for(var r=this._key.words,i=this.cfg.iv,o=0;o<4;o++)r[o]=16711935&(r[o]<<8|r[o]>>>24)|4278255360&(r[o]<<24|r[o]>>>8);var u=this._X=[r[0],r[3]<<16|r[2]>>>16,r[1],r[0]<<16|r[3]>>>16,r[2],r[1]<<16|r[0]>>>16,r[3],r[2]<<16|r[1]>>>16],p=this._C=[r[2]<<16|r[2]>>>16,4294901760&r[0]|65535&r[1],r[3]<<16|r[3]>>>16,4294901760&r[1]|65535&r[2],r[0]<<16|r[0]>>>16,4294901760&r[2]|65535&r[3],r[1]<<16|r[1]>>>16,4294901760&r[3]|65535&r[0]];for(this._b=0,o=0;o<4;o++)nextState.call(this);for(o=0;o<8;o++)p[o]^=u[o+4&7];if(i){var g=i.words,y=g[0],w=g[1],_=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),x=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8),k=_>>>16|4294901760&x,P=x<<16|65535&_;for(p[0]^=_,p[1]^=k,p[2]^=x,p[3]^=P,p[4]^=_,p[5]^=k,p[6]^=x,p[7]^=P,o=0;o<4;o++)nextState.call(this)}},_doProcessBlock:function _doProcessBlock(r,i){var o=this._X;nextState.call(this),p[0]=o[0]^o[5]>>>16^o[3]<<16,p[1]=o[2]^o[7]>>>16^o[5]<<16,p[2]=o[4]^o[1]>>>16^o[7]<<16,p[3]=o[6]^o[3]>>>16^o[1]<<16;for(var u=0;u<4;u++)p[u]=16711935&(p[u]<<8|p[u]>>>24)|4278255360&(p[u]<<24|p[u]>>>8),r[i+u]^=p[u]},blockSize:4,ivSize:2});function nextState(){for(var r=this._X,i=this._C,o=0;o<8;o++)g[o]=i[o];for(i[0]=i[0]+1295307597+this._b|0,i[1]=i[1]+3545052371+(i[0]>>>0<g[0]>>>0?1:0)|0,i[2]=i[2]+886263092+(i[1]>>>0<g[1]>>>0?1:0)|0,i[3]=i[3]+1295307597+(i[2]>>>0<g[2]>>>0?1:0)|0,i[4]=i[4]+3545052371+(i[3]>>>0<g[3]>>>0?1:0)|0,i[5]=i[5]+886263092+(i[4]>>>0<g[4]>>>0?1:0)|0,i[6]=i[6]+1295307597+(i[5]>>>0<g[5]>>>0?1:0)|0,i[7]=i[7]+3545052371+(i[6]>>>0<g[6]>>>0?1:0)|0,this._b=i[7]>>>0<g[7]>>>0?1:0,o=0;o<8;o++){var u=r[o]+i[o],p=65535&u,w=u>>>16,_=((p*p>>>17)+p*w>>>15)+w*w,x=((4294901760&u)*u|0)+((65535&u)*u|0);y[o]=_^x}r[0]=y[0]+(y[7]<<16|y[7]>>>16)+(y[6]<<16|y[6]>>>16)|0,r[1]=y[1]+(y[0]<<8|y[0]>>>24)+y[7]|0,r[2]=y[2]+(y[1]<<16|y[1]>>>16)+(y[0]<<16|y[0]>>>16)|0,r[3]=y[3]+(y[2]<<8|y[2]>>>24)+y[1]|0,r[4]=y[4]+(y[3]<<16|y[3]>>>16)+(y[2]<<16|y[2]>>>16)|0,r[5]=y[5]+(y[4]<<8|y[4]>>>24)+y[3]|0,r[6]=y[6]+(y[5]<<16|y[5]>>>16)+(y[4]<<16|y[4]>>>16)|0,r[7]=y[7]+(y[6]<<8|y[6]>>>24)+y[5]|0}r.Rabbit=i._createHelper(w)}(),u.Rabbit)},76635:function(r,i,o){"use strict";var u;o(14032),o(68067),r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib.StreamCipher,o=r.algo,p=o.RC4=i.extend({_doReset:function _doReset(){for(var r=this._key,i=r.words,o=r.sigBytes,u=this._S=[],p=0;p<256;p++)u[p]=p;p=0;for(var g=0;p<256;p++){var y=p%o,w=i[y>>>2]>>>24-y%4*8&255;g=(g+u[p]+w)%256;var _=u[p];u[p]=u[g],u[g]=_}this._i=this._j=0},_doProcessBlock:function _doProcessBlock(r,i){r[i]^=generateKeystreamWord.call(this)},keySize:8,ivSize:0});function generateKeystreamWord(){for(var r=this._S,i=this._i,o=this._j,u=0,p=0;p<4;p++){o=(o+r[i=(i+1)%256])%256;var g=r[i];r[i]=r[o],r[o]=g,u|=r[(r[i]+r[o])%256]<<24-8*p}return this._i=i,this._j=o,u}r.RC4=i._createHelper(p);var g=o.RC4Drop=p.extend({cfg:p.cfg.extend({drop:192}),_doReset:function _doReset(){p._doReset.call(this);for(var r=this.cfg.drop;r>0;r--)generateKeystreamWord.call(this)}});r.RC4Drop=i._createHelper(g)}(),u.RC4)},76930:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=p.create([0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,7,4,13,1,10,6,15,3,12,0,9,5,2,14,11,8,3,10,14,4,9,15,8,1,2,7,0,6,13,11,5,12,1,9,11,10,0,8,12,4,13,3,7,15,14,5,6,2,4,0,5,9,7,12,2,10,14,1,3,8,11,6,15,13]),_=p.create([5,14,7,0,9,2,11,4,13,6,15,8,1,10,3,12,6,11,3,7,0,13,5,10,14,15,8,12,4,9,1,2,15,5,1,3,7,14,6,9,11,8,12,2,10,0,4,13,8,6,4,1,3,11,15,0,5,12,2,13,9,7,10,14,12,15,10,4,1,5,8,7,6,2,13,14,0,3,9,11]),x=p.create([11,14,15,12,5,8,7,9,11,13,14,15,6,7,9,8,7,6,8,13,11,9,7,15,7,12,15,9,11,7,13,12,11,13,6,7,14,9,13,15,14,8,13,6,5,12,7,5,11,12,14,15,14,15,9,8,9,14,5,6,8,6,5,12,9,15,5,11,6,8,13,12,5,12,13,14,11,8,5,6]),k=p.create([8,9,9,11,13,15,15,5,7,7,8,11,14,14,12,6,9,13,15,7,12,8,9,11,7,7,12,7,6,15,13,11,9,7,15,11,8,6,6,14,12,13,5,14,13,13,7,5,15,5,8,11,14,14,6,14,6,9,12,9,12,5,15,8,8,5,12,9,12,5,14,6,8,13,6,5,15,13,11,11]),P=p.create([0,1518500249,1859775393,2400959708,2840853838]),O=p.create([1352829926,1548603684,1836072691,2053994217,0]),I=y.RIPEMD160=g.extend({_doReset:function _doReset(){this._hash=p.create([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=0;o<16;o++){var u=i+o,p=r[u];r[u]=16711935&(p<<8|p>>>24)|4278255360&(p<<24|p>>>8)}var g,y,I,B,D,R,N,W,U,j,G,X=this._hash.words,K=P.words,Y=O.words,J=w.words,$=_.words,ee=x.words,te=k.words;for(R=g=X[0],N=y=X[1],W=I=X[2],U=B=X[3],j=D=X[4],o=0;o<80;o+=1)G=g+r[i+J[o]]|0,G+=o<16?f1(y,I,B)+K[0]:o<32?f2(y,I,B)+K[1]:o<48?f3(y,I,B)+K[2]:o<64?f4(y,I,B)+K[3]:f5(y,I,B)+K[4],G=(G=rotl(G|=0,ee[o]))+D|0,g=D,D=B,B=rotl(I,10),I=y,y=G,G=R+r[i+$[o]]|0,G+=o<16?f5(N,W,U)+Y[0]:o<32?f4(N,W,U)+Y[1]:o<48?f3(N,W,U)+Y[2]:o<64?f2(N,W,U)+Y[3]:f1(N,W,U)+Y[4],G=(G=rotl(G|=0,te[o]))+j|0,R=j,j=U,U=rotl(W,10),W=N,N=G;G=X[1]+I+U|0,X[1]=X[2]+B+j|0,X[2]=X[3]+D+R|0,X[3]=X[4]+g+N|0,X[4]=X[0]+y+W|0,X[0]=G},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;i[u>>>5]|=128<<24-u%32,i[14+(u+64>>>9<<4)]=16711935&(o<<8|o>>>24)|4278255360&(o<<24|o>>>8),r.sigBytes=4*(i.length+1),this._process();for(var p=this._hash,g=p.words,y=0;y<5;y++){var w=g[y];g[y]=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8)}return p},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});function f1(r,i,o){return r^i^o}function f2(r,i,o){return r&i|~r&o}function f3(r,i,o){return(r|~i)^o}function f4(r,i,o){return r&o|i&~o}function f5(r,i,o){return r^(i|~o)}function rotl(r,i){return r<<i|r>>>32-i}i.RIPEMD160=g._createHelper(I),i.HmacRIPEMD160=g._createHmacHelper(I)}(Math),u.RIPEMD160)},76289:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;r.exports=(k=o(34559),p=(u=k).lib,g=p.WordArray,y=p.Hasher,w=u.algo,_=[],x=w.SHA1=y.extend({_doReset:function _doReset(){this._hash=new g.init([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],x=0;x<80;x++){if(x<16)_[x]=0|r[i+x];else{var k=_[x-3]^_[x-8]^_[x-14]^_[x-16];_[x]=k<<1|k>>>31}var P=(u<<5|u>>>27)+w+_[x];P+=x<20?1518500249+(p&g|~p&y):x<40?1859775393+(p^g^y):x<60?(p&g|p&y|g&y)-1894007588:(p^g^y)-899497514,w=y,y=g,g=p<<30|p>>>2,p=u,u=P}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;return i[u>>>5]|=128<<24-u%32,i[14+(u+64>>>9<<4)]=Math.floor(o/4294967296),i[15+(u+64>>>9<<4)]=o,r.sigBytes=4*i.length,this._process(),this._hash},clone:function clone(){var clone=y.clone.call(this);return clone._hash=this._hash.clone(),clone}}),u.SHA1=y._createHelper(x),u.HmacSHA1=y._createHmacHelper(x),k.SHA1)},75230:function(r,i,o){"use strict";var u,p,g,y,w,_;r.exports=(_=o(34559),o(26739),p=(u=_).lib.WordArray,g=u.algo,y=g.SHA256,w=g.SHA224=y.extend({_doReset:function _doReset(){this._hash=new p.init([3238371032,914150663,812702999,4144912697,4290775857,1750603025,1694076839,3204075428])},_doFinalize:function _doFinalize(){var r=y._doFinalize.call(this);return r.sigBytes-=4,r}}),u.SHA224=y._createHelper(w),u.HmacSHA224=y._createHmacHelper(w),_.SHA224)},26739:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[],_=[];!function(){function isPrime(i){for(var o=r.sqrt(i),u=2;u<=o;u++)if(!(i%u))return!1;return!0}function getFractionalBits(r){return 4294967296*(r-(0|r))|0}for(var i=2,o=0;o<64;)isPrime(i)&&(o<8&&(w[o]=getFractionalBits(r.pow(i,.5))),_[o]=getFractionalBits(r.pow(i,1/3)),o++),i++}();var x=[],k=y.SHA256=g.extend({_doReset:function _doReset(){this._hash=new p.init(w.slice(0))},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],k=o[5],P=o[6],O=o[7],I=0;I<64;I++){if(I<16)x[I]=0|r[i+I];else{var B=x[I-15],D=(B<<25|B>>>7)^(B<<14|B>>>18)^B>>>3,R=x[I-2],N=(R<<15|R>>>17)^(R<<13|R>>>19)^R>>>10;x[I]=D+x[I-7]+N+x[I-16]}var W=u&p^u&g^p&g,U=(u<<30|u>>>2)^(u<<19|u>>>13)^(u<<10|u>>>22),j=O+((w<<26|w>>>6)^(w<<21|w>>>11)^(w<<7|w>>>25))+(w&k^~w&P)+_[I]+x[I];O=P,P=k,k=w,w=y+j|0,y=g,g=p,p=u,u=j+(U+W)|0}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0,o[5]=o[5]+k|0,o[6]=o[6]+P|0,o[7]=o[7]+O|0},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;return o[p>>>5]|=128<<24-p%32,o[14+(p+64>>>9<<4)]=r.floor(u/4294967296),o[15+(p+64>>>9<<4)]=u,i.sigBytes=4*o.length,this._process(),this._hash},clone:function clone(){var clone=g.clone.call(this);return clone._hash=this._hash.clone(),clone}});i.SHA256=g._createHelper(k),i.HmacSHA256=g._createHmacHelper(k)}(Math),u.SHA256)},80767:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),o(26478),function(r){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.x64.Word,w=i.algo,_=[],x=[],k=[];!function(){for(var r=1,i=0,o=0;o<24;o++){_[r+5*i]=(o+1)*(o+2)/2%64;var u=(2*r+3*i)%5;r=i%5,i=u}for(r=0;r<5;r++)for(i=0;i<5;i++)x[r+5*i]=i+(2*r+3*i)%5*5;for(var p=1,g=0;g<24;g++){for(var w=0,P=0,O=0;O<7;O++){if(1&p){var I=(1<<O)-1;I<32?P^=1<<I:w^=1<<I-32}128&p?p=p<<1^113:p<<=1}k[g]=y.create(w,P)}}();var P=[];!function(){for(var r=0;r<25;r++)P[r]=y.create()}();var O=w.SHA3=g.extend({cfg:g.cfg.extend({outputLength:512}),_doReset:function _doReset(){for(var r=this._state=[],i=0;i<25;i++)r[i]=new y.init;this.blockSize=(1600-2*this.cfg.outputLength)/32},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._state,u=this.blockSize/2,p=0;p<u;p++){var g=r[i+2*p],y=r[i+2*p+1];g=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8),y=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),(te=o[p]).high^=y,te.low^=g}for(var w=0;w<24;w++){for(var O=0;O<5;O++){for(var I=0,B=0,D=0;D<5;D++)I^=(te=o[O+5*D]).high,B^=te.low;var R=P[O];R.high=I,R.low=B}for(O=0;O<5;O++){var N=P[(O+4)%5],W=P[(O+1)%5],U=W.high,j=W.low;for(I=N.high^(U<<1|j>>>31),B=N.low^(j<<1|U>>>31),D=0;D<5;D++)(te=o[O+5*D]).high^=I,te.low^=B}for(var G=1;G<25;G++){var X=(te=o[G]).high,K=te.low,Y=_[G];Y<32?(I=X<<Y|K>>>32-Y,B=K<<Y|X>>>32-Y):(I=K<<Y-32|X>>>64-Y,B=X<<Y-32|K>>>64-Y);var J=P[x[G]];J.high=I,J.low=B}var $=P[0],ee=o[0];for($.high=ee.high,$.low=ee.low,O=0;O<5;O++)for(D=0;D<5;D++){var te=o[G=O+5*D],re=P[G],ne=P[(O+1)%5+5*D],ie=P[(O+2)%5+5*D];te.high=re.high^~ne.high&ie.high,te.low=re.low^~ne.low&ie.low}te=o[0];var oe=k[w];te.high^=oe.high,te.low^=oe.low}},_doFinalize:function _doFinalize(){var i=this._data,o=i.words,u=(this._nDataBytes,8*i.sigBytes),g=32*this.blockSize;o[u>>>5]|=1<<24-u%32,o[(r.ceil((u+1)/g)*g>>>5)-1]|=128,i.sigBytes=4*o.length,this._process();for(var y=this._state,w=this.cfg.outputLength/8,_=w/8,x=[],k=0;k<_;k++){var P=y[k],O=P.high,I=P.low;O=16711935&(O<<8|O>>>24)|4278255360&(O<<24|O>>>8),I=16711935&(I<<8|I>>>24)|4278255360&(I<<24|I>>>8),x.push(I),x.push(O)}return new p.init(x,w)},clone:function clone(){for(var clone=g.clone.call(this),r=clone._state=this._state.slice(0),i=0;i<25;i++)r[i]=r[i].clone();return clone}});i.SHA3=g._createHelper(O),i.HmacSHA3=g._createHmacHelper(O)}(Math),u.SHA3)},371:function(r,i,o){"use strict";var u,p,g,y,w,_,x,k;r.exports=(k=o(34559),o(26478),o(97074),p=(u=k).x64,g=p.Word,y=p.WordArray,w=u.algo,_=w.SHA512,x=w.SHA384=_.extend({_doReset:function _doReset(){this._hash=new y.init([new g.init(3418070365,3238371032),new g.init(1654270250,914150663),new g.init(2438529370,812702999),new g.init(355462360,4144912697),new g.init(1731405415,4290775857),new g.init(2394180231,1750603025),new g.init(3675008525,1694076839),new g.init(1203062813,3204075428)])},_doFinalize:function _doFinalize(){var r=_._doFinalize.call(this);return r.sigBytes-=16,r}}),u.SHA384=_._createHelper(x),u.HmacSHA384=_._createHmacHelper(x),k.SHA384)},97074:function(r,i,o){"use strict";var u;r.exports=(u=o(34559),o(26478),function(){var r=u,i=r.lib.Hasher,o=r.x64,p=o.Word,g=o.WordArray,y=r.algo;function X64Word_create(){return p.create.apply(p,arguments)}var w=[X64Word_create(1116352408,3609767458),X64Word_create(1899447441,602891725),X64Word_create(3049323471,3964484399),X64Word_create(3921009573,2173295548),X64Word_create(961987163,4081628472),X64Word_create(1508970993,3053834265),X64Word_create(2453635748,2937671579),X64Word_create(2870763221,3664609560),X64Word_create(3624381080,2734883394),X64Word_create(310598401,1164996542),X64Word_create(607225278,1323610764),X64Word_create(1426881987,3590304994),X64Word_create(1925078388,4068182383),X64Word_create(2162078206,991336113),X64Word_create(2614888103,633803317),X64Word_create(3248222580,3479774868),X64Word_create(3835390401,2666613458),X64Word_create(4022224774,944711139),X64Word_create(264347078,2341262773),X64Word_create(604807628,2007800933),X64Word_create(770255983,1495990901),X64Word_create(1249150122,1856431235),X64Word_create(1555081692,3175218132),X64Word_create(1996064986,2198950837),X64Word_create(2554220882,3999719339),X64Word_create(2821834349,766784016),X64Word_create(2952996808,2566594879),X64Word_create(3210313671,3203337956),X64Word_create(3336571891,1034457026),X64Word_create(3584528711,2466948901),X64Word_create(113926993,3758326383),X64Word_create(338241895,168717936),X64Word_create(666307205,1188179964),X64Word_create(773529912,1546045734),X64Word_create(1294757372,1522805485),X64Word_create(1396182291,2643833823),X64Word_create(1695183700,2343527390),X64Word_create(1986661051,1014477480),X64Word_create(2177026350,1206759142),X64Word_create(2456956037,344077627),X64Word_create(2730485921,1290863460),X64Word_create(2820302411,3158454273),X64Word_create(3259730800,3505952657),X64Word_create(3345764771,106217008),X64Word_create(3516065817,3606008344),X64Word_create(3600352804,1432725776),X64Word_create(4094571909,1467031594),X64Word_create(275423344,851169720),X64Word_create(430227734,3100823752),X64Word_create(506948616,1363258195),X64Word_create(659060556,3750685593),X64Word_create(883997877,3785050280),X64Word_create(958139571,3318307427),X64Word_create(1322822218,3812723403),X64Word_create(1537002063,2003034995),X64Word_create(1747873779,3602036899),X64Word_create(1955562222,1575990012),X64Word_create(2024104815,1125592928),X64Word_create(2227730452,2716904306),X64Word_create(2361852424,442776044),X64Word_create(2428436474,593698344),X64Word_create(2756734187,3733110249),X64Word_create(3204031479,2999351573),X64Word_create(3329325298,3815920427),X64Word_create(3391569614,3928383900),X64Word_create(3515267271,566280711),X64Word_create(3940187606,3454069534),X64Word_create(4118630271,4000239992),X64Word_create(116418474,1914138554),X64Word_create(174292421,2731055270),X64Word_create(289380356,3203993006),X64Word_create(460393269,320620315),X64Word_create(685471733,587496836),X64Word_create(852142971,1086792851),X64Word_create(1017036298,365543100),X64Word_create(1126000580,2618297676),X64Word_create(1288033470,3409855158),X64Word_create(1501505948,4234509866),X64Word_create(1607167915,987167468),X64Word_create(1816402316,1246189591)],_=[];!function(){for(var r=0;r<80;r++)_[r]=X64Word_create()}();var x=y.SHA512=i.extend({_doReset:function _doReset(){this._hash=new g.init([new p.init(1779033703,4089235720),new p.init(3144134277,2227873595),new p.init(1013904242,4271175723),new p.init(2773480762,1595750129),new p.init(1359893119,2917565137),new p.init(2600822924,725511199),new p.init(528734635,4215389547),new p.init(1541459225,327033209)])},_doProcessBlock:function _doProcessBlock(r,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],x=o[4],k=o[5],P=o[6],O=o[7],I=u.high,B=u.low,D=p.high,R=p.low,N=g.high,W=g.low,U=y.high,j=y.low,G=x.high,X=x.low,K=k.high,Y=k.low,J=P.high,$=P.low,ee=O.high,te=O.low,re=I,ne=B,ie=D,oe=R,ae=N,se=W,ce=U,le=j,ue=G,fe=X,he=K,de=Y,pe=J,ge=$,ye=ee,ve=te,me=0;me<80;me++){var be,we,_e=_[me];if(me<16)we=_e.high=0|r[i+2*me],be=_e.low=0|r[i+2*me+1];else{var Se=_[me-15],xe=Se.high,Ce=Se.low,ke=(xe>>>1|Ce<<31)^(xe>>>8|Ce<<24)^xe>>>7,Pe=(Ce>>>1|xe<<31)^(Ce>>>8|xe<<24)^(Ce>>>7|xe<<25),Ae=_[me-2],Ee=Ae.high,Te=Ae.low,Oe=(Ee>>>19|Te<<13)^(Ee<<3|Te>>>29)^Ee>>>6,Ie=(Te>>>19|Ee<<13)^(Te<<3|Ee>>>29)^(Te>>>6|Ee<<26),Be=_[me-7],Fe=Be.high,Le=Be.low,De=_[me-16],Re=De.high,Me=De.low;we=(we=(we=ke+Fe+((be=Pe+Le)>>>0<Pe>>>0?1:0))+Oe+((be+=Ie)>>>0<Ie>>>0?1:0))+Re+((be+=Me)>>>0<Me>>>0?1:0),_e.high=we,_e.low=be}var Ne,We=ue&he^~ue&pe,ze=fe&de^~fe&ge,Ue=re&ie^re&ae^ie&ae,je=ne&oe^ne&se^oe&se,Ge=(re>>>28|ne<<4)^(re<<30|ne>>>2)^(re<<25|ne>>>7),Ve=(ne>>>28|re<<4)^(ne<<30|re>>>2)^(ne<<25|re>>>7),He=(ue>>>14|fe<<18)^(ue>>>18|fe<<14)^(ue<<23|fe>>>9),qe=(fe>>>14|ue<<18)^(fe>>>18|ue<<14)^(fe<<23|ue>>>9),Xe=w[me],Ze=Xe.high,Ke=Xe.low,Ye=ye+He+((Ne=ve+qe)>>>0<ve>>>0?1:0),Je=Ve+je;ye=pe,ve=ge,pe=he,ge=de,he=ue,de=fe,ue=ce+(Ye=(Ye=(Ye=Ye+We+((Ne+=ze)>>>0<ze>>>0?1:0))+Ze+((Ne+=Ke)>>>0<Ke>>>0?1:0))+we+((Ne+=be)>>>0<be>>>0?1:0))+((fe=le+Ne|0)>>>0<le>>>0?1:0)|0,ce=ae,le=se,ae=ie,se=oe,ie=re,oe=ne,re=Ye+(Ge+Ue+(Je>>>0<Ve>>>0?1:0))+((ne=Ne+Je|0)>>>0<Ne>>>0?1:0)|0}B=u.low=B+ne,u.high=I+re+(B>>>0<ne>>>0?1:0),R=p.low=R+oe,p.high=D+ie+(R>>>0<oe>>>0?1:0),W=g.low=W+se,g.high=N+ae+(W>>>0<se>>>0?1:0),j=y.low=j+le,y.high=U+ce+(j>>>0<le>>>0?1:0),X=x.low=X+fe,x.high=G+ue+(X>>>0<fe>>>0?1:0),Y=k.low=Y+de,k.high=K+he+(Y>>>0<de>>>0?1:0),$=P.low=$+ge,P.high=J+pe+($>>>0<ge>>>0?1:0),te=O.low=te+ve,O.high=ee+ye+(te>>>0<ve>>>0?1:0)},_doFinalize:function _doFinalize(){var r=this._data,i=r.words,o=8*this._nDataBytes,u=8*r.sigBytes;return i[u>>>5]|=128<<24-u%32,i[30+(u+128>>>10<<5)]=Math.floor(o/4294967296),i[31+(u+128>>>10<<5)]=o,r.sigBytes=4*i.length,this._process(),this._hash.toX32()},clone:function clone(){var clone=i.clone.call(this);return clone._hash=this._hash.clone(),clone},blockSize:32});r.SHA512=i._createHelper(x),r.HmacSHA512=i._createHmacHelper(x)}(),u.SHA512)},64390:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),o(49300),o(72342),o(36572),o(28395),function(){var r=u,i=r.lib,o=i.WordArray,p=i.BlockCipher,g=r.algo,y=[57,49,41,33,25,17,9,1,58,50,42,34,26,18,10,2,59,51,43,35,27,19,11,3,60,52,44,36,63,55,47,39,31,23,15,7,62,54,46,38,30,22,14,6,61,53,45,37,29,21,13,5,28,20,12,4],w=[14,17,11,24,1,5,3,28,15,6,21,10,23,19,12,4,26,8,16,7,27,20,13,2,41,52,31,37,47,55,30,40,51,45,33,48,44,49,39,56,34,53,46,42,50,36,29,32],_=[1,2,4,6,8,10,12,14,15,17,19,21,23,25,27,28],x=[{0:8421888,268435456:32768,536870912:8421378,805306368:2,1073741824:512,1342177280:8421890,1610612736:8389122,1879048192:8388608,2147483648:514,2415919104:8389120,2684354560:33280,2952790016:8421376,3221225472:32770,3489660928:8388610,3758096384:0,4026531840:33282,134217728:0,402653184:8421890,671088640:33282,939524096:32768,1207959552:8421888,1476395008:512,1744830464:8421378,2013265920:2,2281701376:8389120,2550136832:33280,2818572288:8421376,3087007744:8389122,3355443200:8388610,3623878656:32770,3892314112:514,4160749568:8388608,1:32768,268435457:2,536870913:8421888,805306369:8388608,1073741825:8421378,1342177281:33280,1610612737:512,1879048193:8389122,2147483649:8421890,2415919105:8421376,2684354561:8388610,2952790017:33282,3221225473:514,3489660929:8389120,3758096385:32770,4026531841:0,134217729:8421890,402653185:8421376,671088641:8388608,939524097:512,1207959553:32768,1476395009:8388610,1744830465:2,2013265921:33282,2281701377:32770,2550136833:8389122,2818572289:514,3087007745:8421888,3355443201:8389120,3623878657:0,3892314113:33280,4160749569:8421378},{0:1074282512,16777216:16384,33554432:524288,50331648:1074266128,67108864:1073741840,83886080:1074282496,100663296:1073758208,117440512:16,134217728:540672,150994944:1073758224,167772160:1073741824,184549376:540688,201326592:524304,218103808:0,234881024:16400,251658240:1074266112,8388608:1073758208,25165824:540688,41943040:16,58720256:1073758224,75497472:1074282512,92274688:1073741824,109051904:524288,125829120:1074266128,142606336:524304,159383552:0,176160768:16384,192937984:1074266112,209715200:1073741840,226492416:540672,243269632:1074282496,260046848:16400,268435456:0,285212672:1074266128,301989888:1073758224,318767104:1074282496,335544320:1074266112,352321536:16,369098752:540688,385875968:16384,402653184:16400,419430400:524288,436207616:524304,452984832:1073741840,469762048:540672,486539264:1073758208,503316480:1073741824,520093696:1074282512,276824064:540688,293601280:524288,310378496:1074266112,327155712:16384,343932928:1073758208,360710144:1074282512,377487360:16,394264576:1073741824,411041792:1074282496,427819008:1073741840,444596224:1073758224,461373440:524304,478150656:0,494927872:16400,511705088:1074266128,528482304:540672},{0:260,1048576:0,2097152:67109120,3145728:65796,4194304:65540,5242880:67108868,6291456:67174660,7340032:67174400,8388608:67108864,9437184:67174656,10485760:65792,11534336:67174404,12582912:67109124,13631488:65536,14680064:4,15728640:256,524288:67174656,1572864:67174404,2621440:0,3670016:67109120,4718592:67108868,5767168:65536,6815744:65540,7864320:260,8912896:4,9961472:256,11010048:67174400,12058624:65796,13107200:65792,14155776:67109124,15204352:67174660,16252928:67108864,16777216:67174656,17825792:65540,18874368:65536,19922944:67109120,20971520:256,22020096:67174660,23068672:67108868,24117248:0,25165824:67109124,26214400:67108864,27262976:4,28311552:65792,29360128:67174400,30408704:260,31457280:65796,32505856:67174404,17301504:67108864,18350080:260,19398656:67174656,20447232:0,21495808:65540,22544384:67109120,23592960:256,24641536:67174404,25690112:65536,26738688:67174660,27787264:65796,28835840:67108868,29884416:67109124,30932992:67174400,31981568:4,33030144:65792},{0:2151682048,65536:2147487808,131072:4198464,196608:2151677952,262144:0,327680:4198400,393216:2147483712,458752:4194368,524288:2147483648,589824:4194304,655360:64,720896:2147487744,786432:2151678016,851968:4160,917504:4096,983040:2151682112,32768:2147487808,98304:64,163840:2151678016,229376:2147487744,294912:4198400,360448:2151682112,425984:0,491520:2151677952,557056:4096,622592:2151682048,688128:4194304,753664:4160,819200:2147483648,884736:4194368,950272:4198464,1015808:2147483712,1048576:4194368,1114112:4198400,1179648:2147483712,1245184:0,1310720:4160,1376256:2151678016,1441792:2151682048,1507328:2147487808,1572864:2151682112,1638400:2147483648,1703936:2151677952,1769472:4198464,1835008:2147487744,1900544:4194304,1966080:64,2031616:4096,1081344:2151677952,1146880:2151682112,1212416:0,1277952:4198400,1343488:4194368,1409024:2147483648,1474560:2147487808,1540096:64,1605632:2147483712,1671168:4096,1736704:2147487744,1802240:2151678016,1867776:4160,1933312:2151682048,1998848:4194304,2064384:4198464},{0:128,4096:17039360,8192:262144,12288:536870912,16384:537133184,20480:16777344,24576:553648256,28672:262272,32768:16777216,36864:537133056,40960:536871040,45056:553910400,49152:553910272,53248:0,57344:17039488,61440:553648128,2048:17039488,6144:553648256,10240:128,14336:17039360,18432:262144,22528:537133184,26624:553910272,30720:536870912,34816:537133056,38912:0,43008:553910400,47104:16777344,51200:536871040,55296:553648128,59392:16777216,63488:262272,65536:262144,69632:128,73728:536870912,77824:553648256,81920:16777344,86016:553910272,90112:537133184,94208:16777216,98304:553910400,102400:553648128,106496:17039360,110592:537133056,114688:262272,118784:536871040,122880:0,126976:17039488,67584:553648256,71680:16777216,75776:17039360,79872:537133184,83968:536870912,88064:17039488,92160:128,96256:553910272,100352:262272,104448:553910400,108544:0,112640:553648128,116736:16777344,120832:262144,124928:537133056,129024:536871040},{0:268435464,256:8192,512:270532608,768:270540808,1024:268443648,1280:2097152,1536:2097160,1792:268435456,2048:0,2304:268443656,2560:2105344,2816:8,3072:270532616,3328:2105352,3584:8200,3840:270540800,128:270532608,384:270540808,640:8,896:2097152,1152:2105352,1408:268435464,1664:268443648,1920:8200,2176:2097160,2432:8192,2688:268443656,2944:270532616,3200:0,3456:270540800,3712:2105344,3968:268435456,4096:268443648,4352:270532616,4608:270540808,4864:8200,5120:2097152,5376:268435456,5632:268435464,5888:2105344,6144:2105352,6400:0,6656:8,6912:270532608,7168:8192,7424:268443656,7680:270540800,7936:2097160,4224:8,4480:2105344,4736:2097152,4992:268435464,5248:268443648,5504:8200,5760:270540808,6016:270532608,6272:270540800,6528:270532616,6784:8192,7040:2105352,7296:2097160,7552:0,7808:268435456,8064:268443656},{0:1048576,16:33555457,32:1024,48:1049601,64:34604033,80:0,96:1,112:34603009,128:33555456,144:1048577,160:33554433,176:34604032,192:34603008,208:1025,224:1049600,240:33554432,8:34603009,24:0,40:33555457,56:34604032,72:1048576,88:33554433,104:33554432,120:1025,136:1049601,152:33555456,168:34603008,184:1048577,200:1024,216:34604033,232:1,248:1049600,256:33554432,272:1048576,288:33555457,304:34603009,320:1048577,336:33555456,352:34604032,368:1049601,384:1025,400:34604033,416:1049600,432:1,448:0,464:34603008,480:33554433,496:1024,264:1049600,280:33555457,296:34603009,312:1,328:33554432,344:1048576,360:1025,376:34604032,392:33554433,408:34603008,424:0,440:34604033,456:1049601,472:1024,488:33555456,504:1048577},{0:134219808,1:131072,2:134217728,3:32,4:131104,5:134350880,6:134350848,7:2048,8:134348800,9:134219776,10:133120,11:134348832,12:2080,13:0,14:134217760,15:133152,2147483648:2048,2147483649:134350880,2147483650:134219808,2147483651:134217728,2147483652:134348800,2147483653:133120,2147483654:133152,2147483655:32,2147483656:134217760,2147483657:2080,2147483658:131104,2147483659:134350848,2147483660:0,2147483661:134348832,2147483662:134219776,2147483663:131072,16:133152,17:134350848,18:32,19:2048,20:134219776,21:134217760,22:134348832,23:131072,24:0,25:131104,26:134348800,27:134219808,28:134350880,29:133120,30:2080,31:134217728,2147483664:131072,2147483665:2048,2147483666:134348832,2147483667:133152,2147483668:32,2147483669:134348800,2147483670:134217728,2147483671:134219808,2147483672:134350880,2147483673:134217760,2147483674:134219776,2147483675:0,2147483676:133120,2147483677:2080,2147483678:131104,2147483679:134350848}],k=[4160749569,528482304,33030144,2064384,129024,8064,504,2147483679],P=g.DES=p.extend({_doReset:function _doReset(){for(var r=this._key.words,i=[],o=0;o<56;o++){var u=y[o]-1;i[o]=r[u>>>5]>>>31-u%32&1}for(var p=this._subKeys=[],g=0;g<16;g++){var x=p[g]=[],k=_[g];for(o=0;o<24;o++)x[o/6|0]|=i[(w[o]-1+k)%28]<<31-o%6,x[4+(o/6|0)]|=i[28+(w[o+24]-1+k)%28]<<31-o%6;for(x[0]=x[0]<<1|x[0]>>>31,o=1;o<7;o++)x[o]=x[o]>>>4*(o-1)+3;x[7]=x[7]<<5|x[7]>>>27}var P=this._invSubKeys=[];for(o=0;o<16;o++)P[o]=p[15-o]},encryptBlock:function encryptBlock(r,i){this._doCryptBlock(r,i,this._subKeys)},decryptBlock:function decryptBlock(r,i){this._doCryptBlock(r,i,this._invSubKeys)},_doCryptBlock:function _doCryptBlock(r,i,o){this._lBlock=r[i],this._rBlock=r[i+1],exchangeLR.call(this,4,252645135),exchangeLR.call(this,16,65535),exchangeRL.call(this,2,858993459),exchangeRL.call(this,8,16711935),exchangeLR.call(this,1,1431655765);for(var u=0;u<16;u++){for(var p=o[u],g=this._lBlock,y=this._rBlock,w=0,_=0;_<8;_++)w|=x[_][((y^p[_])&k[_])>>>0];this._lBlock=y,this._rBlock=g^w}var P=this._lBlock;this._lBlock=this._rBlock,this._rBlock=P,exchangeLR.call(this,1,1431655765),exchangeRL.call(this,8,16711935),exchangeRL.call(this,2,858993459),exchangeLR.call(this,16,65535),exchangeLR.call(this,4,252645135),r[i]=this._lBlock,r[i+1]=this._rBlock},keySize:2,ivSize:2,blockSize:2});function exchangeLR(r,i){var o=(this._lBlock>>>r^this._rBlock)&i;this._rBlock^=o,this._lBlock^=o<<r}function exchangeRL(r,i){var o=(this._rBlock>>>r^this._lBlock)&i;this._lBlock^=o,this._rBlock^=o<<r}r.DES=p._createHelper(P);var O=g.TripleDES=p.extend({_doReset:function _doReset(){var r=this._key.words;if(2!==r.length&&4!==r.length&&r.length<6)throw new Error("Invalid key length - 3DES requires the key length to be 64, 128, 192 or >192.");var i=r.slice(0,2),u=r.length<4?r.slice(0,2):r.slice(2,4),p=r.length<6?r.slice(0,2):r.slice(4,6);this._des1=P.createEncryptor(o.create(i)),this._des2=P.createEncryptor(o.create(u)),this._des3=P.createEncryptor(o.create(p))},encryptBlock:function encryptBlock(r,i){this._des1.encryptBlock(r,i),this._des2.decryptBlock(r,i),this._des3.encryptBlock(r,i)},decryptBlock:function decryptBlock(r,i){this._des3.decryptBlock(r,i),this._des2.encryptBlock(r,i),this._des1.decryptBlock(r,i)},keySize:6,ivSize:2,blockSize:2});r.TripleDES=p._createHelper(O)}(),u.TripleDES)},26478:function(r,i,o){"use strict";var u;o(20731),r.exports=(u=o(34559),function(r){var i=u,o=i.lib,p=o.Base,g=o.WordArray,y=i.x64={};y.Word=p.extend({init:function init(r,i){this.high=r,this.low=i}}),y.WordArray=p.extend({init:function init(i,o){i=this.words=i||[],this.sigBytes=o!=r?o:8*i.length},toX32:function toX32(){for(var r=this.words,i=r.length,o=[],u=0;u<i;u++){var p=r[u];o.push(p.high),o.push(p.low)}return g.create(o,this.sigBytes)},clone:function clone(){for(var clone=p.clone.call(this),r=clone.words=this.words.slice(0),i=r.length,o=0;o<i;o++)r[o]=r[o].clone();return clone}})}(),u)},91836:function(r,i,o){"use strict";function _createForOfIteratorHelperLoose(r,i){var o="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(o)return(o=o.call(r)).next.bind(o);if(Array.isArray(r)||(o=function _unsupportedIterableToArray(r,i){if(!r)return;if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);"Object"===o&&r.constructor&&(o=r.constructor.name);if("Map"===o||"Set"===o)return Array.from(r);if("Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o))return _arrayLikeToArray(r,i)}(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0;return function(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}function _regeneratorRuntime(){_regeneratorRuntime=function _regeneratorRuntime(){return i};var r,i={},o=Object.prototype,u=o.hasOwnProperty,p=Object.defineProperty||function(r,i,o){r[i]=o.value},g="function"==typeof Symbol?Symbol:{},y=g.iterator||"@@iterator",w=g.asyncIterator||"@@asyncIterator",_=g.toStringTag||"@@toStringTag";function define(r,i,o){return Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}),r[i]}try{define({},"")}catch(r){define=function define(r,i,o){return r[i]=o}}function wrap(r,i,o,u){var g=i&&i.prototype instanceof Generator?i:Generator,y=Object.create(g.prototype),w=new Context(u||[]);return p(y,"_invoke",{value:makeInvokeMethod(r,o,w)}),y}function tryCatch(r,i,o){try{return{type:"normal",arg:r.call(i,o)}}catch(r){return{type:"throw",arg:r}}}i.wrap=wrap;var x="suspendedStart",k="suspendedYield",P="executing",O="completed",I={};function Generator(){}function GeneratorFunction(){}function GeneratorFunctionPrototype(){}var B={};define(B,y,(function(){return this}));var D=Object.getPrototypeOf,R=D&&D(D(values([])));R&&R!==o&&u.call(R,y)&&(B=R);var N=GeneratorFunctionPrototype.prototype=Generator.prototype=Object.create(B);function defineIteratorMethods(r){["next","throw","return"].forEach((function(i){define(r,i,(function(r){return this._invoke(i,r)}))}))}function AsyncIterator(r,i){function invoke(o,p,g,y){var w=tryCatch(r[o],r,p);if("throw"!==w.type){var _=w.arg,x=_.value;return x&&"object"==typeof x&&u.call(x,"__await")?i.resolve(x.__await).then((function(r){invoke("next",r,g,y)}),(function(r){invoke("throw",r,g,y)})):i.resolve(x).then((function(r){_.value=r,g(_)}),(function(r){return invoke("throw",r,g,y)}))}y(w.arg)}var o;p(this,"_invoke",{value:function value(r,u){function callInvokeWithMethodAndArg(){return new i((function(i,o){invoke(r,u,i,o)}))}return o=o?o.then(callInvokeWithMethodAndArg,callInvokeWithMethodAndArg):callInvokeWithMethodAndArg()}})}function makeInvokeMethod(i,o,u){var p=x;return function(g,y){if(p===P)throw new Error("Generator is already running");if(p===O){if("throw"===g)throw y;return{value:r,done:!0}}for(u.method=g,u.arg=y;;){var w=u.delegate;if(w){var _=maybeInvokeDelegate(w,u);if(_){if(_===I)continue;return _}}if("next"===u.method)u.sent=u._sent=u.arg;else if("throw"===u.method){if(p===x)throw p=O,u.arg;u.dispatchException(u.arg)}else"return"===u.method&&u.abrupt("return",u.arg);p=P;var B=tryCatch(i,o,u);if("normal"===B.type){if(p=u.done?O:k,B.arg===I)continue;return{value:B.arg,done:u.done}}"throw"===B.type&&(p=O,u.method="throw",u.arg=B.arg)}}}function maybeInvokeDelegate(i,o){var u=o.method,p=i.iterator[u];if(p===r)return o.delegate=null,"throw"===u&&i.iterator.return&&(o.method="return",o.arg=r,maybeInvokeDelegate(i,o),"throw"===o.method)||"return"!==u&&(o.method="throw",o.arg=new TypeError("The iterator does not provide a '"+u+"' method")),I;var g=tryCatch(p,i.iterator,o.arg);if("throw"===g.type)return o.method="throw",o.arg=g.arg,o.delegate=null,I;var y=g.arg;return y?y.done?(o[i.resultName]=y.value,o.next=i.nextLoc,"return"!==o.method&&(o.method="next",o.arg=r),o.delegate=null,I):y:(o.method="throw",o.arg=new TypeError("iterator result is not an object"),o.delegate=null,I)}function pushTryEntry(r){var i={tryLoc:r[0]};1 in r&&(i.catchLoc=r[1]),2 in r&&(i.finallyLoc=r[2],i.afterLoc=r[3]),this.tryEntries.push(i)}function resetTryEntry(r){var i=r.completion||{};i.type="normal",delete i.arg,r.completion=i}function Context(r){this.tryEntries=[{tryLoc:"root"}],r.forEach(pushTryEntry,this),this.reset(!0)}function values(i){if(i||""===i){var o=i[y];if(o)return o.call(i);if("function"==typeof i.next)return i;if(!isNaN(i.length)){var p=-1,g=function next(){for(;++p<i.length;)if(u.call(i,p))return next.value=i[p],next.done=!1,next;return next.value=r,next.done=!0,next};return g.next=g}}throw new TypeError(typeof i+" is not iterable")}return GeneratorFunction.prototype=GeneratorFunctionPrototype,p(N,"constructor",{value:GeneratorFunctionPrototype,configurable:!0}),p(GeneratorFunctionPrototype,"constructor",{value:GeneratorFunction,configurable:!0}),GeneratorFunction.displayName=define(GeneratorFunctionPrototype,_,"GeneratorFunction"),i.isGeneratorFunction=function(r){var i="function"==typeof r&&r.constructor;return!!i&&(i===GeneratorFunction||"GeneratorFunction"===(i.displayName||i.name))},i.mark=function(r){return Object.setPrototypeOf?Object.setPrototypeOf(r,GeneratorFunctionPrototype):(r.__proto__=GeneratorFunctionPrototype,define(r,_,"GeneratorFunction")),r.prototype=Object.create(N),r},i.awrap=function(r){return{__await:r}},defineIteratorMethods(AsyncIterator.prototype),define(AsyncIterator.prototype,w,(function(){return this})),i.AsyncIterator=AsyncIterator,i.async=function(r,o,u,p,g){void 0===g&&(g=Promise);var y=new AsyncIterator(wrap(r,o,u,p),g);return i.isGeneratorFunction(o)?y:y.next().then((function(r){return r.done?r.value:y.next()}))},defineIteratorMethods(N),define(N,_,"Generator"),define(N,y,(function(){return this})),define(N,"toString",(function(){return"[object Generator]"})),i.keys=function(r){var i=Object(r),o=[];for(var u in i)o.push(u);return o.reverse(),function next(){for(;o.length;){var r=o.pop();if(r in i)return next.value=r,next.done=!1,next}return next.done=!0,next}},i.values=values,Context.prototype={constructor:Context,reset:function reset(i){if(this.prev=0,this.next=0,this.sent=this._sent=r,this.done=!1,this.delegate=null,this.method="next",this.arg=r,this.tryEntries.forEach(resetTryEntry),!i)for(var o in this)"t"===o.charAt(0)&&u.call(this,o)&&!isNaN(+o.slice(1))&&(this[o]=r)},stop:function stop(){this.done=!0;var r=this.tryEntries[0].completion;if("throw"===r.type)throw r.arg;return this.rval},dispatchException:function dispatchException(i){if(this.done)throw i;var o=this;function handle(u,p){return y.type="throw",y.arg=i,o.next=u,p&&(o.method="next",o.arg=r),!!p}for(var p=this.tryEntries.length-1;p>=0;--p){var g=this.tryEntries[p],y=g.completion;if("root"===g.tryLoc)return handle("end");if(g.tryLoc<=this.prev){var w=u.call(g,"catchLoc"),_=u.call(g,"finallyLoc");if(w&&_){if(this.prev<g.catchLoc)return handle(g.catchLoc,!0);if(this.prev<g.finallyLoc)return handle(g.finallyLoc)}else if(w){if(this.prev<g.catchLoc)return handle(g.catchLoc,!0)}else{if(!_)throw new Error("try statement without catch or finally");if(this.prev<g.finallyLoc)return handle(g.finallyLoc)}}}},abrupt:function abrupt(r,i){for(var o=this.tryEntries.length-1;o>=0;--o){var p=this.tryEntries[o];if(p.tryLoc<=this.prev&&u.call(p,"finallyLoc")&&this.prev<p.finallyLoc){var g=p;break}}g&&("break"===r||"continue"===r)&&g.tryLoc<=i&&i<=g.finallyLoc&&(g=null);var y=g?g.completion:{};return y.type=r,y.arg=i,g?(this.method="next",this.next=g.finallyLoc,I):this.complete(y)},complete:function complete(r,i){if("throw"===r.type)throw r.arg;return"break"===r.type||"continue"===r.type?this.next=r.arg:"return"===r.type?(this.rval=this.arg=r.arg,this.method="return",this.next="end"):"normal"===r.type&&i&&(this.next=i),I},finish:function finish(r){for(var i=this.tryEntries.length-1;i>=0;--i){var o=this.tryEntries[i];if(o.finallyLoc===r)return this.complete(o.completion,o.afterLoc),resetTryEntry(o),I}},catch:function _catch(r){for(var i=this.tryEntries.length-1;i>=0;--i){var o=this.tryEntries[i];if(o.tryLoc===r){var u=o.completion;if("throw"===u.type){var p=u.arg;resetTryEntry(o)}return p}}throw new Error("illegal catch attempt")},delegateYield:function delegateYield(i,o,u){return this.delegate={iterator:values(i),resultName:o,nextLoc:u},"next"===this.method&&(this.arg=r),I}},i}o(35877),o(38178),o(11765),o(24863),o(43448),o(63956),o(71950),o(68067),o(57114),o(42437),o(65292),o(73844),o(69330),o(81755),o(20731),o(14032),o(61726),o(58281),o(6422),o(94712);var u=function(){function StateMachine(r){this.stateTable=r.stateTable,this.accepting=r.accepting,this.tags=r.tags}var r=StateMachine.prototype;return r.match=function match(r){var i,o=this;return(i={})[Symbol.iterator]=_regeneratorRuntime().mark((function _callee(){var i,u,p,g,y,w;return _regeneratorRuntime().wrap((function _callee$(_){for(;;)switch(_.prev=_.next){case 0:i=1,u=null,p=null,g=null,y=0;case 5:if(!(y<r.length)){_.next=21;break}if(w=r[y],g=i,0!==(i=o.stateTable[i][w])){_.next=15;break}if(!(null!=u&&null!=p&&p>=u)){_.next=13;break}return _.next=13,[u,p,o.tags[g]];case 13:i=o.stateTable[1][w],u=null;case 15:0!==i&&null==u&&(u=y),o.accepting[i]&&(p=y),0===i&&(i=1);case 18:y++,_.next=5;break;case 21:if(!(null!=u&&null!=p&&p>=u)){_.next=24;break}return _.next=24,[u,p,o.tags[i]];case 24:case"end":return _.stop()}}),_callee)})),i},r.apply=function apply(r,i){for(var o,u=_createForOfIteratorHelperLoose(this.match(r));!(o=u()).done;)for(var p,g=o.value,y=g[0],w=g[1],_=_createForOfIteratorHelperLoose(g[2]);!(p=_()).done;){var x=p.value;"function"==typeof i[x]&&i[x](y,w,r.slice(y,w+1))}},StateMachine}();r.exports=u},80646:function(r,i,o){"use strict";var u=o(50621).Buffer;o(58028),o(20731),o(14032),o(68067);var p=o(48181),g=o(6729);r.exports=function(){function PNG(r){var i;for(this.data=r,this.pos=8,this.palette=[],this.imgData=[],this.transparency={},this.text={};;){var o=this.readUInt32(),p="";for(i=0;i<4;i++)p+=String.fromCharCode(this.data[this.pos++]);switch(p){case"IHDR":this.width=this.readUInt32(),this.height=this.readUInt32(),this.bits=this.data[this.pos++],this.colorType=this.data[this.pos++],this.compressionMethod=this.data[this.pos++],this.filterMethod=this.data[this.pos++],this.interlaceMethod=this.data[this.pos++];break;case"PLTE":this.palette=this.read(o);break;case"IDAT":for(i=0;i<o;i++)this.imgData.push(this.data[this.pos++]);break;case"tRNS":switch(this.transparency={},this.colorType){case 3:this.transparency.indexed=this.read(o);var g=255-this.transparency.indexed.length;if(g>0)for(i=0;i<g;i++)this.transparency.indexed.push(255);break;case 0:this.transparency.grayscale=this.read(o)[0];break;case 2:this.transparency.rgb=this.read(o)}break;case"tEXt":var y=this.read(o),w=y.indexOf(0),_=String.fromCharCode.apply(String,y.slice(0,w));this.text[_]=String.fromCharCode.apply(String,y.slice(w+1));break;case"IEND":switch(this.colorType){case 0:case 3:case 4:this.colors=1;break;case 2:case 6:this.colors=3}this.hasAlphaChannel=[4,6].includes(this.colorType);var x=this.colors+(this.hasAlphaChannel?1:0);switch(this.pixelBitlength=this.bits*x,this.colors){case 1:this.colorSpace="DeviceGray";break;case 3:this.colorSpace="DeviceRGB"}return void(this.imgData=new u(this.imgData));default:this.pos+=o}if(this.pos+=4,this.pos>this.data.length)throw new Error("Incomplete or corrupt PNG file")}}PNG.decode=function decode(r,i){return p.readFile(r,(function(r,o){return new PNG(o).decode((function(r){return i(r)}))}))},PNG.load=function load(r){return new PNG(p.readFileSync(r))};var r=PNG.prototype;return r.read=function read(r){for(var i=new Array(r),o=0;o<r;o++)i[o]=this.data[this.pos++];return i},r.readUInt32=function readUInt32(){return this.data[this.pos++]<<24|this.data[this.pos++]<<16|this.data[this.pos++]<<8|this.data[this.pos++]},r.readUInt16=function readUInt16(){return this.data[this.pos++]<<8|this.data[this.pos++]},r.decodePixels=function decodePixels(r){var i=this;return g.inflate(this.imgData,(function(o,p){if(o)throw o;var g=i.width,y=i.height,w=i.pixelBitlength/8,_=new u(g*y*w),x=p.length,k=0;function pass(r,i,o,P,O){void 0===O&&(O=!1);for(var I=Math.ceil((g-r)/o),B=Math.ceil((y-i)/P),D=w*I,R=O?_:new u(D*B),N=0,W=0;N<B&&k<x;){var U,j,G,X,K;switch(p[k++]){case 0:for(G=0;G<D;G++)R[W++]=p[k++];break;case 1:for(G=0;G<D;G++)U=p[k++],X=G<w?0:R[W-w],R[W++]=(U+X)%256;break;case 2:for(G=0;G<D;G++)U=p[k++],j=(G-G%w)/w,K=N&&R[(N-1)*D+j*w+G%w],R[W++]=(K+U)%256;break;case 3:for(G=0;G<D;G++)U=p[k++],j=(G-G%w)/w,X=G<w?0:R[W-w],K=N&&R[(N-1)*D+j*w+G%w],R[W++]=(U+Math.floor((X+K)/2))%256;break;case 4:for(G=0;G<D;G++){var Y,J;U=p[k++],j=(G-G%w)/w,X=G<w?0:R[W-w],0===N?K=J=0:(K=R[(N-1)*D+j*w+G%w],J=j&&R[(N-1)*D+(j-1)*w+G%w]);var $=X+K-J,ee=Math.abs($-X),te=Math.abs($-K),re=Math.abs($-J);Y=ee<=te&&ee<=re?X:te<=re?K:J,R[W++]=(U+Y)%256}break;default:throw new Error("Invalid filter algorithm: "+p[k-1])}if(!O){var ne=((i+N*P)*g+r)*w,ie=N*D;for(G=0;G<I;G++){for(var oe=0;oe<w;oe++)_[ne++]=R[ie++];ne+=(o-1)*w}}N++}}return 1===i.interlaceMethod?(pass(0,0,8,8),pass(4,0,8,8),pass(0,4,4,8),pass(2,0,4,4),pass(0,2,2,4),pass(1,0,2,2),pass(0,1,1,2)):pass(0,0,1,1,!0),r(_)}))},r.decodePalette=function decodePalette(){for(var r=this.palette,i=r.length,o=this.transparency.indexed||[],p=new u(o.length+i),g=0,y=0,w=0;w<i;w+=3){var _;p[g++]=r[w],p[g++]=r[w+1],p[g++]=r[w+2],p[g++]=null!=(_=o[y++])?_:255}return p},r.copyToImageData=function copyToImageData(r,i){var o,u,p=this.colors,g=null,y=this.hasAlphaChannel;this.palette.length&&(g=this._decodedPalette||(this._decodedPalette=this.decodePalette()),p=4,y=!0);var w=r.data||r,_=w.length,x=g||i,k=o=0;if(1===p)for(;k<_;){u=g?4*i[k/4]:o;var P=x[u++];w[k++]=P,w[k++]=P,w[k++]=P,w[k++]=y?x[u++]:255,o=u}else for(;k<_;)u=g?4*i[k/4]:o,w[k++]=x[u++],w[k++]=x[u++],w[k++]=x[u++],w[k++]=y?x[u++]:255,o=u},r.decode=function decode(r){var i=this,o=new u(this.width*this.height*4);return this.decodePixels((function(u){return i.copyToImageData(o,u),r(o)}))},PNG}()},16696:function(r,i,o){"use strict";var u=o(9964);o(41584),o(81755),o(14032),o(56912),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(83326),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529);var p,g=o(50621),y=g.Buffer,w={};for(p in g)g.hasOwnProperty(p)&&"SlowBuffer"!==p&&"Buffer"!==p&&(w[p]=g[p]);var _=w.Buffer={};for(p in y)y.hasOwnProperty(p)&&"allocUnsafe"!==p&&"allocUnsafeSlow"!==p&&(_[p]=y[p]);if(w.Buffer.prototype=y.prototype,_.from&&_.from!==Uint8Array.from||(_.from=function(r,i,o){if("number"==typeof r)throw new TypeError('The "value" argument must not be of type number. Received type '+typeof r);if(r&&void 0===r.length)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r);return y(r,i,o)}),_.alloc||(_.alloc=function(r,i,o){if("number"!=typeof r)throw new TypeError('The "size" argument must be of type number. Received type '+typeof r);if(r<0||r>=2*(1<<30))throw new RangeError('The value "'+r+'" is invalid for option "size"');var u=y(r);return i&&0!==i.length?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u}),!w.kStringMaxLength)try{w.kStringMaxLength=u.binding("buffer").kStringMaxLength}catch(r){}w.constants||(w.constants={MAX_LENGTH:w.kMaxLength},w.kStringMaxLength&&(w.constants.MAX_STRING_LENGTH=w.kStringMaxLength)),r.exports=w},57540:function(r,i,o){"use strict";function ownKeys(r,i){var o=Object.keys(r);if(Object.getOwnPropertySymbols){var u=Object.getOwnPropertySymbols(r);i&&(u=u.filter((function(i){return Object.getOwnPropertyDescriptor(r,i).enumerable}))),o.push.apply(o,u)}return o}function _objectSpread(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{};i%2?ownKeys(Object(o),!0).forEach((function(i){_defineProperty(r,i,o[i])})):Object.getOwnPropertyDescriptors?Object.defineProperties(r,Object.getOwnPropertyDescriptors(o)):ownKeys(Object(o)).forEach((function(i){Object.defineProperty(r,i,Object.getOwnPropertyDescriptor(o,i))}))}return r}function _defineProperty(r,i,o){return(i=_toPropertyKey(i))in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,_toPropertyKey(u.key),u)}}function _toPropertyKey(r){var i=function _toPrimitive(r,i){if("object"!=typeof r||null===r)return r;var o=r[Symbol.toPrimitive];if(void 0!==o){var u=o.call(r,i||"default");if("object"!=typeof u)return u;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===i?String:Number)(r)}(r,"string");return"symbol"==typeof i?i:String(i)}o(65292),o(73844),o(5597),o(49063),o(20731),o(7585),o(7283),o(57444),o(28331),o(37309),o(14032),o(42437);var u=o(50621).Buffer,p=o(63779).inspect,g=p&&p.custom||"inspect";r.exports=function(){function BufferList(){!function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,BufferList),this.head=null,this.tail=null,this.length=0}return function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}(BufferList,[{key:"push",value:function push(r){var i={data:r,next:null};this.length>0?this.tail.next=i:this.head=i,this.tail=i,++this.length}},{key:"unshift",value:function unshift(r){var i={data:r,next:this.head};0===this.length&&(this.tail=i),this.head=i,++this.length}},{key:"shift",value:function shift(){if(0!==this.length){var r=this.head.data;return 1===this.length?this.head=this.tail=null:this.head=this.head.next,--this.length,r}}},{key:"clear",value:function clear(){this.head=this.tail=null,this.length=0}},{key:"join",value:function join(r){if(0===this.length)return"";for(var i=this.head,o=""+i.data;i=i.next;)o+=r+i.data;return o}},{key:"concat",value:function concat(r){if(0===this.length)return u.alloc(0);for(var i,o,p,g=u.allocUnsafe(r>>>0),y=this.head,w=0;y;)i=y.data,o=g,p=w,u.prototype.copy.call(i,o,p),w+=y.data.length,y=y.next;return g}},{key:"consume",value:function consume(r,i){var o;return r<this.head.data.length?(o=this.head.data.slice(0,r),this.head.data=this.head.data.slice(r)):o=r===this.head.data.length?this.shift():i?this._getString(r):this._getBuffer(r),o}},{key:"first",value:function first(){return this.head.data}},{key:"_getString",value:function _getString(r){var i=this.head,o=1,u=i.data;for(r-=u.length;i=i.next;){var p=i.data,g=r>p.length?p.length:r;if(g===p.length?u+=p:u+=p.slice(0,r),0===(r-=g)){g===p.length?(++o,i.next?this.head=i.next:this.head=this.tail=null):(this.head=i,i.data=p.slice(g));break}++o}return this.length-=o,u}},{key:"_getBuffer",value:function _getBuffer(r){var i=u.allocUnsafe(r),o=this.head,p=1;for(o.data.copy(i),r-=o.data.length;o=o.next;){var g=o.data,y=r>g.length?g.length:r;if(g.copy(i,i.length-r,0,y),0===(r-=y)){y===g.length?(++p,o.next?this.head=o.next:this.head=this.tail=null):(this.head=o,o.data=g.slice(y));break}++p}return this.length-=p,i}},{key:g,value:function value(r,i){return p(this,_objectSpread(_objectSpread({},i),{},{depth:0,customInspect:!1}))}}]),BufferList}()},72361:function(r,i,o){"use strict";o(41584);var u=o(50621),p=u.Buffer;function copyProps(r,i){for(var o in r)i[o]=r[o]}function SafeBuffer(r,i,o){return p(r,i,o)}p.from&&p.alloc&&p.allocUnsafe&&p.allocUnsafeSlow?r.exports=u:(copyProps(u,i),i.Buffer=SafeBuffer),copyProps(p,SafeBuffer),SafeBuffer.from=function(r,i,o){if("number"==typeof r)throw new TypeError("Argument must not be a number");return p(r,i,o)},SafeBuffer.alloc=function(r,i,o){if("number"!=typeof r)throw new TypeError("Argument must be a number");var u=p(r);return void 0!==i?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u},SafeBuffer.allocUnsafe=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return p(r)},SafeBuffer.allocUnsafeSlow=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return u.SlowBuffer(r)}},41209:function(r,i,o){"use strict";o(81755),o(20731),o(14032),o(56912),o(59735),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529);var u=o(3483),p=o(51014).swap32LE,g=function(){function UnicodeTrie(r){var i="function"==typeof r.readUInt32BE&&"function"==typeof r.slice;if(i||r instanceof Uint8Array){var o;if(i)this.highStart=r.readUInt32LE(0),this.errorValue=r.readUInt32LE(4),o=r.readUInt32LE(8),r=r.slice(12);else{var g=new DataView(r.buffer);this.highStart=g.getUint32(0,!0),this.errorValue=g.getUint32(4,!0),o=g.getUint32(8,!0),r=r.subarray(12)}r=u(r,new Uint8Array(o)),r=u(r,new Uint8Array(o)),p(r),this.data=new Uint32Array(r.buffer)}else{var y=r;this.data=y.data,this.highStart=y.highStart,this.errorValue=y.errorValue}}return UnicodeTrie.prototype.get=function get(r){var i;return r<0||r>1114111?this.errorValue:r<55296||r>56319&&r<=65535?(i=(this.data[r>>5]<<2)+(31&r),this.data[i]):r<=65535?(i=(this.data[2048+(r-55296>>5)]<<2)+(31&r),this.data[i]):r<this.highStart?(i=this.data[2080+(r>>11)],i=((i=this.data[i+(r>>5&63)])<<2)+(31&r),this.data[i]):this.data[this.data.length-4]},UnicodeTrie}();r.exports=g},51014:function(r,i,o){"use strict";o(81755),o(14032),o(56912),o(59735),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529);var u=18===new Uint8Array(new Uint32Array([305419896]).buffer)[0],p=function swap(r,i,o){var u=r[i];r[i]=r[o],r[o]=u};r.exports={swap32LE:function swap32LE(r){u&&function swap32(r){for(var i=r.length,o=0;o<i;o+=4)p(r,o,o+3),p(r,o+1,o+2)}(r)}}},54074:function(r,i,o){"use strict";var u=o(94732);r.exports=u},94732:function(r,i,o){"use strict";r=o.nmd(r),o(39081),o(41584),o(81755),o(20731),o(24863),o(7283),o(37309),o(14032),o(61726),o(6422),o(46467),o(7851),o(72095),o(47259),o(1083),o(94712);r&&void 0!==r.exports&&(r.exports=function SVGtoPDF(r,i,o,u,p){var g={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgrey:[169,169,169],darkgreen:[0,100,0],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgrey:[211,211,211],lightgreen:[144,238,144],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],rebeccapurple:[102,51,153],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0]},y={black:[g.black,1],white:[g.white,1],transparent:[g.black,0]},w={quot:34,amp:38,lt:60,gt:62,apos:39,OElig:338,oelig:339,Scaron:352,scaron:353,Yuml:376,circ:710,tilde:732,ensp:8194,emsp:8195,thinsp:8201,zwnj:8204,zwj:8205,lrm:8206,rlm:8207,ndash:8211,mdash:8212,lsquo:8216,rsquo:8217,sbquo:8218,ldquo:8220,rdquo:8221,bdquo:8222,dagger:8224,Dagger:8225,permil:8240,lsaquo:8249,rsaquo:8250,euro:8364,nbsp:160,iexcl:161,cent:162,pound:163,curren:164,yen:165,brvbar:166,sect:167,uml:168,copy:169,ordf:170,laquo:171,not:172,shy:173,reg:174,macr:175,deg:176,plusmn:177,sup2:178,sup3:179,acute:180,micro:181,para:182,middot:183,cedil:184,sup1:185,ordm:186,raquo:187,frac14:188,frac12:189,frac34:190,iquest:191,Agrave:192,Aacute:193,Acirc:194,Atilde:195,Auml:196,Aring:197,AElig:198,Ccedil:199,Egrave:200,Eacute:201,Ecirc:202,Euml:203,Igrave:204,Iacute:205,Icirc:206,Iuml:207,ETH:208,Ntilde:209,Ograve:210,Oacute:211,Ocirc:212,Otilde:213,Ouml:214,times:215,Oslash:216,Ugrave:217,Uacute:218,Ucirc:219,Uuml:220,Yacute:221,THORN:222,szlig:223,agrave:224,aacute:225,acirc:226,atilde:227,auml:228,aring:229,aelig:230,ccedil:231,egrave:232,eacute:233,ecirc:234,euml:235,igrave:236,iacute:237,icirc:238,iuml:239,eth:240,ntilde:241,ograve:242,oacute:243,ocirc:244,otilde:245,ouml:246,divide:247,oslash:248,ugrave:249,uacute:250,ucirc:251,uuml:252,yacute:253,thorn:254,yuml:255,fnof:402,Alpha:913,Beta:914,Gamma:915,Delta:916,Epsilon:917,Zeta:918,Eta:919,Theta:920,Iota:921,Kappa:922,Lambda:923,Mu:924,Nu:925,Xi:926,Omicron:927,Pi:928,Rho:929,Sigma:931,Tau:932,Upsilon:933,Phi:934,Chi:935,Psi:936,Omega:937,alpha:945,beta:946,gamma:947,delta:948,epsilon:949,zeta:950,eta:951,theta:952,iota:953,kappa:954,lambda:955,mu:956,nu:957,xi:958,omicron:959,pi:960,rho:961,sigmaf:962,sigma:963,tau:964,upsilon:965,phi:966,chi:967,psi:968,omega:969,thetasym:977,upsih:978,piv:982,bull:8226,hellip:8230,prime:8242,Prime:8243,oline:8254,frasl:8260,weierp:8472,image:8465,real:8476,trade:8482,alefsym:8501,larr:8592,uarr:8593,rarr:8594,darr:8595,harr:8596,crarr:8629,lArr:8656,uArr:8657,rArr:8658,dArr:8659,hArr:8660,forall:8704,part:8706,exist:8707,empty:8709,nabla:8711,isin:8712,notin:8713,ni:8715,prod:8719,sum:8721,minus:8722,lowast:8727,radic:8730,prop:8733,infin:8734,ang:8736,and:8743,or:8744,cap:8745,cup:8746,int:8747,there4:8756,sim:8764,cong:8773,asymp:8776,ne:8800,equiv:8801,le:8804,ge:8805,sub:8834,sup:8835,nsub:8836,sube:8838,supe:8839,oplus:8853,otimes:8855,perp:8869,sdot:8901,lceil:8968,rceil:8969,lfloor:8970,rfloor:8971,lang:9001,rang:9002,loz:9674,spades:9824,clubs:9827,hearts:9829,diams:9830},_={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},x={A3:!0,A4:!0,a3:!0,a4:!0},k={color:{inherit:!0,initial:void 0},visibility:{inherit:!0,initial:"visible",values:{hidden:"hidden",collapse:"hidden",visible:"visible"}},fill:{inherit:!0,initial:y.black},stroke:{inherit:!0,initial:"none"},"stop-color":{inherit:!1,initial:y.black},"fill-opacity":{inherit:!0,initial:1},"stroke-opacity":{inherit:!0,initial:1},"stop-opacity":{inherit:!1,initial:1},"fill-rule":{inherit:!0,initial:"nonzero",values:{nonzero:"nonzero",evenodd:"evenodd"}},"clip-rule":{inherit:!0,initial:"nonzero",values:{nonzero:"nonzero",evenodd:"evenodd"}},"stroke-width":{inherit:!0,initial:1},"stroke-dasharray":{inherit:!0,initial:[]},"stroke-dashoffset":{inherit:!0,initial:0},"stroke-miterlimit":{inherit:!0,initial:4},"stroke-linejoin":{inherit:!0,initial:"miter",values:{miter:"miter",round:"round",bevel:"bevel"}},"stroke-linecap":{inherit:!0,initial:"butt",values:{butt:"butt",round:"round",square:"square"}},"font-size":{inherit:!0,initial:16,values:{"xx-small":9,"x-small":10,small:13,medium:16,large:18,"x-large":24,"xx-large":32}},"font-family":{inherit:!0,initial:"sans-serif"},"font-weight":{inherit:!0,initial:"normal",values:{600:"bold",700:"bold",800:"bold",900:"bold",bold:"bold",bolder:"bold",500:"normal",400:"normal",300:"normal",200:"normal",100:"normal",normal:"normal",lighter:"normal"}},"font-style":{inherit:!0,initial:"normal",values:{italic:"italic",oblique:"italic",normal:"normal"}},"text-anchor":{inherit:!0,initial:"start",values:{start:"start",middle:"middle",end:"end"}},direction:{inherit:!0,initial:"ltr",values:{ltr:"ltr",rtl:"rtl"}},"dominant-baseline":{inherit:!0,initial:"baseline",values:{auto:"baseline",baseline:"baseline","before-edge":"before-edge","text-before-edge":"before-edge",middle:"middle",central:"central","after-edge":"after-edge","text-after-edge":"after-edge",ideographic:"ideographic",alphabetic:"alphabetic",hanging:"hanging",mathematical:"mathematical"}},"alignment-baseline":{inherit:!1,initial:void 0,values:{auto:"baseline",baseline:"baseline","before-edge":"before-edge","text-before-edge":"before-edge",middle:"middle",central:"central","after-edge":"after-edge","text-after-edge":"after-edge",ideographic:"ideographic",alphabetic:"alphabetic",hanging:"hanging",mathematical:"mathematical"}},"baseline-shift":{inherit:!0,initial:"baseline",values:{baseline:"baseline",sub:"sub",super:"super"}},"word-spacing":{inherit:!0,initial:0,values:{normal:0}},"letter-spacing":{inherit:!0,initial:0,values:{normal:0}},"text-decoration":{inherit:!1,initial:"none",values:{none:"none",underline:"underline",overline:"overline","line-through":"line-through"}},"xml:space":{inherit:!0,initial:"default",css:"white-space",values:{preserve:"preserve",default:"default",pre:"preserve","pre-line":"preserve","pre-wrap":"preserve",nowrap:"default"}},"marker-start":{inherit:!0,initial:"none"},"marker-mid":{inherit:!0,initial:"none"},"marker-end":{inherit:!0,initial:"none"},opacity:{inherit:!1,initial:1},transform:{inherit:!1,initial:[1,0,0,1,0,0]},display:{inherit:!1,initial:"inline",values:{none:"none",inline:"inline",block:"inline"}},"clip-path":{inherit:!1,initial:"none"},mask:{inherit:!1,initial:"none"},overflow:{inherit:!1,initial:"hidden",values:{hidden:"hidden",scroll:"hidden",visible:"visible"}}};function docBeginGroup(i){var o=new function PDFGroup(){};return o.name="G"+(r._groupCount=(r._groupCount||0)+1),o.resources=r.ref(),o.xobj=r.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:i,Group:{S:"Transparency",CS:"DeviceRGB",I:!0,K:!1},Resources:o.resources}),o.xobj.write(""),o.savedMatrix=r._ctm,o.savedPage=r.page,Ae.push(o),r._ctm=[1,0,0,1,0,0],r.page={width:r.page.width,height:r.page.height,write:function write(r){o.xobj.write(r)},fonts:{},xobjects:{},ext_gstates:{},patterns:{}},o}function docEndGroup(i){if(i!==Ae.pop())throw"Group not matching";Object.keys(r.page.fonts).length&&(i.resources.data.Font=r.page.fonts),Object.keys(r.page.xobjects).length&&(i.resources.data.XObject=r.page.xobjects),Object.keys(r.page.ext_gstates).length&&(i.resources.data.ExtGState=r.page.ext_gstates),Object.keys(r.page.patterns).length&&(i.resources.data.Pattern=r.page.patterns),i.resources.end(),i.xobj.end(),r._ctm=i.savedMatrix,r.page=i.savedPage}function docInsertGroup(i){r.page.xobjects[i.name]=i.xobj,r.addContent("/"+i.name+" Do")}function docApplyMask(i,o){var u="M"+(r._maskCount=(r._maskCount||0)+1),p=r.ref({Type:"ExtGState",CA:1,ca:1,BM:"Normal",SMask:{S:"Luminosity",G:i.xobj,BC:o?[0,0,0]:[1,1,1]}});p.end(),r.page.ext_gstates[u]=p,r.addContent("/"+u+" gs")}function docCreatePattern(r,i,o,u){var p=new function PDFPattern(){};return p.group=r,p.dx=i,p.dy=o,p.matrix=u||[1,0,0,1,0,0],p}function docUsePattern(i,o){var u,p="P"+(r._patternCount=(r._patternCount||0)+1),g=r.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:[0,0,i.dx,i.dy],XStep:i.dx,YStep:i.dy,Matrix:multiplyMatrix(r._ctm,i.matrix),Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],XObject:(u={},u[i.group.name]=i.group.xobj,u)}});g.write("/"+i.group.name+" Do"),g.end(),r.page.patterns[p]=g,o?(r.addContent("/Pattern CS"),r.addContent("/"+p+" SCN")):(r.addContent("/Pattern cs"),r.addContent("/"+p+" scn"))}function docSetTextMode(i,o){var u=i&&o?2:o?1:i?0:3;r.addContent(u+" Tr")}function docFillColor(i){"PDFPattern"===i[0].constructor.name?(r.fillOpacity(i[1]),docUsePattern(i[0],!1)):r.fillColor(i[0],i[1])}function docStrokeColor(i){"PDFPattern"===i[0].constructor.name?(r.strokeOpacity(i[1]),docUsePattern(i[0],!0)):r.strokeColor(i[0],i[1])}function parseXml(r){var i=function SvgNode(r,i,o,u){this.error=u,this.nodeName=r,this.nodeValue=o,this.nodeType=i,this.attributes=Object.create(null),this.childNodes=[],this.parentNode=null,this.id="",this.textContent="",this.classList=[]};i.prototype.getAttribute=function(r){return null!=this.attributes[r]?this.attributes[r]:null},i.prototype.getElementById=function(r){var i=null;return function recursive(o){if(!i&&1===o.nodeType){o.id===r&&(i=o);for(var u=0;u<o.childNodes.length;u++)recursive(o.childNodes[u])}}(this),i},i.prototype.getElementsByTagName=function(r){var i=[];return function recursive(o){if(1===o.nodeType){o.nodeName===r&&i.push(o);for(var u=0;u<o.childNodes.length;u++)recursive(o.childNodes[u])}}(this),i};for(var o,u,p=new P(r.trim()),g=!1,y=function recursive(){var r,o;if(r=p.match(/^<([\w:.-]+)\s*/,!0)){for(var u=new i(r[1],1,null,g);r=p.match(/^([\w:.-]+)(?:\s*=\s*"([^"]*)"|\s*=\s*'([^']*)')?\s*/,!0);){var y=r[1],w=decodeEntities(r[2]||r[3]||"");u.attributes[y]?(_e('parseXml: duplicate attribute "'+y+'"'),g=!0):(u.attributes[y]=w,"id"===y&&(u.id=w),"class"===y&&(u.classList=w.split(" ")))}if(p.match(/^>/)){for(;o=recursive();)u.childNodes.push(o),o.parentNode=u,u.textContent+=3===o.nodeType||4===o.nodeType?o.nodeValue:o.textContent;return(r=p.match(/^<\/([\w:.-]+)\s*>/,!0))?(r[1]===u.nodeName||(_e('parseXml: tag not matching, opening "'+u.nodeName+'" & closing "'+r[1]+'"'),g=!0),u):(_e('parseXml: tag not matching, opening "'+u.nodeName+'" & not closing'),g=!0,u)}if(p.match(/^\/>/))return u;_e('parseXml: tag could not be parsed "'+u.nodeName+'"'),g=!0}else{if(r=p.match(/^<!--[\s\S]*?-->/))return new i(null,8,r,g);if(r=p.match(/^<\?[\s\S]*?\?>/))return new i(null,7,r,g);if(r=p.match(/^<!DOCTYPE\s*([\s\S]*?)>/))return new i(null,10,r,g);if(r=p.match(/^<!\[CDATA\[([\s\S]*?)\]\]>/,!0))return new i("#cdata-section",4,r[1],g);if(r=p.match(/^([^<]+)/,!0))return new i("#text",3,decodeEntities(r[1]),g)}};u=y();)1!==u.nodeType||o?(1===u.nodeType||3===u.nodeType&&""!==u.nodeValue.trim())&&_e("parseXml: data after document end has been discarded"):o=u;return p.matchAll()&&_e("parseXml: parsing error"),o}function decodeEntities(r){return r.replace(/&(?:#([0-9]+)|#[xX]([0-9A-Fa-f]+)|([0-9A-Za-z]+));/g,(function(r,i,o,u){return i?String.fromCharCode(parseInt(i,10)):o?String.fromCharCode(parseInt(o,16)):u&&w[u]?String.fromCharCode(w[u]):r}))}function parseColor(r){var i,o;return r=(r||"").trim(),(i=g[r])?o=[i.slice(),1]:(i=r.match(/^rgba\(\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9.]+)\s*\)$/i))?(i[1]=parseInt(i[1]),i[2]=parseInt(i[2]),i[3]=parseInt(i[3]),i[4]=parseFloat(i[4]),i[1]<256&&i[2]<256&&i[3]<256&&i[4]<=1&&(o=[i.slice(1,4),i[4]])):(i=r.match(/^rgb\(\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*\)$/i))?(i[1]=parseInt(i[1]),i[2]=parseInt(i[2]),i[3]=parseInt(i[3]),i[1]<256&&i[2]<256&&i[3]<256&&(o=[i.slice(1,4),1])):(i=r.match(/^rgb\(\s*([0-9.]+)%\s*,\s*([0-9.]+)%\s*,\s*([0-9.]+)%\s*\)$/i))?(i[1]=2.55*parseFloat(i[1]),i[2]=2.55*parseFloat(i[2]),i[3]=2.55*parseFloat(i[3]),i[1]<256&&i[2]<256&&i[3]<256&&(o=[i.slice(1,4),1])):(i=r.match(/^#([0-9a-f]{2})([0-9a-f]{2})([0-9a-f]{2})$/i))?o=[[parseInt(i[1],16),parseInt(i[2],16),parseInt(i[3],16)],1]:(i=r.match(/^#([0-9a-f])([0-9a-f])([0-9a-f])$/i))&&(o=[[17*parseInt(i[1],16),17*parseInt(i[2],16),17*parseInt(i[3],16)],1]),Ce?Ce(o,r):o}function opacityToColor(r,i,o){var u=r[0].slice(),p=r[1]*i;if(o){for(var g=0;g<r.length;g++)u[g]*=p;return[u,1]}return[u,p]}function multiplyMatrix(){for(var r,i,o=arguments[0],u=1;u<arguments.length;u++)i=arguments[u],o=[(r=o)[0]*i[0]+r[2]*i[1],r[1]*i[0]+r[3]*i[1],r[0]*i[2]+r[2]*i[3],r[1]*i[2]+r[3]*i[3],r[0]*i[4]+r[2]*i[5]+r[4],r[1]*i[4]+r[3]*i[5]+r[5]];return o}function transformPoint(r,i){return[i[0]*r[0]+i[2]*r[1]+i[4],i[1]*r[0]+i[3]*r[1]+i[5]]}function getGlobalMatrix(){for(var i=r._ctm,o=Ae.length-1;o>=0;o--)i=multiplyMatrix(Ae[o].savedMatrix,i);return i}function getPageBBox(){return(new B).M(0,0).L(r.page.width,0).L(r.page.width,r.page.height).L(0,r.page.height).transform(inverseMatrix(getGlobalMatrix())).getBoundingBox()}function inverseMatrix(r){var i=r[0]*r[3]-r[1]*r[2];return[r[3]/i,-r[1]/i,-r[2]/i,r[0]/i,(r[2]*r[5]-r[3]*r[4])/i,(r[1]*r[4]-r[0]*r[5])/i]}function validateMatrix(r){var i=validateNumber(r[0]),o=validateNumber(r[1]),u=validateNumber(r[2]),p=validateNumber(r[3]),g=validateNumber(r[4]),y=validateNumber(r[5]);if(isNotEqual(i*p-o*u,0))return[i,o,u,p,g,y]}function solveEquation(r){var i=r[2]||0,o=r[1]||0,u=r[0]||0;if(isEqual(i,0)&&isEqual(o,0))return[];if(isEqual(i,0))return[-u/o];var p=o*o-4*i*u;return isNotEqual(p,0)&&p>0?[(-o+Math.sqrt(p))/(2*i),(-o-Math.sqrt(p))/(2*i)]:isEqual(p,0)?[-o/(2*i)]:[]}function getCurveValue(r,i){return(i[0]||0)+(i[1]||0)*r+(i[2]||0)*r*r+(i[3]||0)*r*r*r}function isEqual(r,i){return Math.abs(r-i)<1e-10}function isNotEqual(r,i){return Math.abs(r-i)>=1e-10}function validateNumber(r){return r>-1e21&&r<1e21?Math.round(1e6*r)/1e6:0}function parseTranform(r){for(var i,o=new P((r||"").trim()),u=[1,0,0,1,0,0];i=o.match(/^([A-Za-z]+)\s*[(]([^(]+)[)]/,!0);){for(var p=i[1],g=[],y=new P(i[2].trim()),w=void 0;w=y.matchNumber();)g.push(Number(w)),y.matchSeparator();if("matrix"===p&&6===g.length)u=multiplyMatrix(u,[g[0],g[1],g[2],g[3],g[4],g[5]]);else if("translate"===p&&2===g.length)u=multiplyMatrix(u,[1,0,0,1,g[0],g[1]]);else if("translate"===p&&1===g.length)u=multiplyMatrix(u,[1,0,0,1,g[0],0]);else if("scale"===p&&2===g.length)u=multiplyMatrix(u,[g[0],0,0,g[1],0,0]);else if("scale"===p&&1===g.length)u=multiplyMatrix(u,[g[0],0,0,g[0],0,0]);else if("rotate"===p&&3===g.length){var _=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,0,0,1,g[1],g[2]],[Math.cos(_),Math.sin(_),-Math.sin(_),Math.cos(_),0,0],[1,0,0,1,-g[1],-g[2]])}else if("rotate"===p&&1===g.length){var x=g[0]*Math.PI/180;u=multiplyMatrix(u,[Math.cos(x),Math.sin(x),-Math.sin(x),Math.cos(x),0,0])}else if("skewX"===p&&1===g.length){var k=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,0,Math.tan(k),1,0,0])}else{if("skewY"!==p||1!==g.length)return;var O=g[0]*Math.PI/180;u=multiplyMatrix(u,[1,Math.tan(O),0,1,0,0])}o.matchSeparator()}if(!o.matchAll())return u}function parseAspectRatio(r,i,o,u,p,g){var y=(r||"").trim().match(/^(none)$|^x(Min|Mid|Max)Y(Min|Mid|Max)(?:\s+(meet|slice))?$/)||[],w=y[1]||y[4]||"meet",_=y[2]||"Mid",x=y[3]||"Mid",k=i/u,P=o/p,O={Min:0,Mid:.5,Max:1}[_]-(g||0),I={Min:0,Mid:.5,Max:1}[x]-(g||0);return"slice"===w?P=k=Math.max(k,P):"meet"===w&&(P=k=Math.min(k,P)),[k,0,0,P,O*(i-u*k),I*(o-p*P)]}function parseStyleAttr(r){var i=Object.create(null);r=(r||"").trim().split(/;/);for(var o=0;o<r.length;o++){var u=(r[o].split(":")[0]||"").trim(),p=(r[o].split(":")[1]||"").trim();u&&(i[u]=p)}if(i.marker&&(i["marker-start"]||(i["marker-start"]=i.marker),i["marker-mid"]||(i["marker-mid"]=i.marker),i["marker-end"]||(i["marker-end"]=i.marker)),i.font){for(var g=null,y=null,w="normal",_="normal",x="normal",k=i.font.split(/\s+/),P=0;P<k.length;P++)switch(k[P]){case"normal":break;case"italic":case"oblique":w=k[P];break;case"small-caps":x=k[P];break;case"bold":case"bolder":case"lighter":case"100":case"200":case"300":case"400":case"500":case"600":case"700":case"800":case"900":_=k[P];break;default:y?g?g+=" "+k[P]:g=k[P]:y=k[P].split("/")[0]}i["font-style"]||(i["font-style"]=w),i["font-variant"]||(i["font-variant"]=x),i["font-weight"]||(i["font-weight"]=_),i["font-size"]||(i["font-size"]=y),i["font-family"]||(i["font-family"]=g)}return i}function parseSelector(r){for(var i,o=r.split(/(?=[.#])/g),u=[],p=[],g=[],y=0;y<o.length;y++)if(i=o[y].match(/^[#]([_A-Za-z0-9-]+)$/))u.push(i[1]);else if(i=o[y].match(/^[.]([_A-Za-z0-9-]+)$/))p.push(i[1]);else if(i=o[y].match(/^([_A-Za-z0-9-]+)$/))g.push(i[1]);else if("*"!==o[y])return;return{tags:g,ids:u,classes:p,specificity:1e4*u.length+100*p.length+g.length}}function parseStyleSheet(r){for(var i,o=new P(r.trim()),u=[];i=o.match(/^\s*([^\{\}]*?)\s*\{([^\{\}]*?)\}/,!0);)for(var p=i[1].split(/\s*,\s*/g),g=parseStyleAttr(i[2]),y=0;y<p.length;y++){var w=parseSelector(p[y]);w&&u.push({selector:w,css:g})}return u}function matchesSelector(r,i){if(1!==r.nodeType)return!1;for(var o=0;o<i.tags.length;o++)if(i.tags[o]!==r.nodeName)return!1;for(var u=0;u<i.ids.length;u++)if(i.ids[u]!==r.id)return!1;for(var p=0;p<i.classes.length;p++)if(-1===r.classList.indexOf(i.classes[p]))return!1;return!0}function combineArrays(r,i){return r.concat(i.slice(r.length))}function getAscent(r,i){return Math.max(r.ascender,(r.bbox[3]||r.bbox.maxY)*(r.scale||1))*i/1e3}function getDescent(r,i){return Math.min(r.descender,(r.bbox[1]||r.bbox.minY)*(r.scale||1))*i/1e3}function getTextPos(r,i,o){for(var u=r.encode(""+o),p=u[0],g=u[1],y=[],w=0;w<p.length;w++){var _=r.unicode?r.unicode[parseInt(p[w],16)]:[o.charCodeAt(w)];y.push({glyph:p[w],unicode:_,width:g[w].advanceWidth*i/1e3,xOffset:g[w].xOffset*i/1e3,yOffset:g[w].yOffset*i/1e3,xAdvance:g[w].xAdvance*i/1e3,yAdvance:g[w].yAdvance*i/1e3})}return y}function createSVGElement(r,i){switch(r.nodeName){case"use":return new U(r,i);case"symbol":return new j(r,i);case"g":return new G(r,i);case"a":return new X(r,i);case"svg":return new K(r,i);case"image":return new Y(r,i);case"rect":return new te(r,i);case"circle":return new re(r,i);case"ellipse":return new ne(r,i);case"line":return new ie(r,i);case"polyline":return new oe(r,i);case"polygon":return new ae(r,i);case"path":return new se(r,i);case"text":return new ge(r,i);case"tspan":return new de(r,i);case"textPath":return new pe(r,i);case"#text":case"#cdata-section":return new he(r,i);default:return new D(r,i)}}var P=function StringParser(r){this.match=function(i,o){var u=r.match(i);if(u&&0===u.index)return r=r.substring(u[0].length),o?u:u[0]},this.matchSeparator=function(){return this.match(/^(?:\s*,\s*|\s*|)/)},this.matchSpace=function(){return this.match(/^(?:\s*)/)},this.matchLengthUnit=function(){return this.match(/^(?:px|pt|cm|mm|in|pc|em|ex|%|)/)},this.matchNumber=function(){return this.match(/^(?:[-+]?(?:[0-9]+[.][0-9]+|[0-9]+[.]|[.][0-9]+|[0-9]+)(?:[eE][-+]?[0-9]+)?)/)},this.matchAll=function(){return this.match(/^[\s\S]+/)}},O=function BezierSegment(r,i,o,u,p,g,y,w){for(var _=6*Pe,x=[r,-3*r+3*o,3*r-6*o+3*p,3*o-r-3*p+y],k=[i,-3*i+3*u,3*i-6*u+3*g,3*u-i-3*g+w],P=[-3*r+3*o,6*r-12*o+6*p,-3*r+9*o-9*p+3*y],O=[-3*i+3*u,6*i-12*u+6*g,-3*i+9*u-9*g+3*w],I=[0],B=1;B<=_;B++){var D=(B-.5)/_,R=getCurveValue(D,P)/_,N=getCurveValue(D,O)/_,W=Math.sqrt(R*R+N*N);I[B]=I[B-1]+W}this.totalLength=I[_],this.startPoint=[r,i,isEqual(r,o)&&isEqual(i,u)?Math.atan2(g-u,p-o):Math.atan2(u-i,o-r)],this.endPoint=[y,w,isEqual(p,y)&&isEqual(g,w)?Math.atan2(g-u,p-o):Math.atan2(w-g,y-p)],this.getBoundingBox=function(){var r,i=getCurveValue(0,x),o=getCurveValue(0,k),u=getCurveValue(1,x),p=getCurveValue(1,k);i>u&&(r=u,u=i,i=r),o>p&&(r=p,p=o,o=r);for(var g=solveEquation(P),y=0;y<g.length;y++)if(g[y]>=0&&g[y]<=1){var w=getCurveValue(g[y],x);w<i&&(i=w),w>u&&(u=w)}for(var _=solveEquation(O),I=0;I<_.length;I++)if(_[I]>=0&&_[I]<=1){var B=getCurveValue(_[I],k);B<o&&(o=B),B>p&&(p=B)}return[i,o,u,p]},this.getPointAtLength=function(r){if(isEqual(r,0))return this.startPoint;if(isEqual(r,this.totalLength))return this.endPoint;if(!(r<0||r>this.totalLength))for(var i=1;i<=_;i++){var o=I[i-1],u=I[i];if(o<=r&&r<=u){var p=(i-(u-r)/(u-o))/_,g=getCurveValue(p,x),y=getCurveValue(p,k),w=getCurveValue(p,P),B=getCurveValue(p,O);return[g,y,Math.atan2(B,w)]}}}},I=function LineSegment(r,i,o,u){this.totalLength=Math.sqrt((o-r)*(o-r)+(u-i)*(u-i)),this.startPoint=[r,i,Math.atan2(u-i,o-r)],this.endPoint=[o,u,Math.atan2(u-i,o-r)],this.getBoundingBox=function(){return[Math.min(this.startPoint[0],this.endPoint[0]),Math.min(this.startPoint[1],this.endPoint[1]),Math.max(this.startPoint[0],this.endPoint[0]),Math.max(this.startPoint[1],this.endPoint[1])]},this.getPointAtLength=function(r){if(r>=0&&r<=this.totalLength){var i=r/this.totalLength||0;return[this.startPoint[0]+i*(this.endPoint[0]-this.startPoint[0]),this.startPoint[1]+i*(this.endPoint[1]-this.startPoint[1]),this.startPoint[2]]}}},B=function SvgShape(){this.pathCommands=[],this.pathSegments=[],this.startPoint=null,this.endPoint=null,this.totalLength=0;var i,o,u,p=0,g=0,y=0,w=0;this.move=function(r,i){return p=y=r,g=w=i,null},this.line=function(r,i){var o=new I(y,w,r,i);return y=r,w=i,o},this.curve=function(r,i,o,u,p,g){var _=new O(y,w,r,i,o,u,p,g);return y=p,w=g,_},this.close=function(){var r=new I(y,w,p,g);return y=p,w=g,r},this.addCommand=function(r){this.pathCommands.push(r);var i=this[r[0]].apply(this,r.slice(3));i&&(i.hasStart=r[1],i.hasEnd=r[2],this.startPoint=this.startPoint||i.startPoint,this.endPoint=i.endPoint,this.pathSegments.push(i),this.totalLength+=i.totalLength)},this.M=function(r,o){return this.addCommand(["move",!0,!0,r,o]),i="M",this},this.m=function(r,i){return this.M(y+r,w+i)},this.Z=this.z=function(){return this.addCommand(["close",!0,!0]),i="Z",this},this.L=function(r,o){return this.addCommand(["line",!0,!0,r,o]),i="L",this},this.l=function(r,i){return this.L(y+r,w+i)},this.H=function(r){return this.L(r,w)},this.h=function(r){return this.L(y+r,w)},this.V=function(r){return this.L(y,r)},this.v=function(r){return this.L(y,w+r)},this.C=function(r,p,g,y,w,_){return this.addCommand(["curve",!0,!0,r,p,g,y,w,_]),i="C",o=g,u=y,this},this.c=function(r,i,o,u,p,g){return this.C(y+r,w+i,y+o,w+u,y+p,w+g)},this.S=function(r,p,g,_){return this.C(y+("C"===i?y-o:0),w+("C"===i?w-u:0),r,p,g,_)},this.s=function(r,p,g,_){return this.C(y+("C"===i?y-o:0),w+("C"===i?w-u:0),y+r,w+p,y+g,w+_)},this.Q=function(r,p,g,_){var x=y+2/3*(r-y),k=w+2/3*(p-w),P=g+2/3*(r-g),O=_+2/3*(p-_);return this.addCommand(["curve",!0,!0,x,k,P,O,g,_]),i="Q",o=r,u=p,this},this.q=function(r,i,o,u){return this.Q(y+r,w+i,y+o,w+u)},this.T=function(r,p){return this.Q(y+("Q"===i?y-o:0),w+("Q"===i?w-u:0),r,p)},this.t=function(r,p){return this.Q(y+("Q"===i?y-o:0),w+("Q"===i?w-u:0),y+r,w+p)},this.A=function(r,o,u,p,g,_,x){if(isEqual(r,0)||isEqual(o,0))this.addCommand(["line",!0,!0,_,x]);else{u*=Math.PI/180,r=Math.abs(r),o=Math.abs(o),p=1*!!p,g=1*!!g;var k=Math.cos(u)*(y-_)/2+Math.sin(u)*(w-x)/2,P=Math.cos(u)*(w-x)/2-Math.sin(u)*(y-_)/2,O=k*k/(r*r)+P*P/(o*o);O>1&&(r*=Math.sqrt(O),o*=Math.sqrt(O));var I=Math.sqrt(Math.max(0,r*r*o*o-r*r*P*P-o*o*k*k)/(r*r*P*P+o*o*k*k)),B=(p===g?-1:1)*I*r*P/o,D=(p===g?1:-1)*I*o*k/r,R=Math.cos(u)*B-Math.sin(u)*D+(y+_)/2,N=Math.sin(u)*B+Math.cos(u)*D+(w+x)/2,W=Math.atan2((P-D)/o,(k-B)/r),U=Math.atan2((-P-D)/o,(-k-B)/r);0===g&&U-W>0?U-=2*Math.PI:1===g&&U-W<0&&(U+=2*Math.PI);for(var j=Math.ceil(Math.abs(U-W)/(Math.PI/Pe)),G=0;G<j;G++){var X=W+G*(U-W)/j,K=W+(G+1)*(U-W)/j,Y=4/3*Math.tan((K-X)/4),J=R+Math.cos(u)*r*(Math.cos(X)-Y*Math.sin(X))-Math.sin(u)*o*(Math.sin(X)+Y*Math.cos(X)),$=N+Math.sin(u)*r*(Math.cos(X)-Y*Math.sin(X))+Math.cos(u)*o*(Math.sin(X)+Y*Math.cos(X)),ee=R+Math.cos(u)*r*(Math.cos(K)+Y*Math.sin(K))-Math.sin(u)*o*(Math.sin(K)-Y*Math.cos(K)),te=N+Math.sin(u)*r*(Math.cos(K)+Y*Math.sin(K))+Math.cos(u)*o*(Math.sin(K)-Y*Math.cos(K)),re=R+Math.cos(u)*r*Math.cos(K)-Math.sin(u)*o*Math.sin(K),ne=N+Math.sin(u)*r*Math.cos(K)+Math.cos(u)*o*Math.sin(K);this.addCommand(["curve",0===G,G===j-1,J,$,ee,te,re,ne])}}return i="A",this},this.a=function(r,i,o,u,p,g,_){return this.A(r,i,o,u,p,y+g,w+_)},this.path=function(r){for(var i,o,u,p=new P((r||"").trim());i=p.match(/^[astvzqmhlcASTVZQMHLC]/);){p.matchSeparator();for(var g=[];o=x[i+g.length]?p.match(/^[01]/):p.matchNumber();)p.matchSeparator(),g.length===_[i]&&(this[i].apply(this,g),g=[],"M"===i?i="L":"m"===i&&(i="l")),g.push(Number(o));if(g.length!==_[i])return void _e("SvgPath: command "+i+" with "+g.length+" numbers");this[i].apply(this,g)}return(u=p.matchAll())&&_e("SvgPath: unexpected string "+u),this},this.getBoundingBox=function(){for(var r,i=[1/0,1/0,-1/0,-1/0],o=0;o<this.pathSegments.length;o++)(r=this.pathSegments[o].getBoundingBox())[0]<i[0]&&(i[0]=r[0]),r[2]>i[2]&&(i[2]=r[2]),r[1]<i[1]&&(i[1]=r[1]),r[3]>i[3]&&(i[3]=r[3]);return i[0]===1/0&&(i[0]=0),i[1]===1/0&&(i[1]=0),i[2]===-1/0&&(i[2]=0),i[3]===-1/0&&(i[3]=0),i},this.getPointAtLength=function(r){if(r>=0&&r<=this.totalLength){for(var i,o=0;o<this.pathSegments.length;o++){if(i=this.pathSegments[o].getPointAtLength(r))return i;r-=this.pathSegments[o].totalLength}return this.endPoint}},this.transform=function(r){this.pathSegments=[],this.startPoint=null,this.endPoint=null,this.totalLength=0;for(var i=0;i<this.pathCommands.length;i++){for(var o=this.pathCommands.shift(),u=3;u<o.length;u+=2){var p=transformPoint([o[u],o[u+1]],r);o[u]=p[0],o[u+1]=p[1]}this.addCommand(o)}return this},this.mergeShape=function(r){for(var i=0;i<r.pathCommands.length;i++)this.addCommand(r.pathCommands[i].slice());return this},this.clone=function(){return(new SvgShape).mergeShape(this)},this.insertInDocument=function(){for(var i=0;i<this.pathCommands.length;i++){var o=this.pathCommands[i][0],u=this.pathCommands[i].slice(3);switch(o){case"move":r.moveTo(u[0],u[1]);break;case"line":r.lineTo(u[0],u[1]);break;case"curve":r.bezierCurveTo(u[0],u[1],u[2],u[3],u[4],u[5]);break;case"close":r.closePath()}}},this.getSubPaths=function(){for(var r=[],i=new SvgShape,o=0;o<this.pathCommands.length;o++){var u=this.pathCommands[o];"move"===this.pathCommands[o][0]&&0!==o&&(r.push(i),i=new SvgShape),i.addCommand(u)}return r.push(i),r},this.getMarkers=function(){for(var r=[],i=this.getSubPaths(),o=0;o<i.length;o++){for(var u=i[o],p=[],g=0;g<u.pathSegments.length;g++){var y=u.pathSegments[g];if(isNotEqual(y.totalLength,0)||0===g||g===u.pathSegments.length-1){if(y.hasStart){var w=y.getPointAtLength(0),_=p.pop();_&&(w[2]=.5*(_[2]+w[2])),p.push(w)}if(y.hasEnd){var x=y.getPointAtLength(y.totalLength);p.push(x)}}}r=r.concat(p)}return r}},D=function SvgElem(r,o){var u=Object.create(null),p=null;this.name=r.nodeName,this.isOuterElement=r===i||!r.parentNode,this.inherits=o||(this.isOuterElement?null:createSVGElement(r.parentNode,null)),this.stack=this.inherits?this.inherits.stack.concat(r):[r],this.style=parseStyleAttr("function"==typeof r.getAttribute&&r.getAttribute("style")),this.css=we?getComputedStyle(r):function getStyle(r){for(var i=Object.create(null),o=Object.create(null),u=0;u<Oe.length;u++){var p=Oe[u];if(matchesSelector(r,p.selector))for(var g in p.css)o[g]>p.selector.specificity||(i[g]=p.css[g],o[g]=p.selector.specificity)}return i}(r),this.allowedChildren=[],this.attr=function(i){if("function"==typeof r.getAttribute)return r.getAttribute(i)},this.resolveUrl=function(r){var o=(r||"").match(/^\s*(?:url\("(.*)#(.*)"\)|url\('(.*)#(.*)'\)|url\((.*)#(.*)\)|(.*)#(.*))\s*$/)||[],u=o[1]||o[3]||o[5]||o[7],p=o[2]||o[4]||o[6]||o[8];if(p){if(!u){var g=i.getElementById(p);if(g)return-1===this.stack.indexOf(g)?g:void _e('SVGtoPDF: loop of circular references for id "'+p+'"')}if(ke){var y=Ee[u];if(!y){(function isArrayLike(r){return"object"==typeof r&&null!==r&&"number"==typeof r.length})(y=ke(u))||(y=[y]);for(var w=0;w<y.length;w++)"string"==typeof y[w]&&(y[w]=parseXml(y[w]));Ee[u]=y}for(var _=0;_<y.length;_++){var x=y[_].getElementById(p);if(x)return-1===this.stack.indexOf(x)?x:void _e('SVGtoPDF: loop of circular references for id "'+u+"#"+p+'"')}}}},this.computeUnits=function(r,i,o,u){return"%"===i?parseFloat(r)/100*(u||null!=o?o:this.getViewport()):"ex"===i||"em"===i?r*{em:1,ex:.5}[i]*(u?o:this.get("font-size")):r*{"":1,px:1,pt:96/72,cm:96/2.54,mm:96/25.4,in:96,pc:16}[i]},this.computeLength=function(r,i,o,u){var p,g,y=new P((r||"").trim());return"string"!=typeof(p=y.matchNumber())||"string"!=typeof(g=y.matchLengthUnit())||y.matchAll()?o:this.computeUnits(p,g,i,u)},this.computeLengthList=function(r,i,o){for(var u,p,g=new P((r||"").trim()),y=[];"string"==typeof(u=g.matchNumber())&&"string"==typeof(p=g.matchLengthUnit());)y.push(this.computeUnits(u,p,i)),g.matchSeparator();if(!o||!g.matchAll())return y},this.getLength=function(r,i,o){return this.computeLength(this.attr(r),i,o)},this.getLengthList=function(r,i){return this.computeLengthList(this.attr(r),i)},this.getUrl=function(r){return this.resolveUrl(this.attr(r))},this.getNumberList=function(r){for(var i,o=new P((this.attr(r)||"").trim()),u=[];i=o.matchNumber();)u.push(Number(i)),o.matchSeparator();return u.error=o.matchAll(),u},this.getViewbox=function(r,i){var o=this.getNumberList(r);return 4===o.length&&o[2]>=0&&o[3]>=0?o:i},this.getPercent=function(r,i){var o=this.attr(r),u=new P((o||"").trim()),p=u.matchNumber();return p?(u.match("%")&&(p*=.01),u.matchAll()?i:Math.max(0,Math.min(1,p))):i},this.chooseValue=function(r){for(var i=0;i<arguments.length;i++)if(null!=arguments[i]&&arguments[i]==arguments[i])return arguments[i];return arguments[arguments.length-1]},this.get=function(r){if(void 0!==u[r])return u[r];for(var i,o,p=k[r]||{},g=0;g<3;g++){switch(g){case 0:"transform"!==r&&(i=this.css[p.css||r]);break;case 1:i=this.style[r];break;case 2:i=this.attr(r)}if("inherit"===i&&null!=(o=this.inherits?this.inherits.get(r):p.initial))return u[r]=o;if(null!=p.values&&null!=(o=p.values[i]))return u[r]=o;if(null!=i){var y=void 0;switch(r){case"font-size":o=this.computeLength(i,this.inherits?this.inherits.get(r):p.initial,void 0,!0);break;case"baseline-shift":o=this.computeLength(i,this.get("font-size"));break;case"font-family":o=i||void 0;break;case"opacity":case"stroke-opacity":case"fill-opacity":case"stop-opacity":y=parseFloat(i),isNaN(y)||(o=Math.max(0,Math.min(1,y)));break;case"transform":o=parseTranform(i);break;case"stroke-dasharray":if("none"===i)o=[];else if(y=this.computeLengthList(i,this.getViewport(),!0)){for(var w=0,_=!1,x=0;x<y.length;x++)y[x]<0&&(_=!0),w+=y[x];_||(y.length%2==1&&(y=y.concat(y)),o=0===w?[]:y)}break;case"color":o="none"===i||"transparent"===i?"none":parseColor(i);break;case"fill":case"stroke":if("none"===i||"transparent"===i)o="none";else if("currentColor"===i)o=this.get("color");else{if(y=parseColor(i))return y;if(y=(i||"").split(" ")){var P=this.resolveUrl(y[0]),O=parseColor(y[1]);o=null==P?O:"linearGradient"===P.nodeName||"radialGradient"===P.nodeName?new $(P,null,O):"pattern"===P.nodeName?new J(P,null,O):O}}break;case"stop-color":o="none"===i||"transparent"===i?"none":"currentColor"===i?this.get("color"):parseColor(i);break;case"marker-start":case"marker-mid":case"marker-end":case"clip-path":case"mask":o="none"===i?"none":this.resolveUrl(i);break;case"stroke-width":null!=(y=this.computeLength(i,this.getViewport()))&&y>=0&&(o=y);break;case"stroke-miterlimit":null!=(y=parseFloat(i))&&y>=1&&(o=y);break;case"word-spacing":case"letter-spacing":o=this.computeLength(i,this.getViewport());break;case"stroke-dashoffset":if(null!=(o=this.computeLength(i,this.getViewport()))&&o<0)for(var I=this.get("stroke-dasharray"),B=0;B<I.length;B++)o+=I[B]}if(null!=o)return u[r]=o}}return u[r]=p.inherit&&this.inherits?this.inherits.get(r):p.initial},this.getChildren=function(){if(null!=p)return p;for(var i=[],o=0;o<r.childNodes.length;o++){var u=r.childNodes[o];u.error||-1===this.allowedChildren.indexOf(u.nodeName)||i.push(createSVGElement(u,this))}return p=i},this.getParentVWidth=function(){return this.inherits?this.inherits.getVWidth():ve},this.getParentVHeight=function(){return this.inherits?this.inherits.getVHeight():me},this.getParentViewport=function(){return Math.sqrt(.5*this.getParentVWidth()*this.getParentVWidth()+.5*this.getParentVHeight()*this.getParentVHeight())},this.getVWidth=function(){return this.getParentVWidth()},this.getVHeight=function(){return this.getParentVHeight()},this.getViewport=function(){return Math.sqrt(.5*this.getVWidth()*this.getVWidth()+.5*this.getVHeight()*this.getVHeight())},this.getBoundingBox=function(){return this.getBoundingShape().getBoundingBox()}},R=function SvgElemStylable(i,o){D.call(this,i,o),this.transform=function(){r.transform.apply(r,this.getTransformation())},this.clip=function(){if("none"!==this.get("clip-path"))return new le(this.get("clip-path"),null).useMask(this.getBoundingBox()),!0},this.mask=function(){if("none"!==this.get("mask"))return new ue(this.get("mask"),null).useMask(this.getBoundingBox()),!0},this.getFill=function(r,i){var o=this.get("opacity"),u=this.get("fill"),p=this.get("fill-opacity");return r?y.white:"none"!==u&&o&&p?u instanceof $||u instanceof J?u.getPaint(this.getBoundingBox(),p*o,r,i):opacityToColor(u,p*o,i):void 0},this.getStroke=function(r,i){var o=this.get("opacity"),u=this.get("stroke"),p=this.get("stroke-opacity");if(!r&&!isEqual(this.get("stroke-width"),0))return"none"!==u&&o&&p?u instanceof $||u instanceof J?u.getPaint(this.getBoundingBox(),p*o,r,i):opacityToColor(u,p*o,i):void 0}},N=function SvgElemHasChildren(r,i){R.call(this,r,i),this.allowedChildren=["use","g","a","svg","image","rect","circle","ellipse","line","polyline","polygon","path","text"],this.getBoundingShape=function(){for(var r=new B,i=this.getChildren(),o=0;o<i.length;o++)if("none"!==i[o].get("display")&&"function"==typeof i[o].getBoundingShape){var u=i[o].getBoundingShape().clone();"function"==typeof i[o].getTransformation&&u.transform(i[o].getTransformation()),r.mergeShape(u)}return r},this.drawChildren=function(r,i){for(var o=this.getChildren(),u=0;u<o.length;u++)"none"!==o[u].get("display")&&"function"==typeof o[u].drawInDocument&&o[u].drawInDocument(r,i)}},W=function SvgElemContainer(i,o){N.call(this,i,o),this.drawContent=function(i,o){this.transform();var u,p=this.clip(),g=this.mask();(this.get("opacity")<1||p||g)&&!i&&(u=docBeginGroup(getPageBBox())),this.drawChildren(i,o),u&&(docEndGroup(u),r.fillOpacity(this.get("opacity")),docInsertGroup(u))}},U=function SvgElemUse(i,o){W.call(this,i,o);var u=this.getLength("x",this.getVWidth(),0),p=this.getLength("y",this.getVHeight(),0),g=this.getUrl("href")||this.getUrl("xlink:href");g&&(g=createSVGElement(g,this)),this.getChildren=function(){return g?[g]:[]},this.drawInDocument=function(i,o){r.save(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(this.get("transform"),[1,0,0,1,u,p])}},j=function SvgElemSymbol(i,o){W.call(this,i,o);var u=this.getLength("width",this.getParentVWidth(),this.getParentVWidth()),p=this.getLength("height",this.getParentVHeight(),this.getParentVHeight());o instanceof U&&(u=o.getLength("width",o.getParentVWidth(),u),p=o.getLength("height",o.getParentVHeight(),p));var g=(this.attr("preserveAspectRatio")||"").trim(),y=this.getViewbox("viewBox",[0,0,u,p]);this.getVWidth=function(){return y[2]},this.getVHeight=function(){return y[3]},this.drawInDocument=function(i,o){r.save(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(parseAspectRatio(g,u,p,y[2],y[3]),[1,0,0,1,-y[0],-y[1]])}},G=function SvgElemGroup(i,o){W.call(this,i,o),this.drawInDocument=function(i,o){r.save(),!this.link||i||o||this.addLink(),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return this.get("transform")}},X=function SvgElemLink(i,o){o&&o.isText?(de.call(this,i,o),this.allowedChildren=["textPath","tspan","#text","#cdata-section","a"]):G.call(this,i,o),this.link=this.attr("href")||this.attr("xlink:href"),this.addLink=function(){if(this.link.match(/^(?:[a-z][a-z0-9+.-]*:|\/\/)?/i)&&this.getChildren().length){var i=this.getBoundingShape().transform(getGlobalMatrix()).getBoundingBox();!function docInsertLink(i,o,u,p,g){var y=r.ref({Type:"Annot",Subtype:"Link",Rect:[i,o,u,p],Border:[0,0,0],A:{S:"URI",URI:new String(g)}});y.end(),Te.push(y)}(i[0],i[1],i[2],i[3],this.link)}}},K=function SvgElemSvg(i,o){W.call(this,i,o);var u=this.getLength("width",this.getParentVWidth(),this.getParentVWidth()),p=this.getLength("height",this.getParentVHeight(),this.getParentVHeight()),g=this.getLength("x",this.getParentVWidth(),0),y=this.getLength("y",this.getParentVHeight(),0);o instanceof U&&(u=o.getLength("width",o.getParentVWidth(),u),p=o.getLength("height",o.getParentVHeight(),p));var w=this.attr("preserveAspectRatio"),_=this.getViewbox("viewBox",[0,0,u,p]);this.isOuterElement&&be&&(g=y=0,u=ve,p=me,w=be),this.getVWidth=function(){return _[2]},this.getVHeight=function(){return _[3]},this.drawInDocument=function(i,o){r.save(),"hidden"===this.get("overflow")&&((new B).M(g,y).L(g+u,y).L(g+u,y+p).L(g,y+p).Z().transform(this.get("transform")).insertInDocument(),r.clip()),this.drawContent(i,o),r.restore()},this.getTransformation=function(){return multiplyMatrix(this.get("transform"),[1,0,0,1,g,y],parseAspectRatio(w,u,p,_[2],_[3]),[1,0,0,1,-_[0],-_[1]])}},Y=function SVGElemImage(i,o){R.call(this,i,o);var u,p=xe(this.attr("href")||this.attr("xlink:href")||""),g=this.getLength("x",this.getVWidth(),0),w=this.getLength("y",this.getVHeight(),0),_=this.getLength("width",this.getVWidth(),"auto"),x=this.getLength("height",this.getVHeight(),"auto");try{u=r.openImage(p)}catch(r){_e('SVGElemImage: failed to open image "'+p+'" in PDFKit')}u&&("auto"===_&&"auto"!==x?_=x*u.width/u.height:"auto"===x&&"auto"!==_?x=_*u.height/u.width:"auto"===_&&"auto"===x&&(_=u.width,x=u.height)),("auto"===_||_<0)&&(_=0),("auto"===x||x<0)&&(x=0),this.getTransformation=function(){return this.get("transform")},this.getBoundingShape=function(){return(new B).M(g,w).L(g+_,w).M(g+_,w+x).L(g,w+x)},this.drawInDocument=function(i,o){"hidden"!==this.get("visibility")&&u&&(r.save(),this.transform(),"hidden"===this.get("overflow")&&r.rect(g,w,_,x).clip(),this.clip(),this.mask(),r.translate(g,w),r.transform.apply(r,parseAspectRatio(this.attr("preserveAspectRatio"),_,x,u?u.width:_,u?u.height:x)),i?(r.rect(0,0,u.width,u.height),docFillColor(y.white).fill()):(r.fillOpacity(this.get("opacity")),r.image(u,0,0)),r.restore())}},J=function SvgElemPattern(i,o,u){N.call(this,i,o),this.ref=function(){var r=this.getUrl("href")||this.getUrl("xlink:href");if(r&&r.nodeName===i.nodeName)return new SvgElemPattern(r,o,u)}.call(this);var p=this.attr;this.attr=function(r){var i=p.call(this,r);return null!=i||"href"===r||"xlink:href"===r?i:this.ref?this.ref.attr(r):null};var g=this.getChildren;this.getChildren=function(){var r=g.call(this);return r.length>0?r:this.ref?this.ref.getChildren():[]},this.getPaint=function(i,o,p,g){var y="userSpaceOnUse"!==this.attr("patternUnits"),w="objectBoundingBox"===this.attr("patternContentUnits"),_=this.getLength("x",y?1:this.getParentVWidth(),0),x=this.getLength("y",y?1:this.getParentVHeight(),0),k=this.getLength("width",y?1:this.getParentVWidth(),0),P=this.getLength("height",y?1:this.getParentVHeight(),0);w&&!y?(_=(_-i[0])/(i[2]-i[0])||0,x=(x-i[1])/(i[3]-i[1])||0,k=k/(i[2]-i[0])||0,P=P/(i[3]-i[1])||0):!w&&y&&(_=i[0]+_*(i[2]-i[0]),x=i[1]+x*(i[3]-i[1]),k*=i[2]-i[0],P*=i[3]-i[1]);var O=this.getViewbox("viewBox",[0,0,k,P]),I=multiplyMatrix(parseAspectRatio((this.attr("preserveAspectRatio")||"").trim(),k,P,O[2],O[3],0),[1,0,0,1,-O[0],-O[1]]),B=parseTranform(this.attr("patternTransform"));if(w&&(B=multiplyMatrix([i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]],B)),(B=validateMatrix(B=multiplyMatrix(B,[1,0,0,1,_,x])))&&(I=validateMatrix(I))&&(k=validateNumber(k))&&(P=validateNumber(P))){var D=docBeginGroup([0,0,k,P]);return r.transform.apply(r,I),this.drawChildren(p,g),docEndGroup(D),[docCreatePattern(D,k,P,B),o]}return u?[u[0],u[1]*o]:void 0},this.getVWidth=function(){var r="userSpaceOnUse"!==this.attr("patternUnits"),i=this.getLength("width",r?1:this.getParentVWidth(),0);return this.getViewbox("viewBox",[0,0,i,0])[2]},this.getVHeight=function(){var r="userSpaceOnUse"!==this.attr("patternUnits"),i=this.getLength("height",r?1:this.getParentVHeight(),0);return this.getViewbox("viewBox",[0,0,0,i])[3]}},$=function SvgElemGradient(i,o,u){D.call(this,i,o),this.allowedChildren=["stop"],this.ref=function(){var r=this.getUrl("href")||this.getUrl("xlink:href");if(r&&r.nodeName===i.nodeName)return new SvgElemGradient(r,o,u)}.call(this);var p=this.attr;this.attr=function(r){var i=p.call(this,r);return null!=i||"href"===r||"xlink:href"===r?i:this.ref?this.ref.attr(r):null};var g=this.getChildren;this.getChildren=function(){var r=g.call(this);return r.length>0?r:this.ref?this.ref.getChildren():[]},this.getPaint=function(i,o,p,g){var w=this.getChildren();if(0!==w.length){if(1===w.length){var _=w[0],x=_.get("stop-color");if("none"===x)return;return opacityToColor(x,_.get("stop-opacity")*o,g)}var k,P,O,I,B,D,R="userSpaceOnUse"!==this.attr("gradientUnits"),N=parseTranform(this.attr("gradientTransform")),W=this.attr("spreadMethod"),U=0,j=0,G=1;if(R&&(N=multiplyMatrix([i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]],N)),N=validateMatrix(N)){if("linearGradient"===this.name)P=this.getLength("x1",R?1:this.getVWidth(),0),O=this.getLength("x2",R?1:this.getVWidth(),R?1:this.getVWidth()),I=this.getLength("y1",R?1:this.getVHeight(),0),B=this.getLength("y2",R?1:this.getVHeight(),0);else{O=this.getLength("cx",R?1:this.getVWidth(),R?.5:.5*this.getVWidth()),B=this.getLength("cy",R?1:this.getVHeight(),R?.5:.5*this.getVHeight()),D=this.getLength("r",R?1:this.getViewport(),R?.5:.5*this.getViewport()),P=this.getLength("fx",R?1:this.getVWidth(),O),I=this.getLength("fy",R?1:this.getVHeight(),B),D<0&&_e("SvgElemGradient: negative r value");var X=Math.sqrt(Math.pow(O-P,2)+Math.pow(B-I,2)),K=1;X>D&&(P=O+(P-O)*(K=D/X),I=B+(I-B)*K),D=Math.max(D,X*K*1.000001)}if("reflect"===W||"repeat"===W){var Y=inverseMatrix(N),J=transformPoint([i[0],i[1]],Y),$=transformPoint([i[2],i[1]],Y),ee=transformPoint([i[2],i[3]],Y),te=transformPoint([i[0],i[3]],Y);"linearGradient"===this.name?(U=Math.max((J[0]-O)*(O-P)+(J[1]-B)*(B-I),($[0]-O)*(O-P)+($[1]-B)*(B-I),(ee[0]-O)*(O-P)+(ee[1]-B)*(B-I),(te[0]-O)*(O-P)+(te[1]-B)*(B-I))/(Math.pow(O-P,2)+Math.pow(B-I,2)),j=Math.max((J[0]-P)*(P-O)+(J[1]-I)*(I-B),($[0]-P)*(P-O)+($[1]-I)*(I-B),(ee[0]-P)*(P-O)+(ee[1]-I)*(I-B),(te[0]-P)*(P-O)+(te[1]-I)*(I-B))/(Math.pow(O-P,2)+Math.pow(B-I,2))):U=Math.sqrt(Math.max(Math.pow(J[0]-O,2)+Math.pow(J[1]-B,2),Math.pow($[0]-O,2)+Math.pow($[1]-B,2),Math.pow(ee[0]-O,2)+Math.pow(ee[1]-B,2),Math.pow(te[0]-O,2)+Math.pow(te[1]-B,2)))/D-1,U=Math.ceil(U+.5),G=(j=Math.ceil(j+.5))+1+U}k="linearGradient"===this.name?r.linearGradient(P-j*(O-P),I-j*(B-I),O+U*(O-P),B+U*(B-I)):r.radialGradient(P,I,0,O,B,D+U*D);for(var re=0;re<G;re++)for(var ne=0,ie="reflect"!==W||(re-j)%2==0,oe=0;oe<w.length;oe++){var ae=w[ie?oe:w.length-1-oe],se=ae.get("stop-color");"none"===se&&(se=y.transparent),se=opacityToColor(se,ae.get("stop-opacity")*o,g),ne=Math.max(ne,ie?ae.getPercent("offset",0):1-ae.getPercent("offset",0)),0===oe&&4===se[0].length&&(k._colorSpace="DeviceCMYK"),0===oe&&ne>0&&k.stop((re+0)/G,se[0],se[1]),k.stop((re+ne)/(U+j+1),se[0],se[1]),oe===w.length-1&&ne<1&&k.stop((re+1)/G,se[0],se[1])}return k.setTransform.apply(k,N),[k,1]}return u?[u[0],u[1]*o]:void 0}}},ee=function SvgElemBasicShape(i,o){R.call(this,i,o),this.dashScale=1,this.getBoundingShape=function(){return this.shape},this.getTransformation=function(){return this.get("transform")},this.drawInDocument=function(i,o){if("hidden"!==this.get("visibility")&&this.shape){if(r.save(),this.transform(),this.clip(),i)this.shape.insertInDocument(),docFillColor(y.white),r.fill(this.get("clip-rule"));else{var u;this.mask()&&(u=docBeginGroup(getPageBBox()));var p=this.shape.getSubPaths(),g=this.getFill(i,o),w=this.getStroke(i,o),_=this.get("stroke-width"),x=this.get("stroke-linecap");if(g||w){if(g&&docFillColor(g),w){for(var k=0;k<p.length;k++)if(isEqual(p[k].totalLength,0)&&("square"===x||"round"===x)&&_>0&&p[k].startPoint&&p[k].startPoint.length>1){var P=p[k].startPoint[0],O=p[k].startPoint[1];docFillColor(w),"square"===x?r.rect(P-.5*_,O-.5*_,_,_):"round"===x&&r.circle(P,O,.5*_),r.fill()}var I=this.get("stroke-dasharray"),B=this.get("stroke-dashoffset");if(isNotEqual(this.dashScale,1)){for(var D=0;D<I.length;D++)I[D]*=this.dashScale;B*=this.dashScale}docStrokeColor(w),r.lineWidth(_).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(x).dash(I,{phase:B})}for(var R=0;R<p.length;R++)p[R].totalLength>0&&p[R].insertInDocument();g&&w?r.fillAndStroke(this.get("fill-rule")):g?r.fill(this.get("fill-rule")):w&&r.stroke()}var N=this.get("marker-start"),W=this.get("marker-mid"),U=this.get("marker-end");if("none"!==N||"none"!==W||"none"!==U){var j=this.shape.getMarkers();if("none"!==N&&new ce(N,null).drawMarker(!1,o,j[0],_),"none"!==W)for(var G=1;G<j.length-1;G++)new ce(W,null).drawMarker(!1,o,j[G],_);"none"!==U&&new ce(U,null).drawMarker(!1,o,j[j.length-1],_)}u&&(docEndGroup(u),docInsertGroup(u))}r.restore()}}},te=function SvgElemRect(r,i){ee.call(this,r,i);var o=this.getLength("x",this.getVWidth(),0),u=this.getLength("y",this.getVHeight(),0),p=this.getLength("width",this.getVWidth(),0),g=this.getLength("height",this.getVHeight(),0),y=this.getLength("rx",this.getVWidth()),w=this.getLength("ry",this.getVHeight());void 0===y&&void 0===w?y=w=0:void 0===y&&void 0!==w?y=w:void 0!==y&&void 0===w&&(w=y),p>0&&g>0?y&&w?(y=Math.min(y,.5*p),w=Math.min(w,.5*g),this.shape=(new B).M(o+y,u).L(o+p-y,u).A(y,w,0,0,1,o+p,u+w).L(o+p,u+g-w).A(y,w,0,0,1,o+p-y,u+g).L(o+y,u+g).A(y,w,0,0,1,o,u+g-w).L(o,u+w).A(y,w,0,0,1,o+y,u).Z()):this.shape=(new B).M(o,u).L(o+p,u).L(o+p,u+g).L(o,u+g).Z():this.shape=new B},re=function SvgElemCircle(r,i){ee.call(this,r,i);var o=this.getLength("cx",this.getVWidth(),0),u=this.getLength("cy",this.getVHeight(),0),p=this.getLength("r",this.getViewport(),0);this.shape=p>0?(new B).M(o+p,u).A(p,p,0,0,1,o-p,u).A(p,p,0,0,1,o+p,u).Z():new B},ne=function SvgElemEllipse(r,i){ee.call(this,r,i);var o=this.getLength("cx",this.getVWidth(),0),u=this.getLength("cy",this.getVHeight(),0),p=this.getLength("rx",this.getVWidth(),0),g=this.getLength("ry",this.getVHeight(),0);this.shape=p>0&&g>0?(new B).M(o+p,u).A(p,g,0,0,1,o-p,u).A(p,g,0,0,1,o+p,u).Z():new B},ie=function SvgElemLine(r,i){ee.call(this,r,i);var o=this.getLength("x1",this.getVWidth(),0),u=this.getLength("y1",this.getVHeight(),0),p=this.getLength("x2",this.getVWidth(),0),g=this.getLength("y2",this.getVHeight(),0);this.shape=(new B).M(o,u).L(p,g)},oe=function SvgElemPolyline(r,i){ee.call(this,r,i);var o=this.getNumberList("points");this.shape=new B;for(var u=0;u<o.length-1;u+=2)0===u?this.shape.M(o[u],o[u+1]):this.shape.L(o[u],o[u+1]);o.error&&_e("SvgElemPolygon: unexpected string "+o.error),o.length%2==1&&_e("SvgElemPolyline: uneven number of coordinates")},ae=function SvgElemPolygon(r,i){ee.call(this,r,i);var o=this.getNumberList("points");this.shape=new B;for(var u=0;u<o.length-1;u+=2)0===u?this.shape.M(o[u],o[u+1]):this.shape.L(o[u],o[u+1]);this.shape.Z(),o.error&&_e("SvgElemPolygon: unexpected string "+o.error),o.length%2==1&&_e("SvgElemPolygon: uneven number of coordinates")},se=function SvgElemPath(r,i){ee.call(this,r,i),this.shape=(new B).path(this.attr("d"));var o=this.getLength("pathLength",this.getViewport());this.pathLength=o>0?o:void 0,this.dashScale=void 0!==this.pathLength?this.shape.totalLength/this.pathLength:1},ce=function SvgElemMarker(i,o){N.call(this,i,o);var u=this.getLength("markerWidth",this.getParentVWidth(),3),p=this.getLength("markerHeight",this.getParentVHeight(),3),g=this.getViewbox("viewBox",[0,0,u,p]);this.getVWidth=function(){return g[2]},this.getVHeight=function(){return g[3]},this.drawMarker=function(i,o,y,w){r.save();var _=this.attr("orient"),x=this.attr("markerUnits"),k="auto"===_?y[2]:(parseFloat(_)||0)*Math.PI/180,P="userSpaceOnUse"===x?1:w;r.transform(Math.cos(k)*P,Math.sin(k)*P,-Math.sin(k)*P,Math.cos(k)*P,y[0],y[1]);var O,I=this.getLength("refX",this.getVWidth(),0),B=this.getLength("refY",this.getVHeight(),0),D=parseAspectRatio(this.attr("preserveAspectRatio"),u,p,g[2],g[3],.5);"hidden"===this.get("overflow")&&r.rect(D[0]*(g[0]+g[2]/2-I)-u/2,D[3]*(g[1]+g[3]/2-B)-p/2,u,p).clip(),r.transform.apply(r,D),r.translate(-I,-B),this.get("opacity")<1&&!i&&(O=docBeginGroup(getPageBBox())),this.drawChildren(i,o),O&&(docEndGroup(O),r.fillOpacity(this.get("opacity")),docInsertGroup(O)),r.restore()}},le=function SvgElemClipPath(i,o){N.call(this,i,o),this.useMask=function(i){var o=docBeginGroup(getPageBBox());r.save(),"objectBoundingBox"===this.attr("clipPathUnits")&&r.transform(i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]),this.clip(),this.drawChildren(!0,!1),r.restore(),docEndGroup(o),docApplyMask(o,!0)}},ue=function SvgElemMask(i,o){N.call(this,i,o),this.useMask=function(i){var o,u,p,g,y=docBeginGroup(getPageBBox());r.save(),"userSpaceOnUse"===this.attr("maskUnits")?(o=this.getLength("x",this.getVWidth(),-.1*(i[2]-i[0])+i[0]),u=this.getLength("y",this.getVHeight(),-.1*(i[3]-i[1])+i[1]),p=this.getLength("width",this.getVWidth(),1.2*(i[2]-i[0])),g=this.getLength("height",this.getVHeight(),1.2*(i[3]-i[1]))):(o=this.getLength("x",this.getVWidth(),-.1)*(i[2]-i[0])+i[0],u=this.getLength("y",this.getVHeight(),-.1)*(i[3]-i[1])+i[1],p=this.getLength("width",this.getVWidth(),1.2)*(i[2]-i[0]),g=this.getLength("height",this.getVHeight(),1.2)*(i[3]-i[1])),r.rect(o,u,p,g).clip(),"objectBoundingBox"===this.attr("maskContentUnits")&&r.transform(i[2]-i[0],0,0,i[3]-i[1],i[0],i[1]),this.clip(),this.drawChildren(!1,!0),r.restore(),docEndGroup(y),docApplyMask(y,!0)}},fe=function SvgElemTextContainer(i,o){R.call(this,i,o),this.allowedChildren=["tspan","#text","#cdata-section","a"],this.isText=!0,this.getBoundingShape=function(){for(var r=new B,i=0;i<this._pos.length;i++){var o=this._pos[i];if(!o.hidden){var u=o.ascent*Math.sin(o.rotate),p=-o.ascent*Math.cos(o.rotate),g=o.descent*Math.sin(o.rotate),y=-o.descent*Math.cos(o.rotate),w=o.width*Math.cos(o.rotate),_=o.width*Math.sin(o.rotate);r.M(o.x+u,o.y+p).L(o.x+u+w,o.y+p+_).M(o.x+g+w,o.y+y+_).L(o.x+g,o.y+y)}}return r},this.drawTextInDocument=function(i,o){!this.link||i||o||this.addLink(),"underline"===this.get("text-decoration")&&this.decorate(.05*this._font.size,-.075*this._font.size,i,o),"overline"===this.get("text-decoration")&&this.decorate(.05*this._font.size,getAscent(this._font.font,this._font.size)+.075*this._font.size,i,o);var u=this.getFill(i,o),p=this.getStroke(i,o),g=this.get("stroke-width");this._font.fauxBold&&(p?g+=.03*this._font.size:(p=u,g=.03*this._font.size));for(var y,w,_,x,k,P,O,I,B,D=this.getChildren(),R=0;R<D.length;R++){var N=D[R];switch(N.name){case"tspan":case"textPath":case"a":"none"!==N.get("display")&&N.drawTextInDocument(i,o);break;case"#text":case"#cdata-section":if("hidden"===this.get("visibility"))continue;if(u||p||i){u&&docFillColor(u),p&&g&&(docStrokeColor(p),r.lineWidth(g).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(this.get("stroke-linecap")).dash(this.get("stroke-dasharray"),{phase:this.get("stroke-dashoffset")})),I=this._font.font,B=this._font.size,r.page.fonts[I.id]||(r.page.fonts[I.id]=I.ref()),r.addContent("BT").addContent("/"+I.id+" "+B+" Tf"),docSetTextMode(!!u,!!p);for(var W=0,U=N._pos;W<U.length;W++)if(!U[W].hidden&&isNotEqual(U[W].width,0)){var j=Math.cos(U[W].rotate),G=Math.sin(U[W].rotate),X=this._font.fauxItalic?-.25:0;w=j*U[W].scale,_=G*U[W].scale,x=j*X-G,k=G*X+j,P=U[W].x,O=U[W].y,r.addContent(validateNumber(w)+" "+validateNumber(_)+" "+validateNumber(-x)+" "+validateNumber(-k)+" "+validateNumber(P)+" "+validateNumber(O)+" Tm"),y=U[W].glyph,r.addContent("<"+y+"> Tj")}r.addContent("ET")}}}"line-through"===this.get("text-decoration")&&this.decorate(.05*this._font.size,.5*(getAscent(this._font.font,this._font.size)+getDescent(this._font.font,this._font.size)),i,o)},this.decorate=function(i,o,u,p){var g=this.getFill(u,p),y=this.getStroke(u,p);g&&docFillColor(g),y&&(docStrokeColor(y),r.lineWidth(this.get("stroke-width")).miterLimit(this.get("stroke-miterlimit")).lineJoin(this.get("stroke-linejoin")).lineCap(this.get("stroke-linecap")).dash(this.get("stroke-dasharray"),{phase:this.get("stroke-dashoffset")}));for(var w=0,_=this._pos;w<_.length;w++)if(!_[w].hidden&&isNotEqual(_[w].width,0)){var x=(o+i/2)*Math.sin(_[w].rotate),k=-(o+i/2)*Math.cos(_[w].rotate),P=(o-i/2)*Math.sin(_[w].rotate),O=-(o-i/2)*Math.cos(_[w].rotate),I=_[w].width*Math.cos(_[w].rotate),D=_[w].width*Math.sin(_[w].rotate);(new B).M(_[w].x+x,_[w].y+k).L(_[w].x+x+I,_[w].y+k+D).L(_[w].x+P+I,_[w].y+O+D).L(_[w].x+P,_[w].y+O).Z().insertInDocument(),g&&y?r.fillAndStroke():g?r.fill():y&&r.stroke()}}},he=function SvgElemTextNode(r,i){this.name=r.nodeName,this.textContent=r.nodeValue},de=function SvgElemTspan(r,i){fe.call(this,r,i)},pe=function SvgElemTextPath(r,i){var o;if(fe.call(this,r,i),(o=this.attr("path"))&&""!==o.trim()){var u=this.getLength("pathLength",this.getViewport());this.pathObject=(new B).path(o),this.pathLength=u>0?u:this.pathObject.totalLength,this.pathScale=this.pathObject.totalLength/this.pathLength}else if((o=this.getUrl("href")||this.getUrl("xlink:href"))&&"path"===o.nodeName){var p=new se(o,this);this.pathObject=p.shape.clone().transform(p.get("transform")),this.pathLength=this.chooseValue(p.pathLength,this.pathObject.totalLength),this.pathScale=this.pathObject.totalLength/this.pathLength}},ge=function SvgElemText(i,o){fe.call(this,i,o),this.allowedChildren=["textPath","tspan","#text","#cdata-section","a"],function(o){var u,p,g="",y=i.textContent,w=[],_=[],x=0,k=0;function doAnchoring(){if(_.length)for(var r=_[_.length-1],i=_[0],o=r.x+r.width-i.x,g={startltr:0,middleltr:.5,endltr:1,startrtl:1,middlertl:.5,endrtl:0}[u+p]*o||0,y=0;y<_.length;y++)_[y].x-=g;_=[]}function textOnPath(r){var i=r.pathObject,o=r.pathLength,u=r.pathScale;if(i)for(var p=r.getLength("startOffset",o,0),g=0;g<r._pos.length;g++){var y=p+r._pos[g].x+.5*r._pos[g].width;if(y>o||y<0)r._pos[g].hidden=!0;else{var w=i.getPointAtLength(y*u);isNotEqual(u,1)&&(r._pos[g].scale*=u,r._pos[g].width*=u),r._pos[g].x=w[0]-.5*r._pos[g].width*Math.cos(w[2])-r._pos[g].y*Math.sin(w[2]),r._pos[g].y=w[1]-.5*r._pos[g].width*Math.sin(w[2])+r._pos[g].y*Math.cos(w[2]),r._pos[g].rotate=w[2]+r._pos[g].rotate,r._pos[g].continuous=!1}}else for(var _=0;_<r._pos.length;_++)r._pos[_].hidden=!0}!function recursive(i,o){i._x=combineArrays(i.getLengthList("x",i.getVWidth()),o?o._x.slice(o._pos.length):[]),i._y=combineArrays(i.getLengthList("y",i.getVHeight()),o?o._y.slice(o._pos.length):[]),i._dx=combineArrays(i.getLengthList("dx",i.getVWidth()),o?o._dx.slice(o._pos.length):[]),i._dy=combineArrays(i.getLengthList("dy",i.getVHeight()),o?o._dy.slice(o._pos.length):[]),i._rot=combineArrays(i.getNumberList("rotate"),o?o._rot.slice(o._pos.length):[]),i._defRot=i.chooseValue(i._rot[i._rot.length-1],o&&o._defRot,0),"textPath"===i.name&&(i._y=[]);var P={fauxItalic:!1,fauxBold:!1},O=Se(i.get("font-family"),"bold"===i.get("font-weight"),"italic"===i.get("font-style"),P);try{r.font(O)}catch(r){_e('SVGElemText: failed to open font "'+O+'" in PDFKit')}i._pos=[],i._index=0,i._font={font:r._font,size:i.get("font-size"),fauxItalic:P.fauxItalic,fauxBold:P.fauxBold};var I=i.getLength("textLength",i.getVWidth(),void 0),B="spacingAndGlyphs"===i.attr("lengthAdjust"),D=i.get("word-spacing"),R=i.get("letter-spacing"),N=i.get("text-anchor"),W=i.get("direction"),U=function getBaseline(r,i,o,u){var p,g;switch(o){case"middle":p=.5*function getXHeight(r,i){return(r.xHeight||.5*(r.ascender-r.descender))*i/1e3}(r,i);break;case"central":p=.5*(getDescent(r,i)+getAscent(r,i));break;case"after-edge":case"text-after-edge":p=getDescent(r,i);break;case"alphabetic":case"auto":case"baseline":default:p=0;break;case"mathematical":p=.5*getAscent(r,i);break;case"hanging":p=.8*getAscent(r,i);break;case"before-edge":case"text-before-edge":p=getAscent(r,i)}switch(u){case"baseline":g=0;break;case"super":g=.6*i;break;case"sub":g=-.6*i;break;default:g=u}return p-g}(i._font.font,i._font.size,i.get("alignment-baseline")||i.get("dominant-baseline"),i.get("baseline-shift"));"textPath"===i.name&&(doAnchoring(),x=k=0);for(var j=i.getChildren(),G=0;G<j.length;G++){var X=j[G];switch(X.name){case"tspan":case"textPath":case"a":recursive(X,i);break;case"#text":case"#cdata-section":var K=X.textContent,Y=K,J=void 0;X._font=i._font,X._pos=[],y=y.substring(K.length),"preserve"===i.get("xml:space")?Y=Y.replace(/[\s]/g," "):(Y=Y.replace(/[\s]+/g," "),g.match(/[\s]$|^$/)&&(Y=Y.replace(/^[\s]/,"")),y.match(/^[\s]*$/)&&(Y=Y.replace(/[\s]$/,""))),g+=K,J=0===D?[Y]:Y.split(/(\s)/);for(var $=0;$<J.length;$++){for(var ee=getTextPos(i._font.font,i._font.size,J[$]),te=0;te<ee.length;te++){var re=i._index,ne=i._x[re],ie=i._y[re],oe=i._dx[re],ae=i._dy[re],se=i._rot[re],ce=!(0===$&&0===te);void 0!==ne&&(ce=!1,doAnchoring(),x=ne),void 0!==ie&&(ce=!1,doAnchoring(),k=ie),void 0!==oe&&(ce=!1,x+=oe),void 0!==ae&&(ce=!1,k+=ae),void 0===se&&0===i._defRot||(ce=!1);var le={glyph:ee[te].glyph,rotate:Math.PI/180*i.chooseValue(se,i._defRot),x:x+ee[te].xOffset,y:k+U+ee[te].yOffset,width:ee[te].width,ascent:getAscent(i._font.font,i._font.size),descent:getDescent(i._font.font,i._font.size),scale:1,hidden:!1,continuous:ce};_.push(le),X._pos.push(le),i._pos.push(le),i._index+=ee[te].unicode.length,1===_.length&&(u=N,p=W),x+=ee[te].xAdvance+R,k+=ee[te].yAdvance}" "===J[$]&&(x+=D)}break;default:y=y.substring(X.textContent.length)}}if(I&&i._pos.length&&function adjustLength(r,i,o){var u=r[0],p=r[r.length-1],g=u.x,y=p.x+p.width;if(o){var w=i/(y-g);if(w>0&&w<1/0)for(var _=0;_<r.length;_++)r[_].x=g+w*(r[_].x-g),r[_].scale*=w,r[_].width*=w}else if(r.length>=2)for(var k=(i-(y-g))/(r.length-1),P=0;P<r.length;P++)r[P].x+=P*k;x+=i-(y-g)}(i._pos,I,B),"textPath"!==i.name&&"text"!==i.name||doAnchoring(),"textPath"===i.name){w.push(i);var ue=i.pathObject;ue&&(x=ue.endPoint[0],k=ue.endPoint[1])}o&&(o._pos=o._pos.concat(i._pos),o._index+=i._index)}(o,null);for(var P=0;P<w.length;P++)textOnPath(w[P])}(this),this.getTransformation=function(){return this.get("transform")},this.drawInDocument=function(i,o){var u;r.save(),this.transform(),this.clip(),this.mask()&&(u=docBeginGroup(getPageBBox())),this.drawTextInDocument(i,o),u&&(docEndGroup(u),docInsertGroup(u)),r.restore()}},ye=(p=p||{}).assumePt?1:.75,ve=(p.width||r.page.width)/ye,me=(p.height||r.page.height)/ye,be=p.preserveAspectRatio||null,we=p.useCSS&&"undefined"!=typeof SVGElement&&i instanceof SVGElement&&"function"==typeof getComputedStyle,_e=p.warningCallback,Se=p.fontCallback,xe=p.imageCallback,Ce=p.colorCallback,ke=p.documentCallback,Pe=Math.ceil(Math.max(1,p.precision))||3,Ae=[],Ee={},Te=[],Oe=[];if("function"!=typeof _e&&(_e=function warningCallback(r){void 0!==typeof console&&console.warn}),"function"!=typeof Se&&(Se=function fontCallback(i,o,u,p){if(o&&u){if(r._registeredFonts.hasOwnProperty(i+"-BoldItalic"))return i+"-BoldItalic";if(r._registeredFonts.hasOwnProperty(i+"-Italic"))return p.fauxBold=!0,i+"-Italic";if(r._registeredFonts.hasOwnProperty(i+"-Bold"))return p.fauxItalic=!0,i+"-Bold";if(r._registeredFonts.hasOwnProperty(i))return p.fauxBold=!0,p.fauxItalic=!0,i}if(o&&!u){if(r._registeredFonts.hasOwnProperty(i+"-Bold"))return i+"-Bold";if(r._registeredFonts.hasOwnProperty(i))return p.fauxBold=!0,i}if(!o&&u){if(r._registeredFonts.hasOwnProperty(i+"-Italic"))return i+"-Italic";if(r._registeredFonts.hasOwnProperty(i))return p.fauxItalic=!0,i}if(!o&&!u&&r._registeredFonts.hasOwnProperty(i))return i;if(i.match(/(?:^|,)\s*serif\s*$/)){if(o&&u)return"Times-BoldItalic";if(o&&!u)return"Times-Bold";if(!o&&u)return"Times-Italic";if(!o&&!u)return"Times-Roman"}else if(i.match(/(?:^|,)\s*monospace\s*$/)){if(o&&u)return"Courier-BoldOblique";if(o&&!u)return"Courier-Bold";if(!o&&u)return"Courier-Oblique";if(!o&&!u)return"Courier"}else{if(i.match(/(?:^|,)\s*sans-serif\s*$/),o&&u)return"Helvetica-BoldOblique";if(o&&!u)return"Helvetica-Bold";if(!o&&u)return"Helvetica-Oblique";if(!o&&!u)return"Helvetica"}}),"function"!=typeof xe&&(xe=function imageCallback(r){return r.replace(/\s+/g,"")}),"function"!=typeof Ce)Ce=null;else for(var Ie in y){var Be=Ce(y[Ie]);y[Ie][0]=Be[0],y[Ie][1]=Be[1]}if("function"!=typeof ke&&(ke=null),"string"==typeof i&&(i=parseXml(i)),i){for(var Fe=i.getElementsByTagName("style"),Le=0;Le<Fe.length;Le++)Oe=Oe.concat(parseStyleSheet(Fe[Le].textContent));var De=createSVGElement(i,null);if("function"==typeof De.drawInDocument){p.useCSS&&!we&&_e("SVGtoPDF: useCSS option can only be used for SVG *elements* in compatible browsers");var Re=r._fillColor;r.save().translate(o||0,u||0).scale(ye),De.drawInDocument();for(var Me=0;Me<Te.length;Me++)r.page.annotations.push(Te[Me]);r.restore(),r._fillColor=Re}else _e("SVGtoPDF: this element can't be rendered directly: "+i.nodeName)}else _e("SVGtoPDF: the input does not look like a valid SVG")})},48585:function(r,i,o){"use strict";var u="/",p=o(50621).Buffer;Object.defineProperty(i,"__esModule",{value:!0}),i.default=void 0,o(65292),o(73844),o(69330),o(39081),o(41584),o(49063),o(11765),o(58028),o(81755),o(94845),o(80055),o(20731),o(45337),o(10720),o(24863),o(7283),o(70095),o(10849),o(18756),o(57444),o(28331),o(71950),o(37309),o(14032),o(68067),o(73228),o(61726),o(57114),o(26663),o(65578),o(58281),o(6422),o(46467),o(7851),o(72095),o(62046),o(47259),o(18425),o(56912),o(73663),o(29883),o(35471),o(21012),o(88997),o(97464),o(2857),o(94715),o(13624),o(91132),o(62514),o(24597),o(88042),o(4660),o(92451),o(44206),o(66288),o(13250),o(3858),o(84538),o(64793),o(74202),o(52529),o(42437),o(94712);var g=_interopRequireDefault(o(9760)),y=_interopRequireDefault(o(6729)),w=_interopRequireDefault(o(66122)),_=_interopRequireDefault(o(60421)),x=o(64785),k=_interopRequireDefault(o(19304)),P=_interopRequireDefault(o(80646));function _interopRequireDefault(r){return r&&r.__esModule?r:{default:r}}var O=o(48181);function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,u.key,u)}}function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),r}function _defineProperty(r,i,o){return i in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function ownKeys(r,i){var o=Object.keys(r);if(Object.getOwnPropertySymbols){var u=Object.getOwnPropertySymbols(r);i&&(u=u.filter((function(i){return Object.getOwnPropertyDescriptor(r,i).enumerable}))),o.push.apply(o,u)}return o}function _objectSpread2(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{};i%2?ownKeys(Object(o),!0).forEach((function(i){_defineProperty(r,i,o[i])})):Object.getOwnPropertyDescriptors?Object.defineProperties(r,Object.getOwnPropertyDescriptors(o)):ownKeys(Object(o)).forEach((function(i){Object.defineProperty(r,i,Object.getOwnPropertyDescriptor(o,i))}))}return r}function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),i&&_setPrototypeOf(r,i)}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _createSuper(r){var i=function _isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Date.prototype.toString.call(Reflect.construct(Date,[],(function(){}))),!0}catch(r){return!1}}();return function _createSuperInternal(){var o,u=_getPrototypeOf(r);if(i){var p=_getPrototypeOf(this).constructor;o=Reflect.construct(u,arguments,p)}else o=u.apply(this,arguments);return function _possibleConstructorReturn(r,i){return!i||"object"!=typeof i&&"function"!=typeof i?_assertThisInitialized(r):i}(this,o)}}function _slicedToArray(r,i){return function _arrayWithHoles(r){if(Array.isArray(r))return r}(r)||function _iterableToArrayLimit(r,i){if("undefined"==typeof Symbol||!(Symbol.iterator in Object(r)))return;var o=[],u=!0,p=!1,g=void 0;try{for(var y,w=r[Symbol.iterator]();!(u=(y=w.next()).done)&&(o.push(y.value),!i||o.length!==i);u=!0);}catch(r){p=!0,g=r}finally{try{u||null==w.return||w.return()}finally{if(p)throw g}}return o}(r,i)||_unsupportedIterableToArray(r,i)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _toConsumableArray(r){return function _arrayWithoutHoles(r){if(Array.isArray(r))return _arrayLikeToArray(r)}(r)||function _iterableToArray(r){if("undefined"!=typeof Symbol&&Symbol.iterator in Object(r))return Array.from(r)}(r)||_unsupportedIterableToArray(r)||function _nonIterableSpread(){throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _unsupportedIterableToArray(r,i){if(r){if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);return"Object"===o&&r.constructor&&(o=r.constructor.name),"Map"===o||"Set"===o?Array.from(r):"Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o)?_arrayLikeToArray(r,i):void 0}}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}function _createForOfIteratorHelper(r,i){var o;if("undefined"==typeof Symbol||null==r[Symbol.iterator]){if(Array.isArray(r)||(o=_unsupportedIterableToArray(r))||i&&r&&"number"==typeof r.length){o&&(r=o);var u=0,p=function F(){};return{s:p,n:function n(){return u>=r.length?{done:!0}:{done:!1,value:r[u++]}},e:function e(r){throw r},f:p}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}var g,y=!0,w=!1;return{s:function s(){o=r[Symbol.iterator]()},n:function n(){var r=o.next();return y=r.done,r},e:function e(r){w=!0,g=r},f:function f(){try{y||null==o.return||o.return()}finally{if(w)throw g}}}}var I=function(){function PDFAbstractReference(){_classCallCheck(this,PDFAbstractReference)}return _createClass(PDFAbstractReference,[{key:"toString",value:function toString(){throw new Error("Must be implemented by subclasses")}}]),PDFAbstractReference}(),B=function(){function PDFTree(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};_classCallCheck(this,PDFTree),this._items={},this.limits="boolean"!=typeof r.limits||r.limits}return _createClass(PDFTree,[{key:"add",value:function add(r,i){return this._items[r]=i}},{key:"get",value:function get(r){return this._items[r]}},{key:"toString",value:function toString(){var r=this,i=Object.keys(this._items).sort((function(i,o){return r._compareKeys(i,o)})),o=["<<"];if(this.limits&&i.length>1){var u=i[0],p=i[i.length-1];o.push("  /Limits ".concat(W.convert([this._dataForKey(u),this._dataForKey(p)])))}o.push("  /".concat(this._keysName()," ["));var g,y=_createForOfIteratorHelper(i);try{for(y.s();!(g=y.n()).done;){var w=g.value;o.push("    ".concat(W.convert(this._dataForKey(w))," ").concat(W.convert(this._items[w])))}}catch(r){y.e(r)}finally{y.f()}return o.push("]"),o.push(">>"),o.join("\n")}},{key:"_compareKeys",value:function _compareKeys(){throw new Error("Must be implemented by subclasses")}},{key:"_keysName",value:function _keysName(){throw new Error("Must be implemented by subclasses")}},{key:"_dataForKey",value:function _dataForKey(){throw new Error("Must be implemented by subclasses")}}]),PDFTree}(),D=function pad(r,i){return(Array(i+1).join("0")+r).slice(-i)},R=/[\n\r\t\b\f()\\]/g,N={"\n":"\\n","\r":"\\r","\t":"\\t","\b":"\\b","\f":"\\f","\\":"\\\\","(":"\\(",")":"\\)"},W=function(){function PDFObject(){_classCallCheck(this,PDFObject)}return _createClass(PDFObject,null,[{key:"convert",value:function convert(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("string"==typeof r)return"/".concat(r);if(r instanceof String){for(var o,u=r,g=!1,y=0,w=u.length;y<w;y++)if(u.charCodeAt(y)>127){g=!0;break}return o=g?function swapBytes(r){var i=r.length;if(1&i)throw new Error("Buffer length must be even");for(var o=0,u=i-1;o<u;o+=2){var p=r[o];r[o]=r[o+1],r[o+1]=p}return r}(p.from("\ufeff".concat(u),"utf16le")):p.from(u.valueOf(),"ascii"),u=(u=i?i(o).toString("binary"):o.toString("binary")).replace(R,(function(r){return N[r]})),"(".concat(u,")")}if(p.isBuffer(r))return"<".concat(r.toString("hex"),">");if(r instanceof I||r instanceof B)return r.toString();if(r instanceof Date){var _="D:".concat(D(r.getUTCFullYear(),4))+D(r.getUTCMonth()+1,2)+D(r.getUTCDate(),2)+D(r.getUTCHours(),2)+D(r.getUTCMinutes(),2)+D(r.getUTCSeconds(),2)+"Z";return i&&(_=(_=i(p.from(_,"ascii")).toString("binary")).replace(R,(function(r){return N[r]}))),"(".concat(_,")")}if(Array.isArray(r)){var x=r.map((function(r){return PDFObject.convert(r,i)})).join(" ");return"[".concat(x,"]")}if("[object Object]"==={}.toString.call(r)){var k=["<<"];for(var P in r){var O=r[P];k.push("/".concat(P," ").concat(PDFObject.convert(O,i)))}return k.push(">>"),k.join("\n")}return"number"==typeof r?PDFObject.number(r):"".concat(r)}},{key:"number",value:function number(r){if(r>-1e21&&r<1e21)return Math.round(1e6*r)/1e6;throw new Error("unsupported number: ".concat(r))}}]),PDFObject}(),U=function(r){_inherits(PDFReference,r);var i=_createSuper(PDFReference);function PDFReference(r,o){var u,p=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};return _classCallCheck(this,PDFReference),(u=i.call(this)).document=r,u.id=o,u.data=p,u.gen=0,u.compress=u.document.compress&&!u.data.Filter,u.uncompressedLength=0,u.buffer=[],u}return _createClass(PDFReference,[{key:"write",value:function write(r){if(p.isBuffer(r)||(r=p.from(r+"\n","binary")),this.uncompressedLength+=r.length,null==this.data.Length&&(this.data.Length=0),this.buffer.push(r),this.data.Length+=r.length,this.compress)return this.data.Filter="FlateDecode"}},{key:"end",value:function end(r){return r&&this.write(r),this.finalize()}},{key:"finalize",value:function finalize(){this.offset=this.document._offset;var r=this.document._security?this.document._security.getEncryptFn(this.id,this.gen):null;this.buffer.length&&(this.buffer=p.concat(this.buffer),this.compress&&(this.buffer=y.default.deflateSync(this.buffer)),r&&(this.buffer=r(this.buffer)),this.data.Length=this.buffer.length),this.document._write("".concat(this.id," ").concat(this.gen," obj")),this.document._write(W.convert(this.data,r)),this.buffer.length&&(this.document._write("stream"),this.document._write(this.buffer),this.buffer=[],this.document._write("\nendstream")),this.document._write("endobj"),this.document._refEnd(this)}},{key:"toString",value:function toString(){return"".concat(this.id," ").concat(this.gen," R")}}]),PDFReference}(I),j={top:72,left:72,bottom:72,right:72},G={"4A0":[4767.87,6740.79],"2A0":[3370.39,4767.87],A0:[2383.94,3370.39],A1:[1683.78,2383.94],A2:[1190.55,1683.78],A3:[841.89,1190.55],A4:[595.28,841.89],A5:[419.53,595.28],A6:[297.64,419.53],A7:[209.76,297.64],A8:[147.4,209.76],A9:[104.88,147.4],A10:[73.7,104.88],B0:[2834.65,4008.19],B1:[2004.09,2834.65],B2:[1417.32,2004.09],B3:[1000.63,1417.32],B4:[708.66,1000.63],B5:[498.9,708.66],B6:[354.33,498.9],B7:[249.45,354.33],B8:[175.75,249.45],B9:[124.72,175.75],B10:[87.87,124.72],C0:[2599.37,3676.54],C1:[1836.85,2599.37],C2:[1298.27,1836.85],C3:[918.43,1298.27],C4:[649.13,918.43],C5:[459.21,649.13],C6:[323.15,459.21],C7:[229.61,323.15],C8:[161.57,229.61],C9:[113.39,161.57],C10:[79.37,113.39],RA0:[2437.8,3458.27],RA1:[1729.13,2437.8],RA2:[1218.9,1729.13],RA3:[864.57,1218.9],RA4:[609.45,864.57],SRA0:[2551.18,3628.35],SRA1:[1814.17,2551.18],SRA2:[1275.59,1814.17],SRA3:[907.09,1275.59],SRA4:[637.8,907.09],EXECUTIVE:[521.86,756],FOLIO:[612,936],LEGAL:[612,1008],LETTER:[612,792],TABLOID:[792,1224]},X=function(){function PDFPage(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};_classCallCheck(this,PDFPage),this.document=r,this.size=i.size||"letter",this.layout=i.layout||"portrait","number"==typeof i.margin?this.margins={top:i.margin,left:i.margin,bottom:i.margin,right:i.margin}:this.margins=i.margins||j;var o=Array.isArray(this.size)?this.size:G[this.size.toUpperCase()];this.width=o["portrait"===this.layout?0:1],this.height=o["portrait"===this.layout?1:0],this.content=this.document.ref(),this.resources=this.document.ref({ProcSet:["PDF","Text","ImageB","ImageC","ImageI"]}),this.dictionary=this.document.ref({Type:"Page",Parent:this.document._root.data.Pages,MediaBox:[0,0,this.width,this.height],Contents:this.content,Resources:this.resources}),this.markings=[]}return _createClass(PDFPage,[{key:"maxY",value:function maxY(){return this.height-this.margins.bottom}},{key:"write",value:function write(r){return this.content.write(r)}},{key:"end",value:function end(){return this.dictionary.end(),this.resources.end(),this.content.end()}},{key:"fonts",get:function get(){var r=this.resources.data;return null!=r.Font?r.Font:r.Font={}}},{key:"xobjects",get:function get(){var r=this.resources.data;return null!=r.XObject?r.XObject:r.XObject={}}},{key:"ext_gstates",get:function get(){var r=this.resources.data;return null!=r.ExtGState?r.ExtGState:r.ExtGState={}}},{key:"patterns",get:function get(){var r=this.resources.data;return null!=r.Pattern?r.Pattern:r.Pattern={}}},{key:"colorSpaces",get:function get(){var r=this.resources.data;return r.ColorSpace||(r.ColorSpace={})}},{key:"annotations",get:function get(){var r=this.dictionary.data;return null!=r.Annots?r.Annots:r.Annots=[]}},{key:"structParentTreeKey",get:function get(){var r=this.dictionary.data;return null!=r.StructParents?r.StructParents:r.StructParents=this.document.createStructParentTreeNextKey()}}]),PDFPage}(),K=function(r){_inherits(PDFNameTree,r);var i=_createSuper(PDFNameTree);function PDFNameTree(){return _classCallCheck(this,PDFNameTree),i.apply(this,arguments)}return _createClass(PDFNameTree,[{key:"_compareKeys",value:function _compareKeys(r,i){return r.localeCompare(i)}},{key:"_keysName",value:function _keysName(){return"Names"}},{key:"_dataForKey",value:function _dataForKey(r){return new String(r)}}]),PDFNameTree}(B);function inRange(r,i){if(r<i[0])return!1;for(var o=0,u=i.length/2;o<=u;){var p=Math.floor((o+u)/2),g=2*p;if(r>=i[g]&&r<=i[g+1])return!0;r>i[g+1]?o=p+1:u=p-1}return!1}var Y=[545,545,564,591,686,687,751,767,848,863,880,883,886,889,891,893,895,899,907,907,909,909,930,930,975,975,1015,1023,1159,1159,1231,1231,1270,1271,1274,1279,1296,1328,1367,1368,1376,1376,1416,1416,1419,1424,1442,1442,1466,1466,1477,1487,1515,1519,1525,1547,1549,1562,1564,1566,1568,1568,1595,1599,1622,1631,1774,1775,1791,1791,1806,1806,1837,1839,1867,1919,1970,2304,2308,2308,2362,2363,2382,2383,2389,2391,2417,2432,2436,2436,2445,2446,2449,2450,2473,2473,2481,2481,2483,2485,2490,2491,2493,2493,2501,2502,2505,2506,2510,2518,2520,2523,2526,2526,2532,2533,2555,2561,2563,2564,2571,2574,2577,2578,2601,2601,2609,2609,2612,2612,2615,2615,2618,2619,2621,2621,2627,2630,2633,2634,2638,2648,2653,2653,2655,2661,2677,2688,2692,2692,2700,2700,2702,2702,2706,2706,2729,2729,2737,2737,2740,2740,2746,2747,2758,2758,2762,2762,2766,2767,2769,2783,2785,2789,2800,2816,2820,2820,2829,2830,2833,2834,2857,2857,2865,2865,2868,2869,2874,2875,2884,2886,2889,2890,2894,2901,2904,2907,2910,2910,2914,2917,2929,2945,2948,2948,2955,2957,2961,2961,2966,2968,2971,2971,2973,2973,2976,2978,2981,2983,2987,2989,2998,2998,3002,3005,3011,3013,3017,3017,3022,3030,3032,3046,3059,3072,3076,3076,3085,3085,3089,3089,3113,3113,3124,3124,3130,3133,3141,3141,3145,3145,3150,3156,3159,3167,3170,3173,3184,3201,3204,3204,3213,3213,3217,3217,3241,3241,3252,3252,3258,3261,3269,3269,3273,3273,3278,3284,3287,3293,3295,3295,3298,3301,3312,3329,3332,3332,3341,3341,3345,3345,3369,3369,3386,3389,3396,3397,3401,3401,3406,3414,3416,3423,3426,3429,3440,3457,3460,3460,3479,3481,3506,3506,3516,3516,3518,3519,3527,3529,3531,3534,3541,3541,3543,3543,3552,3569,3573,3584,3643,3646,3676,3712,3715,3715,3717,3718,3721,3721,3723,3724,3726,3731,3736,3736,3744,3744,3748,3748,3750,3750,3752,3753,3756,3756,3770,3770,3774,3775,3781,3781,3783,3783,3790,3791,3802,3803,3806,3839,3912,3912,3947,3952,3980,3983,3992,3992,4029,4029,4045,4046,4048,4095,4130,4130,4136,4136,4139,4139,4147,4149,4154,4159,4186,4255,4294,4303,4345,4346,4348,4351,4442,4446,4515,4519,4602,4607,4615,4615,4679,4679,4681,4681,4686,4687,4695,4695,4697,4697,4702,4703,4743,4743,4745,4745,4750,4751,4783,4783,4785,4785,4790,4791,4799,4799,4801,4801,4806,4807,4815,4815,4823,4823,4847,4847,4879,4879,4881,4881,4886,4887,4895,4895,4935,4935,4955,4960,4989,5023,5109,5120,5751,5759,5789,5791,5873,5887,5901,5901,5909,5919,5943,5951,5972,5983,5997,5997,6001,6001,6004,6015,6109,6111,6122,6143,6159,6159,6170,6175,6264,6271,6314,7679,7836,7839,7930,7935,7958,7959,7966,7967,8006,8007,8014,8015,8024,8024,8026,8026,8028,8028,8030,8030,8062,8063,8117,8117,8133,8133,8148,8149,8156,8156,8176,8177,8181,8181,8191,8191,8275,8278,8280,8286,8292,8297,8306,8307,8335,8351,8370,8399,8427,8447,8507,8508,8524,8530,8580,8591,9167,9215,9255,9279,9291,9311,9471,9471,9748,9749,9752,9752,9854,9855,9866,9984,9989,9989,9994,9995,10024,10024,10060,10060,10062,10062,10067,10069,10071,10071,10079,10080,10133,10135,10160,10160,10175,10191,10220,10223,11008,11903,11930,11930,12020,12031,12246,12271,12284,12287,12352,12352,12439,12440,12544,12548,12589,12592,12687,12687,12728,12783,12829,12831,12868,12880,12924,12926,13004,13007,13055,13055,13175,13178,13278,13279,13311,13311,19894,19967,40870,40959,42125,42127,42183,44031,55204,55295,64046,64047,64107,64255,64263,64274,64280,64284,64311,64311,64317,64317,64319,64319,64322,64322,64325,64325,64434,64466,64832,64847,64912,64913,64968,64975,65021,65023,65040,65055,65060,65071,65095,65096,65107,65107,65127,65127,65132,65135,65141,65141,65277,65278,65280,65280,65471,65473,65480,65481,65488,65489,65496,65497,65501,65503,65511,65511,65519,65528,65536,66303,66335,66335,66340,66351,66379,66559,66598,66599,66638,118783,119030,119039,119079,119081,119262,119807,119893,119893,119965,119965,119968,119969,119971,119972,119975,119976,119981,119981,119994,119994,119996,119996,120001,120001,120004,120004,120070,120070,120075,120076,120085,120085,120093,120093,120122,120122,120127,120127,120133,120133,120135,120137,120145,120145,120484,120487,120778,120781,120832,131069,173783,194559,195102,196605,196608,262141,262144,327677,327680,393213,393216,458749,458752,524285,524288,589821,589824,655357,655360,720893,720896,786429,786432,851965,851968,917501,917504,917504,917506,917535,917632,983037],J=function isUnassignedCodePoint(r){return inRange(r,Y)},$=[173,173,847,847,6150,6150,6155,6155,6156,6156,6157,6157,8203,8203,8204,8204,8205,8205,8288,8288,65024,65024,65025,65025,65026,65026,65027,65027,65028,65028,65029,65029,65030,65030,65031,65031,65032,65032,65033,65033,65034,65034,65035,65035,65036,65036,65037,65037,65038,65038,65039,65039,65279,65279],ee=[160,160,5760,5760,8192,8192,8193,8193,8194,8194,8195,8195,8196,8196,8197,8197,8198,8198,8199,8199,8200,8200,8201,8201,8202,8202,8203,8203,8239,8239,8287,8287,12288,12288],te=[128,159,1757,1757,1807,1807,6158,6158,8204,8204,8205,8205,8232,8232,8233,8233,8288,8288,8289,8289,8290,8290,8291,8291,8298,8303,65279,65279,65529,65532,119155,119162],re=[64976,65007,65534,65535,131070,131071,196606,196607,262142,262143,327678,327679,393214,393215,458750,458751,524286,524287,589822,589823,655358,655359,720894,720895,786430,786431,851966,851967,917502,917503,983038,983039,1114110,1114111],ne=[0,31,127,127,832,832,833,833,8206,8206,8207,8207,8234,8234,8235,8235,8236,8236,8237,8237,8238,8238,8298,8298,8299,8299,8300,8300,8301,8301,8302,8302,8303,8303,12272,12283,55296,57343,57344,63743,65529,65529,65530,65530,65531,65531,65532,65532,65533,65533,917505,917505,917536,917631,983040,1048573,1048576,1114109],ie=function isProhibitedCharacter(r){return inRange(r,ee)||inRange(r,ne)||inRange(r,te)||inRange(r,re)},oe=[1470,1470,1472,1472,1475,1475,1488,1514,1520,1524,1563,1563,1567,1567,1569,1594,1600,1610,1645,1647,1649,1749,1757,1757,1765,1766,1786,1790,1792,1805,1808,1808,1810,1836,1920,1957,1969,1969,8207,8207,64285,64285,64287,64296,64298,64310,64312,64316,64318,64318,64320,64321,64323,64324,64326,64433,64467,64829,64848,64911,64914,64967,65008,65020,65136,65140,65142,65276],ae=function isBidirectionalRAL(r){return inRange(r,oe)},se=[65,90,97,122,170,170,181,181,186,186,192,214,216,246,248,544,546,563,592,685,688,696,699,705,720,721,736,740,750,750,890,890,902,902,904,906,908,908,910,929,931,974,976,1013,1024,1154,1162,1230,1232,1269,1272,1273,1280,1295,1329,1366,1369,1375,1377,1415,1417,1417,2307,2307,2309,2361,2365,2368,2377,2380,2384,2384,2392,2401,2404,2416,2434,2435,2437,2444,2447,2448,2451,2472,2474,2480,2482,2482,2486,2489,2494,2496,2503,2504,2507,2508,2519,2519,2524,2525,2527,2529,2534,2545,2548,2554,2565,2570,2575,2576,2579,2600,2602,2608,2610,2611,2613,2614,2616,2617,2622,2624,2649,2652,2654,2654,2662,2671,2674,2676,2691,2691,2693,2699,2701,2701,2703,2705,2707,2728,2730,2736,2738,2739,2741,2745,2749,2752,2761,2761,2763,2764,2768,2768,2784,2784,2790,2799,2818,2819,2821,2828,2831,2832,2835,2856,2858,2864,2866,2867,2870,2873,2877,2878,2880,2880,2887,2888,2891,2892,2903,2903,2908,2909,2911,2913,2918,2928,2947,2947,2949,2954,2958,2960,2962,2965,2969,2970,2972,2972,2974,2975,2979,2980,2984,2986,2990,2997,2999,3001,3006,3007,3009,3010,3014,3016,3018,3020,3031,3031,3047,3058,3073,3075,3077,3084,3086,3088,3090,3112,3114,3123,3125,3129,3137,3140,3168,3169,3174,3183,3202,3203,3205,3212,3214,3216,3218,3240,3242,3251,3253,3257,3262,3262,3264,3268,3271,3272,3274,3275,3285,3286,3294,3294,3296,3297,3302,3311,3330,3331,3333,3340,3342,3344,3346,3368,3370,3385,3390,3392,3398,3400,3402,3404,3415,3415,3424,3425,3430,3439,3458,3459,3461,3478,3482,3505,3507,3515,3517,3517,3520,3526,3535,3537,3544,3551,3570,3572,3585,3632,3634,3635,3648,3654,3663,3675,3713,3714,3716,3716,3719,3720,3722,3722,3725,3725,3732,3735,3737,3743,3745,3747,3749,3749,3751,3751,3754,3755,3757,3760,3762,3763,3773,3773,3776,3780,3782,3782,3792,3801,3804,3805,3840,3863,3866,3892,3894,3894,3896,3896,3902,3911,3913,3946,3967,3967,3973,3973,3976,3979,4030,4037,4039,4044,4047,4047,4096,4129,4131,4135,4137,4138,4140,4140,4145,4145,4152,4152,4160,4183,4256,4293,4304,4344,4347,4347,4352,4441,4447,4514,4520,4601,4608,4614,4616,4678,4680,4680,4682,4685,4688,4694,4696,4696,4698,4701,4704,4742,4744,4744,4746,4749,4752,4782,4784,4784,4786,4789,4792,4798,4800,4800,4802,4805,4808,4814,4816,4822,4824,4846,4848,4878,4880,4880,4882,4885,4888,4894,4896,4934,4936,4954,4961,4988,5024,5108,5121,5750,5761,5786,5792,5872,5888,5900,5902,5905,5920,5937,5941,5942,5952,5969,5984,5996,5998,6e3,6016,6070,6078,6085,6087,6088,6100,6106,6108,6108,6112,6121,6160,6169,6176,6263,6272,6312,7680,7835,7840,7929,7936,7957,7960,7965,7968,8005,8008,8013,8016,8023,8025,8025,8027,8027,8029,8029,8031,8061,8064,8116,8118,8124,8126,8126,8130,8132,8134,8140,8144,8147,8150,8155,8160,8172,8178,8180,8182,8188,8206,8206,8305,8305,8319,8319,8450,8450,8455,8455,8458,8467,8469,8469,8473,8477,8484,8484,8486,8486,8488,8488,8490,8493,8495,8497,8499,8505,8509,8511,8517,8521,8544,8579,9014,9082,9109,9109,9372,9449,12293,12295,12321,12329,12337,12341,12344,12348,12353,12438,12445,12447,12449,12538,12540,12543,12549,12588,12593,12686,12688,12727,12784,12828,12832,12867,12896,12923,12927,12976,12992,13003,13008,13054,13056,13174,13179,13277,13280,13310,13312,19893,19968,40869,40960,42124,44032,55203,55296,64045,64048,64106,64256,64262,64275,64279,65313,65338,65345,65370,65382,65470,65474,65479,65482,65487,65490,65495,65498,65500,66304,66334,66336,66339,66352,66378,66560,66597,66600,66637,118784,119029,119040,119078,119082,119142,119146,119154,119171,119172,119180,119209,119214,119261,119808,119892,119894,119964,119966,119967,119970,119970,119973,119974,119977,119980,119982,119993,119995,119995,119997,12e4,120002,120003,120005,120069,120071,120074,120077,120084,120086,120092,120094,120121,120123,120126,120128,120132,120134,120134,120138,120144,120146,120483,120488,120777,131072,173782,194560,195101,983040,1048573,1048576,1114109],ce=function isBidirectionalL(r){return inRange(r,se)},le=function isNonASCIISpaceCharacter(r){return inRange(r,ee)},ue=function isCommonlyMappedToNothing(r){return inRange(r,$)},fe=function getCodePoint(r){return r.codePointAt(0)},he=function first(r){return r[0]},de=function last(r){return r[r.length-1]};function toCodePoints(r){for(var i=[],o=r.length,u=0;u<o;u+=1){var p=r.charCodeAt(u);if(p>=55296&&p<=56319&&o>u+1){var g=r.charCodeAt(u+1);if(g>=56320&&g<=57343){i.push(1024*(p-55296)+g-56320+65536),u+=1;continue}}i.push(p)}return i}var pe=function(){function PDFSecurity(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if(_classCallCheck(this,PDFSecurity),!i.ownerPassword&&!i.userPassword)throw new Error("None of owner password and user password is defined.");this.document=r,this._setupEncryption(i)}return _createClass(PDFSecurity,null,[{key:"generateFileID",value:function generateFileID(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i="".concat(r.CreationDate.getTime(),"\n");for(var o in r)r.hasOwnProperty(o)&&(i+="".concat(o,": ").concat(r[o].valueOf(),"\n"));return wordArrayToBuffer(w.default.MD5(i))}},{key:"generateRandomWordArray",value:function generateRandomWordArray(r){return w.default.lib.WordArray.random(r)}},{key:"create",value:function create(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return i.ownerPassword||i.userPassword?new PDFSecurity(r,i):null}}]),_createClass(PDFSecurity,[{key:"_setupEncryption",value:function _setupEncryption(r){switch(r.pdfVersion){case"1.4":case"1.5":this.version=2;break;case"1.6":case"1.7":this.version=4;break;case"1.7ext3":this.version=5;break;default:this.version=1}var i={Filter:"Standard"};switch(this.version){case 1:case 2:case 4:this._setupEncryptionV1V2V4(this.version,i,r);break;case 5:this._setupEncryptionV5(i,r)}this.dictionary=this.document.ref(i)}},{key:"_setupEncryptionV1V2V4",value:function _setupEncryptionV1V2V4(r,i,o){var u,p;switch(r){case 1:u=2,this.keyBits=40,p=function getPermissionsR2(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-64;r.printing&&(i|=4);r.modifying&&(i|=8);r.copying&&(i|=16);r.annotating&&(i|=32);return i}(o.permissions);break;case 2:u=3,this.keyBits=128,p=getPermissionsR3(o.permissions);break;case 4:u=4,this.keyBits=128,p=getPermissionsR3(o.permissions)}var g,y=processPasswordR2R3R4(o.userPassword),_=o.ownerPassword?processPasswordR2R3R4(o.ownerPassword):y,x=function getOwnerPasswordR2R3R4(r,i,o,u){for(var p=u,g=r>=3?51:1,y=0;y<g;y++)p=w.default.MD5(p);var _=p.clone();_.sigBytes=i/8;var x=o;g=r>=3?20:1;for(var k=0;k<g;k++){for(var P=Math.ceil(_.sigBytes/4),O=0;O<P;O++)_.words[O]=p.words[O]^(k|k<<8|k<<16|k<<24);x=w.default.RC4.encrypt(x,_).ciphertext}return x}(u,this.keyBits,y,_);this.encryptionKey=function getEncryptionKeyR2R3R4(r,i,o,u,p,g){for(var y=u.clone().concat(p).concat(w.default.lib.WordArray.create([lsbFirstWord(g)],4)).concat(w.default.lib.WordArray.create(o)),_=r>=3?51:1,x=0;x<_;x++)(y=w.default.MD5(y)).sigBytes=i/8;return y}(u,this.keyBits,this.document._id,y,x,p),g=2===u?function getUserPasswordR2(r){return w.default.RC4.encrypt(processPasswordR2R3R4(),r).ciphertext}(this.encryptionKey):function getUserPasswordR3R4(r,i){for(var o=i.clone(),u=w.default.MD5(processPasswordR2R3R4().concat(w.default.lib.WordArray.create(r))),p=0;p<20;p++){for(var g=Math.ceil(o.sigBytes/4),y=0;y<g;y++)o.words[y]=i.words[y]^(p|p<<8|p<<16|p<<24);u=w.default.RC4.encrypt(u,o).ciphertext}return u.concat(w.default.lib.WordArray.create(null,16))}(this.document._id,this.encryptionKey),i.V=r,r>=2&&(i.Length=this.keyBits),4===r&&(i.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV2",Length:this.keyBits/8}},i.StmF="StdCF",i.StrF="StdCF"),i.R=u,i.O=wordArrayToBuffer(x),i.U=wordArrayToBuffer(g),i.P=p}},{key:"_setupEncryptionV5",value:function _setupEncryptionV5(r,i){this.keyBits=256;var o=getPermissionsR3(i.permissions),u=processPasswordR5(i.userPassword),p=i.ownerPassword?processPasswordR5(i.ownerPassword):u;this.encryptionKey=function getEncryptionKeyR5(r){return r(32)}(PDFSecurity.generateRandomWordArray);var g=function getUserPasswordR5(r,i){var o=i(8),u=i(8);return w.default.SHA256(r.clone().concat(o)).concat(o).concat(u)}(u,PDFSecurity.generateRandomWordArray),y=function getUserEncryptionKeyR5(r,i,o){var u=w.default.SHA256(r.clone().concat(i)),p={mode:w.default.mode.CBC,padding:w.default.pad.NoPadding,iv:w.default.lib.WordArray.create(null,16)};return w.default.AES.encrypt(o,u,p).ciphertext}(u,w.default.lib.WordArray.create(g.words.slice(10,12),8),this.encryptionKey),_=function getOwnerPasswordR5(r,i,o){var u=o(8),p=o(8);return w.default.SHA256(r.clone().concat(u).concat(i)).concat(u).concat(p)}(p,g,PDFSecurity.generateRandomWordArray),x=function getOwnerEncryptionKeyR5(r,i,o,u){var p=w.default.SHA256(r.clone().concat(i).concat(o)),g={mode:w.default.mode.CBC,padding:w.default.pad.NoPadding,iv:w.default.lib.WordArray.create(null,16)};return w.default.AES.encrypt(u,p,g).ciphertext}(p,w.default.lib.WordArray.create(_.words.slice(10,12),8),g,this.encryptionKey),k=function getEncryptedPermissionsR5(r,i,o){var u=w.default.lib.WordArray.create([lsbFirstWord(r),4294967295,1415668834],12).concat(o(4)),p={mode:w.default.mode.ECB,padding:w.default.pad.NoPadding};return w.default.AES.encrypt(u,i,p).ciphertext}(o,this.encryptionKey,PDFSecurity.generateRandomWordArray);r.V=5,r.Length=this.keyBits,r.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV3",Length:this.keyBits/8}},r.StmF="StdCF",r.StrF="StdCF",r.R=5,r.O=wordArrayToBuffer(_),r.OE=wordArrayToBuffer(x),r.U=wordArrayToBuffer(g),r.UE=wordArrayToBuffer(y),r.P=o,r.Perms=wordArrayToBuffer(k)}},{key:"getEncryptFn",value:function getEncryptFn(r,i){var o,u;if(this.version<5&&(o=this.encryptionKey.clone().concat(w.default.lib.WordArray.create([(255&r)<<24|(65280&r)<<8|r>>8&65280|255&i,(65280&i)<<16],5))),1===this.version||2===this.version){var p=w.default.MD5(o);return p.sigBytes=Math.min(16,this.keyBits/8+5),function(r){return wordArrayToBuffer(w.default.RC4.encrypt(w.default.lib.WordArray.create(r),p).ciphertext)}}u=4===this.version?w.default.MD5(o.concat(w.default.lib.WordArray.create([1933667412],4))):this.encryptionKey;var g=PDFSecurity.generateRandomWordArray(16),y={mode:w.default.mode.CBC,padding:w.default.pad.Pkcs7,iv:g};return function(r){return wordArrayToBuffer(g.clone().concat(w.default.AES.encrypt(w.default.lib.WordArray.create(r),u,y).ciphertext))}}},{key:"end",value:function end(){this.dictionary.end()}}]),PDFSecurity}();function getPermissionsR3(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-3904;return"lowResolution"===r.printing&&(i|=4),"highResolution"===r.printing&&(i|=2052),r.modifying&&(i|=8),r.copying&&(i|=16),r.annotating&&(i|=32),r.fillingForms&&(i|=256),r.contentAccessibility&&(i|=512),r.documentAssembly&&(i|=1024),i}function processPasswordR2R3R4(){for(var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"",i=p.alloc(32),o=r.length,u=0;u<o&&u<32;){var g=r.charCodeAt(u);if(g>255)throw new Error("Password contains one or more invalid characters.");i[u]=g,u++}for(;u<32;)i[u]=_e[u-o],u++;return w.default.lib.WordArray.create(i)}function processPasswordR5(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"";r=unescape(encodeURIComponent(function saslprep(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if("string"!=typeof r)throw new TypeError("Expected string.");if(0===r.length)return"";var o=toCodePoints(r).map((function(r){return le(r)?32:r})).filter((function(r){return!ue(r)})),u=String.fromCodePoint.apply(null,o).normalize("NFKC"),p=toCodePoints(u);if(p.some(ie))throw new Error("Prohibited character, see https://tools.ietf.org/html/rfc4013#section-2.3");if(!0!==i.allowUnassigned&&p.some(J))throw new Error("Unassigned code point, see https://tools.ietf.org/html/rfc4013#section-2.5");var g=p.some(ae),y=p.some(ce);if(g&&y)throw new Error("String must not contain RandALCat and LCat at the same time, see https://tools.ietf.org/html/rfc3454#section-6");var w=ae(fe(he(u))),_=ae(fe(de(u)));if(g&&(!w||!_))throw new Error("Bidirectional RandALCat character must be the first and the last character of the string, see https://tools.ietf.org/html/rfc3454#section-6");return u}(r)));for(var i=Math.min(127,r.length),o=p.alloc(i),u=0;u<i;u++)o[u]=r.charCodeAt(u);return w.default.lib.WordArray.create(o)}function lsbFirstWord(r){return(255&r)<<24|(65280&r)<<8|r>>8&65280|r>>24&255}function wordArrayToBuffer(r){for(var i=[],o=0;o<r.sigBytes;o++)i.push(r.words[Math.floor(o/4)]>>8*(3-o%4)&255);return p.from(i)}var ge,ye,ve,me,be,we,_e=[40,191,78,94,78,117,138,65,100,0,78,86,255,250,1,8,46,46,0,182,208,104,62,128,47,12,169,254,100,83,105,122],Se=W.number,xe=function(){function PDFGradient(r){_classCallCheck(this,PDFGradient),this.doc=r,this.stops=[],this.embedded=!1,this.transform=[1,0,0,1,0,0]}return _createClass(PDFGradient,[{key:"stop",value:function stop(r,i,o){if(null==o&&(o=1),i=this.doc._normalizeColor(i),0===this.stops.length)if(3===i.length)this._colorSpace="DeviceRGB";else if(4===i.length)this._colorSpace="DeviceCMYK";else{if(1!==i.length)throw new Error("Unknown color space");this._colorSpace="DeviceGray"}else if("DeviceRGB"===this._colorSpace&&3!==i.length||"DeviceCMYK"===this._colorSpace&&4!==i.length||"DeviceGray"===this._colorSpace&&1!==i.length)throw new Error("All gradient stops must use the same color space");return o=Math.max(0,Math.min(1,o)),this.stops.push([r,i,o]),this}},{key:"setTransform",value:function setTransform(r,i,o,u,p,g){return this.transform=[r,i,o,u,p,g],this}},{key:"embed",value:function embed(r){var i,o=this.stops.length;if(0!==o){this.embedded=!0,this.matrix=r;var u=this.stops[o-1];u[0]<1&&this.stops.push([1,u[1],u[2]]);for(var p=[],g=[],y=[],w=0;w<o-1;w++)g.push(0,1),w+2!==o&&p.push(this.stops[w+1][0]),i=this.doc.ref({FunctionType:2,Domain:[0,1],C0:this.stops[w+0][1],C1:this.stops[w+1][1],N:1}),y.push(i),i.end();1===o?i=y[0]:(i=this.doc.ref({FunctionType:3,Domain:[0,1],Functions:y,Bounds:p,Encode:g})).end(),this.id="Sh".concat(++this.doc._gradCount);var _=this.shader(i);_.end();var x=this.doc.ref({Type:"Pattern",PatternType:2,Shading:_,Matrix:this.matrix.map(Se)});if(x.end(),this.stops.some((function(r){return r[2]<1}))){var k=this.opacityGradient();k._colorSpace="DeviceGray";var P,O=_createForOfIteratorHelper(this.stops);try{for(O.s();!(P=O.n()).done;){var I=P.value;k.stop(I[0],[I[2]])}}catch(r){O.e(r)}finally{O.f()}k=k.embed(this.matrix);var B=[0,0,this.doc.page.width,this.doc.page.height],D=this.doc.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:B,Group:{Type:"Group",S:"Transparency",CS:"DeviceGray"},Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:k}}});D.write("/Pattern cs /Sh1 scn"),D.end("".concat(B.join(" ")," re f"));var R=this.doc.ref({Type:"ExtGState",SMask:{Type:"Mask",S:"Luminosity",G:D}});R.end();var N=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:B,XStep:B[2],YStep:B[3],Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:x},ExtGState:{Gs1:R}}});N.write("/Gs1 gs /Pattern cs /Sh1 scn"),N.end("".concat(B.join(" ")," re f")),this.doc.page.patterns[this.id]=N}else this.doc.page.patterns[this.id]=x;return x}}},{key:"apply",value:function apply(r){var i=_slicedToArray(this.doc._ctm,6),o=i[0],u=i[1],p=i[2],g=i[3],y=i[4],w=i[5],_=_slicedToArray(this.transform,6),x=_[0],k=_[1],P=_[2],O=_[3],I=_[4],B=_[5],D=[o*x+p*k,u*x+g*k,o*P+p*O,u*P+g*O,o*I+p*B+y,u*I+g*B+w];this.embedded&&D.join(" ")===this.matrix.join(" ")||this.embed(D),this.doc._setColorSpace("Pattern",r);var R=r?"SCN":"scn";return this.doc.addContent("/".concat(this.id," ").concat(R))}}]),PDFGradient}(),Ce={PDFGradient:xe,PDFLinearGradient:function(r){_inherits(PDFLinearGradient,r);var i=_createSuper(PDFLinearGradient);function PDFLinearGradient(r,o,u,p,g){var y;return _classCallCheck(this,PDFLinearGradient),(y=i.call(this,r)).x1=o,y.y1=u,y.x2=p,y.y2=g,y}return _createClass(PDFLinearGradient,[{key:"shader",value:function shader(r){return this.doc.ref({ShadingType:2,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.x2,this.y2],Function:r,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFLinearGradient(this.doc,this.x1,this.y1,this.x2,this.y2)}}]),PDFLinearGradient}(xe),PDFRadialGradient:function(r){_inherits(PDFRadialGradient,r);var i=_createSuper(PDFRadialGradient);function PDFRadialGradient(r,o,u,p,g,y,w){var _;return _classCallCheck(this,PDFRadialGradient),(_=i.call(this,r)).doc=r,_.x1=o,_.y1=u,_.r1=p,_.x2=g,_.y2=y,_.r2=w,_}return _createClass(PDFRadialGradient,[{key:"shader",value:function shader(r){return this.doc.ref({ShadingType:3,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.r1,this.x2,this.y2,this.r2],Function:r,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFRadialGradient(this.doc,this.x1,this.y1,this.r1,this.x2,this.y2,this.r2)}}]),PDFRadialGradient}(xe)},ke=["DeviceCMYK","DeviceRGB"],Pe=function(){function PDFTilingPattern(r,i,o,u,p){_classCallCheck(this,PDFTilingPattern),this.doc=r,this.bBox=i,this.xStep=o,this.yStep=u,this.stream=p}return _createClass(PDFTilingPattern,[{key:"createPattern",value:function createPattern(){var r=this.doc.ref();r.end();var i=_slicedToArray(this.doc._ctm,6),o=i[0],u=i[1],p=i[2],g=i[3],y=[1*o+0*p,1*u+0*g,0*o+1*p,0*u+1*g,0*o+0*p+i[4],0*u+0*g+i[5]],w=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:2,TilingType:2,BBox:this.bBox,XStep:this.xStep,YStep:this.yStep,Matrix:y.map((function(r){return+r.toFixed(5)})),Resources:r});return w.end(this.stream),w}},{key:"embedPatternColorSpaces",value:function embedPatternColorSpaces(){var r=this;ke.forEach((function(i){var o=r.getPatternColorSpaceId(i);if(!r.doc.page.colorSpaces[o]){var u=r.doc.ref(["Pattern",i]);u.end(),r.doc.page.colorSpaces[o]=u}}))}},{key:"getPatternColorSpaceId",value:function getPatternColorSpaceId(r){return"CsP".concat(r)}},{key:"embed",value:function embed(){this.id||(this.doc._patternCount=this.doc._patternCount+1,this.id="P"+this.doc._patternCount,this.pattern=this.createPattern()),this.doc.page.patterns[this.id]||(this.doc.page.patterns[this.id]=this.pattern)}},{key:"apply",value:function apply(r,i){this.embedPatternColorSpaces(),this.embed();var o=this.doc._normalizeColor(i);if(!o)throw Error("invalid pattern color. (value: ".concat(i,")"));var u=this.getPatternColorSpaceId(this.doc._getColorSpace(o));this.doc._setColorSpace(u,r);var p=r?"SCN":"scn";return this.doc.addContent("".concat(o.join(" ")," /").concat(this.id," ").concat(p))}}]),PDFTilingPattern}(),Ae=Ce.PDFGradient,Ee=Ce.PDFLinearGradient,Te=Ce.PDFRadialGradient,Oe={PDFTilingPattern:Pe}.PDFTilingPattern,Ie={initColor:function initColor(){return this._opacityRegistry={},this._opacityCount=0,this._patternCount=0,this._gradCount=0},_normalizeColor:function _normalizeColor(r){if("string"==typeof r)if("#"===r.charAt(0)){4===r.length&&(r=r.replace(/#([0-9A-F])([0-9A-F])([0-9A-F])/i,"#$1$1$2$2$3$3"));var i=parseInt(r.slice(1),16);r=[i>>16,i>>8&255,255&i]}else Be[r]&&(r=Be[r]);return Array.isArray(r)?(3===r.length?r=r.map((function(r){return r/255})):4===r.length&&(r=r.map((function(r){return r/100}))),r):null},_setColor:function _setColor(r,i){return r instanceof Ae?(r.apply(i),!0):Array.isArray(r)&&r[0]instanceof Oe?(r[0].apply(i,r[1]),!0):this._setColorCore(r,i)},_setColorCore:function _setColorCore(r,i){if(!(r=this._normalizeColor(r)))return!1;var o=i?"SCN":"scn",u=this._getColorSpace(r);return this._setColorSpace(u,i),r=r.join(" "),this.addContent("".concat(r," ").concat(o)),!0},_setColorSpace:function _setColorSpace(r,i){var o=i?"CS":"cs";return this.addContent("/".concat(r," ").concat(o))},_getColorSpace:function _getColorSpace(r){return 4===r.length?"DeviceCMYK":"DeviceRGB"},fillColor:function fillColor(r,i){return this._setColor(r,!1)&&this.fillOpacity(i),this._fillColor=[r,i],this},strokeColor:function strokeColor(r,i){return this._setColor(r,!0)&&this.strokeOpacity(i),this},opacity:function opacity(r){return this._doOpacity(r,r),this},fillOpacity:function fillOpacity(r){return this._doOpacity(r,null),this},strokeOpacity:function strokeOpacity(r){return this._doOpacity(null,r),this},_doOpacity:function _doOpacity(r,i){var o,u;if(null!=r||null!=i){null!=r&&(r=Math.max(0,Math.min(1,r))),null!=i&&(i=Math.max(0,Math.min(1,i)));var p="".concat(r,"_").concat(i);if(this._opacityRegistry[p]){var g=_slicedToArray(this._opacityRegistry[p],2);o=g[0],u=g[1]}else{o={Type:"ExtGState"},null!=r&&(o.ca=r),null!=i&&(o.CA=i),(o=this.ref(o)).end();var y=++this._opacityCount;u="Gs".concat(y),this._opacityRegistry[p]=[o,u]}return this.page.ext_gstates[u]=o,this.addContent("/".concat(u," gs"))}},linearGradient:function linearGradient(r,i,o,u){return new Ee(this,r,i,o,u)},radialGradient:function radialGradient(r,i,o,u,p,g){return new Te(this,r,i,o,u,p,g)},pattern:function pattern(r,i,o,u){return new Oe(this,r,i,o,u)}},Be={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgreen:[0,100,0],darkgrey:[169,169,169],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0],yellowgreen:[154,205,50]};ge=ye=ve=me=be=we=0;var Fe={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},Le={M:function M(r,i){return ge=i[0],ye=i[1],ve=me=null,be=ge,we=ye,r.moveTo(ge,ye)},m:function m(r,i){return ge+=i[0],ye+=i[1],ve=me=null,be=ge,we=ye,r.moveTo(ge,ye)},C:function C(r,i){return ge=i[4],ye=i[5],ve=i[2],me=i[3],r.bezierCurveTo.apply(r,_toConsumableArray(i))},c:function c(r,i){return r.bezierCurveTo(i[0]+ge,i[1]+ye,i[2]+ge,i[3]+ye,i[4]+ge,i[5]+ye),ve=ge+i[2],me=ye+i[3],ge+=i[4],ye+=i[5]},S:function S(r,i){return null===ve&&(ve=ge,me=ye),r.bezierCurveTo(ge-(ve-ge),ye-(me-ye),i[0],i[1],i[2],i[3]),ve=i[0],me=i[1],ge=i[2],ye=i[3]},s:function s(r,i){return null===ve&&(ve=ge,me=ye),r.bezierCurveTo(ge-(ve-ge),ye-(me-ye),ge+i[0],ye+i[1],ge+i[2],ye+i[3]),ve=ge+i[0],me=ye+i[1],ge+=i[2],ye+=i[3]},Q:function Q(r,i){return ve=i[0],me=i[1],ge=i[2],ye=i[3],r.quadraticCurveTo(i[0],i[1],ge,ye)},q:function q(r,i){return r.quadraticCurveTo(i[0]+ge,i[1]+ye,i[2]+ge,i[3]+ye),ve=ge+i[0],me=ye+i[1],ge+=i[2],ye+=i[3]},T:function T(r,i){return null===ve?(ve=ge,me=ye):(ve=ge-(ve-ge),me=ye-(me-ye)),r.quadraticCurveTo(ve,me,i[0],i[1]),ve=ge-(ve-ge),me=ye-(me-ye),ge=i[0],ye=i[1]},t:function t(r,i){return null===ve?(ve=ge,me=ye):(ve=ge-(ve-ge),me=ye-(me-ye)),r.quadraticCurveTo(ve,me,ge+i[0],ye+i[1]),ge+=i[0],ye+=i[1]},A:function A(r,i){return De(r,ge,ye,i),ge=i[5],ye=i[6]},a:function a(r,i){return i[5]+=ge,i[6]+=ye,De(r,ge,ye,i),ge=i[5],ye=i[6]},L:function L(r,i){return ge=i[0],ye=i[1],ve=me=null,r.lineTo(ge,ye)},l:function l(r,i){return ge+=i[0],ye+=i[1],ve=me=null,r.lineTo(ge,ye)},H:function H(r,i){return ge=i[0],ve=me=null,r.lineTo(ge,ye)},h:function h(r,i){return ge+=i[0],ve=me=null,r.lineTo(ge,ye)},V:function V(r,i){return ye=i[0],ve=me=null,r.lineTo(ge,ye)},v:function v(r,i){return ye+=i[0],ve=me=null,r.lineTo(ge,ye)},Z:function Z(r){return r.closePath(),ge=be,ye=we},z:function z(r){return r.closePath(),ge=be,ye=we}},De=function solveArc(r,i,o,u){var p,g=_slicedToArray(u,7),y=g[0],w=g[1],_=g[2],x=g[3],k=g[4],P=g[5],O=g[6],I=_createForOfIteratorHelper(Re(P,O,y,w,x,k,_,i,o));try{for(I.s();!(p=I.n()).done;){var B=p.value,D=Me.apply(void 0,_toConsumableArray(B));r.bezierCurveTo.apply(r,_toConsumableArray(D))}}catch(r){I.e(r)}finally{I.f()}},Re=function arcToSegments(r,i,o,u,p,g,y,w,_){var x=y*(Math.PI/180),k=Math.sin(x),P=Math.cos(x);o=Math.abs(o),u=Math.abs(u);var O=(ve=P*(w-r)*.5+k*(_-i)*.5)*ve/(o*o)+(me=P*(_-i)*.5-k*(w-r)*.5)*me/(u*u);O>1&&(o*=O=Math.sqrt(O),u*=O);var I=P/o,B=k/o,D=-k/u,R=P/u,N=I*w+B*_,W=D*w+R*_,U=I*r+B*i,j=D*r+R*i,G=1/((U-N)*(U-N)+(j-W)*(j-W))-.25;G<0&&(G=0);var X=Math.sqrt(G);g===p&&(X=-X);var K=.5*(N+U)-X*(j-W),Y=.5*(W+j)+X*(U-N),J=Math.atan2(W-Y,N-K),$=Math.atan2(j-Y,U-K)-J;$<0&&1===g?$+=2*Math.PI:$>0&&0===g&&($-=2*Math.PI);for(var ee=Math.ceil(Math.abs($/(.5*Math.PI+.001))),te=[],re=0;re<ee;re++){var ne=J+re*$/ee,ie=J+(re+1)*$/ee;te[re]=[K,Y,ne,ie,o,u,k,P]}return te},Me=function segmentToBezier(r,i,o,u,p,g,y,w){var _=w*p,x=-y*g,k=y*p,P=w*g,O=.5*(u-o),I=8/3*Math.sin(.5*O)*Math.sin(.5*O)/Math.sin(O),B=r+Math.cos(o)-I*Math.sin(o),D=i+Math.sin(o)+I*Math.cos(o),R=r+Math.cos(u),N=i+Math.sin(u),W=R+I*Math.sin(u),U=N-I*Math.cos(u);return[_*B+x*D,k*B+P*D,_*W+x*U,k*W+P*U,_*R+x*N,k*R+P*N]},Ne=function(){function SVGPath(){_classCallCheck(this,SVGPath)}return _createClass(SVGPath,null,[{key:"apply",value:function apply(r,i){var o=function parse(r){var i,o,u=[],p=[],g="",y=!1,w=0,_=_createForOfIteratorHelper(r);try{for(_.s();!(o=_.n()).done;){var x=o.value;if(null!=Fe[x])w=Fe[x],i&&(g.length>0&&(p[p.length]=+g),u[u.length]={cmd:i,args:p},p=[],g="",y=!1),i=x;else if([" ",","].includes(x)||"-"===x&&g.length>0&&"e"!==g[g.length-1]||"."===x&&y){if(0===g.length)continue;p.length===w?(u[u.length]={cmd:i,args:p},p=[+g],"M"===i&&(i="L"),"m"===i&&(i="l")):p[p.length]=+g,y="."===x,g=["-","."].includes(x)?x:""}else g+=x,"."===x&&(y=!0)}}catch(r){_.e(r)}finally{_.f()}return g.length>0&&(p.length===w?(u[u.length]={cmd:i,args:p},p=[+g],"M"===i&&(i="L"),"m"===i&&(i="l")):p[p.length]=+g),u[u.length]={cmd:i,args:p},u}(i);!function apply(r,i){ge=ye=ve=me=be=we=0;for(var o=0;o<r.length;o++){var u=r[o];"function"==typeof Le[u.cmd]&&Le[u.cmd](i,u.args)}}(o,r)}}]),SVGPath}(),We=W.number,ze=(Math.sqrt(2)-1)/3*4,Ue={initVector:function initVector(){return this._ctm=[1,0,0,1,0,0],this._ctmStack=[]},save:function save(){return this._ctmStack.push(this._ctm.slice()),this.addContent("q")},restore:function restore(){return this._ctm=this._ctmStack.pop()||[1,0,0,1,0,0],this.addContent("Q")},closePath:function closePath(){return this.addContent("h")},lineWidth:function lineWidth(r){return this.addContent("".concat(We(r)," w"))},_CAP_STYLES:{BUTT:0,ROUND:1,SQUARE:2},lineCap:function lineCap(r){return"string"==typeof r&&(r=this._CAP_STYLES[r.toUpperCase()]),this.addContent("".concat(r," J"))},_JOIN_STYLES:{MITER:0,ROUND:1,BEVEL:2},lineJoin:function lineJoin(r){return"string"==typeof r&&(r=this._JOIN_STYLES[r.toUpperCase()]),this.addContent("".concat(r," j"))},miterLimit:function miterLimit(r){return this.addContent("".concat(We(r)," M"))},dash:function dash(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=r;if(Array.isArray(r)||(r=[r,i.space||r]),!r.every((function(r){return Number.isFinite(r)&&r>0})))throw new Error("dash(".concat(JSON.stringify(o),", ").concat(JSON.stringify(i),") invalid, lengths must be numeric and greater than zero"));return r=r.map(We).join(" "),this.addContent("[".concat(r,"] ").concat(We(i.phase||0)," d"))},undash:function undash(){return this.addContent("[] 0 d")},moveTo:function moveTo(r,i){return this.addContent("".concat(We(r)," ").concat(We(i)," m"))},lineTo:function lineTo(r,i){return this.addContent("".concat(We(r)," ").concat(We(i)," l"))},bezierCurveTo:function bezierCurveTo(r,i,o,u,p,g){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," ").concat(We(p)," ").concat(We(g)," c"))},quadraticCurveTo:function quadraticCurveTo(r,i,o,u){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," v"))},rect:function rect(r,i,o,u){return this.addContent("".concat(We(r)," ").concat(We(i)," ").concat(We(o)," ").concat(We(u)," re"))},roundedRect:function roundedRect(r,i,o,u,p){null==p&&(p=0);var g=(p=Math.min(p,.5*o,.5*u))*(1-ze);return this.moveTo(r+p,i),this.lineTo(r+o-p,i),this.bezierCurveTo(r+o-g,i,r+o,i+g,r+o,i+p),this.lineTo(r+o,i+u-p),this.bezierCurveTo(r+o,i+u-g,r+o-g,i+u,r+o-p,i+u),this.lineTo(r+p,i+u),this.bezierCurveTo(r+g,i+u,r,i+u-g,r,i+u-p),this.lineTo(r,i+p),this.bezierCurveTo(r,i+g,r+g,i,r+p,i),this.closePath()},ellipse:function ellipse(r,i,o,u){null==u&&(u=o);var p=o*ze,g=u*ze,y=(r-=o)+2*o,w=(i-=u)+2*u,_=r+o,x=i+u;return this.moveTo(r,x),this.bezierCurveTo(r,x-g,_-p,i,_,i),this.bezierCurveTo(_+p,i,y,x-g,y,x),this.bezierCurveTo(y,x+g,_+p,w,_,w),this.bezierCurveTo(_-p,w,r,x+g,r,x),this.closePath()},circle:function circle(r,i,o){return this.ellipse(r,i,o)},arc:function arc(r,i,o,u,p,g){null==g&&(g=!1);var y=2*Math.PI,w=.5*Math.PI,_=p-u;if(Math.abs(_)>y)_=y;else if(0!==_&&g!==_<0){_=(g?-1:1)*y+_}var x=Math.ceil(Math.abs(_)/w),k=_/x,P=k/w*ze*o,O=u,I=-Math.sin(O)*P,B=Math.cos(O)*P,D=r+Math.cos(O)*o,R=i+Math.sin(O)*o;this.moveTo(D,R);for(var N=0;N<x;N++){var W=D+I,U=R+B;O+=k,D=r+Math.cos(O)*o,R=i+Math.sin(O)*o;var j=D-(I=-Math.sin(O)*P),G=R-(B=Math.cos(O)*P);this.bezierCurveTo(W,U,j,G,D,R)}return this},polygon:function polygon(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];this.moveTo.apply(this,_toConsumableArray(i.shift()||[]));for(var u=0,p=i;u<p.length;u++){var g=p[u];this.lineTo.apply(this,_toConsumableArray(g||[]))}return this.closePath()},path:function path(r){return Ne.apply(this,r),this},_windingRule:function _windingRule(r){return/even-?odd/.test(r)?"*":""},fill:function fill(r,i){return/(even-?odd)|(non-?zero)/.test(r)&&(i=r,r=null),r&&this.fillColor(r),this.addContent("f".concat(this._windingRule(i)))},stroke:function stroke(r){return r&&this.strokeColor(r),this.addContent("S")},fillAndStroke:function fillAndStroke(r,i,o){null==i&&(i=r);var u=/(even-?odd)|(non-?zero)/;return u.test(r)&&(o=r,r=null),u.test(i)&&(o=i,i=r),r&&(this.fillColor(r),this.strokeColor(i)),this.addContent("B".concat(this._windingRule(o)))},clip:function clip(r){return this.addContent("W".concat(this._windingRule(r)," n"))},transform:function transform(r,i,o,u,p,g){var y=this._ctm,w=_slicedToArray(y,6),_=w[0],x=w[1],k=w[2],P=w[3],O=w[4],I=w[5];y[0]=_*r+k*i,y[1]=x*r+P*i,y[2]=_*o+k*u,y[3]=x*o+P*u,y[4]=_*p+k*g+O,y[5]=x*p+P*g+I;var B=[r,i,o,u,p,g].map((function(r){return We(r)})).join(" ");return this.addContent("".concat(B," cm"))},translate:function translate(r,i){return this.transform(1,0,0,1,r,i)},rotate:function rotate(r){var i,o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=r*Math.PI/180,p=Math.cos(u),g=Math.sin(u),y=i=0;if(null!=o.origin){var w=_slicedToArray(o.origin,2),_=(y=w[0])*g+(i=w[1])*p;y-=y*p-i*g,i-=_}return this.transform(p,g,-g,p,y,i)},scale:function scale(r,i){var o,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};null==i&&(i=r),"object"==typeof i&&(u=i,i=r);var p=o=0;if(null!=u.origin){var g=_slicedToArray(u.origin,2);p=g[0],o=g[1],p-=r*p,o-=i*o}return this.transform(r,0,0,i,p,o)}},je={402:131,8211:150,8212:151,8216:145,8217:146,8218:130,8220:147,8221:148,8222:132,8224:134,8225:135,8226:149,8230:133,8364:128,8240:137,8249:139,8250:155,710:136,8482:153,338:140,339:156,732:152,352:138,353:154,376:159,381:142,382:158},Ge=".notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n  \nspace         exclam         quotedbl       numbersign\ndollar        percent        ampersand      quotesingle\nparenleft     parenright     asterisk       plus\ncomma         hyphen         period         slash\nzero          one            two            three\nfour          five           six            seven\neight         nine           colon          semicolon\nless          equal          greater        question\n  \nat            A              B              C\nD             E              F              G\nH             I              J              K\nL             M              N              O\nP             Q              R              S\nT             U              V              W\nX             Y              Z              bracketleft\nbackslash     bracketright   asciicircum    underscore\n  \ngrave         a              b              c\nd             e              f              g\nh             i              j              k\nl             m              n              o\np             q              r              s\nt             u              v              w\nx             y              z              braceleft\nbar           braceright     asciitilde     .notdef\n  \nEuro          .notdef        quotesinglbase florin\nquotedblbase  ellipsis       dagger         daggerdbl\ncircumflex    perthousand    Scaron         guilsinglleft\nOE            .notdef        Zcaron         .notdef\n.notdef       quoteleft      quoteright     quotedblleft\nquotedblright bullet         endash         emdash\ntilde         trademark      scaron         guilsinglright\noe            .notdef        zcaron         ydieresis\n  \nspace         exclamdown     cent           sterling\ncurrency      yen            brokenbar      section\ndieresis      copyright      ordfeminine    guillemotleft\nlogicalnot    hyphen         registered     macron\ndegree        plusminus      twosuperior    threesuperior\nacute         mu             paragraph      periodcentered\ncedilla       onesuperior    ordmasculine   guillemotright\nonequarter    onehalf        threequarters  questiondown\n  \nAgrave        Aacute         Acircumflex    Atilde\nAdieresis     Aring          AE             Ccedilla\nEgrave        Eacute         Ecircumflex    Edieresis\nIgrave        Iacute         Icircumflex    Idieresis\nEth           Ntilde         Ograve         Oacute\nOcircumflex   Otilde         Odieresis      multiply\nOslash        Ugrave         Uacute         Ucircumflex\nUdieresis     Yacute         Thorn          germandbls\n  \nagrave        aacute         acircumflex    atilde\nadieresis     aring          ae             ccedilla\negrave        eacute         ecircumflex    edieresis\nigrave        iacute         icircumflex    idieresis\neth           ntilde         ograve         oacute\nocircumflex   otilde         odieresis      divide\noslash        ugrave         uacute         ucircumflex\nudieresis     yacute         thorn          ydieresis".split(/\s+/),Ve=function(){function AFMFont(r){_classCallCheck(this,AFMFont),this.contents=r,this.attributes={},this.glyphWidths={},this.boundingBoxes={},this.kernPairs={},this.parse(),this.charWidths=new Array(256);for(var i=0;i<=255;i++)this.charWidths[i]=this.glyphWidths[Ge[i]];this.bbox=this.attributes.FontBBox.split(/\s+/).map((function(r){return+r})),this.ascender=+(this.attributes.Ascender||0),this.descender=+(this.attributes.Descender||0),this.xHeight=+(this.attributes.XHeight||0),this.capHeight=+(this.attributes.CapHeight||0),this.lineGap=this.bbox[3]-this.bbox[1]-(this.ascender-this.descender)}return _createClass(AFMFont,null,[{key:"open",value:function open(r){return new AFMFont(O.readFileSync(r,"utf8"))}}]),_createClass(AFMFont,[{key:"parse",value:function parse(){var r,i="",o=_createForOfIteratorHelper(this.contents.split("\n"));try{for(o.s();!(r=o.n()).done;){var u,p,g=r.value;if(u=g.match(/^Start(\w+)/))i=u[1];else if(u=g.match(/^End(\w+)/))i="";else switch(i){case"FontMetrics":var y=(u=g.match(/(^\w+)\s+(.*)/))[1],w=u[2];(p=this.attributes[y])?(Array.isArray(p)||(p=this.attributes[y]=[p]),p.push(w)):this.attributes[y]=w;break;case"CharMetrics":if(!/^CH?\s/.test(g))continue;var _=g.match(/\bN\s+(\.?\w+)\s*;/)[1];this.glyphWidths[_]=+g.match(/\bWX\s+(\d+)\s*;/)[1];break;case"KernPairs":(u=g.match(/^KPX\s+(\.?\w+)\s+(\.?\w+)\s+(-?\d+)/))&&(this.kernPairs[u[1]+"\0"+u[2]]=parseInt(u[3]))}}}catch(r){o.e(r)}finally{o.f()}}},{key:"encodeText",value:function encodeText(r){for(var i=[],o=0,u=r.length;o<u;o++){var p=r.charCodeAt(o);p=je[p]||p,i.push(p.toString(16))}return i}},{key:"glyphsForString",value:function glyphsForString(r){for(var i=[],o=0,u=r.length;o<u;o++){var p=r.charCodeAt(o);i.push(this.characterToGlyph(p))}return i}},{key:"characterToGlyph",value:function characterToGlyph(r){return Ge[je[r]||r]||".notdef"}},{key:"widthOfGlyph",value:function widthOfGlyph(r){return this.glyphWidths[r]||0}},{key:"getKernPair",value:function getKernPair(r,i){return this.kernPairs[r+"\0"+i]||0}},{key:"advancesForGlyphs",value:function advancesForGlyphs(r){for(var i=[],o=0;o<r.length;o++){var u=r[o],p=r[o+1];i.push(this.widthOfGlyph(u)+this.getKernPair(u,p))}return i}}]),AFMFont}(),He=function(){function PDFFont(){_classCallCheck(this,PDFFont)}return _createClass(PDFFont,[{key:"encode",value:function encode(){throw new Error("Must be implemented by subclasses")}},{key:"widthOfString",value:function widthOfString(){throw new Error("Must be implemented by subclasses")}},{key:"ref",value:function ref(){return null!=this.dictionary?this.dictionary:this.dictionary=this.document.ref()}},{key:"finalize",value:function finalize(){if(!this.embedded&&null!=this.dictionary)return this.embed(),this.embedded=!0}},{key:"embed",value:function embed(){throw new Error("Must be implemented by subclasses")}},{key:"lineHeight",value:function lineHeight(r,i){null==i&&(i=!1);var o=i?this.lineGap:0;return(this.ascender+o-this.descender)/1e3*r}}]),PDFFont}(),qe={Courier:function Courier(){return O.readFileSync("//data/Courier.afm","utf8")},"Courier-Bold":function CourierBold(){return O.readFileSync("//data/Courier-Bold.afm","utf8")},"Courier-Oblique":function CourierOblique(){return O.readFileSync("//data/Courier-Oblique.afm","utf8")},"Courier-BoldOblique":function CourierBoldOblique(){return O.readFileSync("//data/Courier-BoldOblique.afm","utf8")},Helvetica:function Helvetica(){return O.readFileSync("//data/Helvetica.afm","utf8")},"Helvetica-Bold":function HelveticaBold(){return O.readFileSync("//data/Helvetica-Bold.afm","utf8")},"Helvetica-Oblique":function HelveticaOblique(){return O.readFileSync("//data/Helvetica-Oblique.afm","utf8")},"Helvetica-BoldOblique":function HelveticaBoldOblique(){return O.readFileSync("//data/Helvetica-BoldOblique.afm","utf8")},"Times-Roman":function TimesRoman(){return O.readFileSync("//data/Times-Roman.afm","utf8")},"Times-Bold":function TimesBold(){return O.readFileSync("//data/Times-Bold.afm","utf8")},"Times-Italic":function TimesItalic(){return O.readFileSync("//data/Times-Italic.afm","utf8")},"Times-BoldItalic":function TimesBoldItalic(){return O.readFileSync("//data/Times-BoldItalic.afm","utf8")},Symbol:function Symbol(){return O.readFileSync("//data/Symbol.afm","utf8")},ZapfDingbats:function ZapfDingbats(){return O.readFileSync("//data/ZapfDingbats.afm","utf8")}},Xe=function(r){_inherits(StandardFont,r);var i=_createSuper(StandardFont);function StandardFont(r,o,u){var p;_classCallCheck(this,StandardFont),(p=i.call(this)).document=r,p.name=o,p.id=u,p.font=new Ve(qe[p.name]());var g=p.font;return p.ascender=g.ascender,p.descender=g.descender,p.bbox=g.bbox,p.lineGap=g.lineGap,p.xHeight=g.xHeight,p.capHeight=g.capHeight,p}return _createClass(StandardFont,[{key:"embed",value:function embed(){return this.dictionary.data={Type:"Font",BaseFont:this.name,Subtype:"Type1",Encoding:"WinAnsiEncoding"},this.dictionary.end()}},{key:"encode",value:function encode(r){for(var i=this.font.encodeText(r),o=this.font.glyphsForString("".concat(r)),u=this.font.advancesForGlyphs(o),p=[],g=0;g<o.length;g++){var y=o[g];p.push({xAdvance:u[g],yAdvance:0,xOffset:0,yOffset:0,advanceWidth:this.font.widthOfGlyph(y)})}return[i,p]}},{key:"widthOfString",value:function widthOfString(r,i){var o,u=this.font.glyphsForString("".concat(r)),p=0,g=_createForOfIteratorHelper(this.font.advancesForGlyphs(u));try{for(g.s();!(o=g.n()).done;){p+=o.value}}catch(r){g.e(r)}finally{g.f()}return p*(i/1e3)}}],[{key:"isStandardFont",value:function isStandardFont(r){return r in qe}}]),StandardFont}(He),Ze=function toHex(r){return"0000".concat(r.toString(16)).slice(-4)},Ke=function(r){_inherits(EmbeddedFont,r);var i=_createSuper(EmbeddedFont);function EmbeddedFont(r,o,u){var p;return _classCallCheck(this,EmbeddedFont),(p=i.call(this)).document=r,p.font=o,p.id=u,p.subset=p.font.createSubset(),p.unicode=[[0]],p.widths=[p.font.getGlyph(0).advanceWidth],p.name=p.font.postscriptName,p.scale=1e3/p.font.unitsPerEm,p.ascender=p.font.ascent*p.scale,p.descender=p.font.descent*p.scale,p.xHeight=p.font.xHeight*p.scale,p.capHeight=p.font.capHeight*p.scale,p.lineGap=p.font.lineGap*p.scale,p.bbox=p.font.bbox,!1!==r.options.fontLayoutCache&&(p.layoutCache=Object.create(null)),p}return _createClass(EmbeddedFont,[{key:"layoutRun",value:function layoutRun(r,i){for(var o=this.font.layout(r,i),u=0;u<o.positions.length;u++){var p=o.positions[u];for(var g in p)p[g]*=this.scale;p.advanceWidth=o.glyphs[u].advanceWidth*this.scale}return o}},{key:"layoutCached",value:function layoutCached(r){if(!this.layoutCache)return this.layoutRun(r);var i;if(i=this.layoutCache[r])return i;var o=this.layoutRun(r);return this.layoutCache[r]=o,o}},{key:"layout",value:function layout(r,i,o){if(i)return this.layoutRun(r,i);for(var u=o?null:[],p=o?null:[],g=0,y=0,w=0;w<=r.length;){var _;if(w===r.length&&y<w||(_=r.charAt(w),[" ","\t"].includes(_))){var x=this.layoutCached(r.slice(y,++w));o||(u=u.concat(x.glyphs),p=p.concat(x.positions)),g+=x.advanceWidth,y=w}else w++}return{glyphs:u,positions:p,advanceWidth:g}}},{key:"encode",value:function encode(r,i){for(var o=this.layout(r,i),u=o.glyphs,p=o.positions,g=[],y=0;y<u.length;y++){var w=u[y],_=this.subset.includeGlyph(w.id);g.push("0000".concat(_.toString(16)).slice(-4)),null==this.widths[_]&&(this.widths[_]=w.advanceWidth*this.scale),null==this.unicode[_]&&(this.unicode[_]=w.codePoints)}return[g,p]}},{key:"widthOfString",value:function widthOfString(r,i,o){return this.layout(r,o,!0).advanceWidth*(i/1e3)}},{key:"embed",value:function embed(){var r=this,i=null!=this.subset.cff,o=this.document.ref();i&&(o.data.Subtype="CIDFontType0C"),this.subset.encodeStream().on("data",(function(r){return o.write(r)})).on("end",(function(){return o.end()}));var u=((null!=this.font["OS/2"]?this.font["OS/2"].sFamilyClass:void 0)||0)>>8,g=0;this.font.post.isFixedPitch&&(g|=1),1<=u&&u<=7&&(g|=2),g|=4,10===u&&(g|=8),this.font.head.macStyle.italic&&(g|=64);var y=[1,2,3,4,5,6].map((function(i){return String.fromCharCode((r.id.charCodeAt(i)||73)+17)})).join("")+"+"+this.font.postscriptName,w=this.font.bbox,_=this.document.ref({Type:"FontDescriptor",FontName:y,Flags:g,FontBBox:[w.minX*this.scale,w.minY*this.scale,w.maxX*this.scale,w.maxY*this.scale],ItalicAngle:this.font.italicAngle,Ascent:this.ascender,Descent:this.descender,CapHeight:(this.font.capHeight||this.font.ascent)*this.scale,XHeight:(this.font.xHeight||0)*this.scale,StemV:0});if(i?_.data.FontFile3=o:_.data.FontFile2=o,this.document.subset){var x=p.from("FFFFFFFFC0","hex"),k=this.document.ref();k.write(x),k.end(),_.data.CIDSet=k}_.end();var P={Type:"Font",Subtype:"CIDFontType0",BaseFont:y,CIDSystemInfo:{Registry:new String("Adobe"),Ordering:new String("Identity"),Supplement:0},FontDescriptor:_,W:[0,this.widths]};i||(P.Subtype="CIDFontType2",P.CIDToGIDMap="Identity");var O=this.document.ref(P);return O.end(),this.dictionary.data={Type:"Font",Subtype:"Type0",BaseFont:y,Encoding:"Identity-H",DescendantFonts:[O],ToUnicode:this.toUnicodeCmap()},this.dictionary.end()}},{key:"toUnicodeCmap",value:function toUnicodeCmap(){var r,i=this.document.ref(),o=[],u=_createForOfIteratorHelper(this.unicode);try{for(u.s();!(r=u.n()).done;){var p,g=r.value,y=[],w=_createForOfIteratorHelper(g);try{for(w.s();!(p=w.n()).done;){var _=p.value;_>65535&&(_-=65536,y.push(Ze(_>>>10&1023|55296)),_=56320|1023&_),y.push(Ze(_))}}catch(r){w.e(r)}finally{w.f()}o.push("<".concat(y.join(" "),">"))}}catch(r){u.e(r)}finally{u.f()}return i.end("/CIDInit /ProcSet findresource begin\n12 dict begin\nbegincmap\n/CIDSystemInfo <<\n  /Registry (Adobe)\n  /Ordering (UCS)\n  /Supplement 0\n>> def\n/CMapName /Adobe-Identity-UCS def\n/CMapType 2 def\n1 begincodespacerange\n<0000><ffff>\nendcodespacerange\n1 beginbfrange\n<0000> <".concat(Ze(o.length-1),"> [").concat(o.join(" "),"]\nendbfrange\nendcmap\nCMapName currentdict /CMap defineresource pop\nend\nend")),i}}]),EmbeddedFont}(He),Ye=function(){function PDFFontFactory(){_classCallCheck(this,PDFFontFactory)}return _createClass(PDFFontFactory,null,[{key:"open",value:function open(r,i,o,u){var g;if("string"==typeof i){if(Xe.isStandardFont(i))return new Xe(r,i,u);i=O.readFileSync(i)}if(p.isBuffer(i)?g=_.default.create(i,o):i instanceof Uint8Array?g=_.default.create(p.from(i),o):i instanceof ArrayBuffer&&(g=_.default.create(p.from(new Uint8Array(i)),o)),null==g)throw new Error("Not a supported font format or standard PDF font.");return new Ke(r,g,u)}}]),PDFFontFactory}(),Je={initFonts:function initFonts(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"Helvetica";this._fontFamilies={},this._fontCount=0,this._fontSize=12,this._font=null,this._registeredFonts={},r&&this.font(r)},font:function font(r,i,o){var u,font;if("number"==typeof i&&(o=i,i=null),"string"==typeof r&&this._registeredFonts[r]){u=r;var p=this._registeredFonts[r];r=p.src,i=p.family}else"string"!=typeof(u=i||r)&&(u=null);if(null!=o&&this.fontSize(o),font=this._fontFamilies[u])return this._font=font,this;var g="F".concat(++this._fontCount);return this._font=Ye.open(this,r,i,g),(font=this._fontFamilies[this._font.name])?(this._font=font,this):(u&&(this._fontFamilies[u]=this._font),this._font.name&&(this._fontFamilies[this._font.name]=this._font),this)},fontSize:function fontSize(r){return this._fontSize=r,this},currentLineHeight:function currentLineHeight(r){return null==r&&(r=!1),this._font.lineHeight(this._fontSize,r)},registerFont:function registerFont(r,i,o){return this._registeredFonts[r]={src:i,family:o},this}},Qe=function(r){_inherits(LineWrapper,r);var i=_createSuper(LineWrapper);function LineWrapper(r,o){var u;return _classCallCheck(this,LineWrapper),(u=i.call(this)).document=r,u.indent=o.indent||0,u.characterSpacing=o.characterSpacing||0,u.wordSpacing=0===o.wordSpacing,u.columns=o.columns||1,u.columnGap=null!=o.columnGap?o.columnGap:18,u.lineWidth=(o.width-u.columnGap*(u.columns-1))/u.columns,u.spaceLeft=u.lineWidth,u.startX=u.document.x,u.startY=u.document.y,u.column=1,u.ellipsis=o.ellipsis,u.continuedX=0,u.features=o.features,null!=o.height?(u.height=o.height,u.maxY=u.startY+o.height):u.maxY=u.document.page.maxY(),u.on("firstLine",(function(r){var i=u.continuedX||u.indent;return u.document.x+=i,u.lineWidth-=i,u.once("line",(function(){if(u.document.x-=i,u.lineWidth+=i,r.continued&&!u.continuedX&&(u.continuedX=u.indent),!r.continued)return u.continuedX=0}))})),u.on("lastLine",(function(r){var i=r.align;return"justify"===i&&(r.align="left"),u.lastLine=!0,u.once("line",(function(){return u.document.y+=r.paragraphGap||0,r.align=i,u.lastLine=!1}))})),u}return _createClass(LineWrapper,[{key:"wordWidth",value:function wordWidth(r){return this.document.widthOfString(r,this)+this.characterSpacing+this.wordSpacing}},{key:"eachWord",value:function eachWord(r,i){for(var o,u=new k.default(r),p=null,g=Object.create(null);o=u.nextBreak();){var y,w=r.slice((null!=p?p.position:void 0)||0,o.position),_=null!=g[w]?g[w]:g[w]=this.wordWidth(w);if(_>this.lineWidth+this.continuedX)for(var x=p,P={};w.length;){var O,I;_>this.spaceLeft?(O=Math.ceil(this.spaceLeft/(_/w.length)),I=(_=this.wordWidth(w.slice(0,O)))<=this.spaceLeft&&O<w.length):O=w.length;for(var B=_>this.spaceLeft&&O>0;B||I;)B?B=(_=this.wordWidth(w.slice(0,--O)))>this.spaceLeft&&O>0:(B=(_=this.wordWidth(w.slice(0,++O)))>this.spaceLeft&&O>0,I=_<=this.spaceLeft&&O<w.length);if(0===O&&this.spaceLeft===this.lineWidth&&(O=1),P.required=o.required||O<w.length,y=i(w.slice(0,O),_,P,x),x={required:!1},w=w.slice(O),_=this.wordWidth(w),!1===y)break}else y=i(w,_,o,p);if(!1===y)break;p=o}}},{key:"wrap",value:function wrap(r,i){var o=this;null!=i.indent&&(this.indent=i.indent),null!=i.characterSpacing&&(this.characterSpacing=i.characterSpacing),null!=i.wordSpacing&&(this.wordSpacing=i.wordSpacing),null!=i.ellipsis&&(this.ellipsis=i.ellipsis);var u=this.document.y+this.document.currentLineHeight(!0);(this.document.y>this.maxY||u>this.maxY)&&this.nextSection();var p="",g=0,y=0,w=0,_=this.document.y,x=function emitLine(){return i.textWidth=g+o.wordSpacing*(y-1),i.wordCount=y,i.lineWidth=o.lineWidth,_=o.document.y,o.emit("line",p,i,o),w++};return this.emit("sectionStart",i,this),this.eachWord(r,(function(r,u,w,_){if((null==_||_.required)&&(o.emit("firstLine",i,o),o.spaceLeft=o.lineWidth),u<=o.spaceLeft&&(p+=r,g+=u,y++),w.required||u>o.spaceLeft){var k=o.document.currentLineHeight(!0);if(null!=o.height&&o.ellipsis&&o.document.y+2*k>o.maxY&&o.column>=o.columns){for(!0===o.ellipsis&&(o.ellipsis="…"),p=p.replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);p&&g>o.lineWidth;)p=p.slice(0,-1).replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);g<=o.lineWidth&&(p+=o.ellipsis),g=o.wordWidth(p)}if(w.required&&(u>o.spaceLeft&&(x(),p=r,g=u,y=1),o.emit("lastLine",i,o)),x(),o.document.y+k>o.maxY)if(!o.nextSection())return y=0,p="",!1;return w.required?(o.spaceLeft=o.lineWidth,p="",g=0,y=0):(o.spaceLeft=o.lineWidth-u,p=r,g=u,y=1)}return o.spaceLeft-=u})),y>0&&(this.emit("lastLine",i,this),x()),this.emit("sectionEnd",i,this),!0===i.continued?(w>1&&(this.continuedX=0),this.continuedX+=i.textWidth||0,this.document.y=_):this.document.x=this.startX}},{key:"nextSection",value:function nextSection(r){if(this.emit("sectionEnd",r,this),++this.column>this.columns){if(null!=this.height)return!1;var i;if(this.document.continueOnNewPage(),this.column=1,this.startY=this.document.page.margins.top,this.maxY=this.document.page.maxY(),this.document.x=this.startX,this.document._fillColor)(i=this.document).fillColor.apply(i,_toConsumableArray(this.document._fillColor));this.emit("pageBreak",r,this)}else this.document.x+=this.lineWidth+this.columnGap,this.document.y=this.startY,this.emit("columnBreak",r,this);return this.emit("sectionStart",r,this),!0}}]),LineWrapper}(x.EventEmitter),$e=W.number,et={initText:function initText(){return this._line=this._line.bind(this),this.x=0,this.y=0,this._lineGap=0},lineGap:function lineGap(r){return this._lineGap=r,this},moveDown:function moveDown(r){return null==r&&(r=1),this.y+=this.currentLineHeight(!0)*r+this._lineGap,this},moveUp:function moveUp(r){return null==r&&(r=1),this.y-=this.currentLineHeight(!0)*r+this._lineGap,this},_text:function _text(r,i,o,u,p){var g=this;u=this._initOptions(i,o,u),r=null==r?"":"".concat(r),u.wordSpacing&&(r=r.replace(/\s{2,}/g," "));var y=function addStructure(){u.structParent&&u.structParent.add(g.struct(u.structType||"P",[g.markStructureContent(u.structType||"P")]))};if(u.width){var w=this._wrapper;w||((w=new Qe(this,u)).on("line",p),w.on("firstLine",y)),this._wrapper=u.continued?w:null,this._textOptions=u.continued?u:null,w.wrap(r,u)}else{var _,x=_createForOfIteratorHelper(r.split("\n"));try{for(x.s();!(_=x.n()).done;){var k=_.value;y(),p(k,u)}}catch(r){x.e(r)}finally{x.f()}}return this},text:function text(r,i,o,u){return this._text(r,i,o,u,this._line)},widthOfString:function widthOfString(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return this._font.widthOfString(r,this._fontSize,i.features)+(i.characterSpacing||0)*(r.length-1)},heightOfString:function heightOfString(r,i){var o=this,u=this.x,p=this.y;(i=this._initOptions(i)).height=1/0;var g=i.lineGap||this._lineGap||0;this._text(r,this.x,this.y,i,(function(){return o.y+=o.currentLineHeight(!0)+g}));var y=this.y-p;return this.x=u,this.y=p,y},list:function list(r,i,o,u,p){var g=this,y=(u=this._initOptions(i,o,u)).listType||"bullet",w=Math.round(this._font.ascender/1e3*this._fontSize),_=w/2,x=u.bulletRadius||w/3,k=u.textIndent||("bullet"===y?5*x:2*w),P=u.bulletIndent||("bullet"===y?8*x:2*w),O=1,I=[],B=[],D=[];!function flatten(r){for(var i=1,o=0;o<r.length;o++){var u=r[o];Array.isArray(u)?(O++,flatten(u),O--):(I.push(u),B.push(O),"bullet"!==y&&D.push(i++))}}(r);(p=new Qe(this,u)).on("line",this._line),O=1;var R=0;return p.on("firstLine",(function(){var r,i,o,w,I;if(u.structParent)if(u.structTypes){var N=_slicedToArray(u.structTypes,3);i=N[0],o=N[1],w=N[2]}else i="LI",o="Lbl",w="LBody";if(i?(r=g.struct(i),u.structParent.add(r)):u.structParent&&(r=u.structParent),(I=B[R++])!==O){var W=P*(I-O);g.x+=W,p.lineWidth-=W,O=I}switch(r&&(o||w)&&r.add(g.struct(o||w,[g.markStructureContent(o||w)])),y){case"bullet":g.circle(g.x-k+x,g.y+_,x),g.fill();break;case"numbered":case"lettered":var U=function label(r){switch(y){case"numbered":return"".concat(r,".");case"lettered":var i=String.fromCharCode((r-1)%26+65),o=Math.floor((r-1)/26+1),u=Array(o+1).join(i);return"".concat(u,".")}}(D[R-1]);g._fragment(U,g.x-k,g.y,u)}r&&o&&w&&r.add(g.struct(w,[g.markStructureContent(w)])),r&&r!==u.structParent&&r.end()})),p.on("sectionStart",(function(){var r=k+P*(O-1);return g.x+=r,p.lineWidth-=r})),p.on("sectionEnd",(function(){var r=k+P*(O-1);return g.x-=r,p.lineWidth+=r})),p.wrap(I.join("\n"),u),this},_initOptions:function _initOptions(){var r=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=arguments.length>1?arguments[1]:void 0,o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};"object"==typeof r&&(o=r,r=null);var u=Object.assign({},o);if(this._textOptions)for(var p in this._textOptions){var g=this._textOptions[p];"continued"!==p&&void 0===u[p]&&(u[p]=g)}return null!=r&&(this.x=r),null!=i&&(this.y=i),!1!==u.lineBreak&&(null==u.width&&(u.width=this.page.width-this.x-this.page.margins.right),u.width=Math.max(u.width,0)),u.columns||(u.columns=0),null==u.columnGap&&(u.columnGap=18),u},_line:function _line(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=arguments.length>2?arguments[2]:void 0;this._fragment(r,this.x,this.y,i);var u=i.lineGap||this._lineGap||0;return o?this.y+=this.currentLineHeight(!0)+u:this.x+=this.widthOfString(r)},_fragment:function _fragment(r,i,o,u){var p,g,y,w,_,x,k=this;if(0!==(r="".concat(r).replace(/\n/g,"")).length){var P=u.align||"left",O=u.wordSpacing||0,I=u.characterSpacing||0;if(u.width)switch(P){case"right":_=this.widthOfString(r.replace(/\s+$/,""),u),i+=u.lineWidth-_;break;case"center":i+=u.lineWidth/2-u.textWidth/2;break;case"justify":x=r.trim().split(/\s+/),_=this.widthOfString(r.replace(/\s+/g,""),u);var B=this.widthOfString(" ")+I;O=Math.max(0,(u.lineWidth-_)/Math.max(1,x.length-1)-B)}if("number"==typeof u.baseline)p=-u.baseline;else{switch(u.baseline){case"svg-middle":p=.5*this._font.xHeight;break;case"middle":case"svg-central":p=.5*(this._font.descender+this._font.ascender);break;case"bottom":case"ideographic":p=this._font.descender;break;case"alphabetic":p=0;break;case"mathematical":p=.5*this._font.ascender;break;case"hanging":p=.8*this._font.ascender;break;default:p=this._font.ascender}p=p/1e3*this._fontSize}var D,R=u.textWidth+O*(u.wordCount-1)+I*(r.length-1);if(null!=u.link&&this.link(i,o,R,this.currentLineHeight(),u.link),null!=u.goTo&&this.goTo(i,o,R,this.currentLineHeight(),u.goTo),null!=u.destination&&this.addNamedDestination(u.destination,"XYZ",i,o,null),u.underline){this.save(),u.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var N=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(N);var W=o+this.currentLineHeight()-N;this.moveTo(i,W),this.lineTo(i+R,W),this.stroke(),this.restore()}if(u.strike){this.save(),u.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var U=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(U);var j=o+this.currentLineHeight()/2;this.moveTo(i,j),this.lineTo(i+R,j),this.stroke(),this.restore()}if(this.save(),u.oblique)D="number"==typeof u.oblique?-Math.tan(u.oblique*Math.PI/180):-.25,this.transform(1,0,0,1,i,o),this.transform(1,0,D,1,-D*p,0),this.transform(1,0,0,1,-i,-o);this.transform(1,0,0,-1,0,this.page.height),o=this.page.height-o-p,null==this.page.fonts[this._font.id]&&(this.page.fonts[this._font.id]=this._font.ref()),this.addContent("BT"),this.addContent("1 0 0 1 ".concat($e(i)," ").concat($e(o)," Tm")),this.addContent("/".concat(this._font.id," ").concat($e(this._fontSize)," Tf"));var G=u.fill&&u.stroke?2:u.stroke?1:0;if(G&&this.addContent("".concat(G," Tr")),I&&this.addContent("".concat($e(I)," Tc")),O){x=r.trim().split(/\s+/),O+=this.widthOfString(" ")+I,O*=1e3/this._fontSize,g=[],w=[];var X,K=_createForOfIteratorHelper(x);try{for(K.s();!(X=K.n()).done;){var Y=X.value,J=_slicedToArray(this._font.encode(Y,u.features),2),$=J[0],ee=J[1];g=g.concat($),w=w.concat(ee);var te={},re=w[w.length-1];for(var ne in re){var ie=re[ne];te[ne]=ie}te.xAdvance+=O,w[w.length-1]=te}}catch(r){K.e(r)}finally{K.f()}}else{var oe=_slicedToArray(this._font.encode(r,u.features),2);g=oe[0],w=oe[1]}var ae=this._fontSize/1e3,se=[],ce=0,le=!1,ue=function addSegment(r){if(ce<r){var i=g.slice(ce,r).join(""),o=w[r-1].xAdvance-w[r-1].advanceWidth;se.push("<".concat(i,"> ").concat($e(-o)))}return ce=r},fe=function flush(r){if(ue(r),se.length>0)return k.addContent("[".concat(se.join(" "),"] TJ")),se.length=0};for(y=0;y<w.length;y++){var he=w[y];he.xOffset||he.yOffset?(fe(y),this.addContent("1 0 0 1 ".concat($e(i+he.xOffset*ae)," ").concat($e(o+he.yOffset*ae)," Tm")),fe(y+1),le=!0):(le&&(this.addContent("1 0 0 1 ".concat($e(i)," ").concat($e(o)," Tm")),le=!1),he.xAdvance-he.advanceWidth!=0&&ue(y+1)),i+=he.xAdvance*ae}return fe(y),this.addContent("ET"),this.restore()}}},tt=[65472,65473,65474,65475,65477,65478,65479,65480,65481,65482,65483,65484,65485,65486,65487],rt={1:"DeviceGray",3:"DeviceRGB",4:"DeviceCMYK"},nt=function(){function JPEG(r,i){var o;if(_classCallCheck(this,JPEG),this.data=r,this.label=i,65496!==this.data.readUInt16BE(0))throw"SOI not found in JPEG";for(var u=2;u<this.data.length&&(o=this.data.readUInt16BE(u),u+=2,!tt.includes(o));)u+=this.data.readUInt16BE(u);if(!tt.includes(o))throw"Invalid JPEG.";u+=2,this.bits=this.data[u++],this.height=this.data.readUInt16BE(u),u+=2,this.width=this.data.readUInt16BE(u),u+=2;var p=this.data[u++];this.colorSpace=rt[p],this.obj=null}return _createClass(JPEG,[{key:"embed",value:function embed(r){if(!this.obj)return this.obj=r.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:this.bits,Width:this.width,Height:this.height,ColorSpace:this.colorSpace,Filter:"DCTDecode"}),"DeviceCMYK"===this.colorSpace&&(this.obj.data.Decode=[1,0,1,0,1,0,1,0]),this.obj.end(this.data),this.data=null}}]),JPEG}(),it=function(){function PNGImage(r,i){_classCallCheck(this,PNGImage),this.label=i,this.image=new P.default(r),this.width=this.image.width,this.height=this.image.height,this.imgData=this.image.imgData,this.obj=null}return _createClass(PNGImage,[{key:"embed",value:function embed(r){var i=!1;if(this.document=r,!this.obj){var o=this.image.hasAlphaChannel,u=1===this.image.interlaceMethod;if(this.obj=this.document.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:o?8:this.image.bits,Width:this.width,Height:this.height,Filter:"FlateDecode"}),!o){var g=this.document.ref({Predictor:u?1:15,Colors:this.image.colors,BitsPerComponent:this.image.bits,Columns:this.width});this.obj.data.DecodeParms=g,g.end()}if(0===this.image.palette.length)this.obj.data.ColorSpace=this.image.colorSpace;else{var y=this.document.ref();y.end(p.from(this.image.palette)),this.obj.data.ColorSpace=["Indexed","DeviceRGB",this.image.palette.length/3-1,y]}if(null!=this.image.transparency.grayscale){var w=this.image.transparency.grayscale;this.obj.data.Mask=[w,w]}else if(this.image.transparency.rgb){var _,x=[],k=_createForOfIteratorHelper(this.image.transparency.rgb);try{for(k.s();!(_=k.n()).done;){var P=_.value;x.push(P,P)}}catch(r){k.e(r)}finally{k.f()}this.obj.data.Mask=x}else{if(this.image.transparency.indexed)return i=!0,this.loadIndexedAlphaChannel();if(o)return i=!0,this.splitAlphaChannel()}if(u&&!i)return this.decodeData();this.finalize()}}},{key:"finalize",value:function finalize(){if(this.alphaChannel){var r=this.document.ref({Type:"XObject",Subtype:"Image",Height:this.height,Width:this.width,BitsPerComponent:8,Filter:"FlateDecode",ColorSpace:"DeviceGray",Decode:[0,1]});r.end(this.alphaChannel),this.obj.data.SMask=r}return this.obj.end(this.imgData),this.image=null,this.imgData=null}},{key:"splitAlphaChannel",value:function splitAlphaChannel(){var r=this;return this.image.decodePixels((function(i){for(var o,u,g=r.image.colors,w=r.width*r.height,_=p.alloc(w*g),x=p.alloc(w),k=u=o=0,P=i.length,O=16===r.image.bits?1:0;k<P;){for(var I=0;I<g;I++)_[u++]=i[k++],k+=O;x[o++]=i[k++],k+=O}return r.imgData=y.default.deflateSync(_),r.alphaChannel=y.default.deflateSync(x),r.finalize()}))}},{key:"loadIndexedAlphaChannel",value:function loadIndexedAlphaChannel(){var r=this,i=this.image.transparency.indexed;return this.image.decodePixels((function(o){for(var u=p.alloc(r.width*r.height),g=0,w=0,_=o.length;w<_;w++)u[g++]=i[o[w]];return r.alphaChannel=y.default.deflateSync(u),r.finalize()}))}},{key:"decodeData",value:function decodeData(){var r=this;this.image.decodePixels((function(i){r.imgData=y.default.deflateSync(i),r.finalize()}))}}]),PNGImage}(),ot=function(){function PDFImage(){_classCallCheck(this,PDFImage)}return _createClass(PDFImage,null,[{key:"open",value:function open(r,i){var o;if(p.isBuffer(r))o=r;else if(r instanceof ArrayBuffer)o=p.from(new Uint8Array(r));else{var u;if(u=/^data:.+;base64,(.*)$/.exec(r))o=p.from(u[1],"base64");else if(!(o=O.readFileSync(r)))return}if(255===o[0]&&216===o[1])return new nt(o,i);if(137===o[0]&&"PNG"===o.toString("ascii",1,4))return new it(o,i);throw new Error("Unknown image format.")}}]),PDFImage}(),at={initImages:function initImages(){return this._imageRegistry={},this._imageCount=0},image:function image(r,i,o){var u,p,g,image,y,w,_,x=arguments.length>3&&void 0!==arguments[3]?arguments[3]:{};"object"==typeof i&&(x=i,i=null),i=null!=(w=null!=i?i:x.x)?w:this.x,o=null!=(_=null!=o?o:x.y)?_:this.y,"string"==typeof r&&(image=this._imageRegistry[r]),image||(image=r.width&&r.height?r:this.openImage(r)),image.obj||image.embed(this),null==this.page.xobjects[image.label]&&(this.page.xobjects[image.label]=image.obj);var k=x.width||image.width,P=x.height||image.height;if(x.width&&!x.height){var O=k/image.width;k=image.width*O,P=image.height*O}else if(x.height&&!x.width){var I=P/image.height;k=image.width*I,P=image.height*I}else if(x.scale)k=image.width*x.scale,P=image.height*x.scale;else if(x.fit){var B=_slicedToArray(x.fit,2);p=(g=B[0])/(u=B[1]),(y=image.width/image.height)>p?(k=g,P=g/y):(P=u,k=u*y)}else if(x.cover){var D=_slicedToArray(x.cover,2);p=(g=D[0])/(u=D[1]),(y=image.width/image.height)>p?(P=u,k=u*y):(k=g,P=g/y)}return(x.fit||x.cover)&&("center"===x.align?i=i+g/2-k/2:"right"===x.align&&(i=i+g-k),"center"===x.valign?o=o+u/2-P/2:"bottom"===x.valign&&(o=o+u-P)),null!=x.link&&this.link(i,o,k,P,x.link),null!=x.goTo&&this.goTo(i,o,k,P,x.goTo),null!=x.destination&&this.addNamedDestination(x.destination,"XYZ",i,o,null),this.y===o&&(this.y+=P),this.save(),this.transform(k,0,0,-P,i,o+P),this.addContent("/".concat(image.label," Do")),this.restore(),this},openImage:function openImage(r){var i;return"string"==typeof r&&(i=this._imageRegistry[r]),i||(i=ot.open(r,"I".concat(++this._imageCount)),"string"==typeof r&&(this._imageRegistry[r]=i)),i}},st={annotate:function annotate(r,i,o,u,p){for(var g in p.Type="Annot",p.Rect=this._convertRect(r,i,o,u),p.Border=[0,0,0],"Link"===p.Subtype&&void 0===p.F&&(p.F=4),"Link"!==p.Subtype&&null==p.C&&(p.C=this._normalizeColor(p.color||[0,0,0])),delete p.color,"string"==typeof p.Dest&&(p.Dest=new String(p.Dest)),p){var y=p[g];p[g[0].toUpperCase()+g.slice(1)]=y}var w=this.ref(p);return this.page.annotations.push(w),w.end(),this},note:function note(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Text",g.Contents=new String(p),g.Name="Comment",null==g.color&&(g.color=[243,223,92]),this.annotate(r,i,o,u,g)},goTo:function goTo(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Link",g.A=this.ref({S:"GoTo",D:new String(p)}),g.A.end(),this.annotate(r,i,o,u,g)},link:function link(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};if(g.Subtype="Link","number"==typeof p){var y=this._root.data.Pages.data;if(!(p>=0&&p<y.Kids.length))throw new Error("The document has no page ".concat(p));g.A=this.ref({S:"GoTo",D:[y.Kids[p],"XYZ",null,null,null]}),g.A.end()}else g.A=this.ref({S:"URI",URI:new String(p)}),g.A.end();return this.annotate(r,i,o,u,g)},_markup:function _markup(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},g=_slicedToArray(this._convertRect(r,i,o,u),4),y=g[0],w=g[1],_=g[2],x=g[3];return p.QuadPoints=[y,x,_,x,y,w,_,w],p.Contents=new String,this.annotate(r,i,o,u,p)},highlight:function highlight(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Highlight",null==p.color&&(p.color=[241,238,148]),this._markup(r,i,o,u,p)},underline:function underline(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Underline",this._markup(r,i,o,u,p)},strike:function strike(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="StrikeOut",this._markup(r,i,o,u,p)},lineAnnotation:function lineAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Line",p.Contents=new String,p.L=[r,this.page.height-i,o,this.page.height-u],this.annotate(r,i,o,u,p)},rectAnnotation:function rectAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Square",p.Contents=new String,this.annotate(r,i,o,u,p)},ellipseAnnotation:function ellipseAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Circle",p.Contents=new String,this.annotate(r,i,o,u,p)},textAnnotation:function textAnnotation(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="FreeText",g.Contents=new String(p),g.DA=new String,this.annotate(r,i,o,u,g)},fileAnnotation:function fileAnnotation(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{},y=this.file(p.src,Object.assign({hidden:!0},p));return g.Subtype="FileAttachment",g.FS=y,g.Contents?g.Contents=new String(g.Contents):y.data.Desc&&(g.Contents=y.data.Desc),this.annotate(r,i,o,u,g)},_convertRect:function _convertRect(r,i,o,u){var p=i;i+=u;var g=r+o,y=_slicedToArray(this._ctm,6),w=y[0],_=y[1],x=y[2],k=y[3],P=y[4],O=y[5];return[r=w*r+x*i+P,i=_*r+k*i+O,g=w*g+x*p+P,p=_*g+k*p+O]}},ct=function(){function PDFOutline(r,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{expanded:!1};_classCallCheck(this,PDFOutline),this.document=r,this.options=p,this.outlineData={},null!==u&&(this.outlineData.Dest=[u.dictionary,"Fit"]),null!==i&&(this.outlineData.Parent=i),null!==o&&(this.outlineData.Title=new String(o)),this.dictionary=this.document.ref(this.outlineData),this.children=[]}return _createClass(PDFOutline,[{key:"addItem",value:function addItem(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{expanded:!1},o=new PDFOutline(this.document,this.dictionary,r,this.document.page,i);return this.children.push(o),o}},{key:"endOutline",value:function endOutline(){if(this.children.length>0){this.options.expanded&&(this.outlineData.Count=this.children.length);var r=this.children[0],i=this.children[this.children.length-1];this.outlineData.First=r.dictionary,this.outlineData.Last=i.dictionary;for(var o=0,u=this.children.length;o<u;o++){var p=this.children[o];o>0&&(p.outlineData.Prev=this.children[o-1].dictionary),o<this.children.length-1&&(p.outlineData.Next=this.children[o+1].dictionary),p.endOutline()}}return this.dictionary.end()}}]),PDFOutline}(),lt={initOutline:function initOutline(){return this.outline=new ct(this,null,null,null)},endOutline:function endOutline(){if(this.outline.endOutline(),this.outline.children.length>0)return this._root.data.Outlines=this.outline.dictionary,this._root.data.PageMode="UseOutlines"}},ut=function(){function PDFStructureContent(r,i){_classCallCheck(this,PDFStructureContent),this.refs=[{pageRef:r,mcid:i}]}return _createClass(PDFStructureContent,[{key:"push",value:function push(r){var i=this;r.refs.forEach((function(r){return i.refs.push(r)}))}}]),PDFStructureContent}(),ft=function(){function PDFStructureElement(r,i){var o=this,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{},p=arguments.length>3&&void 0!==arguments[3]?arguments[3]:null;_classCallCheck(this,PDFStructureElement),this.document=r,this._attached=!1,this._ended=!1,this._flushed=!1,this.dictionary=r.ref({S:i});var g=this.dictionary.data;(Array.isArray(u)||this._isValidChild(u))&&(p=u,u={}),void 0!==u.title&&(g.T=new String(u.title)),void 0!==u.lang&&(g.Lang=new String(u.lang)),void 0!==u.alt&&(g.Alt=new String(u.alt)),void 0!==u.expanded&&(g.E=new String(u.expanded)),void 0!==u.actual&&(g.ActualText=new String(u.actual)),this._children=[],p&&(Array.isArray(p)||(p=[p]),p.forEach((function(r){return o.add(r)})),this.end())}return _createClass(PDFStructureElement,[{key:"add",value:function add(r){if(this._ended)throw new Error("Cannot add child to already-ended structure element");if(!this._isValidChild(r))throw new Error("Invalid structure element child");return r instanceof PDFStructureElement&&(r.setParent(this.dictionary),this._attached&&r.setAttached()),r instanceof ut&&this._addContentToParentTree(r),"function"==typeof r&&this._attached&&(r=this._contentForClosure(r)),this._children.push(r),this}},{key:"_addContentToParentTree",value:function _addContentToParentTree(r){var i=this;r.refs.forEach((function(r){var o=r.pageRef,u=r.mcid;i.document.getStructParentTree().get(o.data.StructParents)[u]=i.dictionary}))}},{key:"setParent",value:function setParent(r){if(this.dictionary.data.P)throw new Error("Structure element added to more than one parent");this.dictionary.data.P=r,this._flush()}},{key:"setAttached",value:function setAttached(){var r=this;this._attached||(this._children.forEach((function(i,o){i instanceof PDFStructureElement&&i.setAttached(),"function"==typeof i&&(r._children[o]=r._contentForClosure(i))})),this._attached=!0,this._flush())}},{key:"end",value:function end(){this._ended||(this._children.filter((function(r){return r instanceof PDFStructureElement})).forEach((function(r){return r.end()})),this._ended=!0,this._flush())}},{key:"_isValidChild",value:function _isValidChild(r){return r instanceof PDFStructureElement||r instanceof ut||"function"==typeof r}},{key:"_contentForClosure",value:function _contentForClosure(r){var i=this.document.markStructureContent(this.dictionary.data.S);return r(),this.document.endMarkedContent(),this._addContentToParentTree(i),i}},{key:"_isFlushable",value:function _isFlushable(){return!(!this.dictionary.data.P||!this._ended)&&this._children.every((function(r){return"function"!=typeof r&&(!(r instanceof PDFStructureElement)||r._isFlushable())}))}},{key:"_flush",value:function _flush(){var r=this;!this._flushed&&this._isFlushable()&&(this.dictionary.data.K=[],this._children.forEach((function(i){return r._flushChild(i)})),this.dictionary.end(),this._children=[],this.dictionary.data.K=null,this._flushed=!0)}},{key:"_flushChild",value:function _flushChild(r){var i=this;r instanceof PDFStructureElement&&this.dictionary.data.K.push(r.dictionary),r instanceof ut&&r.refs.forEach((function(r){var o=r.pageRef,u=r.mcid;i.dictionary.data.Pg||(i.dictionary.data.Pg=o),i.dictionary.data.Pg===o?i.dictionary.data.K.push(u):i.dictionary.data.K.push({Type:"MCR",Pg:o,MCID:u})}))}}]),PDFStructureElement}(),ht=function(r){_inherits(PDFNumberTree,r);var i=_createSuper(PDFNumberTree);function PDFNumberTree(){return _classCallCheck(this,PDFNumberTree),i.apply(this,arguments)}return _createClass(PDFNumberTree,[{key:"_compareKeys",value:function _compareKeys(r,i){return parseInt(r)-parseInt(i)}},{key:"_keysName",value:function _keysName(){return"Nums"}},{key:"_dataForKey",value:function _dataForKey(r){return parseInt(r)}}]),PDFNumberTree}(B),dt={initMarkings:function initMarkings(r){this.structChildren=[],r.tagged&&(this.getMarkInfoDictionary().data.Marked=!0,this.getStructTreeRoot())},markContent:function markContent(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("Artifact"===r||i&&i.mcid){var o=0;for(this.page.markings.forEach((function(r){(o||r.structContent||"Artifact"===r.tag)&&o++}));o--;)this.endMarkedContent()}if(!i)return this.page.markings.push({tag:r}),this.addContent("/".concat(r," BMC")),this;this.page.markings.push({tag:r,options:i});var u={};return void 0!==i.mcid&&(u.MCID=i.mcid),"Artifact"===r&&("string"==typeof i.type&&(u.Type=i.type),Array.isArray(i.bbox)&&(u.BBox=[i.bbox[0],this.page.height-i.bbox[3],i.bbox[2],this.page.height-i.bbox[1]]),Array.isArray(i.attached)&&i.attached.every((function(r){return"string"==typeof r}))&&(u.Attached=i.attached)),"Span"===r&&(i.lang&&(u.Lang=new String(i.lang)),i.alt&&(u.Alt=new String(i.alt)),i.expanded&&(u.E=new String(i.expanded)),i.actual&&(u.ActualText=new String(i.actual))),this.addContent("/".concat(r," ").concat(W.convert(u)," BDC")),this},markStructureContent:function markStructureContent(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=this.getStructParentTree().get(this.page.structParentTreeKey),u=o.length;o.push(null),this.markContent(r,_objectSpread2(_objectSpread2({},i),{},{mcid:u}));var p=new ut(this.page.dictionary,u);return this.page.markings.slice(-1)[0].structContent=p,p},endMarkedContent:function endMarkedContent(){return this.page.markings.pop(),this.addContent("EMC"),this},struct:function struct(r){return new ft(this,r,arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},arguments.length>2&&void 0!==arguments[2]?arguments[2]:null)},addStructure:function addStructure(r){var i=this.getStructTreeRoot();return r.setParent(i),r.setAttached(),this.structChildren.push(r),i.data.K||(i.data.K=[]),i.data.K.push(r.dictionary),this},initPageMarkings:function initPageMarkings(r){var i=this;r.forEach((function(r){if(r.structContent){var o=r.structContent,u=i.markStructureContent(r.tag,r.options);o.push(u),i.page.markings.slice(-1)[0].structContent=o}else i.markContent(r.tag,r.options)}))},endPageMarkings:function endPageMarkings(r){var i=r.markings;return i.forEach((function(){return r.write("EMC")})),r.markings=[],i},getMarkInfoDictionary:function getMarkInfoDictionary(){return this._root.data.MarkInfo||(this._root.data.MarkInfo=this.ref({})),this._root.data.MarkInfo},getStructTreeRoot:function getStructTreeRoot(){return this._root.data.StructTreeRoot||(this._root.data.StructTreeRoot=this.ref({Type:"StructTreeRoot",ParentTree:new ht,ParentTreeNextKey:0})),this._root.data.StructTreeRoot},getStructParentTree:function getStructParentTree(){return this.getStructTreeRoot().data.ParentTree},createStructParentTreeNextKey:function createStructParentTreeNextKey(){this.getMarkInfoDictionary();var r=this.getStructTreeRoot(),i=r.data.ParentTreeNextKey++;return r.data.ParentTree.add(i,[]),i},endMarkings:function endMarkings(){var r=this._root.data.StructTreeRoot;r&&(r.end(),this.structChildren.forEach((function(r){return r.end()}))),this._root.data.MarkInfo&&this._root.data.MarkInfo.end()}},pt={readOnly:1,required:2,noExport:4,multiline:4096,password:8192,toggleToOffButton:16384,radioButton:32768,pushButton:65536,combo:131072,edit:262144,sort:524288,multiSelect:2097152,noSpell:4194304},gt={left:0,center:1,right:2},yt={value:"V",defaultValue:"DV"},vt={zip:"0",zipPlus4:"1",zip4:"1",phone:"2",ssn:"3"},mt={nDec:0,sepComma:!1,negStyle:"MinusBlack",currency:"",currencyPrepend:!0},bt={nDec:0,sepComma:!1},wt={initForm:function initForm(){if(!this._font)throw new Error("Must set a font before calling initForm method");this._acroform={fonts:{},defaultFont:this._font.name},this._acroform.fonts[this._font.id]=this._font.ref();var r={Fields:[],NeedAppearances:!0,DA:new String("/".concat(this._font.id," 0 Tf 0 g")),DR:{Font:{}}};r.DR.Font[this._font.id]=this._font.ref();var i=this.ref(r);return this._root.data.AcroForm=i,this},endAcroForm:function endAcroForm(){var r=this;if(this._root.data.AcroForm){if(!Object.keys(this._acroform.fonts).length&&!this._acroform.defaultFont)throw new Error("No fonts specified for PDF form");var i=this._root.data.AcroForm.data.DR.Font;Object.keys(this._acroform.fonts).forEach((function(o){i[o]=r._acroform.fonts[o]})),this._root.data.AcroForm.data.Fields.forEach((function(i){r._endChild(i)})),this._root.data.AcroForm.end()}return this},_endChild:function _endChild(r){var i=this;return Array.isArray(r.data.Kids)&&(r.data.Kids.forEach((function(r){i._endChild(r)})),r.end()),this},formField:function formField(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=this._fieldDict(r,null,i),u=this.ref(o);return this._addToParent(u),u},formAnnotation:function formAnnotation(r,i,o,u,p,g){var y=arguments.length>6&&void 0!==arguments[6]?arguments[6]:{},w=this._fieldDict(r,i,y);w.Subtype="Widget",void 0===w.F&&(w.F=4),this.annotate(o,u,p,g,w);var _=this.page.annotations[this.page.annotations.length-1];return this._addToParent(_)},formText:function formText(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"text",i,o,u,p,g)},formPushButton:function formPushButton(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"pushButton",i,o,u,p,g)},formCombo:function formCombo(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"combo",i,o,u,p,g)},formList:function formList(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"list",i,o,u,p,g)},formRadioButton:function formRadioButton(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"radioButton",i,o,u,p,g)},formCheckbox:function formCheckbox(r,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(r,"checkbox",i,o,u,p,g)},_addToParent:function _addToParent(r){var i=r.data.Parent;return i?(i.data.Kids||(i.data.Kids=[]),i.data.Kids.push(r)):this._root.data.AcroForm.data.Fields.push(r),this},_fieldDict:function _fieldDict(r,i){var o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};if(!this._acroform)throw new Error("Call document.initForms() method before adding form elements to document");var u=Object.assign({},o);return null!==i&&(u=this._resolveType(i,o)),u=this._resolveFlags(u),u=this._resolveJustify(u),u=this._resolveFont(u),u=this._resolveStrings(u),u=this._resolveColors(u),(u=this._resolveFormat(u)).T=new String(r),u.parent&&(u.Parent=u.parent,delete u.parent),u},_resolveType:function _resolveType(r,i){if("text"===r)i.FT="Tx";else if("pushButton"===r)i.FT="Btn",i.pushButton=!0;else if("radioButton"===r)i.FT="Btn",i.radioButton=!0;else if("checkbox"===r)i.FT="Btn";else if("combo"===r)i.FT="Ch",i.combo=!0;else{if("list"!==r)throw new Error("Invalid form annotation type '".concat(r,"'"));i.FT="Ch"}return i},_resolveFormat:function _resolveFormat(r){var i=r.format;if(i&&i.type){var o,u,p="";if(void 0!==vt[i.type])o="AFSpecial_Keystroke",u="AFSpecial_Format",p=vt[i.type];else{var g=i.type.charAt(0).toUpperCase()+i.type.slice(1);if(o="AF".concat(g,"_Keystroke"),u="AF".concat(g,"_Format"),"date"===i.type)o+="Ex",p=String(i.param);else if("time"===i.type)p=String(i.param);else if("number"===i.type){var y=Object.assign({},mt,i);p=String([String(y.nDec),y.sepComma?"0":"1",'"'+y.negStyle+'"',"null",'"'+y.currency+'"',String(y.currencyPrepend)].join(","))}else if("percent"===i.type){var w=Object.assign({},bt,i);p=String([String(w.nDec),w.sepComma?"0":"1"].join(","))}}r.AA=r.AA?r.AA:{},r.AA.K={S:"JavaScript",JS:new String("".concat(o,"(").concat(p,");"))},r.AA.F={S:"JavaScript",JS:new String("".concat(u,"(").concat(p,");"))}}return delete r.format,r},_resolveColors:function _resolveColors(r){var i=this._normalizeColor(r.backgroundColor);return i&&(r.MK||(r.MK={}),r.MK.BG=i),(i=this._normalizeColor(r.borderColor))&&(r.MK||(r.MK={}),r.MK.BC=i),delete r.backgroundColor,delete r.borderColor,r},_resolveFlags:function _resolveFlags(r){var i=0;return Object.keys(r).forEach((function(o){pt[o]&&(i|=pt[o],delete r[o])})),0!==i&&(r.Ff=r.Ff?r.Ff:0,r.Ff|=i),r},_resolveJustify:function _resolveJustify(r){var i=0;return void 0!==r.align&&("number"==typeof gt[r.align]&&(i=gt[r.align]),delete r.align),0!==i&&(r.Q=i),r},_resolveFont:function _resolveFont(r){if(null===this._acroform.fonts[this._font.id]&&(this._acroform.fonts[this._font.id]=this._font.ref()),this._acroform.defaultFont!==this._font.name){r.DR={Font:{}};var i=r.fontSize||0;r.DR.Font[this._font.id]=this._font.ref(),r.DA=new String("/".concat(this._font.id," ").concat(i," Tf 0 g"))}return r},_resolveStrings:function _resolveStrings(r){var i=[];function appendChoices(r){if(Array.isArray(r))for(var o=0;o<r.length;o++)"string"==typeof r[o]?i.push(new String(r[o])):i.push(r[o])}return appendChoices(r.Opt),r.select&&(appendChoices(r.select),delete r.select),i.length&&(r.Opt=i),Object.keys(yt).forEach((function(i){void 0!==r[i]&&(r[yt[i]]=r[i],delete r[i])})),["V","DV"].forEach((function(i){"string"==typeof r[i]&&(r[i]=new String(r[i]))})),r.MK&&r.MK.CA&&(r.MK.CA=new String(r.MK.CA)),r.label&&(r.MK=r.MK?r.MK:{},r.MK.CA=new String(r.label),delete r.label),r}},_t={file:function file(r){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};i.name=i.name||r;var o,u={Type:"EmbeddedFile",Params:{}};if(!r)throw new Error("No src specified");if(p.isBuffer(r))o=r;else if(r instanceof ArrayBuffer)o=p.from(new Uint8Array(r));else{var g;if(g=/^data:(.*);base64,(.*)$/.exec(r))g[1]&&(u.Subtype=g[1].replace("/","#2F")),o=p.from(g[2],"base64");else{if(!(o=O.readFileSync(r)))throw new Error("Could not read contents of file at filepath ".concat(r));var y=O.statSync(r),_=y.birthtime,x=y.ctime;u.Params.CreationDate=_,u.Params.ModDate=x}}i.creationDate instanceof Date&&(u.Params.CreationDate=i.creationDate),i.modifiedDate instanceof Date&&(u.Params.ModDate=i.modifiedDate),i.type&&(u.Subtype=i.type.replace("/","#2F"));var k,P=w.default.MD5(w.default.lib.WordArray.create(new Uint8Array(o)));u.Params.CheckSum=new String(P),u.Params.Size=o.byteLength,this._fileRegistry||(this._fileRegistry={});var file=this._fileRegistry[i.name];file&&function isEqual(r,i){return r.Subtype===i.Subtype&&r.Params.CheckSum.toString()===i.Params.CheckSum.toString()&&r.Params.Size===i.Params.Size&&r.Params.CreationDate===i.Params.CreationDate&&r.Params.ModDate===i.Params.ModDate}(u,file)?k=file.ref:((k=this.ref(u)).end(o),this._fileRegistry[i.name]=_objectSpread2(_objectSpread2({},u),{},{ref:k}));var I={Type:"Filespec",F:new String(i.name),EF:{F:k},UF:new String(i.name)};i.description&&(I.Desc=new String(i.description));var B=this.ref(I);return B.end(),i.hidden||this.addNamedEmbeddedFile(i.name,B),B}};var St={initPDFA:function initPDFA(r){"-"===r.charAt(r.length-3)?(this.subset_conformance=r.charAt(r.length-1).toUpperCase(),this.subset=parseInt(r.charAt(r.length-2))):(this.subset_conformance="B",this.subset=parseInt(r.charAt(r.length-1)))},endSubset:function endSubset(){this._addPdfaMetadata();var r="".concat(u,"/data/sRGB_IEC61966_2_1.icc"),i="".concat(u,"/../color_profiles/sRGB_IEC61966_2_1.icc");this._addColorOutputIntent(O.existsSync(r)?r:i)},_addColorOutputIntent:function _addColorOutputIntent(r){var i=O.readFileSync(r),o=this.ref({Length:i.length,N:3});o.write(i),o.end();var u=this.ref({Type:"OutputIntent",S:"GTS_PDFA1",Info:new String("sRGB IEC61966-2.1"),OutputConditionIdentifier:new String("sRGB IEC61966-2.1"),DestOutputProfile:o});u.end(),this._root.data.OutputIntents=[u]},_getPdfaid:function _getPdfaid(){return'\n        <rdf:Description xmlns:pdfaid="http://www.aiim.org/pdfa/ns/id/" rdf:about="">\n            <pdfaid:part>'.concat(this.subset,"</pdfaid:part>\n            <pdfaid:conformance>").concat(this.subset_conformance,"</pdfaid:conformance>\n        </rdf:Description>\n        ")},_addPdfaMetadata:function _addPdfaMetadata(){this.appendXML(this._getPdfaid())}},xt={_importSubset:function _importSubset(r){Object.assign(this,r)},initSubset:function initSubset(r){switch(r.subset){case"PDF/A-1":case"PDF/A-1a":case"PDF/A-1b":case"PDF/A-2":case"PDF/A-2a":case"PDF/A-2b":case"PDF/A-3":case"PDF/A-3a":case"PDF/A-3b":this._importSubset(St),this.initPDFA(r.subset)}}},Ct=function(){function PDFMetadata(){_classCallCheck(this,PDFMetadata),this._metadata='\n        <?xpacket begin="\ufeff" id="W5M0MpCehiHzreSzNTczkc9d"?>\n            <x:xmpmeta xmlns:x="adobe:ns:meta/">\n                <rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">\n        '}return _createClass(PDFMetadata,[{key:"_closeTags",value:function _closeTags(){this._metadata=this._metadata.concat('\n                </rdf:RDF>\n            </x:xmpmeta>\n        <?xpacket end="w"?>\n        ')}},{key:"append",value:function append(r){var i=!(arguments.length>1&&void 0!==arguments[1])||arguments[1];this._metadata=this._metadata.concat(r),i&&(this._metadata=this._metadata.concat("\n"))}},{key:"getXML",value:function getXML(){return this._metadata}},{key:"getLength",value:function getLength(){return this._metadata.length}},{key:"end",value:function end(){this._closeTags(),this._metadata=this._metadata.trim()}}]),PDFMetadata}(),kt={initMetadata:function initMetadata(){this.metadata=new Ct},appendXML:function appendXML(r){var i=!(arguments.length>1&&void 0!==arguments[1])||arguments[1];this.metadata.append(r,i)},_addInfo:function _addInfo(){this.appendXML('\n        <rdf:Description rdf:about="" xmlns:xmp="http://ns.adobe.com/xap/1.0/">\n            <xmp:CreateDate>'.concat(this.info.CreationDate.toISOString().split(".")[0]+"Z","</xmp:CreateDate>\n            <xmp:CreatorTool>").concat(this.info.Creator,"</xmp:CreatorTool>\n        </rdf:Description>\n        ")),(this.info.Title||this.info.Author||this.info.Subject)&&(this.appendXML('\n            <rdf:Description rdf:about="" xmlns:dc="http://purl.org/dc/elements/1.1/">\n            '),this.info.Title&&this.appendXML('\n                <dc:title>\n                    <rdf:Alt>\n                        <rdf:li xml:lang="x-default">'.concat(this.info.Title,"</rdf:li>\n                    </rdf:Alt>\n                </dc:title>\n                ")),this.info.Author&&this.appendXML("\n                <dc:creator>\n                    <rdf:Seq>\n                        <rdf:li>".concat(this.info.Author,"</rdf:li>\n                    </rdf:Seq>\n                </dc:creator>\n                ")),this.info.Subject&&this.appendXML('\n                <dc:description>\n                    <rdf:Alt>\n                        <rdf:li xml:lang="x-default">'.concat(this.info.Subject,"</rdf:li>\n                    </rdf:Alt>\n                </dc:description>\n                ")),this.appendXML("\n            </rdf:Description>\n            ")),this.appendXML('\n        <rdf:Description rdf:about="" xmlns:pdf="http://ns.adobe.com/pdf/1.3/">\n            <pdf:Producer>'.concat(this.info.Creator,"</pdf:Producer>"),!1),this.info.Keywords&&this.appendXML("\n            <pdf:Keywords>".concat(this.info.Keywords,"</pdf:Keywords>"),!1),this.appendXML("\n        </rdf:Description>\n        ")},endMetadata:function endMetadata(){this._addInfo(),this.metadata.end(),1.3!=this.version&&(this.metadataRef=this.ref({length:this.metadata.getLength(),Type:"Metadata",Subtype:"XML"}),this.metadataRef.compress=!1,this.metadataRef.write(p.from(this.metadata.getXML(),"utf-8")),this.metadataRef.end(),this._root.data.Metadata=this.metadataRef)}},Pt=function(r){_inherits(PDFDocument,r);var i=_createSuper(PDFDocument);function PDFDocument(){var r,o=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};switch(_classCallCheck(this,PDFDocument),(r=i.call(this,o)).options=o,o.pdfVersion){case"1.4":r.version=1.4;break;case"1.5":r.version=1.5;break;case"1.6":r.version=1.6;break;case"1.7":case"1.7ext3":r.version=1.7;break;default:r.version=1.3}r.compress=null==r.options.compress||r.options.compress,r._pageBuffer=[],r._pageBufferStart=0,r._offsets=[],r._waiting=0,r._ended=!1,r._offset=0;var u=r.ref({Type:"Pages",Count:0,Kids:[]}),p=r.ref({Dests:new K});if(r._root=r.ref({Type:"Catalog",Pages:u,Names:p}),r.options.lang&&(r._root.data.Lang=new String(r.options.lang)),r.page=null,r.initMetadata(),r.initColor(),r.initVector(),r.initFonts(o.font),r.initText(),r.initImages(),r.initOutline(),r.initMarkings(o),r.initSubset(o),r.info={Producer:"PDFKit",Creator:"PDFKit",CreationDate:new Date},r.options.info)for(var g in r.options.info){var y=r.options.info[g];r.info[g]=y}return r.options.displayTitle&&(r._root.data.ViewerPreferences=r.ref({DisplayDocTitle:!0})),r._id=pe.generateFileID(r.info),r._security=pe.create(_assertThisInitialized(r),o),r._write("%PDF-".concat(r.version)),r._write("%ÿÿÿÿ"),!1!==r.options.autoFirstPage&&r.addPage(),r}return _createClass(PDFDocument,[{key:"addPage",value:function addPage(r){null==r&&(r=this.options),this.options.bufferPages||this.flushPages(),this.page=new X(this,r),this._pageBuffer.push(this.page);var i=this._root.data.Pages.data;return i.Kids.push(this.page.dictionary),i.Count++,this.x=this.page.margins.left,this.y=this.page.margins.top,this._ctm=[1,0,0,1,0,0],this.transform(1,0,0,-1,0,this.page.height),this.emit("pageAdded"),this}},{key:"continueOnNewPage",value:function continueOnNewPage(r){var i=this.endPageMarkings(this.page);return this.addPage(r),this.initPageMarkings(i),this}},{key:"bufferedPageRange",value:function bufferedPageRange(){return{start:this._pageBufferStart,count:this._pageBuffer.length}}},{key:"switchToPage",value:function switchToPage(r){var i;if(!(i=this._pageBuffer[r-this._pageBufferStart]))throw new Error("switchToPage(".concat(r,") out of bounds, current buffer covers pages ").concat(this._pageBufferStart," to ").concat(this._pageBufferStart+this._pageBuffer.length-1));return this.page=i}},{key:"flushPages",value:function flushPages(){var r=this._pageBuffer;this._pageBuffer=[],this._pageBufferStart+=r.length;var i,o=_createForOfIteratorHelper(r);try{for(o.s();!(i=o.n()).done;){var u=i.value;this.endPageMarkings(u),u.end()}}catch(r){o.e(r)}finally{o.f()}}},{key:"addNamedDestination",value:function addNamedDestination(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];0===o.length&&(o=["XYZ",null,null,null]),"XYZ"===o[0]&&null!==o[2]&&(o[2]=this.page.height-o[2]),o.unshift(this.page.dictionary),this._root.data.Names.data.Dests.add(r,o)}},{key:"addNamedEmbeddedFile",value:function addNamedEmbeddedFile(r,i){this._root.data.Names.data.EmbeddedFiles||(this._root.data.Names.data.EmbeddedFiles=new K({limits:!1})),this._root.data.Names.data.EmbeddedFiles.add(r,i)}},{key:"addNamedJavaScript",value:function addNamedJavaScript(r,i){this._root.data.Names.data.JavaScript||(this._root.data.Names.data.JavaScript=new K);var o={JS:new String(i),S:"JavaScript"};this._root.data.Names.data.JavaScript.add(r,o)}},{key:"ref",value:function ref(r){var ref=new U(this,this._offsets.length+1,r);return this._offsets.push(null),this._waiting++,ref}},{key:"_read",value:function _read(){}},{key:"_write",value:function _write(r){return p.isBuffer(r)||(r=p.from(r+"\n","binary")),this.push(r),this._offset+=r.length}},{key:"addContent",value:function addContent(r){return this.page.write(r),this}},{key:"_refEnd",value:function _refEnd(r){if(this._offsets[r.id-1]=r.offset,0==--this._waiting&&this._ended)return this._finalize(),this._ended=!1}},{key:"write",value:function write(r,i){new Error("PDFDocument#write is deprecated, and will be removed in a future version of PDFKit. Please pipe the document into a Node stream.");return this.pipe(O.createWriteStream(r)),this.end(),this.once("end",i)}},{key:"end",value:function end(){for(var r in this.flushPages(),this._info=this.ref(),this.info){var i=this.info[r];"string"==typeof i&&(i=new String(i));var o=this.ref(i);o.end(),this._info.data[r]=o}for(var u in this._info.end(),this._fontFamilies){this._fontFamilies[u].finalize()}return this.endOutline(),this.endMarkings(),this.subset&&this.endSubset(),this.endMetadata(),this._root.end(),this._root.data.Pages.end(),this._root.data.Names.end(),this.endAcroForm(),this._root.data.ViewerPreferences&&this._root.data.ViewerPreferences.end(),this._security&&this._security.end(),0===this._waiting?this._finalize():this._ended=!0}},{key:"_finalize",value:function _finalize(){var r=this._offset;this._write("xref"),this._write("0 ".concat(this._offsets.length+1)),this._write("0000000000 65535 f ");var i,o=_createForOfIteratorHelper(this._offsets);try{for(o.s();!(i=o.n()).done;){var u=i.value;u="0000000000".concat(u).slice(-10),this._write(u+" 00000 n ")}}catch(r){o.e(r)}finally{o.f()}var p={Size:this._offsets.length+1,Root:this._root,Info:this._info,ID:[this._id,this._id]};return this._security&&(p.Encrypt=this._security.dictionary),this._write("trailer"),this._write(W.convert(p)),this._write("startxref"),this._write("".concat(r)),this._write("%%EOF"),this.push(null)}},{key:"toString",value:function toString(){return"[object PDFDocument]"}}]),PDFDocument}(g.default.Readable),At=function mixin(r){Object.assign(Pt.prototype,r)};At(kt),At(Ie),At(Ue),At(Je),At(et),At(at),At(st),At(lt),At(dt),At(wt),At(_t),At(xt),Pt.LineWrapper=Qe;i.default=Pt},36164:function(r,i,o){var u=o(45314),p=o(72022);void 0===p.pdfMake&&(p.pdfMake=u),r.exports=u},80182:function(r,i,o){"use strict";var u=o(9964);function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(r){return typeof r}:function(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,(p=u.key,g=void 0,g=function _toPrimitive(r,i){if("object"!==_typeof(r)||null===r)return r;var o=r[Symbol.toPrimitive];if(void 0!==o){var u=o.call(r,i||"default");if("object"!==_typeof(u))return u;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===i?String:Number)(r)}(p,"string"),"symbol"===_typeof(g)?g:String(g)),u)}var p,g}function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}var p,g,y=o(35403).codes,w=y.ERR_AMBIGUOUS_ARGUMENT,_=y.ERR_INVALID_ARG_TYPE,x=y.ERR_INVALID_ARG_VALUE,k=y.ERR_INVALID_RETURN_VALUE,P=y.ERR_MISSING_ARGS,O=o(85416),I=o(7187).inspect,B=o(7187).types,D=B.isPromise,R=B.isRegExp,N=o(37596)(),W=o(89636)(),U=o(67913)("RegExp.prototype.test");new Map;function lazyLoadComparison(){var r=o(86781);p=r.isDeepEqual,g=r.isDeepStrictEqual}var j=!1,G=r.exports=ok,X={};function innerFail(r){if(r.message instanceof Error)throw r.message;throw new O(r)}function innerOk(r,i,o,u){if(!o){var p=!1;if(0===i)p=!0,u="No value argument passed to `assert.ok()`";else if(u instanceof Error)throw u;var g=new O({actual:o,expected:!0,message:u,operator:"==",stackStartFn:r});throw g.generatedMessage=p,g}}function ok(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];innerOk.apply(void 0,[ok,i.length].concat(i))}G.fail=function fail(r,i,o,p,g){var y,w=arguments.length;if(0===w)y="Failed";else if(1===w)o=r,r=void 0;else{if(!1===j)j=!0,(u.emitWarning?u.emitWarning:void 0)("assert.fail() with more than one argument is deprecated. Please use assert.strictEqual() instead or only pass a message.","DeprecationWarning","DEP0094");2===w&&(p="!=")}if(o instanceof Error)throw o;var _={actual:r,expected:i,operator:void 0===p?"fail":p,stackStartFn:g||fail};void 0!==o&&(_.message=o);var x=new O(_);throw y&&(x.message=y,x.generatedMessage=!0),x},G.AssertionError=O,G.ok=ok,G.equal=function equal(r,i,o){if(arguments.length<2)throw new P("actual","expected");r!=i&&innerFail({actual:r,expected:i,message:o,operator:"==",stackStartFn:equal})},G.notEqual=function notEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");r==i&&innerFail({actual:r,expected:i,message:o,operator:"!=",stackStartFn:notEqual})},G.deepEqual=function deepEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),p(r,i)||innerFail({actual:r,expected:i,message:o,operator:"deepEqual",stackStartFn:deepEqual})},G.notDeepEqual=function notDeepEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),p(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notDeepEqual",stackStartFn:notDeepEqual})},G.deepStrictEqual=function deepStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison(),g(r,i)||innerFail({actual:r,expected:i,message:o,operator:"deepStrictEqual",stackStartFn:deepStrictEqual})},G.notDeepStrictEqual=function notDeepStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");void 0===p&&lazyLoadComparison();g(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notDeepStrictEqual",stackStartFn:notDeepStrictEqual})},G.strictEqual=function strictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");W(r,i)||innerFail({actual:r,expected:i,message:o,operator:"strictEqual",stackStartFn:strictEqual})},G.notStrictEqual=function notStrictEqual(r,i,o){if(arguments.length<2)throw new P("actual","expected");W(r,i)&&innerFail({actual:r,expected:i,message:o,operator:"notStrictEqual",stackStartFn:notStrictEqual})};var K=_createClass((function Comparison(r,i,o){var u=this;!function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,Comparison),i.forEach((function(i){i in r&&(void 0!==o&&"string"==typeof o[i]&&R(r[i])&&U(r[i],o[i])?u[i]=o[i]:u[i]=r[i])}))}));function expectedException(r,i,o,u){if("function"!=typeof i){if(R(i))return U(i,r);if(2===arguments.length)throw new _("expected",["Function","RegExp"],i);if("object"!==_typeof(r)||null===r){var y=new O({actual:r,expected:i,message:o,operator:"deepStrictEqual",stackStartFn:u});throw y.operator=u.name,y}var w=Object.keys(i);if(i instanceof Error)w.push("name","message");else if(0===w.length)throw new x("error",i,"may not be an empty object");return void 0===p&&lazyLoadComparison(),w.forEach((function(p){"string"==typeof r[p]&&R(i[p])&&U(i[p],r[p])||function compareExceptionKey(r,i,o,u,p,y){if(!(o in r)||!g(r[o],i[o])){if(!u){var w=new K(r,p),_=new K(i,p,r),x=new O({actual:w,expected:_,operator:"deepStrictEqual",stackStartFn:y});throw x.actual=r,x.expected=i,x.operator=y.name,x}innerFail({actual:r,expected:i,message:u,operator:y.name,stackStartFn:y})}}(r,i,p,o,w,u)})),!0}return void 0!==i.prototype&&r instanceof i||!Error.isPrototypeOf(i)&&!0===i.call({},r)}function getActual(r){if("function"!=typeof r)throw new _("fn","Function",r);try{r()}catch(r){return r}return X}function checkIsPromise(r){return D(r)||null!==r&&"object"===_typeof(r)&&"function"==typeof r.then&&"function"==typeof r.catch}function waitForActual(r){return Promise.resolve().then((function(){var i;if("function"==typeof r){if(!checkIsPromise(i=r()))throw new k("instance of Promise","promiseFn",i)}else{if(!checkIsPromise(r))throw new _("promiseFn",["Function","Promise"],r);i=r}return Promise.resolve().then((function(){return i})).then((function(){return X})).catch((function(r){return r}))}))}function expectsError(r,i,o,u){if("string"==typeof o){if(4===arguments.length)throw new _("error",["Object","Error","Function","RegExp"],o);if("object"===_typeof(i)&&null!==i){if(i.message===o)throw new w("error/message",'The error message "'.concat(i.message,'" is identical to the message.'))}else if(i===o)throw new w("error/message",'The error "'.concat(i,'" is identical to the message.'));u=o,o=void 0}else if(null!=o&&"object"!==_typeof(o)&&"function"!=typeof o)throw new _("error",["Object","Error","Function","RegExp"],o);if(i===X){var p="";o&&o.name&&(p+=" (".concat(o.name,")")),p+=u?": ".concat(u):".";var g="rejects"===r.name?"rejection":"exception";innerFail({actual:void 0,expected:o,operator:r.name,message:"Missing expected ".concat(g).concat(p),stackStartFn:r})}if(o&&!expectedException(i,o,u,r))throw i}function expectsNoError(r,i,o,u){if(i!==X){if("string"==typeof o&&(u=o,o=void 0),!o||expectedException(i,o)){var p=u?": ".concat(u):".",g="doesNotReject"===r.name?"rejection":"exception";innerFail({actual:i,expected:o,operator:r.name,message:"Got unwanted ".concat(g).concat(p,"\n")+'Actual message: "'.concat(i&&i.message,'"'),stackStartFn:r})}throw i}}function internalMatch(r,i,o,u,p){if(!R(i))throw new _("regexp","RegExp",i);var g="match"===p;if("string"!=typeof r||U(i,r)!==g){if(o instanceof Error)throw o;var y=!o;o=o||("string"!=typeof r?'The "string" argument must be of type string. Received type '+"".concat(_typeof(r)," (").concat(I(r),")"):(g?"The input did not match the regular expression ":"The input was expected to not match the regular expression ")+"".concat(I(i),". Input:\n\n").concat(I(r),"\n"));var w=new O({actual:r,expected:i,message:o,operator:p,stackStartFn:u});throw w.generatedMessage=y,w}}function strict(){for(var r=arguments.length,i=new Array(r),o=0;o<r;o++)i[o]=arguments[o];innerOk.apply(void 0,[strict,i.length].concat(i))}G.throws=function throws(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];expectsError.apply(void 0,[throws,getActual(r)].concat(o))},G.rejects=function rejects(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];return waitForActual(r).then((function(r){return expectsError.apply(void 0,[rejects,r].concat(o))}))},G.doesNotThrow=function doesNotThrow(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];expectsNoError.apply(void 0,[doesNotThrow,getActual(r)].concat(o))},G.doesNotReject=function doesNotReject(r){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];return waitForActual(r).then((function(r){return expectsNoError.apply(void 0,[doesNotReject,r].concat(o))}))},G.ifError=function ifError(r){if(null!=r){var i="ifError got unwanted exception: ";"object"===_typeof(r)&&"string"==typeof r.message?0===r.message.length&&r.constructor?i+=r.constructor.name:i+=r.message:i+=I(r);var o=new O({actual:r,expected:null,operator:"ifError",message:i,stackStartFn:ifError}),u=r.stack;if("string"==typeof u){var p=u.split("\n");p.shift();for(var g=o.stack.split("\n"),y=0;y<p.length;y++){var w=g.indexOf(p[y]);if(-1!==w){g=g.slice(0,w);break}}o.stack="".concat(g.join("\n"),"\n").concat(p.join("\n"))}throw o}},G.match=function match(r,i,o){internalMatch(r,i,o,match,"match")},G.doesNotMatch=function doesNotMatch(r,i,o){internalMatch(r,i,o,doesNotMatch,"doesNotMatch")},G.strict=N(strict,G,{equal:G.strictEqual,deepEqual:G.deepStrictEqual,notEqual:G.notStrictEqual,notDeepEqual:G.notDeepStrictEqual}),G.strict.strict=G.strict},85416:function(r,i,o){"use strict";var u=o(9964);function ownKeys(r,i){var o=Object.keys(r);if(Object.getOwnPropertySymbols){var u=Object.getOwnPropertySymbols(r);i&&(u=u.filter((function(i){return Object.getOwnPropertyDescriptor(r,i).enumerable}))),o.push.apply(o,u)}return o}function _objectSpread(r){for(var i=1;i<arguments.length;i++){var o=null!=arguments[i]?arguments[i]:{};i%2?ownKeys(Object(o),!0).forEach((function(i){_defineProperty(r,i,o[i])})):Object.getOwnPropertyDescriptors?Object.defineProperties(r,Object.getOwnPropertyDescriptors(o)):ownKeys(Object(o)).forEach((function(i){Object.defineProperty(r,i,Object.getOwnPropertyDescriptor(o,i))}))}return r}function _defineProperty(r,i,o){return(i=_toPropertyKey(i))in r?Object.defineProperty(r,i,{value:o,enumerable:!0,configurable:!0,writable:!0}):r[i]=o,r}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,_toPropertyKey(u.key),u)}}function _toPropertyKey(r){var i=function _toPrimitive(r,i){if("object"!==_typeof(r)||null===r)return r;var o=r[Symbol.toPrimitive];if(void 0!==o){var u=o.call(r,i||"default");if("object"!==_typeof(u))return u;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===i?String:Number)(r)}(r,"string");return"symbol"===_typeof(i)?i:String(i)}function _possibleConstructorReturn(r,i){if(i&&("object"===_typeof(i)||"function"==typeof i))return i;if(void 0!==i)throw new TypeError("Derived constructors may only return object or undefined");return _assertThisInitialized(r)}function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}function _wrapNativeSuper(r){var i="function"==typeof Map?new Map:void 0;return _wrapNativeSuper=function _wrapNativeSuper(r){if(null===r||!function _isNativeFunction(r){return-1!==Function.toString.call(r).indexOf("[native code]")}(r))return r;if("function"!=typeof r)throw new TypeError("Super expression must either be null or a function");if(void 0!==i){if(i.has(r))return i.get(r);i.set(r,Wrapper)}function Wrapper(){return _construct(r,arguments,_getPrototypeOf(this).constructor)}return Wrapper.prototype=Object.create(r.prototype,{constructor:{value:Wrapper,enumerable:!1,writable:!0,configurable:!0}}),_setPrototypeOf(Wrapper,r)},_wrapNativeSuper(r)}function _construct(r,i,o){return _construct=_isNativeReflectConstruct()?Reflect.construct.bind():function _construct(r,i,o){var u=[null];u.push.apply(u,i);var p=new(Function.bind.apply(r,u));return o&&_setPrototypeOf(p,o.prototype),p},_construct.apply(null,arguments)}function _isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Boolean.prototype.valueOf.call(Reflect.construct(Boolean,[],(function(){}))),!0}catch(r){return!1}}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf.bind():function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(r){return typeof r}:function(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}var p=o(7187).inspect,g=o(35403).codes.ERR_INVALID_ARG_TYPE;function endsWith(r,i,o){return(void 0===o||o>r.length)&&(o=r.length),r.substring(o-i.length,o)===i}var y="",w="",_="",x="",k={deepStrictEqual:"Expected values to be strictly deep-equal:",strictEqual:"Expected values to be strictly equal:",strictEqualObject:'Expected "actual" to be reference-equal to "expected":',deepEqual:"Expected values to be loosely deep-equal:",equal:"Expected values to be loosely equal:",notDeepStrictEqual:'Expected "actual" not to be strictly deep-equal to:',notStrictEqual:'Expected "actual" to be strictly unequal to:',notStrictEqualObject:'Expected "actual" not to be reference-equal to "expected":',notDeepEqual:'Expected "actual" not to be loosely deep-equal to:',notEqual:'Expected "actual" to be loosely unequal to:',notIdentical:"Values identical but not reference-equal:"};function copyError(r){var i=Object.keys(r),o=Object.create(Object.getPrototypeOf(r));return i.forEach((function(i){o[i]=r[i]})),Object.defineProperty(o,"message",{value:r.message}),o}function inspectValue(r){return p(r,{compact:!1,customInspect:!1,depth:1e3,maxArrayLength:1/0,showHidden:!1,breakLength:1/0,showProxy:!1,sorted:!0,getters:!0})}function createErrDiff(r,i,o){var p="",g="",P=0,O="",I=!1,B=inspectValue(r),D=B.split("\n"),R=inspectValue(i).split("\n"),N=0,W="";if("strictEqual"===o&&"object"===_typeof(r)&&"object"===_typeof(i)&&null!==r&&null!==i&&(o="strictEqualObject"),1===D.length&&1===R.length&&D[0]!==R[0]){var U=D[0].length+R[0].length;if(U<=10){if(!("object"===_typeof(r)&&null!==r||"object"===_typeof(i)&&null!==i||0===r&&0===i))return"".concat(k[o],"\n\n")+"".concat(D[0]," !== ").concat(R[0],"\n")}else if("strictEqualObject"!==o){if(U<(u.stderr&&u.stderr.isTTY?u.stderr.columns:80)){for(;D[0][N]===R[0][N];)N++;N>2&&(W="\n  ".concat(function repeat(r,i){if(i=Math.floor(i),0==r.length||0==i)return"";var o=r.length*i;for(i=Math.floor(Math.log(i)/Math.log(2));i;)r+=r,i--;return r+r.substring(0,o-r.length)}(" ",N),"^"),N=0)}}}for(var j=D[D.length-1],G=R[R.length-1];j===G&&(N++<2?O="\n  ".concat(j).concat(O):p=j,D.pop(),R.pop(),0!==D.length&&0!==R.length);)j=D[D.length-1],G=R[R.length-1];var X=Math.max(D.length,R.length);if(0===X){var K=B.split("\n");if(K.length>30)for(K[26]="".concat(y,"...").concat(x);K.length>27;)K.pop();return"".concat(k.notIdentical,"\n\n").concat(K.join("\n"),"\n")}N>3&&(O="\n".concat(y,"...").concat(x).concat(O),I=!0),""!==p&&(O="\n  ".concat(p).concat(O),p="");var Y=0,J=k[o]+"\n".concat(w,"+ actual").concat(x," ").concat(_,"- expected").concat(x),$=" ".concat(y,"...").concat(x," Lines skipped");for(N=0;N<X;N++){var ee=N-P;if(D.length<N+1)ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(R[N-2]),Y++),g+="\n  ".concat(R[N-1]),Y++),P=N,p+="\n".concat(_,"-").concat(x," ").concat(R[N]),Y++;else if(R.length<N+1)ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(D[N-2]),Y++),g+="\n  ".concat(D[N-1]),Y++),P=N,g+="\n".concat(w,"+").concat(x," ").concat(D[N]),Y++;else{var te=R[N],re=D[N],ne=re!==te&&(!endsWith(re,",")||re.slice(0,-1)!==te);ne&&endsWith(te,",")&&te.slice(0,-1)===re&&(ne=!1,re+=","),ne?(ee>1&&N>2&&(ee>4?(g+="\n".concat(y,"...").concat(x),I=!0):ee>3&&(g+="\n  ".concat(D[N-2]),Y++),g+="\n  ".concat(D[N-1]),Y++),P=N,g+="\n".concat(w,"+").concat(x," ").concat(re),p+="\n".concat(_,"-").concat(x," ").concat(te),Y+=2):(g+=p,p="",1!==ee&&0!==N||(g+="\n  ".concat(re),Y++))}if(Y>20&&N<X-2)return"".concat(J).concat($,"\n").concat(g,"\n").concat(y,"...").concat(x).concat(p,"\n")+"".concat(y,"...").concat(x)}return"".concat(J).concat(I?$:"","\n").concat(g).concat(p).concat(O).concat(W)}var P=function(r,i){!function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),Object.defineProperty(r,"prototype",{writable:!1}),i&&_setPrototypeOf(r,i)}(AssertionError,r);var o=function _createSuper(r){var i=_isNativeReflectConstruct();return function _createSuperInternal(){var o,u=_getPrototypeOf(r);if(i){var p=_getPrototypeOf(this).constructor;o=Reflect.construct(u,arguments,p)}else o=u.apply(this,arguments);return _possibleConstructorReturn(this,o)}}(AssertionError);function AssertionError(r){var i;if(function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,AssertionError),"object"!==_typeof(r)||null===r)throw new g("options","Object",r);var p=r.message,P=r.operator,O=r.stackStartFn,I=r.actual,B=r.expected,D=Error.stackTraceLimit;if(Error.stackTraceLimit=0,null!=p)i=o.call(this,String(p));else if(u.stderr&&u.stderr.isTTY&&(u.stderr&&u.stderr.getColorDepth&&1!==u.stderr.getColorDepth()?(y="[34m",w="[32m",x="[39m",_="[31m"):(y="",w="",x="",_="")),"object"===_typeof(I)&&null!==I&&"object"===_typeof(B)&&null!==B&&"stack"in I&&I instanceof Error&&"stack"in B&&B instanceof Error&&(I=copyError(I),B=copyError(B)),"deepStrictEqual"===P||"strictEqual"===P)i=o.call(this,createErrDiff(I,B,P));else if("notDeepStrictEqual"===P||"notStrictEqual"===P){var R=k[P],N=inspectValue(I).split("\n");if("notStrictEqual"===P&&"object"===_typeof(I)&&null!==I&&(R=k.notStrictEqualObject),N.length>30)for(N[26]="".concat(y,"...").concat(x);N.length>27;)N.pop();i=1===N.length?o.call(this,"".concat(R," ").concat(N[0])):o.call(this,"".concat(R,"\n\n").concat(N.join("\n"),"\n"))}else{var W=inspectValue(I),U="",j=k[P];"notDeepEqual"===P||"notEqual"===P?(W="".concat(k[P],"\n\n").concat(W)).length>1024&&(W="".concat(W.slice(0,1021),"...")):(U="".concat(inspectValue(B)),W.length>512&&(W="".concat(W.slice(0,509),"...")),U.length>512&&(U="".concat(U.slice(0,509),"...")),"deepEqual"===P||"equal"===P?W="".concat(j,"\n\n").concat(W,"\n\nshould equal\n\n"):U=" ".concat(P," ").concat(U)),i=o.call(this,"".concat(W).concat(U))}return Error.stackTraceLimit=D,i.generatedMessage=!p,Object.defineProperty(_assertThisInitialized(i),"name",{value:"AssertionError [ERR_ASSERTION]",enumerable:!1,writable:!0,configurable:!0}),i.code="ERR_ASSERTION",i.actual=I,i.expected=B,i.operator=P,Error.captureStackTrace&&Error.captureStackTrace(_assertThisInitialized(i),O),i.stack,i.name="AssertionError",_possibleConstructorReturn(i)}return function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}(AssertionError,[{key:"toString",value:function toString(){return"".concat(this.name," [").concat(this.code,"]: ").concat(this.message)}},{key:i,value:function value(r,i){return p(this,_objectSpread(_objectSpread({},i),{},{customInspect:!1,depth:0}))}}]),AssertionError}(_wrapNativeSuper(Error),p.custom);r.exports=P},35403:function(r,i,o){"use strict";function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(r){return typeof r}:function(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}function _defineProperties(r,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(r,(p=u.key,g=void 0,g=function _toPrimitive(r,i){if("object"!==_typeof(r)||null===r)return r;var o=r[Symbol.toPrimitive];if(void 0!==o){var u=o.call(r,i||"default");if("object"!==_typeof(u))return u;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===i?String:Number)(r)}(p,"string"),"symbol"===_typeof(g)?g:String(g)),u)}var p,g}function _setPrototypeOf(r,i){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(r,i){return r.__proto__=i,r},_setPrototypeOf(r,i)}function _createSuper(r){var i=function _isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Boolean.prototype.valueOf.call(Reflect.construct(Boolean,[],(function(){}))),!0}catch(r){return!1}}();return function _createSuperInternal(){var o,u=_getPrototypeOf(r);if(i){var p=_getPrototypeOf(this).constructor;o=Reflect.construct(u,arguments,p)}else o=u.apply(this,arguments);return function _possibleConstructorReturn(r,i){if(i&&("object"===_typeof(i)||"function"==typeof i))return i;if(void 0!==i)throw new TypeError("Derived constructors may only return object or undefined");return function _assertThisInitialized(r){if(void 0===r)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return r}(r)}(this,o)}}function _getPrototypeOf(r){return _getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf.bind():function _getPrototypeOf(r){return r.__proto__||Object.getPrototypeOf(r)},_getPrototypeOf(r)}var u,p,g={};function createErrorType(r,i,o){o||(o=Error);var u=function(o){!function _inherits(r,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");r.prototype=Object.create(i&&i.prototype,{constructor:{value:r,writable:!0,configurable:!0}}),Object.defineProperty(r,"prototype",{writable:!1}),i&&_setPrototypeOf(r,i)}(NodeError,o);var u=_createSuper(NodeError);function NodeError(o,p,g){var y;return function _classCallCheck(r,i){if(!(r instanceof i))throw new TypeError("Cannot call a class as a function")}(this,NodeError),y=u.call(this,function getMessage(r,o,u){return"string"==typeof i?i:i(r,o,u)}(o,p,g)),y.code=r,y}return function _createClass(r,i,o){return i&&_defineProperties(r.prototype,i),o&&_defineProperties(r,o),Object.defineProperty(r,"prototype",{writable:!1}),r}(NodeError)}(o);g[r]=u}function oneOf(r,i){if(Array.isArray(r)){var o=r.length;return r=r.map((function(r){return String(r)})),o>2?"one of ".concat(i," ").concat(r.slice(0,o-1).join(", "),", or ")+r[o-1]:2===o?"one of ".concat(i," ").concat(r[0]," or ").concat(r[1]):"of ".concat(i," ").concat(r[0])}return"of ".concat(i," ").concat(String(r))}createErrorType("ERR_AMBIGUOUS_ARGUMENT",'The "%s" argument is ambiguous. %s',TypeError),createErrorType("ERR_INVALID_ARG_TYPE",(function(r,i,p){var g,y;if(void 0===u&&(u=o(80182)),u("string"==typeof r,"'name' must be a string"),"string"==typeof i&&function startsWith(r,i,o){return r.substr(!o||o<0?0:+o,i.length)===i}(i,"not ")?(g="must not be",i=i.replace(/^not /,"")):g="must be",function endsWith(r,i,o){return(void 0===o||o>r.length)&&(o=r.length),r.substring(o-i.length,o)===i}(r," argument"))y="The ".concat(r," ").concat(g," ").concat(oneOf(i,"type"));else{var w=function includes(r,i,o){return"number"!=typeof o&&(o=0),!(o+i.length>r.length)&&-1!==r.indexOf(i,o)}(r,".")?"property":"argument";y='The "'.concat(r,'" ').concat(w," ").concat(g," ").concat(oneOf(i,"type"))}return y+=". Received type ".concat(_typeof(p))}),TypeError),createErrorType("ERR_INVALID_ARG_VALUE",(function(r,i){var u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:"is invalid";void 0===p&&(p=o(7187));var g=p.inspect(i);return g.length>128&&(g="".concat(g.slice(0,128),"...")),"The argument '".concat(r,"' ").concat(u,". Received ").concat(g)}),TypeError,RangeError),createErrorType("ERR_INVALID_RETURN_VALUE",(function(r,i,o){var u;return u=o&&o.constructor&&o.constructor.name?"instance of ".concat(o.constructor.name):"type ".concat(_typeof(o)),"Expected ".concat(r,' to be returned from the "').concat(i,'"')+" function but got ".concat(u,".")}),TypeError),createErrorType("ERR_MISSING_ARGS",(function(){for(var r=arguments.length,i=new Array(r),p=0;p<r;p++)i[p]=arguments[p];void 0===u&&(u=o(80182)),u(i.length>0,"At least one arg needs to be specified");var g="The ",y=i.length;switch(i=i.map((function(r){return'"'.concat(r,'"')})),y){case 1:g+="".concat(i[0]," argument");break;case 2:g+="".concat(i[0]," and ").concat(i[1]," arguments");break;default:g+=i.slice(0,y-1).join(", "),g+=", and ".concat(i[y-1]," arguments")}return"".concat(g," must be specified")}),TypeError),r.exports.codes=g},86781:function(r,i,o){"use strict";function _slicedToArray(r,i){return function _arrayWithHoles(r){if(Array.isArray(r))return r}(r)||function _iterableToArrayLimit(r,i){var o=null==r?null:"undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(null!=o){var u,p,g,y,w=[],_=!0,x=!1;try{if(g=(o=o.call(r)).next,0===i){if(Object(o)!==o)return;_=!1}else for(;!(_=(u=g.call(o)).done)&&(w.push(u.value),w.length!==i);_=!0);}catch(r){x=!0,p=r}finally{try{if(!_&&null!=o.return&&(y=o.return(),Object(y)!==y))return}finally{if(x)throw p}}return w}}(r,i)||function _unsupportedIterableToArray(r,i){if(!r)return;if("string"==typeof r)return _arrayLikeToArray(r,i);var o=Object.prototype.toString.call(r).slice(8,-1);"Object"===o&&r.constructor&&(o=r.constructor.name);if("Map"===o||"Set"===o)return Array.from(r);if("Arguments"===o||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(o))return _arrayLikeToArray(r,i)}(r,i)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _arrayLikeToArray(r,i){(null==i||i>r.length)&&(i=r.length);for(var o=0,u=new Array(i);o<i;o++)u[o]=r[o];return u}function _typeof(r){return _typeof="function"==typeof Symbol&&"symbol"==typeof Symbol.iterator?function(r){return typeof r}:function(r){return r&&"function"==typeof Symbol&&r.constructor===Symbol&&r!==Symbol.prototype?"symbol":typeof r},_typeof(r)}var u=void 0!==/a/g.flags,p=function arrayFromSet(r){var i=[];return r.forEach((function(r){return i.push(r)})),i},g=function arrayFromMap(r){var i=[];return r.forEach((function(r,o){return i.push([o,r])})),i},y=Object.is?Object.is:o(98527),w=Object.getOwnPropertySymbols?Object.getOwnPropertySymbols:function(){return[]},_=Number.isNaN?Number.isNaN:o(7051);function uncurryThis(r){return r.call.bind(r)}var x=uncurryThis(Object.prototype.hasOwnProperty),k=uncurryThis(Object.prototype.propertyIsEnumerable),P=uncurryThis(Object.prototype.toString),O=o(7187).types,I=O.isAnyArrayBuffer,B=O.isArrayBufferView,D=O.isDate,R=O.isMap,N=O.isRegExp,W=O.isSet,U=O.isNativeError,j=O.isBoxedPrimitive,G=O.isNumberObject,X=O.isStringObject,K=O.isBooleanObject,Y=O.isBigIntObject,J=O.isSymbolObject,$=O.isFloat32Array,ee=O.isFloat64Array;function isNonIndex(r){if(0===r.length||r.length>10)return!0;for(var i=0;i<r.length;i++){var o=r.charCodeAt(i);if(o<48||o>57)return!0}return 10===r.length&&r>=Math.pow(2,32)}function getOwnNonIndexProperties(r){return Object.keys(r).filter(isNonIndex).concat(w(r).filter(Object.prototype.propertyIsEnumerable.bind(r)))}function compare(r,i){if(r===i)return 0;for(var o=r.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(r[p]!==i[p]){o=r[p],u=i[p];break}return o<u?-1:u<o?1:0}var te=0,re=1,ne=2,ie=3;function innerDeepEqual(r,i,o,p){if(r===i)return 0!==r||(!o||y(r,i));if(o){if("object"!==_typeof(r))return"number"==typeof r&&_(r)&&_(i);if("object"!==_typeof(i)||null===r||null===i)return!1;if(Object.getPrototypeOf(r)!==Object.getPrototypeOf(i))return!1}else{if(null===r||"object"!==_typeof(r))return(null===i||"object"!==_typeof(i))&&r==i;if(null===i||"object"!==_typeof(i))return!1}var g=P(r);if(g!==P(i))return!1;if(Array.isArray(r)){if(r.length!==i.length)return!1;var w=getOwnNonIndexProperties(r),x=getOwnNonIndexProperties(i);return w.length===x.length&&keyCheck(r,i,o,p,re,w)}if("[object Object]"===g&&(!R(r)&&R(i)||!W(r)&&W(i)))return!1;if(D(r)){if(!D(i)||Date.prototype.getTime.call(r)!==Date.prototype.getTime.call(i))return!1}else if(N(r)){if(!N(i)||!function areSimilarRegExps(r,i){return u?r.source===i.source&&r.flags===i.flags:RegExp.prototype.toString.call(r)===RegExp.prototype.toString.call(i)}(r,i))return!1}else if(U(r)||r instanceof Error){if(r.message!==i.message||r.name!==i.name)return!1}else{if(B(r)){if(o||!$(r)&&!ee(r)){if(!function areSimilarTypedArrays(r,i){return r.byteLength===i.byteLength&&0===compare(new Uint8Array(r.buffer,r.byteOffset,r.byteLength),new Uint8Array(i.buffer,i.byteOffset,i.byteLength))}(r,i))return!1}else if(!function areSimilarFloatArrays(r,i){if(r.byteLength!==i.byteLength)return!1;for(var o=0;o<r.byteLength;o++)if(r[o]!==i[o])return!1;return!0}(r,i))return!1;var k=getOwnNonIndexProperties(r),O=getOwnNonIndexProperties(i);return k.length===O.length&&keyCheck(r,i,o,p,te,k)}if(W(r))return!(!W(i)||r.size!==i.size)&&keyCheck(r,i,o,p,ne);if(R(r))return!(!R(i)||r.size!==i.size)&&keyCheck(r,i,o,p,ie);if(I(r)){if(!function areEqualArrayBuffers(r,i){return r.byteLength===i.byteLength&&0===compare(new Uint8Array(r),new Uint8Array(i))}(r,i))return!1}else if(j(r)&&!function isEqualBoxedPrimitive(r,i){return G(r)?G(i)&&y(Number.prototype.valueOf.call(r),Number.prototype.valueOf.call(i)):X(r)?X(i)&&String.prototype.valueOf.call(r)===String.prototype.valueOf.call(i):K(r)?K(i)&&Boolean.prototype.valueOf.call(r)===Boolean.prototype.valueOf.call(i):Y(r)?Y(i)&&BigInt.prototype.valueOf.call(r)===BigInt.prototype.valueOf.call(i):J(i)&&Symbol.prototype.valueOf.call(r)===Symbol.prototype.valueOf.call(i)}(r,i))return!1}return keyCheck(r,i,o,p,te)}function getEnumerables(r,i){return i.filter((function(i){return k(r,i)}))}function keyCheck(r,i,o,u,y,_){if(5===arguments.length){_=Object.keys(r);var P=Object.keys(i);if(_.length!==P.length)return!1}for(var O=0;O<_.length;O++)if(!x(i,_[O]))return!1;if(o&&5===arguments.length){var I=w(r);if(0!==I.length){var B=0;for(O=0;O<I.length;O++){var D=I[O];if(k(r,D)){if(!k(i,D))return!1;_.push(D),B++}else if(k(i,D))return!1}var R=w(i);if(I.length!==R.length&&getEnumerables(i,R).length!==B)return!1}else{var N=w(i);if(0!==N.length&&0!==getEnumerables(i,N).length)return!1}}if(0===_.length&&(y===te||y===re&&0===r.length||0===r.size))return!0;if(void 0===u)u={val1:new Map,val2:new Map,position:0};else{var W=u.val1.get(r);if(void 0!==W){var U=u.val2.get(i);if(void 0!==U)return W===U}u.position++}u.val1.set(r,u.position),u.val2.set(i,u.position);var j=function objEquiv(r,i,o,u,y,w){var _=0;if(w===ne){if(!function setEquiv(r,i,o,u){for(var g=null,y=p(r),w=0;w<y.length;w++){var _=y[w];if("object"===_typeof(_)&&null!==_)null===g&&(g=new Set),g.add(_);else if(!i.has(_)){if(o)return!1;if(!setMightHaveLoosePrim(r,i,_))return!1;null===g&&(g=new Set),g.add(_)}}if(null!==g){for(var x=p(i),k=0;k<x.length;k++){var P=x[k];if("object"===_typeof(P)&&null!==P){if(!setHasEqualElement(g,P,o,u))return!1}else if(!o&&!r.has(P)&&!setHasEqualElement(g,P,o,u))return!1}return 0===g.size}return!0}(r,i,o,y))return!1}else if(w===ie){if(!function mapEquiv(r,i,o,u){for(var p=null,y=g(r),w=0;w<y.length;w++){var _=_slicedToArray(y[w],2),x=_[0],k=_[1];if("object"===_typeof(x)&&null!==x)null===p&&(p=new Set),p.add(x);else{var P=i.get(x);if(void 0===P&&!i.has(x)||!innerDeepEqual(k,P,o,u)){if(o)return!1;if(!mapMightHaveLoosePrim(r,i,x,k,u))return!1;null===p&&(p=new Set),p.add(x)}}}if(null!==p){for(var O=g(i),I=0;I<O.length;I++){var B=_slicedToArray(O[I],2),D=B[0],R=B[1];if("object"===_typeof(D)&&null!==D){if(!mapHasEqualEntry(p,r,D,R,o,u))return!1}else if(!(o||r.has(D)&&innerDeepEqual(r.get(D),R,!1,u)||mapHasEqualEntry(p,r,D,R,!1,u)))return!1}return 0===p.size}return!0}(r,i,o,y))return!1}else if(w===re)for(;_<r.length;_++){if(!x(r,_)){if(x(i,_))return!1;for(var k=Object.keys(r);_<k.length;_++){var P=k[_];if(!x(i,P)||!innerDeepEqual(r[P],i[P],o,y))return!1}return k.length===Object.keys(i).length}if(!x(i,_)||!innerDeepEqual(r[_],i[_],o,y))return!1}for(_=0;_<u.length;_++){var O=u[_];if(!innerDeepEqual(r[O],i[O],o,y))return!1}return!0}(r,i,o,_,u,y);return u.val1.delete(r),u.val2.delete(i),j}function setHasEqualElement(r,i,o,u){for(var g=p(r),y=0;y<g.length;y++){var w=g[y];if(innerDeepEqual(i,w,o,u))return r.delete(w),!0}return!1}function findLooseMatchingPrimitives(r){switch(_typeof(r)){case"undefined":return null;case"object":return;case"symbol":return!1;case"string":r=+r;case"number":if(_(r))return!1}return!0}function setMightHaveLoosePrim(r,i,o){var u=findLooseMatchingPrimitives(o);return null!=u?u:i.has(u)&&!r.has(u)}function mapMightHaveLoosePrim(r,i,o,u,p){var g=findLooseMatchingPrimitives(o);if(null!=g)return g;var y=i.get(g);return!(void 0===y&&!i.has(g)||!innerDeepEqual(u,y,!1,p))&&(!r.has(g)&&innerDeepEqual(u,y,!1,p))}function mapHasEqualEntry(r,i,o,u,g,y){for(var w=p(r),_=0;_<w.length;_++){var x=w[_];if(innerDeepEqual(o,x,g,y)&&innerDeepEqual(u,i.get(x),g,y))return r.delete(x),!0}return!1}r.exports={isDeepEqual:function isDeepEqual(r,i){return innerDeepEqual(r,i,false)},isDeepStrictEqual:function isDeepStrictEqual(r,i){return innerDeepEqual(r,i,true)}}},32504:function(r,i){"use strict";i.byteLength=function byteLength(r){var i=getLens(r),o=i[0],u=i[1];return 3*(o+u)/4-u},i.toByteArray=function toByteArray(r){var i,o,g=getLens(r),y=g[0],w=g[1],_=new p(function _byteLength(r,i,o){return 3*(i+o)/4-o}(0,y,w)),x=0,k=w>0?y-4:y;for(o=0;o<k;o+=4)i=u[r.charCodeAt(o)]<<18|u[r.charCodeAt(o+1)]<<12|u[r.charCodeAt(o+2)]<<6|u[r.charCodeAt(o+3)],_[x++]=i>>16&255,_[x++]=i>>8&255,_[x++]=255&i;2===w&&(i=u[r.charCodeAt(o)]<<2|u[r.charCodeAt(o+1)]>>4,_[x++]=255&i);1===w&&(i=u[r.charCodeAt(o)]<<10|u[r.charCodeAt(o+1)]<<4|u[r.charCodeAt(o+2)]>>2,_[x++]=i>>8&255,_[x++]=255&i);return _},i.fromByteArray=function fromByteArray(r){for(var i,u=r.length,p=u%3,g=[],y=16383,w=0,_=u-p;w<_;w+=y)g.push(encodeChunk(r,w,w+y>_?_:w+y));1===p?(i=r[u-1],g.push(o[i>>2]+o[i<<4&63]+"==")):2===p&&(i=(r[u-2]<<8)+r[u-1],g.push(o[i>>10]+o[i>>4&63]+o[i<<2&63]+"="));return g.join("")};for(var o=[],u=[],p="undefined"!=typeof Uint8Array?Uint8Array:Array,g="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",y=0;y<64;++y)o[y]=g[y],u[g.charCodeAt(y)]=y;function getLens(r){var i=r.length;if(i%4>0)throw new Error("Invalid string. Length must be a multiple of 4");var o=r.indexOf("=");return-1===o&&(o=i),[o,o===i?0:4-o%4]}function encodeChunk(r,i,u){for(var p,g,y=[],w=i;w<u;w+=3)p=(r[w]<<16&16711680)+(r[w+1]<<8&65280)+(255&r[w+2]),y.push(o[(g=p)>>18&63]+o[g>>12&63]+o[g>>6&63]+o[63&g]);return y.join("")}u["-".charCodeAt(0)]=62,u["_".charCodeAt(0)]=63},34097:function(r){var i=4096,o=new Uint32Array([0,1,3,7,15,31,63,127,255,511,1023,2047,4095,8191,16383,32767,65535,131071,262143,524287,1048575,2097151,4194303,8388607,16777215]);function BrotliBitReader(r){this.buf_=new Uint8Array(8224),this.input_=r,this.reset()}BrotliBitReader.READ_SIZE=i,BrotliBitReader.IBUF_MASK=8191,BrotliBitReader.prototype.reset=function(){this.buf_ptr_=0,this.val_=0,this.pos_=0,this.bit_pos_=0,this.bit_end_pos_=0,this.eos_=0,this.readMoreInput();for(var r=0;r<4;r++)this.val_|=this.buf_[this.pos_]<<8*r,++this.pos_;return this.bit_end_pos_>0},BrotliBitReader.prototype.readMoreInput=function(){if(!(this.bit_end_pos_>256))if(this.eos_){if(this.bit_pos_>this.bit_end_pos_)throw new Error("Unexpected end of input "+this.bit_pos_+" "+this.bit_end_pos_)}else{var r=this.buf_ptr_,o=this.input_.read(this.buf_,r,i);if(o<0)throw new Error("Unexpected end of input");if(o<i){this.eos_=1;for(var u=0;u<32;u++)this.buf_[r+o+u]=0}if(0===r){for(u=0;u<32;u++)this.buf_[8192+u]=this.buf_[u];this.buf_ptr_=i}else this.buf_ptr_=0;this.bit_end_pos_+=o<<3}},BrotliBitReader.prototype.fillBitWindow=function(){for(;this.bit_pos_>=8;)this.val_>>>=8,this.val_|=this.buf_[8191&this.pos_]<<24,++this.pos_,this.bit_pos_=this.bit_pos_-8>>>0,this.bit_end_pos_=this.bit_end_pos_-8>>>0},BrotliBitReader.prototype.readBits=function(r){32-this.bit_pos_<r&&this.fillBitWindow();var i=this.val_>>>this.bit_pos_&o[r];return this.bit_pos_+=r,i},r.exports=BrotliBitReader},7043:function(r,i){i.lookup=new Uint8Array([0,0,0,0,0,0,0,0,0,4,4,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,12,16,12,12,20,12,16,24,28,12,12,32,12,36,12,44,44,44,44,44,44,44,44,44,44,32,32,24,40,28,12,12,48,52,52,52,48,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,24,12,28,12,12,12,56,60,60,60,56,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,24,12,28,12,0,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,0,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,56,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,6,6,6,6,7,7,7,7,8,8,8,8,9,9,9,9,10,10,10,10,11,11,11,11,12,12,12,12,13,13,13,13,14,14,14,14,15,15,15,15,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,22,22,22,22,23,23,23,23,24,24,24,24,25,25,25,25,26,26,26,26,27,27,27,27,28,28,28,28,29,29,29,29,30,30,30,30,31,31,31,31,32,32,32,32,33,33,33,33,34,34,34,34,35,35,35,35,36,36,36,36,37,37,37,37,38,38,38,38,39,39,39,39,40,40,40,40,41,41,41,41,42,42,42,42,43,43,43,43,44,44,44,44,45,45,45,45,46,46,46,46,47,47,47,47,48,48,48,48,49,49,49,49,50,50,50,50,51,51,51,51,52,52,52,52,53,53,53,53,54,54,54,54,55,55,55,55,56,56,56,56,57,57,57,57,58,58,58,58,59,59,59,59,60,60,60,60,61,61,61,61,62,62,62,62,63,63,63,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),i.lookupOffsets=new Uint16Array([1024,1536,1280,1536,0,256,768,512])},20980:function(r,i,o){var u=o(98197).z,p=o(98197).y,g=o(34097),y=o(80614),w=o(81561).z,_=o(81561).u,x=o(7043),k=o(42210),P=o(87984),O=8,I=16,B=256,D=704,R=26,N=6,W=2,U=8,j=255,G=1080,X=18,K=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),Y=16,J=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),$=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),ee=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function DecodeWindowBits(r){var i;return 0===r.readBits(1)?16:(i=r.readBits(3))>0?17+i:(i=r.readBits(3))>0?8+i:17}function DecodeVarLenUint8(r){if(r.readBits(1)){var i=r.readBits(3);return 0===i?1:r.readBits(i)+(1<<i)}return 0}function MetaBlockLength(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function DecodeMetaBlockLength(r){var i,o,u,p=new MetaBlockLength;if(p.input_end=r.readBits(1),p.input_end&&r.readBits(1))return p;if(7===(i=r.readBits(2)+4)){if(p.is_metadata=!0,0!==r.readBits(1))throw new Error("Invalid reserved bit");if(0===(o=r.readBits(2)))return p;for(u=0;u<o;u++){var g=r.readBits(8);if(u+1===o&&o>1&&0===g)throw new Error("Invalid size byte");p.meta_block_length|=g<<8*u}}else for(u=0;u<i;++u){var y=r.readBits(4);if(u+1===i&&i>4&&0===y)throw new Error("Invalid size nibble");p.meta_block_length|=y<<4*u}return++p.meta_block_length,p.input_end||p.is_metadata||(p.is_uncompressed=r.readBits(1)),p}function ReadSymbol(r,i,o){var u;return o.fillBitWindow(),(u=r[i+=o.val_>>>o.bit_pos_&j].bits-U)>0&&(o.bit_pos_+=U,i+=r[i].value,i+=o.val_>>>o.bit_pos_&(1<<u)-1),o.bit_pos_+=r[i].bits,r[i].value}function ReadHuffmanCode(r,i,o,u){var p,g,y=new Uint8Array(r);if(u.readMoreInput(),1===(g=u.readBits(2))){for(var x=r-1,k=0,P=new Int32Array(4),B=u.readBits(2)+1;x;)x>>=1,++k;for(D=0;D<B;++D)P[D]=u.readBits(k)%r,y[P[D]]=2;switch(y[P[0]]=1,B){case 1:break;case 3:if(P[0]===P[1]||P[0]===P[2]||P[1]===P[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(P[0]===P[1])throw new Error("[ReadHuffmanCode] invalid symbols");y[P[1]]=1;break;case 4:if(P[0]===P[1]||P[0]===P[2]||P[0]===P[3]||P[1]===P[2]||P[1]===P[3]||P[2]===P[3])throw new Error("[ReadHuffmanCode] invalid symbols");u.readBits(1)?(y[P[2]]=3,y[P[3]]=3):y[P[0]]=2}}else{var D,R=new Uint8Array(X),N=32,W=0,j=[new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,1),new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,5)];for(D=g;D<X&&N>0;++D){var G,Y=K[D],J=0;u.fillBitWindow(),J+=u.val_>>>u.bit_pos_&15,u.bit_pos_+=j[J].bits,G=j[J].value,R[Y]=G,0!==G&&(N-=32>>G,++W)}if(1!==W&&0!==N)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function ReadHuffmanCodeLengths(r,i,o,u){for(var p=0,g=O,y=0,x=0,k=32768,P=[],B=0;B<32;B++)P.push(new w(0,0));for(_(P,0,5,r,X);p<i&&k>0;){var D,R=0;if(u.readMoreInput(),u.fillBitWindow(),R+=u.val_>>>u.bit_pos_&31,u.bit_pos_+=P[R].bits,(D=255&P[R].value)<I)y=0,o[p++]=D,0!==D&&(g=D,k-=32768>>D);else{var N,W,U=D-14,j=0;if(D===I&&(j=g),x!==j&&(y=0,x=j),N=y,y>0&&(y-=2,y<<=U),p+(W=(y+=u.readBits(U)+3)-N)>i)throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var G=0;G<W;G++)o[p+G]=x;p+=W,0!==x&&(k-=W<<15-x)}}if(0!==k)throw new Error("[ReadHuffmanCodeLengths] space = "+k);for(;p<i;p++)o[p]=0}(R,r,y,u)}if(0===(p=_(i,o,U,y,r)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return p}function ReadBlockLength(r,i,o){var u,p;return u=ReadSymbol(r,i,o),p=k.kBlockLengthPrefixCode[u].nbits,k.kBlockLengthPrefixCode[u].offset+o.readBits(p)}function TranslateShortCodes(r,i,o){var u;return r<Y?(o+=J[r],u=i[o&=3]+$[r]):u=r-Y+1,u}function MoveToFront(r,i){for(var o=r[i],u=i;u;--u)r[u]=r[u-1];r[0]=o}function HuffmanTreeGroup(r,i){this.alphabet_size=r,this.num_htrees=i,this.codes=new Array(i+i*ee[r+31>>>5]),this.htrees=new Uint32Array(i)}function DecodeContextMap(r,i){var o,u,p={num_htrees:null,context_map:null},g=0;i.readMoreInput();var y=p.num_htrees=DecodeVarLenUint8(i)+1,_=p.context_map=new Uint8Array(r);if(y<=1)return p;for(i.readBits(1)&&(g=i.readBits(4)+1),o=[],u=0;u<G;u++)o[u]=new w(0,0);for(ReadHuffmanCode(y+g,o,0,i),u=0;u<r;){var x;if(i.readMoreInput(),0===(x=ReadSymbol(o,0,i)))_[u]=0,++u;else if(x<=g)for(var k=1+(1<<x)+i.readBits(x);--k;){if(u>=r)throw new Error("[DecodeContextMap] i >= context_map_size");_[u]=0,++u}else _[u]=x-g,++u}return i.readBits(1)&&function InverseMoveToFrontTransform(r,i){var o,u=new Uint8Array(256);for(o=0;o<256;++o)u[o]=o;for(o=0;o<i;++o){var p=r[o];r[o]=u[p],p&&MoveToFront(u,p)}}(_,r),p}function DecodeBlockType(r,i,o,u,p,g,y){var w,_=2*o,x=o,k=ReadSymbol(i,o*G,y);(w=0===k?p[_+(1&g[x])]:1===k?p[_+(g[x]-1&1)]+1:k-2)>=r&&(w-=r),u[o]=w,p[_+(1&g[x])]=w,++g[x]}function CopyUncompressedBlockToOutput(r,i,o,u,p,y){var w,_=p+1,x=o&p,k=y.pos_&g.IBUF_MASK;if(i<8||y.bit_pos_+(i<<3)<y.bit_end_pos_)for(;i-- >0;)y.readMoreInput(),u[x++]=y.readBits(8),x===_&&(r.write(u,_),x=0);else{if(y.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;y.bit_pos_<32;)u[x]=y.val_>>>y.bit_pos_,y.bit_pos_+=8,++x,--i;if(k+(w=y.bit_end_pos_-y.bit_pos_>>3)>g.IBUF_MASK){for(var P=g.IBUF_MASK+1-k,O=0;O<P;O++)u[x+O]=y.buf_[k+O];w-=P,x+=P,i-=P,k=0}for(O=0;O<w;O++)u[x+O]=y.buf_[k+O];if(i-=w,(x+=w)>=_){r.write(u,_),x-=_;for(O=0;O<x;O++)u[O]=u[_+O]}for(;x+i>=_;){if(w=_-x,y.input_.read(u,x,w)<w)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");r.write(u,_),i-=w,x=0}if(y.input_.read(u,x,i)<i)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");y.reset()}}function JumpToByteBoundary(r){var i=r.bit_pos_+7&-8;return 0==r.readBits(i-r.bit_pos_)}function BrotliDecompressedSize(r){var i=new u(r),o=new g(i);return DecodeWindowBits(o),DecodeMetaBlockLength(o).meta_block_length}function BrotliDecompress(r,i){var o,u,p,_,O,I,U,j,X,K,J=0,$=0,ee=0,te=[16,15,11,4],re=0,ne=0,ie=0,oe=[new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0)],ae=128+g.READ_SIZE;p=(1<<(u=DecodeWindowBits(K=new g(r))))-16,O=(_=1<<u)-1,I=new Uint8Array(_+ae+y.maxDictionaryWordLength),U=_,j=[],X=[];for(var se=0;se<3*G;se++)j[se]=new w(0,0),X[se]=new w(0,0);for(;!$;){var ce,le,ue,fe,he,de,pe,ge,ye,ve,me,be=0,we=[1<<28,1<<28,1<<28],_e=[0],Se=[1,1,1],xe=[0,1,0,1,0,1],Ce=[0],ke=null,Pe=null,Ae=0,Ee=null,Te=0,Oe=0,Ie=0;for(o=0;o<3;++o)oe[o].codes=null,oe[o].htrees=null;K.readMoreInput();var Be=DecodeMetaBlockLength(K);if(J+(be=Be.meta_block_length)>i.buffer.length){var Fe=new Uint8Array(J+be);Fe.set(i.buffer),i.buffer=Fe}if($=Be.input_end,ce=Be.is_uncompressed,Be.is_metadata)for(JumpToByteBoundary(K);be>0;--be)K.readMoreInput(),K.readBits(8);else if(0!==be)if(ce)K.bit_pos_=K.bit_pos_+7&-8,CopyUncompressedBlockToOutput(i,be,J,I,O,K),J+=be;else{for(o=0;o<3;++o)Se[o]=DecodeVarLenUint8(K)+1,Se[o]>=2&&(ReadHuffmanCode(Se[o]+2,j,o*G,K),ReadHuffmanCode(R,X,o*G,K),we[o]=ReadBlockLength(X,o*G,K),Ce[o]=1);for(K.readMoreInput(),fe=(1<<(le=K.readBits(2)))-1,he=(ue=Y+(K.readBits(4)<<le))+(48<<le),ke=new Uint8Array(Se[0]),o=0;o<Se[0];++o)K.readMoreInput(),ke[o]=K.readBits(2)<<1;var Le=DecodeContextMap(Se[0]<<N,K);pe=Le.num_htrees,de=Le.context_map;var De=DecodeContextMap(Se[2]<<W,K);for(ye=De.num_htrees,ge=De.context_map,oe[0]=new HuffmanTreeGroup(B,pe),oe[1]=new HuffmanTreeGroup(D,Se[1]),oe[2]=new HuffmanTreeGroup(he,ye),o=0;o<3;++o)oe[o].decode(K);for(Pe=0,Ee=0,ve=ke[_e[0]],Oe=x.lookupOffsets[ve],Ie=x.lookupOffsets[ve+1],me=oe[1].htrees[0];be>0;){var Re,Me,Ne,We,ze,Ue,je,Ge,Ve,He,qe,Xe;for(K.readMoreInput(),0===we[1]&&(DecodeBlockType(Se[1],j,1,_e,xe,Ce,K),we[1]=ReadBlockLength(X,G,K),me=oe[1].htrees[_e[1]]),--we[1],(Me=(Re=ReadSymbol(oe[1].codes,me,K))>>6)>=2?(Me-=2,je=-1):je=0,Ne=k.kInsertRangeLut[Me]+(Re>>3&7),We=k.kCopyRangeLut[Me]+(7&Re),ze=k.kInsertLengthPrefixCode[Ne].offset+K.readBits(k.kInsertLengthPrefixCode[Ne].nbits),Ue=k.kCopyLengthPrefixCode[We].offset+K.readBits(k.kCopyLengthPrefixCode[We].nbits),ne=I[J-1&O],ie=I[J-2&O],Ve=0;Ve<ze;++Ve)K.readMoreInput(),0===we[0]&&(DecodeBlockType(Se[0],j,0,_e,xe,Ce,K),we[0]=ReadBlockLength(X,0,K),Pe=_e[0]<<N,ve=ke[_e[0]],Oe=x.lookupOffsets[ve],Ie=x.lookupOffsets[ve+1]),Ae=de[Pe+(x.lookup[Oe+ne]|x.lookup[Ie+ie])],--we[0],ie=ne,ne=ReadSymbol(oe[0].codes,oe[0].htrees[Ae],K),I[J&O]=ne,(J&O)===O&&i.write(I,_),++J;if((be-=ze)<=0)break;if(je<0)if(K.readMoreInput(),0===we[2]&&(DecodeBlockType(Se[2],j,2,_e,xe,Ce,K),we[2]=ReadBlockLength(X,2*G,K),Ee=_e[2]<<W),--we[2],Te=ge[Ee+(255&(Ue>4?3:Ue-2))],(je=ReadSymbol(oe[2].codes,oe[2].htrees[Te],K))>=ue)Xe=(je-=ue)&fe,je=ue+((Ze=(2+(1&(je>>=le))<<(qe=1+(je>>1)))-4)+K.readBits(qe)<<le)+Xe;if((Ge=TranslateShortCodes(je,te,re))<0)throw new Error("[BrotliDecompress] invalid distance");if(He=J&O,Ge>(ee=J<p&&ee!==p?J:p)){if(!(Ue>=y.minDictionaryWordLength&&Ue<=y.maxDictionaryWordLength))throw new Error("Invalid backward reference. pos: "+J+" distance: "+Ge+" len: "+Ue+" bytes left: "+be);var Ze=y.offsetsByLength[Ue],Ke=Ge-ee-1,Ye=y.sizeBitsByLength[Ue],Je=Ke>>Ye;if(Ze+=(Ke&(1<<Ye)-1)*Ue,!(Je<P.kNumTransforms))throw new Error("Invalid backward reference. pos: "+J+" distance: "+Ge+" len: "+Ue+" bytes left: "+be);var Qe=P.transformDictionaryWord(I,He,Ze,Ue,Je);if(J+=Qe,be-=Qe,(He+=Qe)>=U){i.write(I,_);for(var $e=0;$e<He-U;$e++)I[$e]=I[U+$e]}}else{if(je>0&&(te[3&re]=Ge,++re),Ue>be)throw new Error("Invalid backward reference. pos: "+J+" distance: "+Ge+" len: "+Ue+" bytes left: "+be);for(Ve=0;Ve<Ue;++Ve)I[J&O]=I[J-Ge&O],(J&O)===O&&i.write(I,_),++J,--be}ne=I[J-1&O],ie=I[J-2&O]}J&=1073741823}}i.write(I,J&O)}HuffmanTreeGroup.prototype.decode=function(r){var i,o=0;for(i=0;i<this.num_htrees;++i)this.htrees[i]=o,o+=ReadHuffmanCode(this.alphabet_size,this.codes,o,r)},i.BrotliDecompressBuffer=function BrotliDecompressBuffer(r,i){var o=new u(r);null==i&&(i=BrotliDecompressedSize(r));var g=new Uint8Array(i),y=new p(g);return BrotliDecompress(o,y),y.pos<y.buffer.length&&(y.buffer=y.buffer.subarray(0,y.pos)),y.buffer},y.init()},84543:function(r,i,o){var u=o(32504);i.init=function(){return(0,o(20980).BrotliDecompressBuffer)(u.toByteArray(o(13501)))}},13501:function(r){r.exports="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"},80614:function(r,i,o){var u=o(84543);i.init=function(){i.dictionary=u.init()},i.offsetsByLength=new Uint32Array([0,0,0,0,0,4096,9216,21504,35840,44032,53248,63488,74752,87040,93696,100864,104704,106752,108928,113536,115968,118528,119872,121280,122016]),i.sizeBitsByLength=new Uint8Array([0,0,0,0,10,10,11,11,10,10,10,10,10,9,9,8,7,7,8,7,7,6,6,5,5]),i.minDictionaryWordLength=4,i.maxDictionaryWordLength=24},81561:function(r,i){function HuffmanCode(r,i){this.bits=r,this.value=i}i.z=HuffmanCode;var o=15;function GetNextKey(r,i){for(var o=1<<i-1;r&o;)o>>=1;return(r&o-1)+o}function ReplicateValue(r,i,o,u,p){do{r[i+(u-=o)]=new HuffmanCode(p.bits,p.value)}while(u>0)}function NextTableBitSize(r,i,u){for(var p=1<<i-u;i<o&&!((p-=r[i])<=0);)++i,p<<=1;return i-u}i.u=function(r,i,u,p,g){var y,w,_,x,k,P,O,I,B,D,R=i,N=new Int32Array(16),W=new Int32Array(16);for(D=new Int32Array(g),w=0;w<g;w++)N[p[w]]++;for(W[1]=0,y=1;y<o;y++)W[y+1]=W[y]+N[y];for(w=0;w<g;w++)0!==p[w]&&(D[W[p[w]]++]=w);if(B=I=1<<(O=u),1===W[15]){for(_=0;_<B;++_)r[i+_]=new HuffmanCode(0,65535&D[0]);return B}for(_=0,w=0,y=1,x=2;y<=u;++y,x<<=1)for(;N[y]>0;--N[y])ReplicateValue(r,i+_,x,I,new HuffmanCode(255&y,65535&D[w++])),_=GetNextKey(_,y);for(P=B-1,k=-1,y=u+1,x=2;y<=o;++y,x<<=1)for(;N[y]>0;--N[y])(_&P)!==k&&(i+=I,B+=I=1<<(O=NextTableBitSize(N,y,u)),r[R+(k=_&P)]=new HuffmanCode(O+u&255,i-R-k&65535)),ReplicateValue(r,i+(_>>u),x,I,new HuffmanCode(y-u&255,65535&D[w++])),_=GetNextKey(_,y);return B}},42210:function(r,i){function PrefixCodeRange(r,i){this.offset=r,this.nbits=i}i.kBlockLengthPrefixCode=[new PrefixCodeRange(1,2),new PrefixCodeRange(5,2),new PrefixCodeRange(9,2),new PrefixCodeRange(13,2),new PrefixCodeRange(17,3),new PrefixCodeRange(25,3),new PrefixCodeRange(33,3),new PrefixCodeRange(41,3),new PrefixCodeRange(49,4),new PrefixCodeRange(65,4),new PrefixCodeRange(81,4),new PrefixCodeRange(97,4),new PrefixCodeRange(113,5),new PrefixCodeRange(145,5),new PrefixCodeRange(177,5),new PrefixCodeRange(209,5),new PrefixCodeRange(241,6),new PrefixCodeRange(305,6),new PrefixCodeRange(369,7),new PrefixCodeRange(497,8),new PrefixCodeRange(753,9),new PrefixCodeRange(1265,10),new PrefixCodeRange(2289,11),new PrefixCodeRange(4337,12),new PrefixCodeRange(8433,13),new PrefixCodeRange(16625,24)],i.kInsertLengthPrefixCode=[new PrefixCodeRange(0,0),new PrefixCodeRange(1,0),new PrefixCodeRange(2,0),new PrefixCodeRange(3,0),new PrefixCodeRange(4,0),new PrefixCodeRange(5,0),new PrefixCodeRange(6,1),new PrefixCodeRange(8,1),new PrefixCodeRange(10,2),new PrefixCodeRange(14,2),new PrefixCodeRange(18,3),new PrefixCodeRange(26,3),new PrefixCodeRange(34,4),new PrefixCodeRange(50,4),new PrefixCodeRange(66,5),new PrefixCodeRange(98,5),new PrefixCodeRange(130,6),new PrefixCodeRange(194,7),new PrefixCodeRange(322,8),new PrefixCodeRange(578,9),new PrefixCodeRange(1090,10),new PrefixCodeRange(2114,12),new PrefixCodeRange(6210,14),new PrefixCodeRange(22594,24)],i.kCopyLengthPrefixCode=[new PrefixCodeRange(2,0),new PrefixCodeRange(3,0),new PrefixCodeRange(4,0),new PrefixCodeRange(5,0),new PrefixCodeRange(6,0),new PrefixCodeRange(7,0),new PrefixCodeRange(8,0),new PrefixCodeRange(9,0),new PrefixCodeRange(10,1),new PrefixCodeRange(12,1),new PrefixCodeRange(14,2),new PrefixCodeRange(18,2),new PrefixCodeRange(22,3),new PrefixCodeRange(30,3),new PrefixCodeRange(38,4),new PrefixCodeRange(54,4),new PrefixCodeRange(70,5),new PrefixCodeRange(102,5),new PrefixCodeRange(134,6),new PrefixCodeRange(198,7),new PrefixCodeRange(326,8),new PrefixCodeRange(582,9),new PrefixCodeRange(1094,10),new PrefixCodeRange(2118,24)],i.kInsertRangeLut=[0,0,8,8,0,16,8,16,16],i.kCopyRangeLut=[0,8,0,8,16,0,16,8,16]},98197:function(r,i){function BrotliInput(r){this.buffer=r,this.pos=0}function BrotliOutput(r){this.buffer=r,this.pos=0}BrotliInput.prototype.read=function(r,i,o){this.pos+o>this.buffer.length&&(o=this.buffer.length-this.pos);for(var u=0;u<o;u++)r[i+u]=this.buffer[this.pos+u];return this.pos+=o,o},i.z=BrotliInput,BrotliOutput.prototype.write=function(r,i){if(this.pos+i>this.buffer.length)throw new Error("Output buffer is not large enough");return this.buffer.set(r.subarray(0,i),this.pos),this.pos+=i,i},i.y=BrotliOutput},87984:function(r,i,o){var u=o(80614),p=10,g=11;function Transform(r,i,o){this.prefix=new Uint8Array(r.length),this.transform=i,this.suffix=new Uint8Array(o.length);for(var u=0;u<r.length;u++)this.prefix[u]=r.charCodeAt(u);for(u=0;u<o.length;u++)this.suffix[u]=o.charCodeAt(u)}var y=[new Transform("",0,""),new Transform("",0," "),new Transform(" ",0," "),new Transform("",12,""),new Transform("",p," "),new Transform("",0," the "),new Transform(" ",0,""),new Transform("s ",0," "),new Transform("",0," of "),new Transform("",p,""),new Transform("",0," and "),new Transform("",13,""),new Transform("",1,""),new Transform(", ",0," "),new Transform("",0,", "),new Transform(" ",p," "),new Transform("",0," in "),new Transform("",0," to "),new Transform("e ",0," "),new Transform("",0,'"'),new Transform("",0,"."),new Transform("",0,'">'),new Transform("",0,"\n"),new Transform("",3,""),new Transform("",0,"]"),new Transform("",0," for "),new Transform("",14,""),new Transform("",2,""),new Transform("",0," a "),new Transform("",0," that "),new Transform(" ",p,""),new Transform("",0,". "),new Transform(".",0,""),new Transform(" ",0,", "),new Transform("",15,""),new Transform("",0," with "),new Transform("",0,"'"),new Transform("",0," from "),new Transform("",0," by "),new Transform("",16,""),new Transform("",17,""),new Transform(" the ",0,""),new Transform("",4,""),new Transform("",0,". The "),new Transform("",g,""),new Transform("",0," on "),new Transform("",0," as "),new Transform("",0," is "),new Transform("",7,""),new Transform("",1,"ing "),new Transform("",0,"\n\t"),new Transform("",0,":"),new Transform(" ",0,". "),new Transform("",0,"ed "),new Transform("",20,""),new Transform("",18,""),new Transform("",6,""),new Transform("",0,"("),new Transform("",p,", "),new Transform("",8,""),new Transform("",0," at "),new Transform("",0,"ly "),new Transform(" the ",0," of "),new Transform("",5,""),new Transform("",9,""),new Transform(" ",p,", "),new Transform("",p,'"'),new Transform(".",0,"("),new Transform("",g," "),new Transform("",p,'">'),new Transform("",0,'="'),new Transform(" ",0,"."),new Transform(".com/",0,""),new Transform(" the ",0," of the "),new Transform("",p,"'"),new Transform("",0,". This "),new Transform("",0,","),new Transform(".",0," "),new Transform("",p,"("),new Transform("",p,"."),new Transform("",0," not "),new Transform(" ",0,'="'),new Transform("",0,"er "),new Transform(" ",g," "),new Transform("",0,"al "),new Transform(" ",g,""),new Transform("",0,"='"),new Transform("",g,'"'),new Transform("",p,". "),new Transform(" ",0,"("),new Transform("",0,"ful "),new Transform(" ",p,". "),new Transform("",0,"ive "),new Transform("",0,"less "),new Transform("",g,"'"),new Transform("",0,"est "),new Transform(" ",p,"."),new Transform("",g,'">'),new Transform(" ",0,"='"),new Transform("",p,","),new Transform("",0,"ize "),new Transform("",g,"."),new Transform("Â ",0,""),new Transform(" ",0,","),new Transform("",p,'="'),new Transform("",g,'="'),new Transform("",0,"ous "),new Transform("",g,", "),new Transform("",p,"='"),new Transform(" ",p,","),new Transform(" ",g,'="'),new Transform(" ",g,", "),new Transform("",g,","),new Transform("",g,"("),new Transform("",g,". "),new Transform(" ",g,"."),new Transform("",g,"='"),new Transform(" ",g,". "),new Transform(" ",p,'="'),new Transform(" ",g,"='"),new Transform(" ",p,"='")];function ToUpperCase(r,i){return r[i]<192?(r[i]>=97&&r[i]<=122&&(r[i]^=32),1):r[i]<224?(r[i+1]^=32,2):(r[i+2]^=5,3)}i.kTransforms=y,i.kNumTransforms=y.length,i.transformDictionaryWord=function(r,i,o,w,_){var x,k=y[_].prefix,P=y[_].suffix,O=y[_].transform,I=O<12?0:O-11,B=0,D=i;I>w&&(I=w);for(var R=0;R<k.length;)r[i++]=k[R++];for(o+=I,w-=I,O<=9&&(w-=O),B=0;B<w;B++)r[i++]=u.dictionary[o+B];if(x=i-w,O===p)ToUpperCase(r,x);else if(O===g)for(;w>0;){var N=ToUpperCase(r,x);x+=N,w-=N}for(var W=0;W<P.length;)r[i++]=P[W++];return i-D}},34460:function(r,i,o){r.exports=o(20980).BrotliDecompressBuffer},72908:function(r,i,o){"use strict";var u=o(50621).Buffer,p=o(9964),g=o(80182),y=o(37468),w=o(22925),_=o(2269),x=o(81607);for(var k in x)i[k]=x[k];i.NONE=0,i.DEFLATE=1,i.INFLATE=2,i.GZIP=3,i.GUNZIP=4,i.DEFLATERAW=5,i.INFLATERAW=6,i.UNZIP=7;function Zlib(r){if("number"!=typeof r||r<i.DEFLATE||r>i.UNZIP)throw new TypeError("Bad argument");this.dictionary=null,this.err=0,this.flush=0,this.init_done=!1,this.level=0,this.memLevel=0,this.mode=r,this.strategy=0,this.windowBits=0,this.write_in_progress=!1,this.pending_close=!1,this.gzip_id_bytes_read=0}Zlib.prototype.close=function(){this.write_in_progress?this.pending_close=!0:(this.pending_close=!1,g(this.init_done,"close before init"),g(this.mode<=i.UNZIP),this.mode===i.DEFLATE||this.mode===i.GZIP||this.mode===i.DEFLATERAW?w.deflateEnd(this.strm):this.mode!==i.INFLATE&&this.mode!==i.GUNZIP&&this.mode!==i.INFLATERAW&&this.mode!==i.UNZIP||_.inflateEnd(this.strm),this.mode=i.NONE,this.dictionary=null)},Zlib.prototype.write=function(r,i,o,u,p,g,y){return this._write(!0,r,i,o,u,p,g,y)},Zlib.prototype.writeSync=function(r,i,o,u,p,g,y){return this._write(!1,r,i,o,u,p,g,y)},Zlib.prototype._write=function(r,o,y,w,_,x,k,P){if(g.equal(arguments.length,8),g(this.init_done,"write before init"),g(this.mode!==i.NONE,"already finalized"),g.equal(!1,this.write_in_progress,"write already in progress"),g.equal(!1,this.pending_close,"close is pending"),this.write_in_progress=!0,g.equal(!1,void 0===o,"must provide flush value"),this.write_in_progress=!0,o!==i.Z_NO_FLUSH&&o!==i.Z_PARTIAL_FLUSH&&o!==i.Z_SYNC_FLUSH&&o!==i.Z_FULL_FLUSH&&o!==i.Z_FINISH&&o!==i.Z_BLOCK)throw new Error("Invalid flush value");if(null==y&&(y=u.alloc(0),_=0,w=0),this.strm.avail_in=_,this.strm.input=y,this.strm.next_in=w,this.strm.avail_out=P,this.strm.output=x,this.strm.next_out=k,this.flush=o,!r)return this._process(),this._checkError()?this._afterSync():void 0;var O=this;return p.nextTick((function(){O._process(),O._after()})),this},Zlib.prototype._afterSync=function(){var r=this.strm.avail_out,i=this.strm.avail_in;return this.write_in_progress=!1,[i,r]},Zlib.prototype._process=function(){var r=null;switch(this.mode){case i.DEFLATE:case i.GZIP:case i.DEFLATERAW:this.err=w.deflate(this.strm,this.flush);break;case i.UNZIP:switch(this.strm.avail_in>0&&(r=this.strm.next_in),this.gzip_id_bytes_read){case 0:if(null===r)break;if(31!==this.strm.input[r]){this.mode=i.INFLATE;break}if(this.gzip_id_bytes_read=1,r++,1===this.strm.avail_in)break;case 1:if(null===r)break;139===this.strm.input[r]?(this.gzip_id_bytes_read=2,this.mode=i.GUNZIP):this.mode=i.INFLATE;break;default:throw new Error("invalid number of gzip magic number bytes read")}case i.INFLATE:case i.GUNZIP:case i.INFLATERAW:for(this.err=_.inflate(this.strm,this.flush),this.err===i.Z_NEED_DICT&&this.dictionary&&(this.err=_.inflateSetDictionary(this.strm,this.dictionary),this.err===i.Z_OK?this.err=_.inflate(this.strm,this.flush):this.err===i.Z_DATA_ERROR&&(this.err=i.Z_NEED_DICT));this.strm.avail_in>0&&this.mode===i.GUNZIP&&this.err===i.Z_STREAM_END&&0!==this.strm.next_in[0];)this.reset(),this.err=_.inflate(this.strm,this.flush);break;default:throw new Error("Unknown mode "+this.mode)}},Zlib.prototype._checkError=function(){switch(this.err){case i.Z_OK:case i.Z_BUF_ERROR:if(0!==this.strm.avail_out&&this.flush===i.Z_FINISH)return this._error("unexpected end of file"),!1;break;case i.Z_STREAM_END:break;case i.Z_NEED_DICT:return null==this.dictionary?this._error("Missing dictionary"):this._error("Bad dictionary"),!1;default:return this._error("Zlib error"),!1}return!0},Zlib.prototype._after=function(){if(this._checkError()){var r=this.strm.avail_out,i=this.strm.avail_in;this.write_in_progress=!1,this.callback(i,r),this.pending_close&&this.close()}},Zlib.prototype._error=function(r){this.strm.msg&&(r=this.strm.msg),this.onerror(r,this.err),this.write_in_progress=!1,this.pending_close&&this.close()},Zlib.prototype.init=function(r,o,u,p,y){g(4===arguments.length||5===arguments.length,"init(windowBits, level, memLevel, strategy, [dictionary])"),g(r>=8&&r<=15,"invalid windowBits"),g(o>=-1&&o<=9,"invalid compression level"),g(u>=1&&u<=9,"invalid memlevel"),g(p===i.Z_FILTERED||p===i.Z_HUFFMAN_ONLY||p===i.Z_RLE||p===i.Z_FIXED||p===i.Z_DEFAULT_STRATEGY,"invalid strategy"),this._init(o,r,u,p,y),this._setDictionary()},Zlib.prototype.params=function(){throw new Error("deflateParams Not supported")},Zlib.prototype.reset=function(){this._reset(),this._setDictionary()},Zlib.prototype._init=function(r,o,u,p,g){switch(this.level=r,this.windowBits=o,this.memLevel=u,this.strategy=p,this.flush=i.Z_NO_FLUSH,this.err=i.Z_OK,this.mode!==i.GZIP&&this.mode!==i.GUNZIP||(this.windowBits+=16),this.mode===i.UNZIP&&(this.windowBits+=32),this.mode!==i.DEFLATERAW&&this.mode!==i.INFLATERAW||(this.windowBits=-1*this.windowBits),this.strm=new y,this.mode){case i.DEFLATE:case i.GZIP:case i.DEFLATERAW:this.err=w.deflateInit2(this.strm,this.level,i.Z_DEFLATED,this.windowBits,this.memLevel,this.strategy);break;case i.INFLATE:case i.GUNZIP:case i.INFLATERAW:case i.UNZIP:this.err=_.inflateInit2(this.strm,this.windowBits);break;default:throw new Error("Unknown mode "+this.mode)}this.err!==i.Z_OK&&this._error("Init error"),this.dictionary=g,this.write_in_progress=!1,this.init_done=!0},Zlib.prototype._setDictionary=function(){if(null!=this.dictionary){switch(this.err=i.Z_OK,this.mode){case i.DEFLATE:case i.DEFLATERAW:this.err=w.deflateSetDictionary(this.strm,this.dictionary)}this.err!==i.Z_OK&&this._error("Failed to set dictionary")}},Zlib.prototype._reset=function(){switch(this.err=i.Z_OK,this.mode){case i.DEFLATE:case i.DEFLATERAW:case i.GZIP:this.err=w.deflateReset(this.strm);break;case i.INFLATE:case i.INFLATERAW:case i.GUNZIP:this.err=_.inflateReset(this.strm)}this.err!==i.Z_OK&&this._error("Failed to reset stream")},i.Zlib=Zlib},6729:function(r,i,o){"use strict";var u=o(9964),p=o(50621).Buffer,g=o(9760).Transform,y=o(72908),w=o(7187),_=o(80182).ok,x=o(50621).kMaxLength,k="Cannot create final Buffer. It would be larger than 0x"+x.toString(16)+" bytes";y.Z_MIN_WINDOWBITS=8,y.Z_MAX_WINDOWBITS=15,y.Z_DEFAULT_WINDOWBITS=15,y.Z_MIN_CHUNK=64,y.Z_MAX_CHUNK=1/0,y.Z_DEFAULT_CHUNK=16384,y.Z_MIN_MEMLEVEL=1,y.Z_MAX_MEMLEVEL=9,y.Z_DEFAULT_MEMLEVEL=8,y.Z_MIN_LEVEL=-1,y.Z_MAX_LEVEL=9,y.Z_DEFAULT_LEVEL=y.Z_DEFAULT_COMPRESSION;for(var P=Object.keys(y),O=0;O<P.length;O++){var I=P[O];I.match(/^Z/)&&Object.defineProperty(i,I,{enumerable:!0,value:y[I],writable:!1})}for(var B={Z_OK:y.Z_OK,Z_STREAM_END:y.Z_STREAM_END,Z_NEED_DICT:y.Z_NEED_DICT,Z_ERRNO:y.Z_ERRNO,Z_STREAM_ERROR:y.Z_STREAM_ERROR,Z_DATA_ERROR:y.Z_DATA_ERROR,Z_MEM_ERROR:y.Z_MEM_ERROR,Z_BUF_ERROR:y.Z_BUF_ERROR,Z_VERSION_ERROR:y.Z_VERSION_ERROR},D=Object.keys(B),R=0;R<D.length;R++){var N=D[R];B[B[N]]=N}function zlibBuffer(r,i,o){var u=[],g=0;function flow(){for(var i;null!==(i=r.read());)u.push(i),g+=i.length;r.once("readable",flow)}function onEnd(){var i,y=null;g>=x?y=new RangeError(k):i=p.concat(u,g),u=[],r.close(),o(y,i)}r.on("error",(function onError(i){r.removeListener("end",onEnd),r.removeListener("readable",flow),o(i)})),r.on("end",onEnd),r.end(i),flow()}function zlibBufferSync(r,i){if("string"==typeof i&&(i=p.from(i)),!p.isBuffer(i))throw new TypeError("Not a string or buffer");var o=r._finishFlushFlag;return r._processChunk(i,o)}function Deflate(r){if(!(this instanceof Deflate))return new Deflate(r);Zlib.call(this,r,y.DEFLATE)}function Inflate(r){if(!(this instanceof Inflate))return new Inflate(r);Zlib.call(this,r,y.INFLATE)}function Gzip(r){if(!(this instanceof Gzip))return new Gzip(r);Zlib.call(this,r,y.GZIP)}function Gunzip(r){if(!(this instanceof Gunzip))return new Gunzip(r);Zlib.call(this,r,y.GUNZIP)}function DeflateRaw(r){if(!(this instanceof DeflateRaw))return new DeflateRaw(r);Zlib.call(this,r,y.DEFLATERAW)}function InflateRaw(r){if(!(this instanceof InflateRaw))return new InflateRaw(r);Zlib.call(this,r,y.INFLATERAW)}function Unzip(r){if(!(this instanceof Unzip))return new Unzip(r);Zlib.call(this,r,y.UNZIP)}function isValidFlushFlag(r){return r===y.Z_NO_FLUSH||r===y.Z_PARTIAL_FLUSH||r===y.Z_SYNC_FLUSH||r===y.Z_FULL_FLUSH||r===y.Z_FINISH||r===y.Z_BLOCK}function Zlib(r,o){var u=this;if(this._opts=r=r||{},this._chunkSize=r.chunkSize||i.Z_DEFAULT_CHUNK,g.call(this,r),r.flush&&!isValidFlushFlag(r.flush))throw new Error("Invalid flush flag: "+r.flush);if(r.finishFlush&&!isValidFlushFlag(r.finishFlush))throw new Error("Invalid flush flag: "+r.finishFlush);if(this._flushFlag=r.flush||y.Z_NO_FLUSH,this._finishFlushFlag=void 0!==r.finishFlush?r.finishFlush:y.Z_FINISH,r.chunkSize&&(r.chunkSize<i.Z_MIN_CHUNK||r.chunkSize>i.Z_MAX_CHUNK))throw new Error("Invalid chunk size: "+r.chunkSize);if(r.windowBits&&(r.windowBits<i.Z_MIN_WINDOWBITS||r.windowBits>i.Z_MAX_WINDOWBITS))throw new Error("Invalid windowBits: "+r.windowBits);if(r.level&&(r.level<i.Z_MIN_LEVEL||r.level>i.Z_MAX_LEVEL))throw new Error("Invalid compression level: "+r.level);if(r.memLevel&&(r.memLevel<i.Z_MIN_MEMLEVEL||r.memLevel>i.Z_MAX_MEMLEVEL))throw new Error("Invalid memLevel: "+r.memLevel);if(r.strategy&&r.strategy!=i.Z_FILTERED&&r.strategy!=i.Z_HUFFMAN_ONLY&&r.strategy!=i.Z_RLE&&r.strategy!=i.Z_FIXED&&r.strategy!=i.Z_DEFAULT_STRATEGY)throw new Error("Invalid strategy: "+r.strategy);if(r.dictionary&&!p.isBuffer(r.dictionary))throw new Error("Invalid dictionary: it should be a Buffer instance");this._handle=new y.Zlib(o);var w=this;this._hadError=!1,this._handle.onerror=function(r,o){_close(w),w._hadError=!0;var u=new Error(r);u.errno=o,u.code=i.codes[o],w.emit("error",u)};var _=i.Z_DEFAULT_COMPRESSION;"number"==typeof r.level&&(_=r.level);var x=i.Z_DEFAULT_STRATEGY;"number"==typeof r.strategy&&(x=r.strategy),this._handle.init(r.windowBits||i.Z_DEFAULT_WINDOWBITS,_,r.memLevel||i.Z_DEFAULT_MEMLEVEL,x,r.dictionary),this._buffer=p.allocUnsafe(this._chunkSize),this._offset=0,this._level=_,this._strategy=x,this.once("end",this.close),Object.defineProperty(this,"_closed",{get:function(){return!u._handle},configurable:!0,enumerable:!0})}function _close(r,i){i&&u.nextTick(i),r._handle&&(r._handle.close(),r._handle=null)}function emitCloseNT(r){r.emit("close")}Object.defineProperty(i,"codes",{enumerable:!0,value:Object.freeze(B),writable:!1}),i.Deflate=Deflate,i.Inflate=Inflate,i.Gzip=Gzip,i.Gunzip=Gunzip,i.DeflateRaw=DeflateRaw,i.InflateRaw=InflateRaw,i.Unzip=Unzip,i.createDeflate=function(r){return new Deflate(r)},i.createInflate=function(r){return new Inflate(r)},i.createDeflateRaw=function(r){return new DeflateRaw(r)},i.createInflateRaw=function(r){return new InflateRaw(r)},i.createGzip=function(r){return new Gzip(r)},i.createGunzip=function(r){return new Gunzip(r)},i.createUnzip=function(r){return new Unzip(r)},i.deflate=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Deflate(i),r,o)},i.deflateSync=function(r,i){return zlibBufferSync(new Deflate(i),r)},i.gzip=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Gzip(i),r,o)},i.gzipSync=function(r,i){return zlibBufferSync(new Gzip(i),r)},i.deflateRaw=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new DeflateRaw(i),r,o)},i.deflateRawSync=function(r,i){return zlibBufferSync(new DeflateRaw(i),r)},i.unzip=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Unzip(i),r,o)},i.unzipSync=function(r,i){return zlibBufferSync(new Unzip(i),r)},i.inflate=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Inflate(i),r,o)},i.inflateSync=function(r,i){return zlibBufferSync(new Inflate(i),r)},i.gunzip=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Gunzip(i),r,o)},i.gunzipSync=function(r,i){return zlibBufferSync(new Gunzip(i),r)},i.inflateRaw=function(r,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new InflateRaw(i),r,o)},i.inflateRawSync=function(r,i){return zlibBufferSync(new InflateRaw(i),r)},w.inherits(Zlib,g),Zlib.prototype.params=function(r,o,p){if(r<i.Z_MIN_LEVEL||r>i.Z_MAX_LEVEL)throw new RangeError("Invalid compression level: "+r);if(o!=i.Z_FILTERED&&o!=i.Z_HUFFMAN_ONLY&&o!=i.Z_RLE&&o!=i.Z_FIXED&&o!=i.Z_DEFAULT_STRATEGY)throw new TypeError("Invalid strategy: "+o);if(this._level!==r||this._strategy!==o){var g=this;this.flush(y.Z_SYNC_FLUSH,(function(){_(g._handle,"zlib binding closed"),g._handle.params(r,o),g._hadError||(g._level=r,g._strategy=o,p&&p())}))}else u.nextTick(p)},Zlib.prototype.reset=function(){return _(this._handle,"zlib binding closed"),this._handle.reset()},Zlib.prototype._flush=function(r){this._transform(p.alloc(0),"",r)},Zlib.prototype.flush=function(r,i){var o=this,g=this._writableState;("function"==typeof r||void 0===r&&!i)&&(i=r,r=y.Z_FULL_FLUSH),g.ended?i&&u.nextTick(i):g.ending?i&&this.once("end",i):g.needDrain?i&&this.once("drain",(function(){return o.flush(r,i)})):(this._flushFlag=r,this.write(p.alloc(0),"",i))},Zlib.prototype.close=function(r){_close(this,r),u.nextTick(emitCloseNT,this)},Zlib.prototype._transform=function(r,i,o){var u,g=this._writableState,w=(g.ending||g.ended)&&(!r||g.length===r.length);return null===r||p.isBuffer(r)?this._handle?(w?u=this._finishFlushFlag:(u=this._flushFlag,r.length>=g.length&&(this._flushFlag=this._opts.flush||y.Z_NO_FLUSH)),void this._processChunk(r,u,o)):o(new Error("zlib binding closed")):o(new Error("invalid input"))},Zlib.prototype._processChunk=function(r,i,o){var u=r&&r.length,g=this._chunkSize-this._offset,y=0,w=this,P="function"==typeof o;if(!P){var O,I=[],B=0;this.on("error",(function(r){O=r})),_(this._handle,"zlib binding closed");do{var D=this._handle.writeSync(i,r,y,u,this._buffer,this._offset,g)}while(!this._hadError&&callback(D[0],D[1]));if(this._hadError)throw O;if(B>=x)throw _close(this),new RangeError(k);var R=p.concat(I,B);return _close(this),R}_(this._handle,"zlib binding closed");var N=this._handle.write(i,r,y,u,this._buffer,this._offset,g);function callback(x,k){if(this&&(this.buffer=null,this.callback=null),!w._hadError){var O=g-k;if(_(O>=0,"have should not go down"),O>0){var D=w._buffer.slice(w._offset,w._offset+O);w._offset+=O,P?w.push(D):(I.push(D),B+=D.length)}if((0===k||w._offset>=w._chunkSize)&&(g=w._chunkSize,w._offset=0,w._buffer=p.allocUnsafe(w._chunkSize)),0===k){if(y+=u-x,u=x,!P)return!0;var R=w._handle.write(i,r,y,u,w._buffer,w._offset,w._chunkSize);return R.callback=callback,void(R.buffer=r)}if(!P)return!1;o()}}N.buffer=r,N.callback=callback},w.inherits(Deflate,Zlib),w.inherits(Inflate,Zlib),w.inherits(Gzip,Zlib),w.inherits(Gunzip,Zlib),w.inherits(DeflateRaw,Zlib),w.inherits(InflateRaw,Zlib),w.inherits(Unzip,Zlib)},67913:function(r,i,o){"use strict";var u=o(28651),p=o(26601),g=p(u("String.prototype.indexOf"));r.exports=function callBoundIntrinsic(r,i){var o=u(r,!!i);return"function"==typeof o&&g(r,".prototype.")>-1?p(o):o}},26601:function(r,i,o){"use strict";var u=o(5049),p=o(28651),g=o(86255),y=o(96785),w=p("%Function.prototype.apply%"),_=p("%Function.prototype.call%"),x=p("%Reflect.apply%",!0)||u.call(_,w),k=o(56649),P=p("%Math.max%");r.exports=function callBind(r){if("function"!=typeof r)throw new y("a function is required");var i=x(u,_,arguments);return g(i,1+P(0,r.length-(arguments.length-1)),!0)};var O=function applyBind(){return x(u,w,arguments)};k?k(r.exports,"apply",{value:O}):r.exports.apply=O},41613:function(r,i,o){var u=o(50621).Buffer,p=function(){"use strict";function clone(r,i,o,p){"object"==typeof i&&(o=i.depth,p=i.prototype,i.filter,i=i.circular);var g=[],y=[],w=void 0!==u;return void 0===i&&(i=!0),void 0===o&&(o=1/0),function _clone(r,o){if(null===r)return null;if(0==o)return r;var _,x;if("object"!=typeof r)return r;if(clone.__isArray(r))_=[];else if(clone.__isRegExp(r))_=new RegExp(r.source,__getRegExpFlags(r)),r.lastIndex&&(_.lastIndex=r.lastIndex);else if(clone.__isDate(r))_=new Date(r.getTime());else{if(w&&u.isBuffer(r))return _=u.allocUnsafe?u.allocUnsafe(r.length):new u(r.length),r.copy(_),_;void 0===p?(x=Object.getPrototypeOf(r),_=Object.create(x)):(_=Object.create(p),x=p)}if(i){var k=g.indexOf(r);if(-1!=k)return y[k];g.push(r),y.push(_)}for(var P in r){var O;x&&(O=Object.getOwnPropertyDescriptor(x,P)),O&&null==O.set||(_[P]=_clone(r[P],o-1))}return _}(r,o)}function __objToStr(r){return Object.prototype.toString.call(r)}function __getRegExpFlags(r){var i="";return r.global&&(i+="g"),r.ignoreCase&&(i+="i"),r.multiline&&(i+="m"),i}return clone.clonePrototype=function clonePrototype(r){if(null===r)return null;var c=function(){};return c.prototype=r,new c},clone.__objToStr=__objToStr,clone.__isDate=function __isDate(r){return"object"==typeof r&&"[object Date]"===__objToStr(r)},clone.__isArray=function __isArray(r){return"object"==typeof r&&"[object Array]"===__objToStr(r)},clone.__isRegExp=function __isRegExp(r){return"object"==typeof r&&"[object RegExp]"===__objToStr(r)},clone.__getRegExpFlags=__getRegExpFlags,clone}();r.exports&&(r.exports=p)},83043:function(r,i,o){o(59883);var u=o(11206);r.exports=u.Object.values},42075:function(r,i,o){o(94910),o(81755),o(14032),o(68067),o(77074),o(44455),o(58605),o(58281);var u=o(11206);r.exports=u.Promise},98168:function(r,i,o){var u=o(90780);o(84151),o(98443),o(49261),o(67858),r.exports=u},32631:function(r,i,o){var u=o(32010),p=o(94578),g=o(68664),y=u.TypeError;r.exports=function(r){if(p(r))return r;throw y(g(r)+" is not a function")}},69075:function(r,i,o){var u=o(32010),p=o(20884),g=o(68664),y=u.TypeError;r.exports=function(r){if(p(r))return r;throw y(g(r)+" is not a constructor")}},58659:function(r,i,o){var u=o(32010),p=o(94578),g=u.String,y=u.TypeError;r.exports=function(r){if("object"==typeof r||p(r))return r;throw y("Can't set "+g(r)+" as a prototype")}},71156:function(r,i,o){var u=o(38688),p=o(10819),g=o(95892),y=u("unscopables"),w=Array.prototype;null==w[y]&&g.f(w,y,{configurable:!0,value:p(null)}),r.exports=function(r){w[y][r]=!0}},36352:function(r,i,o){"use strict";var u=o(69510).charAt;r.exports=function(r,i,o){return i+(o?u(r,i).length:1)}},2868:function(r,i,o){var u=o(32010),p=o(70176),g=u.TypeError;r.exports=function(r,i){if(p(i,r))return r;throw g("Incorrect invocation")}},34984:function(r,i,o){var u=o(32010),p=o(24517),g=u.String,y=u.TypeError;r.exports=function(r){if(p(r))return r;throw y(g(r)+" is not an object")}},92642:function(r,i,o){"use strict";var u=o(43162),p=o(74841),g=o(45495),y=Math.min;r.exports=[].copyWithin||function copyWithin(r,i){var o=u(this),w=g(o),_=p(r,w),x=p(i,w),k=arguments.length>2?arguments[2]:void 0,P=y((void 0===k?w:p(k,w))-x,w-_),O=1;for(x<_&&_<x+P&&(O=-1,x+=P-1,_+=P-1);P-- >0;)x in o?o[_]=o[x]:delete o[_],_+=O,x+=O;return o}},72864:function(r,i,o){"use strict";var u=o(43162),p=o(74841),g=o(45495);r.exports=function fill(r){for(var i=u(this),o=g(i),y=arguments.length,w=p(y>1?arguments[1]:void 0,o),_=y>2?arguments[2]:void 0,x=void 0===_?o:p(_,o);x>w;)i[w++]=r;return i}},82938:function(r,i,o){"use strict";var u=o(91102).forEach,p=o(81007)("forEach");r.exports=p?[].forEach:function forEach(r){return u(this,r,arguments.length>1?arguments[1]:void 0)}},34269:function(r){r.exports=function(r,i){for(var o=0,u=i.length,p=new r(u);u>o;)p[o]=i[o++];return p}},95717:function(r,i,o){"use strict";var u=o(32010),p=o(25567),g=o(2834),y=o(43162),w=o(97738),_=o(89564),x=o(20884),k=o(45495),P=o(38639),O=o(15892),I=o(13872),B=u.Array;r.exports=function from(r){var i=y(r),o=x(this),u=arguments.length,D=u>1?arguments[1]:void 0,R=void 0!==D;R&&(D=p(D,u>2?arguments[2]:void 0));var N,W,U,j,G,X,K=I(i),Y=0;if(!K||this==B&&_(K))for(N=k(i),W=o?new this(N):B(N);N>Y;Y++)X=R?D(i[Y],Y):i[Y],P(W,Y,X);else for(G=(j=O(i,K)).next,W=o?new this:[];!(U=g(G,j)).done;Y++)X=R?w(j,D,[U.value,Y],!0):U.value,P(W,Y,X);return W.length=Y,W}},12636:function(r,i,o){var u=o(98086),p=o(74841),g=o(45495),createMethod=function(r){return function(i,o,y){var w,_=u(i),x=g(_),k=p(y,x);if(r&&o!=o){for(;x>k;)if((w=_[k++])!=w)return!0}else for(;x>k;k++)if((r||k in _)&&_[k]===o)return r||k||0;return!r&&-1}};r.exports={includes:createMethod(!0),indexOf:createMethod(!1)}},91102:function(r,i,o){var u=o(25567),p=o(38347),g=o(7514),y=o(43162),w=o(45495),_=o(45744),x=p([].push),createMethod=function(r){var i=1==r,o=2==r,p=3==r,k=4==r,P=6==r,O=7==r,I=5==r||P;return function(B,D,R,N){for(var W,U,j=y(B),G=g(j),X=u(D,R),K=w(G),Y=0,J=N||_,$=i?J(B,K):o||O?J(B,0):void 0;K>Y;Y++)if((I||Y in G)&&(U=X(W=G[Y],Y,j),r))if(i)$[Y]=U;else if(U)switch(r){case 3:return!0;case 5:return W;case 6:return Y;case 2:x($,W)}else switch(r){case 4:return!1;case 7:x($,W)}return P?-1:p||k?k:$}};r.exports={forEach:createMethod(0),map:createMethod(1),filter:createMethod(2),some:createMethod(3),every:createMethod(4),find:createMethod(5),findIndex:createMethod(6),filterReject:createMethod(7)}},84320:function(r,i,o){"use strict";var u=o(58448),p=o(98086),g=o(26882),y=o(45495),w=o(81007),_=Math.min,x=[].lastIndexOf,k=!!x&&1/[1].lastIndexOf(1,-0)<0,P=w("lastIndexOf"),O=k||!P;r.exports=O?function lastIndexOf(r){if(k)return u(x,this,arguments)||0;var i=p(this),o=y(i),w=o-1;for(arguments.length>1&&(w=_(w,g(arguments[1]))),w<0&&(w=o+w);w>=0;w--)if(w in i&&i[w]===r)return w||0;return-1}:x},56280:function(r,i,o){var u=o(47044),p=o(38688),g=o(70091),y=p("species");r.exports=function(r){return g>=51||!u((function(){var i=[];return(i.constructor={})[y]=function(){return{foo:1}},1!==i[r](Boolean).foo}))}},81007:function(r,i,o){"use strict";var u=o(47044);r.exports=function(r,i){var o=[][r];return!!o&&u((function(){o.call(null,i||function(){throw 1},1)}))}},32843:function(r,i,o){var u=o(32010),p=o(32631),g=o(43162),y=o(7514),w=o(45495),_=u.TypeError,createMethod=function(r){return function(i,o,u,x){p(o);var k=g(i),P=y(k),O=w(k),I=r?O-1:0,B=r?-1:1;if(u<2)for(;;){if(I in P){x=P[I],I+=B;break}if(I+=B,r?I<0:O<=I)throw _("Reduce of empty array with no initial value")}for(;r?I>=0:O>I;I+=B)I in P&&(x=o(x,P[I],I,k));return x}};r.exports={left:createMethod(!1),right:createMethod(!0)}},73163:function(r,i,o){var u=o(38347);r.exports=u([].slice)},43977:function(r,i,o){var u=o(73163),p=Math.floor,mergeSort=function(r,i){var o=r.length,g=p(o/2);return o<8?insertionSort(r,i):merge(r,mergeSort(u(r,0,g),i),mergeSort(u(r,g),i),i)},insertionSort=function(r,i){for(var o,u,p=r.length,g=1;g<p;){for(u=g,o=r[g];u&&i(r[u-1],o)>0;)r[u]=r[--u];u!==g++&&(r[u]=o)}return r},merge=function(r,i,o,u){for(var p=i.length,g=o.length,y=0,w=0;y<p||w<g;)r[y+w]=y<p&&w<g?u(i[y],o[w])<=0?i[y++]:o[w++]:y<p?i[y++]:o[w++];return r};r.exports=mergeSort},98578:function(r,i,o){var u=o(32010),p=o(59113),g=o(20884),y=o(24517),w=o(38688)("species"),_=u.Array;r.exports=function(r){var i;return p(r)&&(i=r.constructor,(g(i)&&(i===_||p(i.prototype))||y(i)&&null===(i=i[w]))&&(i=void 0)),void 0===i?_:i}},45744:function(r,i,o){var u=o(98578);r.exports=function(r,i){return new(u(r))(0===i?0:i)}},97738:function(r,i,o){var u=o(34984),p=o(50194);r.exports=function(r,i,o,g){try{return g?i(u(o)[0],o[1]):i(o)}catch(i){p(r,"throw",i)}}},46769:function(r,i,o){var u=o(38688)("iterator"),p=!1;try{var g=0,y={next:function(){return{done:!!g++}},return:function(){p=!0}};y[u]=function(){return this},Array.from(y,(function(){throw 2}))}catch(r){}r.exports=function(r,i){if(!i&&!p)return!1;var o=!1;try{var g={};g[u]=function(){return{next:function(){return{done:o=!0}}}},r(g)}catch(r){}return o}},93975:function(r,i,o){var u=o(38347),p=u({}.toString),g=u("".slice);r.exports=function(r){return g(p(r),8,-1)}},52564:function(r,i,o){var u=o(32010),p=o(24663),g=o(94578),y=o(93975),w=o(38688)("toStringTag"),_=u.Object,x="Arguments"==y(function(){return arguments}());r.exports=p?y:function(r){var i,o,u;return void 0===r?"Undefined":null===r?"Null":"string"==typeof(o=function(r,i){try{return r[i]}catch(r){}}(i=_(r),w))?o:x?y(i):"Object"==(u=y(i))&&g(i.callee)?"Arguments":u}},34074:function(r,i,o){var u=o(38347),p=o(73163),g=u("".replace),y=u("".split),w=u([].join),_=String(Error("zxcasd").stack),x=/\n\s*at [^:]*:[^\n]*/,k=x.test(_),P=/@[^\n]*\n/.test(_)&&!/zxcasd/.test(_);r.exports=function(r,i){if("string"!=typeof r)return r;if(k)for(;i--;)r=g(r,x,"");else if(P)return w(p(y(r,"\n"),i),"\n");return r}},9649:function(r,i,o){"use strict";var u=o(95892).f,p=o(10819),g=o(15341),y=o(25567),w=o(2868),_=o(80383),x=o(97001),k=o(51334),P=o(15567),O=o(62148).fastKey,I=o(70172),B=I.set,D=I.getterFor;r.exports={getConstructor:function(r,i,o,x){var k=r((function(r,u){w(r,I),B(r,{type:i,index:p(null),first:void 0,last:void 0,size:0}),P||(r.size=0),null!=u&&_(u,r[x],{that:r,AS_ENTRIES:o})})),I=k.prototype,R=D(i),define=function(r,i,o){var u,p,g=R(r),y=getEntry(r,i);return y?y.value=o:(g.last=y={index:p=O(i,!0),key:i,value:o,previous:u=g.last,next:void 0,removed:!1},g.first||(g.first=y),u&&(u.next=y),P?g.size++:r.size++,"F"!==p&&(g.index[p]=y)),r},getEntry=function(r,i){var o,u=R(r),p=O(i);if("F"!==p)return u.index[p];for(o=u.first;o;o=o.next)if(o.key==i)return o};return g(I,{clear:function clear(){for(var r=R(this),i=r.index,o=r.first;o;)o.removed=!0,o.previous&&(o.previous=o.previous.next=void 0),delete i[o.index],o=o.next;r.first=r.last=void 0,P?r.size=0:this.size=0},delete:function(r){var i=this,o=R(i),u=getEntry(i,r);if(u){var p=u.next,g=u.previous;delete o.index[u.index],u.removed=!0,g&&(g.next=p),p&&(p.previous=g),o.first==u&&(o.first=p),o.last==u&&(o.last=g),P?o.size--:i.size--}return!!u},forEach:function forEach(r){for(var i,o=R(this),u=y(r,arguments.length>1?arguments[1]:void 0);i=i?i.next:o.first;)for(u(i.value,i.key,this);i&&i.removed;)i=i.previous},has:function has(r){return!!getEntry(this,r)}}),g(I,o?{get:function get(r){var i=getEntry(this,r);return i&&i.value},set:function set(r,i){return define(this,0===r?0:r,i)}}:{add:function add(r){return define(this,r=0===r?0:r,r)}}),P&&u(I,"size",{get:function(){return R(this).size}}),k},setStrong:function(r,i,o){var u=i+" Iterator",p=D(i),g=D(u);x(r,i,(function(r,i){B(this,{type:u,target:r,state:p(r),kind:i,last:void 0})}),(function(){for(var r=g(this),i=r.kind,o=r.last;o&&o.removed;)o=o.previous;return r.target&&(r.last=o=o?o.next:r.state.first)?"keys"==i?{value:o.key,done:!1}:"values"==i?{value:o.value,done:!1}:{value:[o.key,o.value],done:!1}:(r.target=void 0,{value:void 0,done:!0})}),o?"entries":"values",!o,!0),k(i)}}},36673:function(r,i,o){"use strict";var u=o(56475),p=o(32010),g=o(38347),y=o(39599),w=o(13711),_=o(62148),x=o(80383),k=o(2868),P=o(94578),O=o(24517),I=o(47044),B=o(46769),D=o(15216),R=o(51868);r.exports=function(r,i,o){var N=-1!==r.indexOf("Map"),W=-1!==r.indexOf("Weak"),U=N?"set":"add",j=p[r],G=j&&j.prototype,X=j,K={},fixMethod=function(r){var i=g(G[r]);w(G,r,"add"==r?function add(r){return i(this,0===r?0:r),this}:"delete"==r?function(r){return!(W&&!O(r))&&i(this,0===r?0:r)}:"get"==r?function get(r){return W&&!O(r)?void 0:i(this,0===r?0:r)}:"has"==r?function has(r){return!(W&&!O(r))&&i(this,0===r?0:r)}:function set(r,o){return i(this,0===r?0:r,o),this})};if(y(r,!P(j)||!(W||G.forEach&&!I((function(){(new j).entries().next()})))))X=o.getConstructor(i,r,N,U),_.enable();else if(y(r,!0)){var Y=new X,J=Y[U](W?{}:-0,1)!=Y,$=I((function(){Y.has(1)})),ee=B((function(r){new j(r)})),te=!W&&I((function(){for(var r=new j,i=5;i--;)r[U](i,i);return!r.has(-0)}));ee||((X=i((function(r,i){k(r,G);var o=R(new j,r,X);return null!=i&&x(i,o[U],{that:o,AS_ENTRIES:N}),o}))).prototype=G,G.constructor=X),($||te)&&(fixMethod("delete"),fixMethod("has"),N&&fixMethod("get")),(te||J)&&fixMethod(U),W&&G.clear&&delete G.clear}return K[r]=X,u({global:!0,forced:X!=j},K),D(X,r),W||o.setStrong(X,r,N),X}},2675:function(r,i,o){var u=o(20340),p=o(21594),g=o(72062),y=o(95892);r.exports=function(r,i){for(var o=p(i),w=y.f,_=g.f,x=0;x<o.length;x++){var k=o[x];u(r,k)||w(r,k,_(i,k))}}},91151:function(r,i,o){var u=o(38688)("match");r.exports=function(r){var i=/./;try{"/./"[r](i)}catch(o){try{return i[u]=!1,"/./"[r](i)}catch(r){}}return!1}},68494:function(r,i,o){var u=o(47044);r.exports=!u((function(){function F(){}return F.prototype.constructor=null,Object.getPrototypeOf(new F)!==F.prototype}))},91159:function(r,i,o){var u=o(38347),p=o(83943),g=o(25096),y=/"/g,w=u("".replace);r.exports=function(r,i,o,u){var _=g(p(r)),x="<"+i;return""!==o&&(x+=" "+o+'="'+w(g(u),y,"&quot;")+'"'),x+">"+_+"</"+i+">"}},13945:function(r,i,o){"use strict";var u=o(5844).IteratorPrototype,p=o(10819),g=o(97841),y=o(15216),w=o(15366),returnThis=function(){return this};r.exports=function(r,i,o){var _=i+" Iterator";return r.prototype=p(u,{next:g(1,o)}),y(r,_,!1,!0),w[_]=returnThis,r}},48914:function(r,i,o){var u=o(15567),p=o(95892),g=o(97841);r.exports=u?function(r,i,o){return p.f(r,i,g(1,o))}:function(r,i,o){return r[i]=o,r}},97841:function(r){r.exports=function(r,i){return{enumerable:!(1&r),configurable:!(2&r),writable:!(4&r),value:i}}},38639:function(r,i,o){"use strict";var u=o(63918),p=o(95892),g=o(97841);r.exports=function(r,i,o){var y=u(i);y in r?p.f(r,y,g(0,o)):r[y]=o}},53087:function(r,i,o){"use strict";var u=o(32010),p=o(34984),g=o(39629),y=u.TypeError;r.exports=function(r){if(p(this),"string"===r||"default"===r)r="string";else if("number"!==r)throw y("Incorrect hint");return g(this,r)}},97001:function(r,i,o){"use strict";var u=o(56475),p=o(2834),g=o(63432),y=o(7081),w=o(94578),_=o(13945),x=o(69548),k=o(3840),P=o(15216),O=o(48914),I=o(13711),B=o(38688),D=o(15366),R=o(5844),N=y.PROPER,W=y.CONFIGURABLE,U=R.IteratorPrototype,j=R.BUGGY_SAFARI_ITERATORS,G=B("iterator"),X="keys",K="values",Y="entries",returnThis=function(){return this};r.exports=function(r,i,o,y,B,R,J){_(o,i,y);var $,ee,te,getIterationMethod=function(r){if(r===B&&ae)return ae;if(!j&&r in ie)return ie[r];switch(r){case X:return function keys(){return new o(this,r)};case K:return function values(){return new o(this,r)};case Y:return function entries(){return new o(this,r)}}return function(){return new o(this)}},re=i+" Iterator",ne=!1,ie=r.prototype,oe=ie[G]||ie["@@iterator"]||B&&ie[B],ae=!j&&oe||getIterationMethod(B),se="Array"==i&&ie.entries||oe;if(se&&($=x(se.call(new r)))!==Object.prototype&&$.next&&(g||x($)===U||(k?k($,U):w($[G])||I($,G,returnThis)),P($,re,!0,!0),g&&(D[re]=returnThis)),N&&B==K&&oe&&oe.name!==K&&(!g&&W?O(ie,"name",K):(ne=!0,ae=function values(){return p(oe,this)})),B)if(ee={values:getIterationMethod(K),keys:R?ae:getIterationMethod(X),entries:getIterationMethod(Y)},J)for(te in ee)(j||ne||!(te in ie))&&I(ie,te,ee[te]);else u({target:i,proto:!0,forced:j||ne},ee);return g&&!J||ie[G]===ae||I(ie,G,ae,{name:B}),D[i]=ae,ee}},46042:function(r,i,o){var u=o(11206),p=o(20340),g=o(75960),y=o(95892).f;r.exports=function(r){var i=u.Symbol||(u.Symbol={});p(i,r)||y(i,r,{value:g.f(r)})}},15567:function(r,i,o){var u=o(47044);r.exports=!u((function(){return 7!=Object.defineProperty({},1,{get:function(){return 7}})[1]}))},12072:function(r,i,o){var u=o(32010),p=o(24517),g=u.document,y=p(g)&&p(g.createElement);r.exports=function(r){return y?g.createElement(r):{}}},23327:function(r){r.exports={CSSRuleList:0,CSSStyleDeclaration:0,CSSValueList:0,ClientRectList:0,DOMRectList:0,DOMStringList:0,DOMTokenList:1,DataTransferItemList:0,FileList:0,HTMLAllCollection:0,HTMLCollection:0,HTMLFormElement:0,HTMLSelectElement:0,MediaList:0,MimeTypeArray:0,NamedNodeMap:0,NodeList:1,PaintRequestList:0,Plugin:0,PluginArray:0,SVGLengthList:0,SVGNumberList:0,SVGPathSegList:0,SVGPointList:0,SVGStringList:0,SVGTransformList:0,SourceBufferList:0,StyleSheetList:0,TextTrackCueList:0,TextTrackList:0,TouchList:0}},67797:function(r,i,o){var u=o(12072)("span").classList,p=u&&u.constructor&&u.constructor.prototype;r.exports=p===Object.prototype?void 0:p},3809:function(r,i,o){var u=o(40715).match(/firefox\/(\d+)/i);r.exports=!!u&&+u[1]},3157:function(r){r.exports="object"==typeof window},21983:function(r,i,o){var u=o(40715);r.exports=/MSIE|Trident/.test(u)},70573:function(r,i,o){var u=o(40715),p=o(32010);r.exports=/ipad|iphone|ipod/i.test(u)&&void 0!==p.Pebble},17716:function(r,i,o){var u=o(40715);r.exports=/(?:ipad|iphone|ipod).*applewebkit/i.test(u)},95053:function(r,i,o){var u=o(93975),p=o(32010);r.exports="process"==u(p.process)},664:function(r,i,o){var u=o(40715);r.exports=/web0s(?!.*chrome)/i.test(u)},40715:function(r,i,o){var u=o(38486);r.exports=u("navigator","userAgent")||""},70091:function(r,i,o){var u,p,g=o(32010),y=o(40715),w=g.process,_=g.Deno,x=w&&w.versions||_&&_.version,k=x&&x.v8;k&&(p=(u=k.split("."))[0]>0&&u[0]<4?1:+(u[0]+u[1])),!p&&y&&(!(u=y.match(/Edge\/(\d+)/))||u[1]>=74)&&(u=y.match(/Chrome\/(\d+)/))&&(p=+u[1]),r.exports=p},41731:function(r,i,o){var u=o(40715).match(/AppleWebKit\/(\d+)\./);r.exports=!!u&&+u[1]},2416:function(r){r.exports=["constructor","hasOwnProperty","isPrototypeOf","propertyIsEnumerable","toLocaleString","toString","valueOf"]},45144:function(r,i,o){var u=o(47044),p=o(97841);r.exports=!u((function(){var r=Error("a");return!("stack"in r)||(Object.defineProperty(r,"stack",p(1,7)),7!==r.stack)}))},56475:function(r,i,o){var u=o(32010),p=o(72062).f,g=o(48914),y=o(13711),w=o(7421),_=o(2675),x=o(39599);r.exports=function(r,i){var o,k,P,O,I,B=r.target,D=r.global,R=r.stat;if(o=D?u:R?u[B]||w(B,{}):(u[B]||{}).prototype)for(k in i){if(O=i[k],P=r.noTargetGet?(I=p(o,k))&&I.value:o[k],!x(D?k:B+(R?".":"#")+k,r.forced)&&void 0!==P){if(typeof O==typeof P)continue;_(O,P)}(r.sham||P&&P.sham)&&g(O,"sham",!0),y(o,k,O,r)}}},47044:function(r){r.exports=function(r){try{return!!r()}catch(r){return!0}}},11813:function(r,i,o){"use strict";o(61726);var u=o(38347),p=o(13711),g=o(49820),y=o(47044),w=o(38688),_=o(48914),x=w("species"),k=RegExp.prototype;r.exports=function(r,i,o,P){var O=w(r),I=!y((function(){var i={};return i[O]=function(){return 7},7!=""[r](i)})),B=I&&!y((function(){var i=!1,o=/a/;return"split"===r&&((o={}).constructor={},o.constructor[x]=function(){return o},o.flags="",o[O]=/./[O]),o.exec=function(){return i=!0,null},o[O](""),!i}));if(!I||!B||o){var D=u(/./[O]),R=i(O,""[r],(function(r,i,o,p,y){var w=u(r),_=i.exec;return _===g||_===k.exec?I&&!y?{done:!0,value:D(i,o,p)}:{done:!0,value:w(o,i,p)}:{done:!1}}));p(String.prototype,r,R[0]),p(k,O,R[1])}P&&_(k[O],"sham",!0)}},55481:function(r,i,o){var u=o(47044);r.exports=!u((function(){return Object.isExtensible(Object.preventExtensions({}))}))},58448:function(r){var i=Function.prototype,o=i.apply,u=i.bind,p=i.call;r.exports="object"==typeof Reflect&&Reflect.apply||(u?p.bind(o):function(){return p.apply(o,arguments)})},25567:function(r,i,o){var u=o(38347),p=o(32631),g=u(u.bind);r.exports=function(r,i){return p(r),void 0===i?r:g?g(r,i):function(){return r.apply(i,arguments)}}},5481:function(r,i,o){"use strict";var u=o(32010),p=o(38347),g=o(32631),y=o(24517),w=o(20340),_=o(73163),x=u.Function,k=p([].concat),P=p([].join),O={};r.exports=x.bind||function bind(r){var i=g(this),o=i.prototype,u=_(arguments,1),p=function bound(){var o=k(u,_(arguments));return this instanceof p?function(r,i,o){if(!w(O,i)){for(var u=[],p=0;p<i;p++)u[p]="a["+p+"]";O[i]=x("C,a","return new C("+P(u,",")+")")}return O[i](r,o)}(i,o.length,o):i.apply(r,o)};return y(o)&&(p.prototype=o),p}},2834:function(r){var i=Function.prototype.call;r.exports=i.bind?i.bind(i):function(){return i.apply(i,arguments)}},7081:function(r,i,o){var u=o(15567),p=o(20340),g=Function.prototype,y=u&&Object.getOwnPropertyDescriptor,w=p(g,"name"),_=w&&"something"===function something(){}.name,x=w&&(!u||u&&y(g,"name").configurable);r.exports={EXISTS:w,PROPER:_,CONFIGURABLE:x}},38347:function(r){var i=Function.prototype,o=i.bind,u=i.call,p=o&&o.bind(u);r.exports=o?function(r){return r&&p(u,r)}:function(r){return r&&function(){return u.apply(r,arguments)}}},38486:function(r,i,o){var u=o(32010),p=o(94578);r.exports=function(r,i){return arguments.length<2?(o=u[r],p(o)?o:void 0):u[r]&&u[r][i];var o}},13872:function(r,i,o){var u=o(52564),p=o(51839),g=o(15366),y=o(38688)("iterator");r.exports=function(r){if(null!=r)return p(r,y)||p(r,"@@iterator")||g[u(r)]}},15892:function(r,i,o){var u=o(32010),p=o(2834),g=o(32631),y=o(34984),w=o(68664),_=o(13872),x=u.TypeError;r.exports=function(r,i){var o=arguments.length<2?_(r):i;if(g(o))return y(p(o,r));throw x(w(r)+" is not iterable")}},51839:function(r,i,o){var u=o(32631);r.exports=function(r,i){var o=r[i];return null==o?void 0:u(o)}},29519:function(r,i,o){var u=o(38347),p=o(43162),g=Math.floor,y=u("".charAt),w=u("".replace),_=u("".slice),x=/\$([$&'`]|\d{1,2}|<[^>]*>)/g,k=/\$([$&'`]|\d{1,2})/g;r.exports=function(r,i,o,u,P,O){var I=o+r.length,B=u.length,D=k;return void 0!==P&&(P=p(P),D=x),w(O,D,(function(p,w){var x;switch(y(w,0)){case"$":return"$";case"&":return r;case"`":return _(i,0,o);case"'":return _(i,I);case"<":x=P[_(w,1,-1)];break;default:var k=+w;if(0===k)return p;if(k>B){var O=g(k/10);return 0===O?p:O<=B?void 0===u[O-1]?y(w,1):u[O-1]+y(w,1):p}x=u[k-1]}return void 0===x?"":x}))}},32010:function(r,i,o){var check=function(r){return r&&r.Math==Math&&r};r.exports=check("object"==typeof globalThis&&globalThis)||check("object"==typeof window&&window)||check("object"==typeof self&&self)||check("object"==typeof o.g&&o.g)||function(){return this}()||Function("return this")()},20340:function(r,i,o){var u=o(38347),p=o(43162),g=u({}.hasOwnProperty);r.exports=Object.hasOwn||function hasOwn(r,i){return g(p(r),i)}},90682:function(r){r.exports={}},61144:function(r,i,o){var u=o(32010);r.exports=function(r,i){var o=u.console;o&&o.error&&(1==arguments.length?o.error(r):o.error(r,i))}},520:function(r,i,o){var u=o(38486);r.exports=u("document","documentElement")},18904:function(r,i,o){var u=o(15567),p=o(47044),g=o(12072);r.exports=!u&&!p((function(){return 7!=Object.defineProperty(g("div"),"a",{get:function(){return 7}}).a}))},64397:function(r,i,o){var u=o(32010).Array,p=Math.abs,g=Math.pow,y=Math.floor,w=Math.log,_=Math.LN2;r.exports={pack:function(r,i,o){var x,k,P,O=u(o),I=8*o-i-1,B=(1<<I)-1,D=B>>1,R=23===i?g(2,-24)-g(2,-77):0,N=r<0||0===r&&1/r<0?1:0,W=0;for((r=p(r))!=r||r===1/0?(k=r!=r?1:0,x=B):(x=y(w(r)/_),r*(P=g(2,-x))<1&&(x--,P*=2),(r+=x+D>=1?R/P:R*g(2,1-D))*P>=2&&(x++,P/=2),x+D>=B?(k=0,x=B):x+D>=1?(k=(r*P-1)*g(2,i),x+=D):(k=r*g(2,D-1)*g(2,i),x=0));i>=8;O[W++]=255&k,k/=256,i-=8);for(x=x<<i|k,I+=i;I>0;O[W++]=255&x,x/=256,I-=8);return O[--W]|=128*N,O},unpack:function(r,i){var o,u=r.length,p=8*u-i-1,y=(1<<p)-1,w=y>>1,_=p-7,x=u-1,k=r[x--],P=127&k;for(k>>=7;_>0;P=256*P+r[x],x--,_-=8);for(o=P&(1<<-_)-1,P>>=-_,_+=i;_>0;o=256*o+r[x],x--,_-=8);if(0===P)P=1-w;else{if(P===y)return o?NaN:k?-1/0:1/0;o+=g(2,i),P-=w}return(k?-1:1)*o*g(2,P-i)}}},7514:function(r,i,o){var u=o(32010),p=o(38347),g=o(47044),y=o(93975),w=u.Object,_=p("".split);r.exports=g((function(){return!w("z").propertyIsEnumerable(0)}))?function(r){return"String"==y(r)?_(r,""):w(r)}:w},51868:function(r,i,o){var u=o(94578),p=o(24517),g=o(3840);r.exports=function(r,i,o){var y,w;return g&&u(y=i.constructor)&&y!==o&&p(w=y.prototype)&&w!==o.prototype&&g(r,w),r}},10447:function(r,i,o){var u=o(38347),p=o(94578),g=o(55480),y=u(Function.toString);p(g.inspectSource)||(g.inspectSource=function(r){return y(r)}),r.exports=g.inspectSource},87811:function(r,i,o){var u=o(24517),p=o(48914);r.exports=function(r,i){u(i)&&"cause"in i&&p(r,"cause",i.cause)}},62148:function(r,i,o){var u=o(56475),p=o(38347),g=o(90682),y=o(24517),w=o(20340),_=o(95892).f,x=o(6611),k=o(8807),P=o(46859),O=o(55481),I=!1,B=P("meta"),D=0,R=Object.isExtensible||function(){return!0},setMetadata=function(r){_(r,B,{value:{objectID:"O"+D++,weakData:{}}})},N=r.exports={enable:function(){N.enable=function(){},I=!0;var r=x.f,i=p([].splice),o={};o[B]=1,r(o).length&&(x.f=function(o){for(var u=r(o),p=0,g=u.length;p<g;p++)if(u[p]===B){i(u,p,1);break}return u},u({target:"Object",stat:!0,forced:!0},{getOwnPropertyNames:k.f}))},fastKey:function(r,i){if(!y(r))return"symbol"==typeof r?r:("string"==typeof r?"S":"P")+r;if(!w(r,B)){if(!R(r))return"F";if(!i)return"E";setMetadata(r)}return r[B].objectID},getWeakData:function(r,i){if(!w(r,B)){if(!R(r))return!0;if(!i)return!1;setMetadata(r)}return r[B].weakData},onFreeze:function(r){return O&&I&&R(r)&&!w(r,B)&&setMetadata(r),r}};g[B]=!0},70172:function(r,i,o){var u,p,g,y=o(26168),w=o(32010),_=o(38347),x=o(24517),k=o(48914),P=o(20340),O=o(55480),I=o(82194),B=o(90682),D="Object already initialized",R=w.TypeError,N=w.WeakMap;if(y||O.state){var W=O.state||(O.state=new N),U=_(W.get),j=_(W.has),G=_(W.set);u=function(r,i){if(j(W,r))throw new R(D);return i.facade=r,G(W,r,i),i},p=function(r){return U(W,r)||{}},g=function(r){return j(W,r)}}else{var X=I("state");B[X]=!0,u=function(r,i){if(P(r,X))throw new R(D);return i.facade=r,k(r,X,i),i},p=function(r){return P(r,X)?r[X]:{}},g=function(r){return P(r,X)}}r.exports={set:u,get:p,has:g,enforce:function(r){return g(r)?p(r):u(r,{})},getterFor:function(r){return function(i){var o;if(!x(i)||(o=p(i)).type!==r)throw R("Incompatible receiver, "+r+" required");return o}}}},89564:function(r,i,o){var u=o(38688),p=o(15366),g=u("iterator"),y=Array.prototype;r.exports=function(r){return void 0!==r&&(p.Array===r||y[g]===r)}},59113:function(r,i,o){var u=o(93975);r.exports=Array.isArray||function isArray(r){return"Array"==u(r)}},94578:function(r){r.exports=function(r){return"function"==typeof r}},20884:function(r,i,o){var u=o(38347),p=o(47044),g=o(94578),y=o(52564),w=o(38486),_=o(10447),noop=function(){},x=[],k=w("Reflect","construct"),P=/^\s*(?:class|function)\b/,O=u(P.exec),I=!P.exec(noop),isConstructorModern=function(r){if(!g(r))return!1;try{return k(noop,x,r),!0}catch(r){return!1}};r.exports=!k||p((function(){var r;return isConstructorModern(isConstructorModern.call)||!isConstructorModern(Object)||!isConstructorModern((function(){r=!0}))||r}))?function(r){if(!g(r))return!1;switch(y(r)){case"AsyncFunction":case"GeneratorFunction":case"AsyncGeneratorFunction":return!1}return I||!!O(P,_(r))}:isConstructorModern},39599:function(r,i,o){var u=o(47044),p=o(94578),g=/#|\.prototype\./,isForced=function(r,i){var o=w[y(r)];return o==x||o!=_&&(p(i)?u(i):!!i)},y=isForced.normalize=function(r){return String(r).replace(g,".").toLowerCase()},w=isForced.data={},_=isForced.NATIVE="N",x=isForced.POLYFILL="P";r.exports=isForced},17506:function(r,i,o){var u=o(24517),p=Math.floor;r.exports=Number.isInteger||function isInteger(r){return!u(r)&&isFinite(r)&&p(r)===r}},24517:function(r,i,o){var u=o(94578);r.exports=function(r){return"object"==typeof r?null!==r:u(r)}},63432:function(r){r.exports=!1},28831:function(r,i,o){var u=o(24517),p=o(93975),g=o(38688)("match");r.exports=function(r){var i;return u(r)&&(void 0!==(i=r[g])?!!i:"RegExp"==p(r))}},46290:function(r,i,o){var u=o(32010),p=o(38486),g=o(94578),y=o(70176),w=o(9567),_=u.Object;r.exports=w?function(r){return"symbol"==typeof r}:function(r){var i=p("Symbol");return g(i)&&y(i.prototype,_(r))}},80383:function(r,i,o){var u=o(32010),p=o(25567),g=o(2834),y=o(34984),w=o(68664),_=o(89564),x=o(45495),k=o(70176),P=o(15892),O=o(13872),I=o(50194),B=u.TypeError,Result=function(r,i){this.stopped=r,this.result=i},D=Result.prototype;r.exports=function(r,i,o){var u,R,N,W,U,j,G,X=o&&o.that,K=!(!o||!o.AS_ENTRIES),Y=!(!o||!o.IS_ITERATOR),J=!(!o||!o.INTERRUPTED),$=p(i,X),stop=function(r){return u&&I(u,"normal",r),new Result(!0,r)},callFn=function(r){return K?(y(r),J?$(r[0],r[1],stop):$(r[0],r[1])):J?$(r,stop):$(r)};if(Y)u=r;else{if(!(R=O(r)))throw B(w(r)+" is not iterable");if(_(R)){for(N=0,W=x(r);W>N;N++)if((U=callFn(r[N]))&&k(D,U))return U;return new Result(!1)}u=P(r,R)}for(j=u.next;!(G=g(j,u)).done;){try{U=callFn(G.value)}catch(r){I(u,"throw",r)}if("object"==typeof U&&U&&k(D,U))return U}return new Result(!1)}},50194:function(r,i,o){var u=o(2834),p=o(34984),g=o(51839);r.exports=function(r,i,o){var y,w;p(r);try{if(!(y=g(r,"return"))){if("throw"===i)throw o;return o}y=u(y,r)}catch(r){w=!0,y=r}if("throw"===i)throw o;if(w)throw y;return p(y),o}},5844:function(r,i,o){"use strict";var u,p,g,y=o(47044),w=o(94578),_=o(10819),x=o(69548),k=o(13711),P=o(38688),O=o(63432),I=P("iterator"),B=!1;[].keys&&("next"in(g=[].keys())?(p=x(x(g)))!==Object.prototype&&(u=p):B=!0),null==u||y((function(){var r={};return u[I].call(r)!==r}))?u={}:O&&(u=_(u)),w(u[I])||k(u,I,(function(){return this})),r.exports={IteratorPrototype:u,BUGGY_SAFARI_ITERATORS:B}},15366:function(r){r.exports={}},45495:function(r,i,o){var u=o(23417);r.exports=function(r){return u(r.length)}},59804:function(r,i,o){var u,p,g,y,w,_,x,k,P=o(32010),O=o(25567),I=o(72062).f,B=o(6616).set,D=o(17716),R=o(70573),N=o(664),W=o(95053),U=P.MutationObserver||P.WebKitMutationObserver,j=P.document,G=P.process,X=P.Promise,K=I(P,"queueMicrotask"),Y=K&&K.value;Y||(u=function(){var r,i;for(W&&(r=G.domain)&&r.exit();p;){i=p.fn,p=p.next;try{i()}catch(r){throw p?y():g=void 0,r}}g=void 0,r&&r.enter()},D||W||N||!U||!j?!R&&X&&X.resolve?((x=X.resolve(void 0)).constructor=X,k=O(x.then,x),y=function(){k(u)}):W?y=function(){G.nextTick(u)}:(B=O(B,P),y=function(){B(u)}):(w=!0,_=j.createTextNode(""),new U(u).observe(_,{characterData:!0}),y=function(){_.data=w=!w})),r.exports=Y||function(r){var i={fn:r,next:void 0};g&&(g.next=i),p||(p=i,y()),g=i}},5155:function(r,i,o){var u=o(32010);r.exports=u.Promise},46887:function(r,i,o){var u=o(70091),p=o(47044);r.exports=!!Object.getOwnPropertySymbols&&!p((function(){var r=Symbol();return!String(r)||!(Object(r)instanceof Symbol)||!Symbol.sham&&u&&u<41}))},26168:function(r,i,o){var u=o(32010),p=o(94578),g=o(10447),y=u.WeakMap;r.exports=p(y)&&/native code/.test(g(y))},56614:function(r,i,o){"use strict";var u=o(32631),PromiseCapability=function(r){var i,o;this.promise=new r((function(r,u){if(void 0!==i||void 0!==o)throw TypeError("Bad Promise constructor");i=r,o=u})),this.resolve=u(i),this.reject=u(o)};r.exports.f=function(r){return new PromiseCapability(r)}},86392:function(r,i,o){var u=o(25096);r.exports=function(r,i){return void 0===r?arguments.length<2?"":i:u(r)}},93666:function(r,i,o){var u=o(32010),p=o(28831),g=u.TypeError;r.exports=function(r){if(p(r))throw g("The method doesn't accept regular expressions");return r}},59805:function(r,i,o){var u=o(32010).isFinite;r.exports=Number.isFinite||function isFinite(r){return"number"==typeof r&&u(r)}},87146:function(r,i,o){"use strict";var u=o(15567),p=o(38347),g=o(2834),y=o(47044),w=o(84675),_=o(61146),x=o(55574),k=o(43162),P=o(7514),O=Object.assign,I=Object.defineProperty,B=p([].concat);r.exports=!O||y((function(){if(u&&1!==O({b:1},O(I({},"a",{enumerable:!0,get:function(){I(this,"b",{value:3,enumerable:!1})}}),{b:2})).b)return!0;var r={},i={},o=Symbol(),p="abcdefghijklmnopqrst";return r[o]=7,p.split("").forEach((function(r){i[r]=r})),7!=O({},r)[o]||w(O({},i)).join("")!=p}))?function assign(r,i){for(var o=k(r),p=arguments.length,y=1,O=_.f,I=x.f;p>y;)for(var D,R=P(arguments[y++]),N=O?B(w(R),O(R)):w(R),W=N.length,U=0;W>U;)D=N[U++],u&&!g(I,R,D)||(o[D]=R[D]);return o}:O},10819:function(r,i,o){var u,p=o(34984),g=o(10196),y=o(2416),w=o(90682),_=o(520),x=o(12072),k=o(82194),P="prototype",O="script",I=k("IE_PROTO"),EmptyConstructor=function(){},scriptTag=function(r){return"<"+O+">"+r+"</"+O+">"},NullProtoObjectViaActiveX=function(r){r.write(scriptTag("")),r.close();var i=r.parentWindow.Object;return r=null,i},NullProtoObject=function(){try{u=new ActiveXObject("htmlfile")}catch(r){}var r,i,o;NullProtoObject="undefined"!=typeof document?document.domain&&u?NullProtoObjectViaActiveX(u):(i=x("iframe"),o="java"+O+":",i.style.display="none",_.appendChild(i),i.src=String(o),(r=i.contentWindow.document).open(),r.write(scriptTag("document.F=Object")),r.close(),r.F):NullProtoObjectViaActiveX(u);for(var p=y.length;p--;)delete NullProtoObject[P][y[p]];return NullProtoObject()};w[I]=!0,r.exports=Object.create||function create(r,i){var o;return null!==r?(EmptyConstructor[P]=p(r),o=new EmptyConstructor,EmptyConstructor[P]=null,o[I]=r):o=NullProtoObject(),void 0===i?o:g(o,i)}},10196:function(r,i,o){var u=o(15567),p=o(95892),g=o(34984),y=o(98086),w=o(84675);r.exports=u?Object.defineProperties:function defineProperties(r,i){g(r);for(var o,u=y(i),_=w(i),x=_.length,k=0;x>k;)p.f(r,o=_[k++],u[o]);return r}},95892:function(r,i,o){var u=o(32010),p=o(15567),g=o(18904),y=o(34984),w=o(63918),_=u.TypeError,x=Object.defineProperty;i.f=p?x:function defineProperty(r,i,o){if(y(r),i=w(i),y(o),g)try{return x(r,i,o)}catch(r){}if("get"in o||"set"in o)throw _("Accessors not supported");return"value"in o&&(r[i]=o.value),r}},72062:function(r,i,o){var u=o(15567),p=o(2834),g=o(55574),y=o(97841),w=o(98086),_=o(63918),x=o(20340),k=o(18904),P=Object.getOwnPropertyDescriptor;i.f=u?P:function getOwnPropertyDescriptor(r,i){if(r=w(r),i=_(i),k)try{return P(r,i)}catch(r){}if(x(r,i))return y(!p(g.f,r,i),r[i])}},8807:function(r,i,o){var u=o(93975),p=o(98086),g=o(6611).f,y=o(73163),w="object"==typeof window&&window&&Object.getOwnPropertyNames?Object.getOwnPropertyNames(window):[];r.exports.f=function getOwnPropertyNames(r){return w&&"Window"==u(r)?function(r){try{return g(r)}catch(r){return y(w)}}(r):g(p(r))}},6611:function(r,i,o){var u=o(64429),p=o(2416).concat("length","prototype");i.f=Object.getOwnPropertyNames||function getOwnPropertyNames(r){return u(r,p)}},61146:function(r,i){i.f=Object.getOwnPropertySymbols},69548:function(r,i,o){var u=o(32010),p=o(20340),g=o(94578),y=o(43162),w=o(82194),_=o(68494),x=w("IE_PROTO"),k=u.Object,P=k.prototype;r.exports=_?k.getPrototypeOf:function(r){var i=y(r);if(p(i,x))return i[x];var o=i.constructor;return g(o)&&i instanceof o?o.prototype:i instanceof k?P:null}},70176:function(r,i,o){var u=o(38347);r.exports=u({}.isPrototypeOf)},64429:function(r,i,o){var u=o(38347),p=o(20340),g=o(98086),y=o(12636).indexOf,w=o(90682),_=u([].push);r.exports=function(r,i){var o,u=g(r),x=0,k=[];for(o in u)!p(w,o)&&p(u,o)&&_(k,o);for(;i.length>x;)p(u,o=i[x++])&&(~y(k,o)||_(k,o));return k}},84675:function(r,i,o){var u=o(64429),p=o(2416);r.exports=Object.keys||function keys(r){return u(r,p)}},55574:function(r,i){"use strict";var o={}.propertyIsEnumerable,u=Object.getOwnPropertyDescriptor,p=u&&!o.call({1:2},1);i.f=p?function propertyIsEnumerable(r){var i=u(this,r);return!!i&&i.enumerable}:o},3840:function(r,i,o){var u=o(38347),p=o(34984),g=o(58659);r.exports=Object.setPrototypeOf||("__proto__"in{}?function(){var r,i=!1,o={};try{(r=u(Object.getOwnPropertyDescriptor(Object.prototype,"__proto__").set))(o,[]),i=o instanceof Array}catch(r){}return function setPrototypeOf(o,u){return p(o),g(u),i?r(o,u):o.__proto__=u,o}}():void 0)},80754:function(r,i,o){var u=o(15567),p=o(38347),g=o(84675),y=o(98086),w=p(o(55574).f),_=p([].push),createMethod=function(r){return function(i){for(var o,p=y(i),x=g(p),k=x.length,P=0,O=[];k>P;)o=x[P++],u&&!w(p,o)||_(O,r?[o,p[o]]:p[o]);return O}};r.exports={entries:createMethod(!0),values:createMethod(!1)}},52598:function(r,i,o){"use strict";var u=o(24663),p=o(52564);r.exports=u?{}.toString:function toString(){return"[object "+p(this)+"]"}},39629:function(r,i,o){var u=o(32010),p=o(2834),g=o(94578),y=o(24517),w=u.TypeError;r.exports=function(r,i){var o,u;if("string"===i&&g(o=r.toString)&&!y(u=p(o,r)))return u;if(g(o=r.valueOf)&&!y(u=p(o,r)))return u;if("string"!==i&&g(o=r.toString)&&!y(u=p(o,r)))return u;throw w("Can't convert object to primitive value")}},21594:function(r,i,o){var u=o(38486),p=o(38347),g=o(6611),y=o(61146),w=o(34984),_=p([].concat);r.exports=u("Reflect","ownKeys")||function ownKeys(r){var i=g.f(w(r)),o=y.f;return o?_(i,o(r)):i}},11206:function(r,i,o){var u=o(32010);r.exports=u},61900:function(r){r.exports=function(r){try{return{error:!1,value:r()}}catch(r){return{error:!0,value:r}}}},28617:function(r,i,o){var u=o(34984),p=o(24517),g=o(56614);r.exports=function(r,i){if(u(r),p(i)&&i.constructor===r)return i;var o=g.f(r);return(0,o.resolve)(i),o.promise}},15341:function(r,i,o){var u=o(13711);r.exports=function(r,i,o){for(var p in i)u(r,p,i[p],o);return r}},13711:function(r,i,o){var u=o(32010),p=o(94578),g=o(20340),y=o(48914),w=o(7421),_=o(10447),x=o(70172),k=o(7081).CONFIGURABLE,P=x.get,O=x.enforce,I=String(String).split("String");(r.exports=function(r,i,o,_){var x,P=!!_&&!!_.unsafe,B=!!_&&!!_.enumerable,D=!!_&&!!_.noTargetGet,R=_&&void 0!==_.name?_.name:i;p(o)&&("Symbol("===String(R).slice(0,7)&&(R="["+String(R).replace(/^Symbol\(([^)]*)\)/,"$1")+"]"),(!g(o,"name")||k&&o.name!==R)&&y(o,"name",R),(x=O(o)).source||(x.source=I.join("string"==typeof R?R:""))),r!==u?(P?!D&&r[i]&&(B=!0):delete r[i],B?r[i]=o:y(r,i,o)):B?r[i]=o:w(i,o)})(Function.prototype,"toString",(function toString(){return p(this)&&P(this).source||_(this)}))},66723:function(r,i,o){var u=o(32010),p=o(2834),g=o(34984),y=o(94578),w=o(93975),_=o(49820),x=u.TypeError;r.exports=function(r,i){var o=r.exec;if(y(o)){var u=p(o,r,i);return null!==u&&g(u),u}if("RegExp"===w(r))return p(_,r,i);throw x("RegExp#exec called on incompatible receiver")}},49820:function(r,i,o){"use strict";var u,p,g=o(2834),y=o(38347),w=o(25096),_=o(21182),x=o(74846),k=o(464),P=o(10819),O=o(70172).get,I=o(84030),B=o(97739),D=k("native-string-replace",String.prototype.replace),R=RegExp.prototype.exec,N=R,W=y("".charAt),U=y("".indexOf),j=y("".replace),G=y("".slice),X=(p=/b*/g,g(R,u=/a/,"a"),g(R,p,"a"),0!==u.lastIndex||0!==p.lastIndex),K=x.UNSUPPORTED_Y||x.BROKEN_CARET,Y=void 0!==/()??/.exec("")[1];(X||Y||K||I||B)&&(N=function exec(r){var i,o,u,p,y,x,k,I=this,B=O(I),J=w(r),$=B.raw;if($)return $.lastIndex=I.lastIndex,i=g(N,$,J),I.lastIndex=$.lastIndex,i;var ee=B.groups,te=K&&I.sticky,re=g(_,I),ne=I.source,ie=0,oe=J;if(te&&(re=j(re,"y",""),-1===U(re,"g")&&(re+="g"),oe=G(J,I.lastIndex),I.lastIndex>0&&(!I.multiline||I.multiline&&"\n"!==W(J,I.lastIndex-1))&&(ne="(?: "+ne+")",oe=" "+oe,ie++),o=new RegExp("^(?:"+ne+")",re)),Y&&(o=new RegExp("^"+ne+"$(?!\\s)",re)),X&&(u=I.lastIndex),p=g(R,te?o:I,oe),te?p?(p.input=G(p.input,ie),p[0]=G(p[0],ie),p.index=I.lastIndex,I.lastIndex+=p[0].length):I.lastIndex=0:X&&p&&(I.lastIndex=I.global?p.index+p[0].length:u),Y&&p&&p.length>1&&g(D,p[0],o,(function(){for(y=1;y<arguments.length-2;y++)void 0===arguments[y]&&(p[y]=void 0)})),p&&ee)for(p.groups=x=P(null),y=0;y<ee.length;y++)x[(k=ee[y])[0]]=p[k[1]];return p}),r.exports=N},21182:function(r,i,o){"use strict";var u=o(34984);r.exports=function(){var r=u(this),i="";return r.global&&(i+="g"),r.ignoreCase&&(i+="i"),r.multiline&&(i+="m"),r.dotAll&&(i+="s"),r.unicode&&(i+="u"),r.sticky&&(i+="y"),i}},74846:function(r,i,o){var u=o(47044),p=o(32010).RegExp;i.UNSUPPORTED_Y=u((function(){var r=p("a","y");return r.lastIndex=2,null!=r.exec("abcd")})),i.BROKEN_CARET=u((function(){var r=p("^r","gy");return r.lastIndex=2,null!=r.exec("str")}))},84030:function(r,i,o){var u=o(47044),p=o(32010).RegExp;r.exports=u((function(){var r=p(".","s");return!(r.dotAll&&r.exec("\n")&&"s"===r.flags)}))},97739:function(r,i,o){var u=o(47044),p=o(32010).RegExp;r.exports=u((function(){var r=p("(?<a>b)","g");return"b"!==r.exec("b").groups.a||"bc"!=="b".replace(r,"$<a>c")}))},83943:function(r,i,o){var u=o(32010).TypeError;r.exports=function(r){if(null==r)throw u("Can't call method on "+r);return r}},7421:function(r,i,o){var u=o(32010),p=Object.defineProperty;r.exports=function(r,i){try{p(u,r,{value:i,configurable:!0,writable:!0})}catch(o){u[r]=i}return i}},51334:function(r,i,o){"use strict";var u=o(38486),p=o(95892),g=o(38688),y=o(15567),w=g("species");r.exports=function(r){var i=u(r),o=p.f;y&&i&&!i[w]&&o(i,w,{configurable:!0,get:function(){return this}})}},15216:function(r,i,o){var u=o(95892).f,p=o(20340),g=o(38688)("toStringTag");r.exports=function(r,i,o){r&&!p(r=o?r:r.prototype,g)&&u(r,g,{configurable:!0,value:i})}},82194:function(r,i,o){var u=o(464),p=o(46859),g=u("keys");r.exports=function(r){return g[r]||(g[r]=p(r))}},55480:function(r,i,o){var u=o(32010),p=o(7421),g="__core-js_shared__",y=u[g]||p(g,{});r.exports=y},464:function(r,i,o){var u=o(63432),p=o(55480);(r.exports=function(r,i){return p[r]||(p[r]=void 0!==i?i:{})})("versions",[]).push({version:"3.19.0",mode:u?"pure":"global",copyright:"© 2021 Denis Pushkarev (zloirock.ru)"})},27754:function(r,i,o){var u=o(34984),p=o(69075),g=o(38688)("species");r.exports=function(r,i){var o,y=u(r).constructor;return void 0===y||null==(o=u(y)[g])?i:p(o)}},7452:function(r,i,o){var u=o(47044);r.exports=function(r){return u((function(){var i=""[r]('"');return i!==i.toLowerCase()||i.split('"').length>3}))}},69510:function(r,i,o){var u=o(38347),p=o(26882),g=o(25096),y=o(83943),w=u("".charAt),_=u("".charCodeAt),x=u("".slice),createMethod=function(r){return function(i,o){var u,k,P=g(y(i)),O=p(o),I=P.length;return O<0||O>=I?r?"":void 0:(u=_(P,O))<55296||u>56319||O+1===I||(k=_(P,O+1))<56320||k>57343?r?w(P,O):u:r?x(P,O,O+2):k-56320+(u-55296<<10)+65536}};r.exports={codeAt:createMethod(!1),charAt:createMethod(!0)}},34858:function(r,i,o){"use strict";var u=o(32010),p=o(26882),g=o(25096),y=o(83943),w=u.RangeError;r.exports=function repeat(r){var i=g(y(this)),o="",u=p(r);if(u<0||u==1/0)throw w("Wrong number of repetitions");for(;u>0;(u>>>=1)&&(i+=i))1&u&&(o+=i);return o}},68899:function(r,i,o){var u=o(7081).PROPER,p=o(47044),g=o(43187);r.exports=function(r){return p((function(){return!!g[r]()||"​᠎"!=="​᠎"[r]()||u&&g[r].name!==r}))}},29841:function(r,i,o){var u=o(38347),p=o(83943),g=o(25096),y=o(43187),w=u("".replace),_="["+y+"]",x=RegExp("^"+_+_+"*"),k=RegExp(_+_+"*$"),createMethod=function(r){return function(i){var o=g(p(i));return 1&r&&(o=w(o,x,"")),2&r&&(o=w(o,k,"")),o}};r.exports={start:createMethod(1),end:createMethod(2),trim:createMethod(3)}},6616:function(r,i,o){var u,p,g,y,w=o(32010),_=o(58448),x=o(25567),k=o(94578),P=o(20340),O=o(47044),I=o(520),B=o(73163),D=o(12072),R=o(17716),N=o(95053),W=w.setImmediate,U=w.clearImmediate,j=w.process,G=w.Dispatch,X=w.Function,K=w.MessageChannel,Y=w.String,J=0,$={},ee="onreadystatechange";try{u=w.location}catch(r){}var run=function(r){if(P($,r)){var i=$[r];delete $[r],i()}},runner=function(r){return function(){run(r)}},listener=function(r){run(r.data)},post=function(r){w.postMessage(Y(r),u.protocol+"//"+u.host)};W&&U||(W=function setImmediate(r){var i=B(arguments,1);return $[++J]=function(){_(k(r)?r:X(r),void 0,i)},p(J),J},U=function clearImmediate(r){delete $[r]},N?p=function(r){j.nextTick(runner(r))}:G&&G.now?p=function(r){G.now(runner(r))}:K&&!R?(y=(g=new K).port2,g.port1.onmessage=listener,p=x(y.postMessage,y)):w.addEventListener&&k(w.postMessage)&&!w.importScripts&&u&&"file:"!==u.protocol&&!O(post)?(p=post,w.addEventListener("message",listener,!1)):p=ee in D("script")?function(r){I.appendChild(D("script"))[ee]=function(){I.removeChild(this),run(r)}}:function(r){setTimeout(runner(r),0)}),r.exports={set:W,clear:U}},16679:function(r,i,o){var u=o(38347);r.exports=u(1..valueOf)},74841:function(r,i,o){var u=o(26882),p=Math.max,g=Math.min;r.exports=function(r,i){var o=u(r);return o<0?p(o+i,0):g(o,i)}},71265:function(r,i,o){var u=o(32010),p=o(26882),g=o(23417),y=u.RangeError;r.exports=function(r){if(void 0===r)return 0;var i=p(r),o=g(i);if(i!==o)throw y("Wrong length or index");return o}},98086:function(r,i,o){var u=o(7514),p=o(83943);r.exports=function(r){return u(p(r))}},26882:function(r){var i=Math.ceil,o=Math.floor;r.exports=function(r){var u=+r;return u!=u||0===u?0:(u>0?o:i)(u)}},23417:function(r,i,o){var u=o(26882),p=Math.min;r.exports=function(r){return r>0?p(u(r),9007199254740991):0}},43162:function(r,i,o){var u=o(32010),p=o(83943),g=u.Object;r.exports=function(r){return g(p(r))}},80670:function(r,i,o){var u=o(32010),p=o(64913),g=u.RangeError;r.exports=function(r,i){var o=p(r);if(o%i)throw g("Wrong offset");return o}},64913:function(r,i,o){var u=o(32010),p=o(26882),g=u.RangeError;r.exports=function(r){var i=p(r);if(i<0)throw g("The argument can't be less than 0");return i}},2644:function(r,i,o){var u=o(32010),p=o(2834),g=o(24517),y=o(46290),w=o(51839),_=o(39629),x=o(38688),k=u.TypeError,P=x("toPrimitive");r.exports=function(r,i){if(!g(r)||y(r))return r;var o,u=w(r,P);if(u){if(void 0===i&&(i="default"),o=p(u,r,i),!g(o)||y(o))return o;throw k("Can't convert object to primitive value")}return void 0===i&&(i="number"),_(r,i)}},63918:function(r,i,o){var u=o(2644),p=o(46290);r.exports=function(r){var i=u(r,"string");return p(i)?i:i+""}},24663:function(r,i,o){var u={};u[o(38688)("toStringTag")]="z",r.exports="[object z]"===String(u)},25096:function(r,i,o){var u=o(32010),p=o(52564),g=u.String;r.exports=function(r){if("Symbol"===p(r))throw TypeError("Cannot convert a Symbol value to a string");return g(r)}},68664:function(r,i,o){var u=o(32010).String;r.exports=function(r){try{return u(r)}catch(r){return"Object"}}},98828:function(r,i,o){"use strict";var u=o(56475),p=o(32010),g=o(2834),y=o(15567),w=o(28834),_=o(36597),x=o(89987),k=o(2868),P=o(97841),O=o(48914),I=o(17506),B=o(23417),D=o(71265),R=o(80670),N=o(63918),W=o(20340),U=o(52564),j=o(24517),G=o(46290),X=o(10819),K=o(70176),Y=o(3840),J=o(6611).f,$=o(83590),ee=o(91102).forEach,te=o(51334),re=o(95892),ne=o(72062),ie=o(70172),oe=o(51868),ae=ie.get,se=ie.set,ce=re.f,le=ne.f,ue=Math.round,fe=p.RangeError,he=x.ArrayBuffer,de=he.prototype,pe=x.DataView,ge=_.NATIVE_ARRAY_BUFFER_VIEWS,ye=_.TYPED_ARRAY_CONSTRUCTOR,ve=_.TYPED_ARRAY_TAG,me=_.TypedArray,be=_.TypedArrayPrototype,we=_.aTypedArrayConstructor,_e=_.isTypedArray,Se="BYTES_PER_ELEMENT",xe="Wrong length",fromList=function(r,i){we(r);for(var o=0,u=i.length,p=new r(u);u>o;)p[o]=i[o++];return p},addGetter=function(r,i){ce(r,i,{get:function(){return ae(this)[i]}})},isArrayBuffer=function(r){var i;return K(de,r)||"ArrayBuffer"==(i=U(r))||"SharedArrayBuffer"==i},isTypedArrayIndex=function(r,i){return _e(r)&&!G(i)&&i in r&&I(+i)&&i>=0},Ce=function getOwnPropertyDescriptor(r,i){return i=N(i),isTypedArrayIndex(r,i)?P(2,r[i]):le(r,i)},ke=function defineProperty(r,i,o){return i=N(i),!(isTypedArrayIndex(r,i)&&j(o)&&W(o,"value"))||W(o,"get")||W(o,"set")||o.configurable||W(o,"writable")&&!o.writable||W(o,"enumerable")&&!o.enumerable?ce(r,i,o):(r[i]=o.value,r)};y?(ge||(ne.f=Ce,re.f=ke,addGetter(be,"buffer"),addGetter(be,"byteOffset"),addGetter(be,"byteLength"),addGetter(be,"length")),u({target:"Object",stat:!0,forced:!ge},{getOwnPropertyDescriptor:Ce,defineProperty:ke}),r.exports=function(r,i,o){var y=r.match(/\d+$/)[0]/8,_=r+(o?"Clamped":"")+"Array",x="get"+r,P="set"+r,I=p[_],N=I,W=N&&N.prototype,U={},addElement=function(r,i){ce(r,i,{get:function(){return function(r,i){var o=ae(r);return o.view[x](i*y+o.byteOffset,!0)}(this,i)},set:function(r){return function(r,i,u){var p=ae(r);o&&(u=(u=ue(u))<0?0:u>255?255:255&u),p.view[P](i*y+p.byteOffset,u,!0)}(this,i,r)},enumerable:!0})};ge?w&&(N=i((function(r,i,o,u){return k(r,W),oe(j(i)?isArrayBuffer(i)?void 0!==u?new I(i,R(o,y),u):void 0!==o?new I(i,R(o,y)):new I(i):_e(i)?fromList(N,i):g($,N,i):new I(D(i)),r,N)})),Y&&Y(N,me),ee(J(I),(function(r){r in N||O(N,r,I[r])})),N.prototype=W):(N=i((function(r,i,o,u){k(r,W);var p,w,_,x=0,P=0;if(j(i)){if(!isArrayBuffer(i))return _e(i)?fromList(N,i):g($,N,i);p=i,P=R(o,y);var O=i.byteLength;if(void 0===u){if(O%y)throw fe(xe);if((w=O-P)<0)throw fe(xe)}else if((w=B(u)*y)+P>O)throw fe(xe);_=w/y}else _=D(i),p=new he(w=_*y);for(se(r,{buffer:p,byteOffset:P,byteLength:w,length:_,view:new pe(p)});x<_;)addElement(r,x++)})),Y&&Y(N,me),W=N.prototype=X(be)),W.constructor!==N&&O(W,"constructor",N),O(W,ye,N),ve&&O(W,ve,_),U[_]=N,u({global:!0,forced:N!=I,sham:!ge},U),Se in N||O(N,Se,y),Se in W||O(W,Se,y),te(_)}):r.exports=function(){}},28834:function(r,i,o){var u=o(32010),p=o(47044),g=o(46769),y=o(36597).NATIVE_ARRAY_BUFFER_VIEWS,w=u.ArrayBuffer,_=u.Int8Array;r.exports=!y||!p((function(){_(1)}))||!p((function(){new _(-1)}))||!g((function(r){new _,new _(null),new _(1.5),new _(r)}),!0)||p((function(){return 1!==new _(new w(2),1,void 0).length}))},59610:function(r,i,o){var u=o(34269),p=o(34815);r.exports=function(r,i){return u(p(r),i)}},83590:function(r,i,o){var u=o(25567),p=o(2834),g=o(69075),y=o(43162),w=o(45495),_=o(15892),x=o(13872),k=o(89564),P=o(36597).aTypedArrayConstructor;r.exports=function from(r){var i,o,O,I,B,D,R=g(this),N=y(r),W=arguments.length,U=W>1?arguments[1]:void 0,j=void 0!==U,G=x(N);if(G&&!k(G))for(D=(B=_(N,G)).next,N=[];!(I=p(D,B)).done;)N.push(I.value);for(j&&W>2&&(U=u(U,arguments[2])),o=w(N),O=new(P(R))(o),i=0;o>i;i++)O[i]=j?U(N[i],i):N[i];return O}},34815:function(r,i,o){var u=o(36597),p=o(27754),g=u.TYPED_ARRAY_CONSTRUCTOR,y=u.aTypedArrayConstructor;r.exports=function(r){return y(p(r,r[g]))}},46859:function(r,i,o){var u=o(38347),p=0,g=Math.random(),y=u(1..toString);r.exports=function(r){return"Symbol("+(void 0===r?"":r)+")_"+y(++p+g,36)}},9567:function(r,i,o){var u=o(46887);r.exports=u&&!Symbol.sham&&"symbol"==typeof Symbol.iterator},75960:function(r,i,o){var u=o(38688);i.f=u},38688:function(r,i,o){var u=o(32010),p=o(464),g=o(20340),y=o(46859),w=o(46887),_=o(9567),x=p("wks"),k=u.Symbol,P=k&&k.for,O=_?k:k&&k.withoutSetter||y;r.exports=function(r){if(!g(x,r)||!w&&"string"!=typeof x[r]){var i="Symbol."+r;w&&g(k,r)?x[r]=k[r]:x[r]=_&&P?P(i):O(i)}return x[r]}},43187:function(r){r.exports="\t\n\v\f\r                　\u2028\u2029\ufeff"},94910:function(r,i,o){"use strict";var u=o(56475),p=o(32010),g=o(70176),y=o(69548),w=o(3840),_=o(2675),x=o(10819),k=o(48914),P=o(97841),O=o(34074),I=o(87811),B=o(80383),D=o(86392),R=o(45144),N=p.Error,W=[].push,U=function AggregateError(r,i){var o=g(j,this)?this:x(j),u=arguments.length>2?arguments[2]:void 0;w&&(o=w(new N(void 0),y(o))),k(o,"message",D(i,"")),R&&k(o,"stack",O(o.stack,1)),I(o,u);var p=[];return B(r,W,{that:p}),k(o,"errors",p),o};w?w(U,N):_(U,N);var j=U.prototype=x(N.prototype,{constructor:P(1,U),message:P(1,""),name:P(1,"AggregateError")});u({global:!0},{AggregateError:U})},39081:function(r,i,o){"use strict";var u=o(56475),p=o(32010),g=o(47044),y=o(59113),w=o(24517),_=o(43162),x=o(45495),k=o(38639),P=o(45744),O=o(56280),I=o(38688),B=o(70091),D=I("isConcatSpreadable"),R=9007199254740991,N="Maximum allowed index exceeded",W=p.TypeError,U=B>=51||!g((function(){var r=[];return r[D]=!1,r.concat()[0]!==r})),j=O("concat"),isConcatSpreadable=function(r){if(!w(r))return!1;var i=r[D];return void 0!==i?!!i:y(r)};u({target:"Array",proto:!0,forced:!U||!j},{concat:function concat(r){var i,o,u,p,g,y=_(this),w=P(y,0),O=0;for(i=-1,u=arguments.length;i<u;i++)if(isConcatSpreadable(g=-1===i?y:arguments[i])){if(O+(p=x(g))>R)throw W(N);for(o=0;o<p;o++,O++)o in g&&k(w,O,g[o])}else{if(O>=R)throw W(N);k(w,O++,g)}return w.length=O,w}})},68626:function(r,i,o){var u=o(56475),p=o(92642),g=o(71156);u({target:"Array",proto:!0},{copyWithin:p}),g("copyWithin")},41584:function(r,i,o){var u=o(56475),p=o(72864),g=o(71156);u({target:"Array",proto:!0},{fill:p}),g("fill")},49063:function(r,i,o){"use strict";var u=o(56475),p=o(91102).filter;u({target:"Array",proto:!0,forced